<?php
$config = require_once 'config/config.php';
$pdo = $config['pdo'];

echo "=== VERIFYING WALLET HISTORY DISPLAY ===\n\n";

// Check current wallet_transactions for agent 10
$stmt = $pdo->prepare("
    SELECT id, transaction_type, category, amount, description, reference_type, reference_id, created_at
    FROM wallet_transactions 
    WHERE user_id = 10 
    ORDER BY created_at DESC 
    LIMIT 10
");
$stmt->execute();
$transactions = $stmt->fetchAll();

echo "Recent wallet transactions for agent 10:\n";
foreach ($transactions as $tx) {
    $sign = $tx['transaction_type'] === 'credit' ? '+' : '-';
    echo "  {$tx['created_at']} | {$tx['category']} | {$sign}฿{$tx['amount']} | {$tx['description']}\n";
}

// Check what the controller logic would return
echo "\n=== SIMULATING CONTROLLER LOGIC ===\n";

$uid = 10;
$useWalletTransactions = true;

if ($useWalletTransactions) {
    $wWhere = ['wt.user_id = :uid'];
    $wParams = [':uid' => $uid];
    $wWhereSql = 'WHERE ' . implode(' AND ', $wWhere);
    
    // Count
    $countSql = "SELECT COUNT(*) FROM wallet_transactions wt $wWhereSql";
    $stc = $pdo->prepare($countSql);
    $stc->execute($wParams);
    $total = (int)($stc->fetchColumn() ?? 0);
    
    echo "Total transactions: $total\n";
    
    // Get first page (10 items)
    $sql = "SELECT wt.* FROM wallet_transactions wt $wWhereSql ORDER BY wt.created_at DESC, wt.id DESC LIMIT 10 OFFSET 0";
    $st = $pdo->prepare($sql);
    foreach ($wParams as $k=>$v) { $st->bindValue($k, $v); }
    $st->execute();
    $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    
    echo "Retrieved " . count($rows) . " transactions for display\n";
    
    // Process like the controller does
    $transactions = [];
    foreach ($rows as $tx) {
        $isCredit = (string)($tx['transaction_type'] ?? '') === 'credit';
        $serviceType = $tx['reference_type'] ? str_replace('_', ' ', (string)$tx['reference_type']) : null;
        $reference = $tx['reference_type'] && $tx['reference_id'] ? strtoupper((string)$tx['reference_type']).'-'.(string)$tx['reference_id'] : ('TXN-'.(string)$tx['id']);
        
        $transactions[] = [
            'id' => (int)$tx['id'],
            'transaction_type' => $isCredit ? 'credit' : 'debit',
            'category' => $tx['reference_type'] ? (string)$tx['reference_type'] : 'general',
            'amount' => $isCredit ? (float)$tx['amount'] : -(float)$tx['amount'],
            'description' => (string)($tx['description'] ?? 'Transaction'),
            'reference' => $reference,
            'service_type' => $serviceType,
            'status' => (string)($tx['status'] ?? 'completed'),
            'created_at' => (string)($tx['created_at'] ?? ''),
            'running_balance' => (float)($tx['balance_after'] ?? 0)
        ];
    }
    
    echo "\nProcessed transactions for display:\n";
    foreach ($transactions as $tx) {
        $sign = $tx['amount'] >= 0 ? '+' : '';
        echo "  {$tx['created_at']} | {$tx['service_type']} | {$sign}฿{$tx['amount']} | {$tx['description']} | Balance: ฿{$tx['running_balance']}\n";
    }
    
    // Check for refunds and admin credits specifically
    $refunds = array_filter($transactions, fn($tx) => $tx['category'] === 'refund');
    $adminCredits = array_filter($transactions, fn($tx) => $tx['category'] === 'admin_credit');
    
    echo "\nRefund transactions found: " . count($refunds) . "\n";
    foreach ($refunds as $tx) {
        echo "  - {$tx['description']} (+฿{$tx['amount']})\n";
    }
    
    echo "\nAdmin credit transactions found: " . count($adminCredits) . "\n";
    foreach ($adminCredits as $tx) {
        echo "  - {$tx['description']} (+฿{$tx['amount']})\n";
    }
}

echo "\n=== STATS CALCULATION ===\n";

// Calculate stats like the controller does
$stt = $pdo->prepare("
    SELECT 
        SUM(CASE WHEN transaction_type = 'credit' THEN amount ELSE 0 END) as csum,
        SUM(CASE WHEN transaction_type = 'debit' THEN amount ELSE 0 END) as dsum,
        SUM(CASE WHEN transaction_type = 'debit' AND DATE(created_at) = CURDATE() THEN amount ELSE 0 END) as today_spend,
        SUM(CASE WHEN category = 'admin_credit' THEN amount ELSE 0 END) as admin_credits,
        SUM(CASE WHEN category = 'refund' THEN amount ELSE 0 END) as total_refunds,
        SUM(CASE WHEN transaction_type = 'debit' AND YEAR(created_at) = YEAR(CURDATE()) AND MONTH(created_at) = MONTH(CURDATE()) THEN amount ELSE 0 END) as month_spend
    FROM wallet_transactions WHERE user_id = :uid
");
$stt->execute([':uid' => $uid]);
$ag = $stt->fetch(\PDO::FETCH_ASSOC) ?: [];

echo "Stats calculated:\n";
echo "  Total Credits: ฿" . number_format((float)($ag['csum'] ?? 0), 2) . "\n";
echo "  Total Debits: ฿" . number_format((float)($ag['dsum'] ?? 0), 2) . "\n";
echo "  Admin Credits: ฿" . number_format((float)($ag['admin_credits'] ?? 0), 2) . "\n";
echo "  Total Refunds: ฿" . number_format((float)($ag['total_refunds'] ?? 0), 2) . "\n";
echo "  Today's Spending: ฿" . number_format((float)($ag['today_spend'] ?? 0), 2) . "\n";
echo "  This Month's Spending: ฿" . number_format((float)($ag['month_spend'] ?? 0), 2) . "\n";

echo "\nWallet history should now display refunds and admin credits correctly!\n";
?>
