/*
  Reusable drag-and-drop image uploader with instant preview and thumbnail generation.
  Usage:
    1) Include in your page:
       <link rel="stylesheet" href="/assets/css/media-uploader.css">
       <script src="/assets/js/media-uploader.js" defer></script>
    2) Add container:
       <div id="uploader" class="mu" data-upload-url="/admin/media/upload"></div>
    3) Provide CSRF token via:
       - <meta name="csrf" content="<?= $csrf ?>"> OR
       - window.CSRF = '...'; OR
       - <input type="hidden" name="csrf" value="..."> inside the container
*/
(function(){
  function getCsrf(container){
    const meta = document.querySelector('meta[name=csrf]');
    if (meta && meta.content) return meta.content;
    if (window.CSRF) return window.CSRF;
    const input = container.querySelector('input[name=csrf]');
    if (input && input.value) return input.value;
    return '';
  }

  function h(tag, attrs, ...children){
    const el = document.createElement(tag);
    if (attrs){
      Object.entries(attrs).forEach(([k,v])=>{
        if (k === 'class') el.className = v;
        else if (k.startsWith('on') && typeof v === 'function') el.addEventListener(k.substring(2), v);
        else el.setAttribute(k, v);
      });
    }
    children.forEach(ch=>{
      if (ch==null) return;
      if (typeof ch === 'string') el.appendChild(document.createTextNode(ch));
      else el.appendChild(ch);
    });
    return el;
  }

  function previewFile(file){
    return new Promise((resolve)=>{
      const reader = new FileReader();
      reader.onload = e => resolve(e.target.result);
      reader.readAsDataURL(file);
    });
  }

  async function uploadFiles(container, files){
    const uploadUrl = container.getAttribute('data-upload-url') || '/admin/media/upload';
    const csrf = getCsrf(container);
    const form = new FormData();
    for (const f of files) form.append('files[]', f);
    if (csrf) form.append('csrf', csrf);
    const res = await fetch(uploadUrl, {
      method: 'POST',
      headers: { 'X-Requested-With': 'XMLHttpRequest', 'Accept': 'application/json', ...(csrf ? {'X-CSRF-Token': csrf} : {}) },
      body: form,
    });
    const ctype = res.headers.get('content-type') || '';
    if (!res.ok) {
      // Try to extract server error
      if (ctype.includes('application/json')) {
        const j = await res.json().catch(()=>({}));
        const msg = (j && (j.error || j.message)) ? (j.error || j.message) : `Upload failed (${res.status})`;
        throw new Error(msg);
      } else {
        const t = await res.text().catch(()=> '');
        const snippet = t ? t.slice(0, 200) : '';
        throw new Error(snippet || `Upload failed (${res.status})`);
      }
    }
    let data;
    if (ctype.includes('application/json')) {
      data = await res.json();
    } else {
      const t = await res.text();
      throw new Error(t ? t.slice(0,200) : 'Unexpected non-JSON response');
    }
    if (!data.ok) throw new Error(data.error || 'Upload error');
    return data.files || [];
  }

  function init(container){
    if (container.__muInit) return; container.__muInit = true;
    const input = h('input', { type:'file', accept:'image/*', multiple:'multiple', class:'mu-input' });
    const grid = h('div', {class:'mu-grid'});
    const zone = h('div', {class:'mu-zone'},
      h('div', {class:'mu-instructions'}, 'Drag & drop images here or click to choose')
    );
    container.appendChild(zone);
    container.appendChild(input);
    container.appendChild(grid);

    // Prefill existing images if media_json is present
    try {
      // Priority: data-initial attribute on container, fallback to #media_json hidden input
      let initial = container.getAttribute('data-initial');
      if (!initial) {
        const hidden = document.getElementById('media_json');
        if (hidden && hidden.value) initial = hidden.value;
      }
      if (initial) {
        const files = JSON.parse(initial);
        if (Array.isArray(files)) {
          files.forEach(function(file){
            if (!file || (!file.originalUrl && !file.thumbUrl)) return;
            const url = file.thumbUrl || file.originalUrl;
            const it = h('div', {class:'mu-item'},
              h('img', {src:url, alt:''}),
              h('div', {class:'mu-status'}, 'Ready')
            );
            // store urls for consumers
            it.dataset.originalUrl = file.originalUrl || '';
            it.dataset.thumbUrl = file.thumbUrl || '';
            grid.appendChild(it);
          });
        }
      }
    } catch (e) { /* ignore malformed JSON */ }

    async function handleFiles(fileList){
      const arr = Array.from(fileList);
      // instant local previews
      for (const f of arr){
        if (!f.type.startsWith('image/')) continue;
        const url = await previewFile(f);
        const item = h('div', {class:'mu-item uploading'},
          h('img', {src:url, alt:f.name}),
          h('div', {class:'mu-status'}, 'Uploading...')
        );
        grid.appendChild(item);
      }
      try{
        const uploaded = await uploadFiles(container, arr);
        // replace statuses
        const items = grid.querySelectorAll('.mu-item.uploading');
        uploaded.forEach((file, idx)=>{
          const it = items[idx]; if (!it) return;
          it.classList.remove('uploading');
          const img = it.querySelector('img');
          if (img) img.src = file.thumbUrl || file.originalUrl;
          const st = it.querySelector('.mu-status');
          if (st) st.textContent = 'Uploaded';
          // store urls for consumers
          it.dataset.originalUrl = file.originalUrl;
          it.dataset.thumbUrl = file.thumbUrl;
        });
      } catch(e){
        const err = h('div', {class:'mu-error'}, e.message || 'Upload failed');
        container.appendChild(err);
        setTimeout(()=> err.remove(), 4000);
      }
    }

    zone.addEventListener('click', ()=> input.click());
    zone.addEventListener('dragover', (e)=>{ e.preventDefault(); zone.classList.add('drag'); });
    zone.addEventListener('dragleave', ()=> zone.classList.remove('drag'));
    zone.addEventListener('drop', (e)=>{ e.preventDefault(); zone.classList.remove('drag'); if (e.dataTransfer?.files) handleFiles(e.dataTransfer.files); });
    input.addEventListener('change', ()=>{ if (input.files) handleFiles(input.files); input.value=''; });
  }

  function autoInit(){
    document.querySelectorAll('.mu').forEach(init);
  }
  if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', autoInit); else autoInit();
})();
