<?php
// Debug script to check wallet transactions setup
$config = require_once __DIR__ . '/config/config.php';

try {
    $pdo = $config['pdo'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "=== WALLET TRANSACTIONS DEBUG ===\n\n";
    
    // 1. Check if wallet_transactions table exists
    echo "1. Checking wallet_transactions table:\n";
    $stmt = $pdo->query("SHOW TABLES LIKE 'wallet_transactions'");
    if ($stmt->rowCount() > 0) {
        echo "✓ wallet_transactions table exists\n";
        
        // Check table structure
        $stmt = $pdo->query("DESCRIBE wallet_transactions");
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
        echo "   Columns: " . implode(', ', $columns) . "\n";
        
        // Check row count
        $stmt = $pdo->query("SELECT COUNT(*) FROM wallet_transactions");
        $count = $stmt->fetchColumn();
        echo "   Row count: $count\n";
        
        if ($count > 0) {
            echo "   Sample data:\n";
            $stmt = $pdo->query("SELECT id, user_id, transaction_type, category, amount, description, created_at FROM wallet_transactions ORDER BY created_at DESC LIMIT 3");
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                echo "   - ID:{$row['id']} User:{$row['user_id']} {$row['transaction_type']} {$row['category']} ฿{$row['amount']} '{$row['description']}' {$row['created_at']}\n";
            }
        }
    } else {
        echo "✗ wallet_transactions table does not exist\n";
    }
    
    // 2. Check if function exists
    echo "\n2. Checking insert_wallet_transaction function:\n";
    $stmt = $pdo->query("SHOW FUNCTION STATUS WHERE Name = 'insert_wallet_transaction'");
    if ($stmt->rowCount() > 0) {
        echo "✓ insert_wallet_transaction function exists\n";
    } else {
        echo "✗ insert_wallet_transaction function does not exist\n";
    }
    
    // 3. Check if triggers exist
    echo "\n3. Checking triggers:\n";
    $stmt = $pdo->query("SHOW TRIGGERS LIKE 'taxi_bookings'");
    $refundTrigger = false;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        if ($row['Trigger'] === 'taxi_refund_wallet_transaction') {
            $refundTrigger = true;
            echo "✓ taxi_refund_wallet_transaction trigger exists\n";
        }
    }
    if (!$refundTrigger) {
        echo "✗ taxi_refund_wallet_transaction trigger does not exist\n";
    }
    
    $stmt = $pdo->query("SHOW TRIGGERS LIKE 'wallets'");
    $adminTrigger = false;
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        if ($row['Trigger'] === 'admin_credit_wallet_transaction') {
            $adminTrigger = true;
            echo "✓ admin_credit_wallet_transaction trigger exists\n";
        }
    }
    if (!$adminTrigger) {
        echo "✗ admin_credit_wallet_transaction trigger does not exist\n";
    }
    
    // 4. Check current agent's wallet data
    echo "\n4. Checking agent wallet data (assuming agent ID 10):\n";
    $agentId = 10; // Change this to actual logged-in agent ID
    
    $stmt = $pdo->prepare("SELECT * FROM wallets WHERE user_id = ?");
    $stmt->execute([$agentId]);
    $wallet = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($wallet) {
        echo "   Wallet balance: ฿{$wallet['balance']}\n";
    } else {
        echo "   No wallet found for agent $agentId\n";
    }
    
    // Check wallet_ledger entries
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM wallet_ledger wl JOIN wallets w ON w.id = wl.wallet_id WHERE w.user_id = ?");
    $stmt->execute([$agentId]);
    $ledgerCount = $stmt->fetchColumn();
    echo "   wallet_ledger entries: $ledgerCount\n";
    
    // Check wallet_transactions entries
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM wallet_transactions WHERE user_id = ?");
    $stmt->execute([$agentId]);
    $transactionCount = $stmt->fetchColumn();
    echo "   wallet_transactions entries: $transactionCount\n";
    
    // 5. Test function manually
    echo "\n5. Testing function manually:\n";
    try {
        $stmt = $pdo->prepare("SELECT insert_wallet_transaction(?, 'credit', 'admin_credit', 100.00, 'Test Admin Credit', 'TEST_123', 'admin_action', NULL, JSON_OBJECT('test', true))");
        $stmt->execute([$agentId]);
        $result = $stmt->fetchColumn();
        echo "✓ Function test successful, transaction ID: $result\n";
        
        // Clean up test transaction
        $pdo->prepare("DELETE FROM wallet_transactions WHERE id = ?")->execute([$result]);
        $pdo->prepare("UPDATE wallets SET balance = balance - 100.00 WHERE user_id = ?")->execute([$agentId]);
        echo "   Test transaction cleaned up\n";
    } catch (Exception $e) {
        echo "✗ Function test failed: " . $e->getMessage() . "\n";
    }
    
} catch (Exception $e) {
    echo "Database connection failed: " . $e->getMessage() . "\n";
}
