-- Comprehensive Wallet Transaction System
-- Date: 2025-09-11
-- Purpose: Create complete wallet transaction tracking for enterprise-grade wallet management

-- Create wallet_transactions table for complete transaction history
CREATE TABLE IF NOT EXISTS `wallet_transactions` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) NOT NULL,
  `transaction_type` enum('credit','debit') NOT NULL,
  `category` enum('admin_credit','booking_payment','taxi_payment','refund','transfer_in','transfer_out','adjustment','topup') NOT NULL,
  `amount` decimal(12,2) NOT NULL,
  `balance_before` decimal(12,2) NOT NULL DEFAULT 0.00,
  `balance_after` decimal(12,2) NOT NULL DEFAULT 0.00,
  `currency` char(3) NOT NULL DEFAULT 'THB',
  `description` varchar(255) NOT NULL,
  `reference_id` varchar(100) DEFAULT NULL,
  `reference_type` enum('order','taxi_booking','hotel_booking','admin_action','transfer','refund') DEFAULT NULL,
  `admin_user_id` int(11) DEFAULT NULL COMMENT 'Admin who performed the action',
  `partner_id` int(11) DEFAULT NULL COMMENT 'Partner involved in transaction',
  `metadata` json DEFAULT NULL COMMENT 'Additional transaction details',
  `status` enum('pending','completed','failed','cancelled') NOT NULL DEFAULT 'completed',
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NULL DEFAULT NULL ON UPDATE current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_transaction_type` (`transaction_type`),
  KEY `idx_category` (`category`),
  KEY `idx_reference` (`reference_id`, `reference_type`),
  KEY `idx_created_at` (`created_at`),
  KEY `idx_admin_user` (`admin_user_id`),
  KEY `idx_partner` (`partner_id`),
  FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Migrate existing data from orders (wallet payments)
INSERT INTO `wallet_transactions` (
  `user_id`, 
  `transaction_type`, 
  `category`, 
  `amount`, 
  `balance_before`, 
  `balance_after`, 
  `description`, 
  `reference_id`, 
  `reference_type`, 
  `created_at`
)
SELECT 
  o.user_id,
  'debit' as transaction_type,
  'booking_payment' as category,
  o.total_amount as amount,
  0.00 as balance_before, -- We'll calculate this later
  0.00 as balance_after,  -- We'll calculate this later
  CONCAT('Order Payment #', o.id) as description,
  o.id as reference_id,
  'order' as reference_type,
  o.created_at
FROM orders o 
WHERE o.payment_method = 'wallet' 
AND o.status = 'paid'
AND NOT EXISTS (
  SELECT 1 FROM wallet_transactions wt 
  WHERE wt.reference_id = o.id AND wt.reference_type = 'order'
);

-- Migrate existing data from taxi_bookings (wallet payments)
INSERT INTO `wallet_transactions` (
  `user_id`, 
  `transaction_type`, 
  `category`, 
  `amount`, 
  `balance_before`, 
  `balance_after`, 
  `description`, 
  `reference_id`, 
  `reference_type`, 
  `created_at`
)
SELECT 
  tb.agent_id as user_id,
  'debit' as transaction_type,
  'taxi_payment' as category,
  tb.amount_total as amount,
  0.00 as balance_before,
  0.00 as balance_after,
  CONCAT('Taxi Booking Payment - ', tb.booking_code) as description,
  tb.id as reference_id,
  'taxi_booking' as reference_type,
  tb.created_at
FROM taxi_bookings tb 
WHERE tb.payment_method = 'wallet' 
AND tb.status IN ('confirmed', 'completed')
AND NOT EXISTS (
  SELECT 1 FROM wallet_transactions wt 
  WHERE wt.reference_id = tb.id AND wt.reference_type = 'taxi_booking'
);

-- Create sample admin credit transactions for demonstration
-- (These would normally be created when admin adds funds)
INSERT INTO `wallet_transactions` (
  `user_id`, 
  `transaction_type`, 
  `category`, 
  `amount`, 
  `balance_before`, 
  `balance_after`, 
  `description`, 
  `reference_id`, 
  `reference_type`, 
  `admin_user_id`,
  `metadata`,
  `created_at`
)
SELECT 
  w.user_id,
  'credit' as transaction_type,
  'admin_credit' as category,
  w.balance as amount, -- Current balance as initial credit
  0.00 as balance_before,
  w.balance as balance_after,
  'Initial wallet funding by admin' as description,
  CONCAT('ADMIN_CREDIT_', w.user_id) as reference_id,
  'admin_action' as reference_type,
  3 as admin_user_id, -- Assuming admin user ID 3
  JSON_OBJECT('reason', 'Initial funding', 'admin_note', 'System migration') as metadata,
  DATE_SUB(NOW(), INTERVAL 30 DAY) as created_at -- 30 days ago
FROM wallets w 
WHERE w.balance > 0
AND NOT EXISTS (
  SELECT 1 FROM wallet_transactions wt 
  WHERE wt.user_id = w.user_id AND wt.category = 'admin_credit'
);

-- Update balance_before and balance_after for proper running balance
-- This is a simplified approach - in production, you'd calculate running balances properly
UPDATE wallet_transactions wt1
JOIN (
  SELECT 
    wt.id,
    COALESCE(
      (SELECT SUM(
        CASE 
          WHEN wt2.transaction_type = 'credit' THEN wt2.amount 
          ELSE -wt2.amount 
        END
      ) 
      FROM wallet_transactions wt2 
      WHERE wt2.user_id = wt.user_id 
      AND wt2.created_at < wt.created_at
      ), 0
    ) as calculated_balance_before
  FROM wallet_transactions wt
) calc ON wt1.id = calc.id
SET 
  wt1.balance_before = calc.calculated_balance_before,
  wt1.balance_after = calc.calculated_balance_before + 
    CASE 
      WHEN wt1.transaction_type = 'credit' THEN wt1.amount 
      ELSE -wt1.amount 
    END;

-- Create helper function to insert wallet transactions with proper balance calculation
DELIMITER $$
CREATE OR REPLACE FUNCTION insert_wallet_transaction(
    p_user_id INT,
    p_transaction_type ENUM('credit','debit'),
    p_category ENUM('admin_credit','booking_payment','taxi_payment','refund','transfer_in','transfer_out','adjustment','topup'),
    p_amount DECIMAL(12,2),
    p_description VARCHAR(255),
    p_reference_id VARCHAR(100),
    p_reference_type ENUM('order','taxi_booking','hotel_booking','admin_action','transfer','refund'),
    p_admin_user_id INT,
    p_metadata JSON
) RETURNS INT
READS SQL DATA
MODIFIES SQL DATA
DETERMINISTIC
BEGIN
    DECLARE current_balance DECIMAL(12,2) DEFAULT 0.00;
    DECLARE new_balance DECIMAL(12,2) DEFAULT 0.00;
    DECLARE transaction_id INT DEFAULT 0;
    
    -- Get current wallet balance
    SELECT COALESCE(balance, 0.00) INTO current_balance 
    FROM wallets 
    WHERE user_id = p_user_id 
    LIMIT 1;
    
    -- Calculate new balance
    IF p_transaction_type = 'credit' THEN
        SET new_balance = current_balance + p_amount;
    ELSE
        SET new_balance = current_balance - p_amount;
    END IF;
    
    -- Insert transaction record
    INSERT INTO wallet_transactions (
        user_id, transaction_type, category, amount, 
        balance_before, balance_after, description, 
        reference_id, reference_type, admin_user_id, metadata, 
        status, created_at
    ) VALUES (
        p_user_id, p_transaction_type, p_category, p_amount,
        current_balance, new_balance, p_description,
        p_reference_id, p_reference_type, p_admin_user_id, p_metadata,
        'completed', NOW()
    );
    
    SET transaction_id = LAST_INSERT_ID();
    
    -- Update wallet balance (wallets table may not have updated_at column)
    UPDATE wallets 
    SET balance = new_balance 
    WHERE user_id = p_user_id;
    
    RETURN transaction_id;
END$$
DELIMITER ;

-- Create triggers for automatic wallet transaction logging

-- Trigger for taxi refunds
DELIMITER $$
CREATE OR REPLACE TRIGGER taxi_refund_wallet_transaction
AFTER UPDATE ON taxi_bookings
FOR EACH ROW
BEGIN
    DECLARE refund_amount DECIMAL(12,2) DEFAULT 0.00;
    DECLARE transaction_id INT DEFAULT 0;
    
    -- Check if this is a refund update
    IF (NEW.status = 'refunded' OR NEW.payment_status = 'refunded' OR NEW.refund_amount IS NOT NULL) 
       AND (OLD.status != 'refunded' AND OLD.payment_status != 'refunded' AND OLD.refund_amount IS NULL) THEN
        
        -- Determine refund amount
        SET refund_amount = COALESCE(NEW.refund_amount, NEW.amount_total);
        
        -- Only process if refund amount > 0 and payment was via wallet
        IF refund_amount > 0 AND NEW.payment_method = 'wallet' THEN
            -- Insert wallet transaction for refund
            SET transaction_id = insert_wallet_transaction(
                NEW.agent_id,
                'credit',
                'refund',
                refund_amount,
                CONCAT('Taxi Booking Refund - ', COALESCE(NEW.booking_code, NEW.id)),
                NEW.id,
                'taxi_booking',
                NULL,
                JSON_OBJECT(
                    'original_amount', NEW.amount_total,
                    'refund_amount', refund_amount,
                    'booking_code', NEW.booking_code,
                    'refund_reason', 'Taxi booking cancelled'
                )
            );
        END IF;
    END IF;
END$$
DELIMITER ;

-- Trigger for admin credits (when wallets table is updated by admin)
DELIMITER $$
CREATE OR REPLACE TRIGGER admin_credit_wallet_transaction
AFTER UPDATE ON wallets
FOR EACH ROW
BEGIN
    DECLARE credit_amount DECIMAL(12,2) DEFAULT 0.00;
    DECLARE transaction_id INT DEFAULT 0;
    
    -- Check if balance increased (admin credit)
    IF NEW.balance > OLD.balance THEN
        SET credit_amount = NEW.balance - OLD.balance;
        
        -- Insert wallet transaction for admin credit
        SET transaction_id = insert_wallet_transaction(
            NEW.user_id,
            'credit',
            'admin_credit',
            credit_amount,
            CONCAT('Admin Credit - Balance adjustment of ฿', FORMAT(credit_amount, 2)),
            CONCAT('ADMIN_CREDIT_', UNIX_TIMESTAMP()),
            'admin_action',
            NULL, -- We don't track which admin did this yet
            JSON_OBJECT(
                'previous_balance', OLD.balance,
                'new_balance', NEW.balance,
                'credit_amount', credit_amount,
                'admin_note', 'Manual balance adjustment'
            )
        );
    END IF;
END$$
DELIMITER ;
