<?php
// config/config.php
// Loads .env, sets config array, initializes PDO, session, and error handling

if (!function_exists('env')) {
    function env($key, $default = null) {
        static $env = null;
        if ($env === null) {
            $envPath = dirname(__DIR__) . DIRECTORY_SEPARATOR . '.env';
            if (file_exists($envPath)) {
                $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                foreach ($lines as $line) {
                    if (str_starts_with(trim($line), '#')) continue;
                    [$k, $v] = array_pad(explode('=', $line, 2), 2, '');
                    $v = trim($v);
                    $v = trim($v, "\"'");
                    $env[trim($k)] = $v;
                }
            } else {
                $env = [];
            }
        }
        return $env[$key] ?? $default;
    }
}

// Error reporting
$debug = env('APP_DEBUG', 'false') === 'true';
ini_set('display_errors', $debug ? '1' : '0');
error_reporting($debug ? E_ALL : 0);

// Timezone (affects all date()/strtotime() usage)
$appTz = env('APP_TIMEZONE', 'Asia/Kolkata');
if (@date_default_timezone_set($appTz) === false) {
    // Fallback to UTC if timezone string invalid
    date_default_timezone_set('UTC');
}

// Session
session_name(env('SESSION_NAME', 'b2b_session'));
// Cookie hardening
ini_set('session.cookie_httponly', '1');
ini_set('session.use_only_cookies', '1');
ini_set('session.cookie_samesite', 'Lax');
if (!empty($_SERVER['HTTPS']) || str_starts_with(env('APP_URL',''), 'https')) {
    ini_set('session.cookie_secure', '1');
}
if (session_status() === PHP_SESSION_NONE) session_start();

// CSRF
define('CSRF_TTL', (int)env('CSRF_TOKEN_TTL', 3600));

// Database (PDO)
try {
    $dsn = 'mysql:host=' . env('DB_HOST', 'localhost') . ';port=' . env('DB_PORT', 3306) . ';dbname=' . env('DB_DATABASE', 'b2b_travel') . ';charset=utf8mb4';
    $pdo = new PDO($dsn, env('DB_USERNAME', 'root'), env('DB_PASSWORD', ''));
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    http_response_code(500);
    die('Database connection failed. Configure .env and import database/schema.sql.');
}

// Autoload vendor if available (PHPMailer/Stripe/Twilio via Composer)
$vendor = dirname(__DIR__) . DIRECTORY_SEPARATOR . 'vendor' . DIRECTORY_SEPARATOR . 'autoload.php';
if (file_exists($vendor)) require_once $vendor;

// App helper
define('BASE_PATH', dirname(__DIR__));
define('APP_URL', rtrim(env('APP_URL', ''), '/'));

// Simple container
if (!isset($pdo) || !($pdo instanceof PDO)) {
    http_response_code(500);
    die('Invalid PDO in config/config.php');
}

// Always return an array so public/index.php receives a valid $cfg
return [
    'pdo'   => $pdo,
    'debug' => $debug,
];
