<?php
/** @var array $city @var array $checkin @var array $checkout @var int $adults @var int $rooms */
$city = $city ?? '';
$checkin = $checkin ?? '';
$checkout = $checkout ?? '';
$adults = isset($adults) ? (int)$adults : 2;
$rooms = isset($rooms) ? (int)$rooms : 1;
$title = $title ?? 'Hotels';
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title><?= htmlspecialchars($title) ?></title>
  <script src="https://cdn.tailwindcss.com"></script>
  <style>
    .layout{display:grid;grid-template-columns:300px minmax(0,1fr);gap:20px;align-items:start}
    @media(max-width:980px){.layout{grid-template-columns:1fr}}
    .toolbar{display:flex;flex-wrap:wrap;gap:12px;align-items:center;justify-content:space-between;margin:12px 0;position:sticky;top:80px;background:rgba(255,255,255,.95);backdrop-filter:saturate(180%) blur(10px);z-index:25;padding:8px 6px;border-bottom:1px solid #e2e8f0}
    .hotel-card{display:flex;gap:16px;padding:16px;border:1px solid #e5e7eb;border-radius:12px;background:#fff;transition:box-shadow .15s ease;align-items:flex-start}
    .hotel-card:hover{box-shadow:0 6px 18px rgba(20,33,61,.08)}
    .thumb-wrap{position:relative;width:160px;height:120px;flex:0 0 auto}
    .hotel-thumb{width:100%;height:100%;background:#f3f4f6;border-radius:10px;object-fit:cover;display:block}
    .ribbon{position:absolute;top:10px;left:-8px;background:#1d4ed8;color:#fff;padding:4px 12px;font-size:.72rem;font-weight:800;border-radius:6px;box-shadow:0 4px 10px rgba(0,0,0,.15);transform:rotate(-12deg);transform-origin:left top;letter-spacing:.2px}
    .ribbon::after{content:'';position:absolute;right:-10px;top:0;border-top:14px solid #1d4ed8;border-right:10px solid transparent}
    .hotel-meta{flex:1 1 auto;min-width:0}
    .price{flex:0 0 260px;text-align:right;display:flex;flex-direction:column;align-items:flex-end;justify-content:flex-start;gap:6px}
    .actions{margin-top:8px;display:flex;flex-direction:column;gap:8px;width:100%;align-items:stretch}
    .actions .btn{display:flex;justify-content:center;width:100%}
    .grid{display:grid;grid-template-columns:repeat(1,1fr);gap:16px}
    @media(min-width:980px){.grid{grid-template-columns:repeat(2,1fr)}}
    .form-row{display:flex;flex-wrap:wrap;gap:12px;margin-bottom:8px}
    .form-row input,.form-row select{padding:10px 12px;border:1px solid #d1d5db;border-radius:8px;min-height:44px}
    .btn{background:#2563eb;color:#fff;border:none;padding:10px 14px;border-radius:8px;cursor:pointer}
    .btn:disabled{opacity:.6;cursor:not-allowed}
    .badge{background:#eef2ff;color:#3730a3;border-radius:999px;padding:4px 8px;font-size:.78rem;display:inline-flex;align-items:center;white-space:nowrap;line-height:1;font-weight:600}
    .badge-featured{background:#fef3c7;color:#92400e;border-radius:999px;padding:4px 8px;font-size:.78rem;display:inline-flex;align-items:center;gap:6px;font-weight:700}
    .stars{color:#f59e0b;font-size:.95rem}
    .hotel-title{font-weight:700;font-size:1.05rem;line-height:1.2;margin:2px 0 4px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap}
    .location{color:#475569;font-size:.9rem;display:flex;align-items:center;gap:6px;flex-wrap:wrap}
    .location .dot{color:#94a3b8}
    .loc-text{white-space:nowrap}
    .strike{color:#94a3b8;text-decoration:line-through;margin-right:6px}
    .pill{display:inline-flex;align-items:center;gap:6px;padding:4px 8px;border-radius:999px;background:#f1f5f9;color:#334155;font-size:.78rem}
    .pager{display:flex;gap:8px;justify-content:center;margin:18px 0}
    .pager button{padding:8px 12px;border:1px solid #e2e8f0;background:#fff;border-radius:8px;cursor:pointer}
    .pager button[disabled]{opacity:.6;cursor:not-allowed}
    .filter-card{border:1px solid #e5e7eb;border-radius:12px;background:#fff;padding:12px}
    #filters{position:sticky;top:80px;max-height:calc(100vh - 100px);overflow:auto}
    .filter-card h4{margin:8px 0;font-size:1rem}
    .filter-row{display:flex;gap:8px;align-items:center;margin:6px 0}
    .muted{color:#64748b}
    /* modal */
    .modal-backdrop{position:fixed;inset:0;background:rgba(0,0,0,.4);display:none;align-items:center;justify-content:center;z-index:50}
    .modal{background:#fff;border-radius:12px;max-width:520px;width:92%;padding:16px;border:1px solid #e5e7eb;box-shadow:0 10px 30px rgba(0,0,0,.15)}
    .modal h3{margin:0 0 8px 0}
    .row{display:flex;gap:8px;align-items:center;margin:8px 0}
    .toast{position:fixed;right:16px;bottom:16px;background:#111827;color:#fff;padding:10px 12px;border-radius:8px;display:none;z-index:60}
    /* Responsive enhancements */
    @media(max-width:640px){
      .hotel-card{flex-direction:column}
      .thumb-wrap{width:100%;height:auto}
      .hotel-thumb{width:100%;height:auto;aspect-ratio:4/3}
      .price{text-align:left;margin-top:8px;min-width:0;align-items:stretch}
    }
    @media(max-width:360px){
      .badge{font-size:.72rem}
    }
    /* Main column min height to keep footer away */
    main{min-height:50vh}
  </style>
</head>
<body>
  <div class="container mx-auto px-4 md:px-6" style="max-width:1200px">
    <h1 style="margin:12px 0;">Hotels</h1>

    <form id="searchForm" class="form-row" onsubmit="return false;">
      <input type="text" id="city" name="city" placeholder="City (e.g., Bangkok)" value="<?= htmlspecialchars($city) ?>">
      <input type="date" id="checkin" name="checkin" value="<?= htmlspecialchars($checkin) ?>">
      <input type="date" id="checkout" name="checkout" value="<?= htmlspecialchars($checkout) ?>">
      <select id="adults" name="adults">
        <?php for($i=1;$i<=6;$i++): ?>
          <option value="<?= $i ?>" <?= $adults===$i?'selected':'' ?>>Adults: <?= $i ?></option>
        <?php endfor; ?>
      </select>
      <select id="rooms" name="rooms">
        <?php for($i=1;$i<=6;$i++): ?>
          <option value="<?= $i ?>" <?= $rooms===$i?'selected':'' ?>>Rooms: <?= $i ?></option>
        <?php endfor; ?>
      </select>
      <button class="btn" id="btnSearch">Search</button>
    </form>

    <div class="toolbar">
      <div id="meta" class="pill"></div>
      <div>
        <label for="sort" style="font-size:.85rem;color:#64748b;margin-right:6px">Sort by</label>
        <select id="sort">
          <option value="featured">Featured</option>
          <option value="price_asc">Price (low to high)</option>
          <option value="price_desc">Price (high to low)</option>
          <option value="stars_desc">Stars</option>
          <option value="distance_asc">Distance (closest first)</option>
        </select>
      </div>
      <button id="btnToggleFilters" class="btn md:hidden" title="Filters">Filters</button>
    </div>

    <div class="layout">
      <aside id="filters" class="hidden md:block">
        <div class="filter-card">
          <h4>Price per night (THB)</h4>
          <div class="filter-row">
            <input type="number" id="min_price" placeholder="Min" style="width:100px">
            <span class="muted">–</span>
            <input type="number" id="max_price" placeholder="Max" style="width:100px">
          </div>
        </div>
        <div class="filter-card">
          <h4>Star rating</h4>
          <div class="filter-row"><label><input type="checkbox" class="stars" value="5"> 5 stars</label></div>
          <div class="filter-row"><label><input type="checkbox" class="stars" value="4"> 4 stars</label></div>
          <div class="filter-row"><label><input type="checkbox" class="stars" value="3"> 3 stars</label></div>
          <div class="filter-row"><label><input type="checkbox" class="stars" value="2"> 2 stars</label></div>
          <div class="filter-row"><label><input type="checkbox" class="stars" value="1"> 1 star</label></div>
        </div>
        <div class="filter-card">
          <h4>Facilities</h4>
          <div class="filter-row"><label><input type="checkbox" class="amenity" value="wifi"> Free WiFi</label></div>
          <div class="filter-row"><label><input type="checkbox" class="amenity" value="parking"> Parking</label></div>
          <div class="filter-row"><label><input type="checkbox" class="amenity" value="pool"> Swimming pool</label></div>
          <div class="filter-row"><label><input type="checkbox" class="amenity" value="spa"> Spa</label></div>
          <div class="filter-row"><label><input type="checkbox" class="amenity" value="gym"> Fitness center</label></div>
        </div>
        <div class="filter-card">
          <h4>Policies</h4>
          <div class="filter-row"><label><input type="checkbox" id="free_cancellation" value="1"> Free cancellation</label></div>
          <div class="filter-row"><label><input type="checkbox" id="pay_at_property" value="1"> Pay at property</label></div>
        </div>
        <div class="filter-card">
          <h4>Review score</h4>
          <select id="min_review" style="width:100%">
            <option value="">Any</option>
            <option value="7">7.0+</option>
            <option value="8">8.0+</option>
            <option value="9">9.0+</option>
          </select>
        </div>
        <div class="filter-card">
          <h4>Distance center (opt)</h4>
          <div class="filter-row"><input type="number" step="0.000001" id="center_lat" placeholder="Lat" style="width:120px"><input type="number" step="0.000001" id="center_lng" placeholder="Lng" style="width:120px"></div>
          <div class="muted" style="font-size:.8rem">Set lat/lng to enable distance sort.</div>
        </div>
        <div class="filter-card">
          <button class="btn" id="btnApplyFilters" style="width:100%">Apply filters</button>
        </div>
      </aside>
      <main>
        <div id="results" class="grid"></div>
      </main>
    </div>
    <div id="pager" class="pager" style="display:none">
      <button id="prev">Prev</button>
      <span id="pageInfo" class="pill"></span>
      <button id="next">Next</button>
    </div>
  </div>
  <!-- Quick Book Modal (moved above script so elements exist when JS runs) -->
  <div class="modal-backdrop" id="qbBackdrop">
    <div class="modal" id="qbModal">
      <div style="display:flex;justify-content:space-between;align-items:center;gap:8px">
        <h3>Quick Book</h3>
        <button id="qbClose" class="btn" style="background:#64748b">Close</button>
      </div>
      <div class="muted" id="qbHotelTitle" style="margin-bottom:8px"></div>
      <div class="row">
        <label style="min-width:90px">Room</label>
        <select id="qbRoom" style="flex:1"></select>
      </div>
      <div class="row">
        <label style="min-width:90px">Qty rooms</label>
        <input type="number" id="qbQty" min="1" value="1" style="width:100px">
      </div>
      <div class="row" style="justify-content:flex-end">
        <button class="btn" id="qbSubmit">Book</button>
      </div>
    </div>
  </div>
  <div class="toast" id="toast"></div>

  <script>
  (function(){
    const form = document.getElementById('searchForm');
    const btn = document.getElementById('btnSearch');
    const results = document.getElementById('results');
    const meta = document.getElementById('meta');
    const sortSel = document.getElementById('sort');
    const pager = document.getElementById('pager');
    const btnToggleFilters = document.getElementById('btnToggleFilters');
    const asideFilters = document.getElementById('filters');
    const prevBtn = document.getElementById('prev');
    const nextBtn = document.getElementById('next');
    const pageInfo = document.getElementById('pageInfo');
    const btnApplyFilters = document.getElementById('btnApplyFilters');
    let lastItems = [];
    let currentPage = 1;
    const pageSize = 10;
    const STORAGE_KEY = 'agent_hotels_search_state_v1';

    function fmt(n){ return new Intl.NumberFormat('en-US',{minimumFractionDigits:0,maximumFractionDigits:0}).format(n); }
    function qsel(id){ return document.getElementById(id).value.trim(); }

    function thumbnailUrl(h){
      let p = h.thumb || '';
      if (!p) return 'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="160" height="120"><rect width="100%" height="100%" fill="%23f1f5f9"/><text x="50%" y="50%" font-size="12" fill="%2399a" text-anchor="middle" dominant-baseline="middle">No image</text></svg>';
      if (p.startsWith('http')) return p;
      // normalize Windows backslashes and duplicate slashes
      p = String(p).replace(/\\/g, '/');
      p = p.replace(/\/{2,}/g, '/');
      // drop leading './' if any
      if (p.startsWith('./')) p = p.slice(1);
      // ensure leading slash and fix stray '/\' case
      if (!p.startsWith('/')) p = '/' + p;
      p = p.replace('/\\', '/');
      return p;
    }

    function sortItems(items, sort){
      const list = [...items];
      if (sort === 'price_asc') list.sort((a,b)=>((a.nightly_from_agent??a.nightly_from??a.base_price??0)-(b.nightly_from_agent??b.nightly_from??b.base_price??0)));
      else if (sort === 'price_desc') list.sort((a,b)=>((b.nightly_from_agent??b.nightly_from??b.base_price??0)-(a.nightly_from_agent??a.nightly_from??a.base_price??0)));
      else if (sort === 'stars_desc') list.sort((a,b)=>((b.stars||0)-(a.stars||0)));
      else if (sort === 'distance_asc') list.sort((a,b)=>((a.distance_km??99999)-(b.distance_km??99999)));
      return list; // featured order kept by API for 'featured'
    }

    function paginate(items){
      const total = items.length;
      const pages = Math.max(1, Math.ceil(total / pageSize));
      if (currentPage > pages) currentPage = pages;
      const start = (currentPage - 1) * pageSize;
      const end = start + pageSize;
      const slice = items.slice(start, end);
      pager.style.display = pages > 1 ? '' : 'none';
      prevBtn.disabled = currentPage <= 1;
      nextBtn.disabled = currentPage >= pages;
      pageInfo.textContent = `Page ${currentPage} / ${pages}`;
      return slice;
    }

    function render(items, m){
      meta.textContent = `City: ${m.city} | ${m.checkin} → ${m.checkout} | Nights: ${m.nights} | Rooms: ${m.rooms}`;
      if(!items || items.length===0){ results.innerHTML = '<p>No hotels found.</p>'; pager.style.display='none'; return; }
      const sorted = sortItems(items, sortSel.value);
      const pageItems = paginate(sorted);
      results.innerHTML = pageItems.map(h=>{
        const baseNight = h.base_price ?? h.nightly_from ?? 0;
        const nightlyVendor = h.nightly_from_vendor != null ? h.nightly_from_vendor : baseNight;
        const nightlyAgent = h.nightly_from_agent != null ? h.nightly_from_agent : baseNight;
        const nightlyCustomer = h.nightly_from_customer != null ? h.nightly_from_customer : baseNight;
        const priceNight = nightlyAgent;
        const total = h.total_estimate || (priceNight * (m.nights||1) * (m.rooms||1));
        const stars = '★'.repeat(h.stars||0);
        const lblMap = { top_choice: 'Top Choice', recommended: 'Recommended', trusted_partner: 'Trusted Partner', best_value: 'Best Value' };
        const ribbonText = lblMap[(h.featured_label||'recommended')] || 'Recommended';
        return `
          <div class="hotel-card">
            <div class="thumb-wrap">
              ${h.featured_agent ? `<div class=\"ribbon\">${ribbonText}</div>` : ''}
              <img class="hotel-thumb" src="${thumbnailUrl(h)}" alt="thumb" loading="lazy" referrerpolicy="no-referrer" />
            </div>
            <div class="hotel-meta">
              <div class="hotel-title">${h.name}</div>
              <div class="location"><span class="stars">${stars}</span><span class="dot">•</span><span class="loc-text">${h.city}${h.country? ', '+h.country:''}</span></div>
              <div style="display:flex;gap:8px;flex-wrap:wrap;margin:6px 0;align-items:center">
                <span class="badge inline-flex items-center gap-1 px-2 py-1 rounded-full bg-indigo-50 text-indigo-800 text-xs">Agent: THB ${fmt(nightlyAgent)}</span>
                ${h.review_score_avg!=null? `<span class=\"pill inline-flex items-center gap-1 px-2 py-1 rounded-full bg-slate-100 text-slate-700 text-xs\">Review ${h.review_score_avg.toFixed(1)}</span>`:''}
                ${h.pay_at_property? `<span class=\"pill inline-flex items-center gap-1 px-2 py-1 rounded-full bg-slate-100 text-slate-700 text-xs\">Pay at property</span>`:''}
                ${h.distance_km!=null? `<span class=\"pill inline-flex items-center gap-1 px-2 py-1 rounded-full bg-slate-100 text-slate-700 text-xs\">${h.distance_km.toFixed(1)} km</span>`:''}
              </div>
              <div id="bkcode-${h.id}" style="margin-top:4px"></div>
            </div>
            <div class="price">
              <div style="font-size:.9rem;color:#64748b;">${baseNight>nightlyAgent ? `<span class='strike'>THB ${fmt(baseNight)}</span>`:''}<b>THB ${fmt(priceNight)}</b> / night</div>
              <div style="font-size:1.1rem;font-weight:800;">THB ${fmt(total||0)}</div>
              <div style="font-size:.85rem;color:#64748b;">for ${m.rooms} room(s), ${m.nights} night(s)</div>
              <div class="actions">
                <a class="btn bg-blue-600 hover:bg-blue-700 text-white px-3 py-2 rounded-lg" href="/agent/hotel/detail?id=${h.id}&checkin=${encodeURIComponent(m.checkin)}&checkout=${encodeURIComponent(m.checkout)}">Check Availability</a>
                <button class="btn bg-blue-600 hover:bg-blue-700 text-white px-3 py-2 rounded-lg" data-quick-book="${h.id}">Quick Book</button>
              </div>
            </div>
          </div>`;
      }).join('');
      // bind quick book buttons
      document.querySelectorAll('button[data-quick-book]').forEach(btn=>{
        btn.addEventListener('click',()=>openQuickBook(btn.getAttribute('data-quick-book')));
      });
    }

    function collectFilters(){
      const stars = Array.from(document.querySelectorAll('input.stars:checked')).map(i=>i.value).join(',');
      const amenities = Array.from(document.querySelectorAll('input.amenity:checked')).map(i=>i.value).join(',');
      const min_price = document.getElementById('min_price').value.trim();
      const max_price = document.getElementById('max_price').value.trim();
      const free_cancellation = document.getElementById('free_cancellation').checked ? '1' : '';
      const pay_at_property = document.getElementById('pay_at_property').checked ? '1' : '';
      const min_review = document.getElementById('min_review').value.trim();
      const center_lat = document.getElementById('center_lat').value.trim();
      const center_lng = document.getElementById('center_lng').value.trim();
      return {stars, amenities, min_price, max_price, free_cancellation, pay_at_property, min_review, center_lat, center_lng};
    }

    function applyFiltersToFormFromUrl(){
      const sp = new URLSearchParams(location.search);
      const setVal = (id,key=id)=>{ const v = sp.get(key); if(v!==null) document.getElementById(id).value = v; };
      // main fields
      setVal('city'); setVal('checkin'); setVal('checkout'); setVal('adults'); setVal('rooms');
      // filters
      const stars = sp.get('stars');
      if(stars){ const set = new Set(stars.split(',')); document.querySelectorAll('input.stars').forEach(i=>{ i.checked = set.has(i.value); }); }
      const amenities = sp.get('amenities');
      if(amenities){ const set = new Set(amenities.split(',')); document.querySelectorAll('input.amenity').forEach(i=>{ i.checked = set.has(i.value); }); }
      setVal('min_price'); setVal('max_price'); setVal('min_review'); setVal('center_lat'); setVal('center_lng');
      const fc = sp.get('free_cancellation'); if(fc!==null) document.getElementById('free_cancellation').checked = fc==='1';
      const pap = sp.get('pay_at_property'); if(pap!==null) document.getElementById('pay_at_property').checked = pap==='1';
      const sort = sp.get('sort'); if(sort){ sortSel.value = sort; }
    }

    function getPageState(){
      const f = collectFilters();
      return {
        city: qsel('city'),
        checkin: qsel('checkin'),
        checkout: qsel('checkout'),
        adults: document.getElementById('adults').value,
        rooms: document.getElementById('rooms').value,
        sort: (sortSel && sortSel.value)||'',
        filters: f,
      };
    }
    function savePageState(){
      try{ sessionStorage.setItem(STORAGE_KEY, JSON.stringify(getPageState())); }catch(e){}
    }
    function loadPageState(){
      try{
        const raw = sessionStorage.getItem(STORAGE_KEY);
        if(!raw) return null;
        return JSON.parse(raw);
      }catch(e){ return null; }
    }
    function applyStateToForm(state){
      if(!state) return;
      const set = (id,val)=>{ if(val!=null) document.getElementById(id).value = String(val); };
      set('city', state.city); set('checkin', state.checkin); set('checkout', state.checkout);
      set('adults', state.adults); set('rooms', state.rooms);
      if(state.sort && sortSel) sortSel.value = state.sort;
      const f = state.filters||{};
      // checkboxes from saved csv values
      if(f.stars){ const S=new Set(String(f.stars).split(',')); document.querySelectorAll('input.stars').forEach(i=>i.checked=S.has(i.value)); }
      if(f.amenities){ const S=new Set(String(f.amenities).split(',')); document.querySelectorAll('input.amenity').forEach(i=>i.checked=S.has(i.value)); }
      set('min_price', f.min_price||''); set('max_price', f.max_price||'');
      document.getElementById('free_cancellation').checked = f.free_cancellation==='1';
      document.getElementById('pay_at_property').checked = f.pay_at_property==='1';
      set('min_review', f.min_review||''); set('center_lat', f.center_lat||''); set('center_lng', f.center_lng||'');
    }

    function buildPageQuery(){
      const city = qsel('city');
      const checkin = qsel('checkin');
      const checkout = qsel('checkout');
      const adults = document.getElementById('adults').value;
      const rooms = document.getElementById('rooms').value;
      const f = collectFilters();
      const params = new URLSearchParams();
      if(city) params.set('city', city);
      if(checkin) params.set('checkin', checkin);
      if(checkout) params.set('checkout', checkout);
      if(adults) params.set('adults', adults);
      if(rooms) params.set('rooms', rooms);
      if(f.stars) params.set('stars', f.stars);
      if(f.amenities) params.set('amenities', f.amenities);
      if(f.min_price) params.set('min_price', f.min_price);
      if(f.max_price) params.set('max_price', f.max_price);
      if(f.free_cancellation) params.set('free_cancellation', '1');
      if(f.pay_at_property) params.set('pay_at_property', '1');
      if(f.min_review) params.set('min_review', f.min_review);
      if(f.center_lat && f.center_lng){ params.set('center_lat', f.center_lat); params.set('center_lng', f.center_lng); }
      if(sortSel && sortSel.value) params.set('sort', sortSel.value);
      return params.toString();
    }

    function buildUrl(base){
      const f = collectFilters();
      const params = [];
      if (f.stars) params.push('stars='+encodeURIComponent(f.stars));
      if (f.amenities) params.push('amenities='+encodeURIComponent(f.amenities));
      if (f.min_price) params.push('min_price='+encodeURIComponent(f.min_price));
      if (f.max_price) params.push('max_price='+encodeURIComponent(f.max_price));
      if (f.free_cancellation) params.push('free_cancellation=1');
      if (f.pay_at_property) params.push('pay_at_property=1');
      if (f.min_review) params.push('min_review='+encodeURIComponent(f.min_review));
      if (f.center_lat && f.center_lng) { params.push('center_lat='+encodeURIComponent(f.center_lat)); params.push('center_lng='+encodeURIComponent(f.center_lng)); }
      return base + (params.length? '&'+params.join('&') : '');
    }

    async function load(){
      const city = qsel('city');
      const checkin = qsel('checkin');
      const checkout = qsel('checkout');
      const adults = document.getElementById('adults').value;
      const rooms = document.getElementById('rooms').value;
      if(!city||!checkin||!checkout){ results.innerHTML = '<p>Please enter city, check-in and check-out.</p>'; return; }
      btn.disabled = true; results.innerHTML = '<p>Loading…</p>';
      try{
        let url = `/b2b/api/hotels/search?city=${encodeURIComponent(city)}&checkin=${encodeURIComponent(checkin)}&checkout=${encodeURIComponent(checkout)}&adults=${adults}&rooms=${rooms}`;
        url = buildUrl(url);
        const res = await fetch(url, {credentials:'same-origin'});
        if(!res.ok){ throw new Error('API error'); }
        const json = await res.json();
        lastItems = json.data||[];
        currentPage = 1;
        render(lastItems, json.meta||{city,checkin,checkout,rooms,adults,nights:1});
        // Update page URL with current state so refresh preserves it
        const pageQs = buildPageQuery();
        const newUrl = pageQs ? (`${location.pathname}?${pageQs}`) : location.pathname;
        history.replaceState(null, '', newUrl);
        // Also save to sessionStorage for navigation away and back
        savePageState();
      }catch(e){
        results.innerHTML = '<p>Failed to load hotels.</p>';
      }finally{ btn.disabled = false; }
    }

    form.addEventListener('submit', (e)=>{ e.preventDefault(); savePageState(); load(); });
    sortSel.addEventListener('change', ()=>{
      if(lastItems.length) render(lastItems, { city:qsel('city'), checkin:qsel('checkin'), checkout:qsel('checkout'), rooms:document.getElementById('rooms').value, adults:document.getElementById('adults').value, nights:1 });
      // Also reflect sort in URL
      const pageQs = buildPageQuery();
      history.replaceState(null, '', pageQs ? (`${location.pathname}?${pageQs}`) : location.pathname);
      savePageState();
    });
    btnApplyFilters.addEventListener('click', (e)=>{ e.preventDefault(); savePageState(); load(); });
    prevBtn.addEventListener('click', ()=>{ if(currentPage>1){ currentPage--; render(lastItems, { city:qsel('city'), checkin:qsel('checkin'), checkout:qsel('checkout'), rooms:document.getElementById('rooms').value, adults:document.getElementById('adults').value, nights:1 }); }});
    nextBtn.addEventListener('click', ()=>{ currentPage++; render(lastItems, { city:qsel('city'), checkin:qsel('checkin'), checkout:qsel('checkout'), rooms:document.getElementById('rooms').value, adults:document.getElementById('adults').value, nights:1 }); });

    // Mobile filter toggle
    if(btnToggleFilters && asideFilters){
      btnToggleFilters.addEventListener('click', ()=>{
        const isHidden = asideFilters.classList.contains('hidden');
        if(isHidden){ asideFilters.classList.remove('hidden'); }
        else { asideFilters.classList.add('hidden'); }
      });
    }

    // Quick Book modal logic
    const mb = document.getElementById('qbBackdrop');
    const mq = document.getElementById('qbModal');
    const mqTitle = document.getElementById('qbHotelTitle');
    const mqRoom = document.getElementById('qbRoom');
    const mqQty = document.getElementById('qbQty');
    const mqBtn = document.getElementById('qbSubmit');
    const toast = document.getElementById('toast');

    function showToast(msg){ toast.textContent = msg; toast.style.display='block'; setTimeout(()=>toast.style.display='none', 6000); }
    function openQuickBook(hotelId){
      const checkin = qsel('checkin');
      const checkout = qsel('checkout');
      if(!checkin || !checkout){ showToast('Select check-in and check-out first'); return; }
      mqRoom.innerHTML = '<option value="">Loading…</option>';
      mqQty.value = '1';
      mqBtn.disabled = true; mqBtn.textContent = 'Book';
      fetch(`/b2b/api/hotels/details?id=${encodeURIComponent(hotelId)}&checkin=${encodeURIComponent(checkin)}&checkout=${encodeURIComponent(checkout)}`, {credentials:'same-origin'})
        .then(r=>r.json()).then(json=>{
          mqTitle.textContent = (json.hotel && json.hotel.name) ? json.hotel.name : `Hotel #${hotelId}`;
          const rooms = (json.rooms||[]).filter(r=>r);
          if(rooms.length===0){ mqRoom.innerHTML = '<option value="">No rooms available</option>'; mqBtn.disabled=true; }
          else{
            mqRoom.innerHTML = rooms.map(r=>`<option value="${r.id}">${r.name} — THB ${fmt(r.nightly_from_agent??r.nightly_from_base??0)}/night</option>`).join('');
            mqBtn.disabled=false;
          }
          mb.style.display='flex';
          mq.setAttribute('data-hotel-id', hotelId);
        }).catch(()=>{ showToast('Failed to load rooms'); });
    }
    const qbCloseBtn = document.getElementById('qbClose');
    if(qbCloseBtn){ qbCloseBtn.addEventListener('click', ()=>{ if(mb) mb.style.display='none'; }); }
    if(mb){ mb.addEventListener('click', (e)=>{ if(e.target===mb) mb.style.display='none'; }); }
    if(mqBtn) mqBtn.addEventListener('click', async ()=>{
      const hotelId = mq.getAttribute('data-hotel-id');
      const checkin = qsel('checkin');
      const checkout = qsel('checkout');
      const adults = document.getElementById('adults').value;
      const roomsQty = document.getElementById('rooms').value;
      const roomId = mqRoom.value;
      const qty = mqQty.value;
      if(!roomId){ showToast('Select a room'); return; }
      mqBtn.disabled=true; mqBtn.textContent='Processing…';
      try{
        // 1) Quote
        const qres = await fetch('/b2b/api/hotels/quote',{method:'POST',headers:{'Content-Type':'application/json'},credentials:'same-origin',body:JSON.stringify({hotel_id:parseInt(hotelId,10),checkin,checkout,adults:parseInt(adults,10),rooms:parseInt(roomsQty,10),selections:[{room_id:parseInt(roomId,10),rate_plan_id:null,qty_rooms:parseInt(qty,10)}]})});
        if(!qres.ok){ throw new Error('Quote failed'); }
        const qjson = await qres.json();
        const quoteCode = qjson.quote_code;
        if(!quoteCode) throw new Error('No quote_code');
        // 2) Book (default pay at property)
        const bres = await fetch('/b2b/api/hotels/book',{method:'POST',headers:{'Content-Type':'application/json'},credentials:'same-origin',body:JSON.stringify({quote_code:quoteCode,payment:{method:'none',pay_at_property:1}})});
        if(!bres.ok){ throw new Error('Booking failed'); }
        const bjson = await bres.json();
        const code = bjson.booking_code;
        if(!code) throw new Error('No booking_code');
        mb.style.display='none';
        showToast(`Booked! Code: ${code}`);
        // show copy dialog
        const btnCopy = document.createElement('button');
        btnCopy.className='btn'; btnCopy.style.marginLeft='8px'; btnCopy.textContent='Copy';
        btnCopy.onclick=()=>{ navigator.clipboard.writeText(code); showToast('Booking code copied'); };
        toast.innerHTML = `Booked! Code: ${code} `; toast.appendChild(btnCopy); toast.style.display='block';
        // render code under badge on the card
        const holder = document.getElementById(`bkcode-${hotelId}`);
        if(holder){
          holder.innerHTML = `<span class="pill">Booking: ${code}</span>`;
        }
      }catch(e){
        showToast(e.message||'Failed');
      }finally{
        mqBtn.disabled=false; mqBtn.textContent='Book';
      }
    });

    // On first load, prefill from URL or fallback to sessionStorage, then auto-load if sufficient params
    applyFiltersToFormFromUrl();
    (function(){
      const sp = new URLSearchParams(location.search);
      let city = sp.get('city'), ci = sp.get('checkin'), co = sp.get('checkout');
      if(!(city && ci && co)){
        const st = loadPageState();
        if(st && st.city && st.checkin && st.checkout){ applyStateToForm(st); city=st.city; ci=st.checkin; co=st.checkout; }
      }
      if(city && ci && co){ load(); }
    })();
  })();
  </script>
</body>
</html>
