<?php
// app/Views/agent/taxi_detail_new.php
// Enterprise-level professional taxi booking page
$pageTitle = htmlspecialchars((string)($ride['name'] ?? 'Transfer Booking'));
require __DIR__ . '/partials/header.php';
?>

<div class="min-h-screen bg-gray-50">
  <!-- Navigation Bar -->
  <div class="bg-white border-b border-gray-200">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
      <div class="flex items-center justify-between h-16">
        <nav class="flex items-center space-x-4 text-sm text-gray-500">
          <a href="/b2b/agent" class="hover:text-gray-900">Dashboard</a>
          <span>/</span>
          <a href="/b2b/agent/taxi" class="hover:text-gray-900">Transfers</a>
          <span>/</span>
          <span class="text-gray-900 font-medium">Booking Details</span>
        </nav>
        <a href="/b2b/agent/taxi" class="inline-flex items-center px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50">
          ← Back to Transfers
        </a>
      </div>
    </div>
  </div>

  <!-- Main Content -->
  <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
      
      <!-- Vehicle Summary Card -->
      <div class="lg:col-span-1">
        <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden sticky top-8">
          <!-- Vehicle Image -->
          <?php $thumb = (string)($ride['thumbnail_url'] ?? ($ride['cover_path'] ?? '')); ?>
          <div class="aspect-w-16 aspect-h-9">
            <img src="<?= $thumb !== '' ? htmlspecialchars($thumb) : ('https://picsum.photos/seed/'.urlencode((string)($ride['id'] ?? rand(1,9999))).'/400/240') ?>" 
                 alt="Vehicle" class="w-full h-48 object-cover">
          </div>
          
          <!-- Vehicle Details -->
          <div class="p-6">
            <div class="flex items-center justify-between mb-4">
              <h3 class="text-lg font-semibold text-gray-900"><?= htmlspecialchars((string)($ride['name'] ?? 'Transfer')) ?></h3>
              <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                Available
              </span>
            </div>
            
            <div class="space-y-3 mb-6">
              <div class="flex items-center text-sm text-gray-600">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path>
                </svg>
                <?= htmlspecialchars((string)($ride['vehicle_type'] ?? '')) ?> • <?= (int)($ride['capacity'] ?? 0) ?> passengers
              </div>
            </div>

            <!-- Service Features -->
            <div class="border-t border-gray-200 pt-4 mb-6">
              <h4 class="text-sm font-medium text-gray-900 mb-3">Included Services</h4>
              <ul class="space-y-2 text-sm text-gray-600">
                <li class="flex items-center">
                  <svg class="w-4 h-4 mr-2 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"></path>
                  </svg>
                  Professional Driver
                </li>
                <li class="flex items-center">
                  <svg class="w-4 h-4 mr-2 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"></path>
                  </svg>
                  Door-to-Door Service
                </li>
                <li class="flex items-center">
                  <svg class="w-4 h-4 mr-2 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"></path>
                  </svg>
                  Free Cancellation
                </li>
                <li class="flex items-center">
                  <svg class="w-4 h-4 mr-2 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"></path>
                  </svg>
                  Instant Confirmation
                </li>
              </ul>
            </div>

            <!-- Pricing -->
            <div class="border-t border-gray-200 pt-4">
              <div class="flex items-center justify-between mb-2">
                <span class="text-sm text-gray-600">Agent Price</span>
                <span class="text-2xl font-bold text-gray-900" id="totalPrice">
                  ฿<?= number_format((float) (($pricing['agent_price'] ?? null) !== null ? $pricing['agent_price'] : ($ride['base_price'] ?? 0)), 0) ?>
                </span>
              </div>
              <div class="flex items-center justify-between text-xs text-gray-500">
                <span>Customer Price</span>
                <span>฿<?= number_format((float) (($pricing['customer_price'] ?? null) !== null ? $pricing['customer_price'] : ($ride['base_price'] ?? 0)), 0) ?></span>
              </div>
            </div>
          </div>
        </div>
      </div>

      <!-- Booking Form -->
      <div class="lg:col-span-2">
        <div class="bg-white rounded-lg shadow-sm border border-gray-200">
          <div class="px-6 py-4 border-b border-gray-200">
            <h2 class="text-xl font-semibold text-gray-900">Booking Details</h2>
            <p class="mt-1 text-sm text-gray-600">Complete your transfer booking information</p>
          </div>

          <form id="bookingForm" method="post" action="/b2b/agent/taxi/payment" class="p-6 space-y-6">
            <input type="hidden" name="taxi_id" value="<?= (int)($ride['id'] ?? 0) ?>">
            <input type="hidden" name="from" id="fromHidden" value="">
            <input type="hidden" name="to" id="toHidden" value="">
            <input type="hidden" name="stops_json" id="stopsJson" value="">
            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf) ?>">

            <!-- Trip Information -->
            <div class="space-y-4">
              <h3 class="text-lg font-medium text-gray-900">Trip Information</h3>
              
              <!-- Date and Time -->
              <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Departure Date</label>
                  <input type="date" name="date" id="dateInput" required
                         class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Departure Time</label>
                  <input type="time" name="time" required
                         class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                </div>
              </div>

              <!-- Passengers -->
              <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Number of Passengers</label>
                <div class="flex items-center space-x-3">
                  <button type="button" onclick="changePax(-1)" class="w-8 h-8 rounded-full border border-gray-300 flex items-center justify-center hover:bg-gray-50">-</button>
                  <input type="number" name="pax" id="paxInput" value="1" min="1" max="<?= (int)($ride['capacity'] ?? 0) ?>" readonly
                         class="w-16 text-center px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  <button type="button" onclick="changePax(1)" class="w-8 h-8 rounded-full border border-gray-300 flex items-center justify-center hover:bg-gray-50">+</button>
                  <span class="text-sm text-gray-500">Max: <?= (int)($ride['capacity'] ?? 0) ?></span>
                </div>
              </div>
            </div>

            <!-- Pickup Location -->
            <div class="space-y-4">
              <h3 class="text-lg font-medium text-gray-900">Pickup Location</h3>
              <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Location Type</label>
                  <select name="pickup_type" id="pickupType" required
                          class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                    <option value="airport">Airport</option>
                    <option value="hotel">Hotel</option>
                    <option value="apartment">Apartment</option>
                    <option value="address">Address</option>
                  </select>
                </div>
                <div id="pickupDetails">
                  <div id="pickupAirport">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Airport Code</label>
                    <input type="text" name="pickup_airport_code" placeholder="e.g., BKK"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  </div>
                  <div id="pickupHotel" class="hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Hotel Name</label>
                    <input type="text" name="pickup_hotel_name" placeholder="Hotel name"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  </div>
                  <div id="pickupApartment" class="hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Apartment Name</label>
                    <input type="text" name="pickup_apartment_name" placeholder="Apartment name"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  </div>
                  <div id="pickupAddress" class="hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Address</label>
                    <input type="text" name="pickup_address" placeholder="Full address"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  </div>
                </div>
              </div>
            </div>

            <!-- Drop-off Location -->
            <div class="space-y-4">
              <h3 class="text-lg font-medium text-gray-900">Drop-off Location</h3>
              <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Location Type</label>
                  <select name="drop_type" id="dropType" required
                          class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                    <option value="airport">Airport</option>
                    <option value="hotel">Hotel</option>
                    <option value="apartment">Apartment</option>
                    <option value="address">Address</option>
                  </select>
                </div>
                <div id="dropDetails">
                  <div id="dropAirport">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Airport Code</label>
                    <input type="text" name="drop_airport_code" placeholder="e.g., BKK"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  </div>
                  <div id="dropHotel" class="hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Hotel Name</label>
                    <input type="text" name="drop_hotel_name" placeholder="Hotel name"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  </div>
                  <div id="dropApartment" class="hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Apartment Name</label>
                    <input type="text" name="drop_apartment_name" placeholder="Apartment name"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  </div>
                  <div id="dropAddress" class="hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Address</label>
                    <input type="text" name="drop_address" placeholder="Full address"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                  </div>
                </div>
              </div>
            </div>

            <!-- Customer Information -->
            <div class="space-y-4">
              <h3 class="text-lg font-medium text-gray-900">Customer Information</h3>
              <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Full Name *</label>
                  <input type="text" name="customer_name" required
                         class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                  <input type="email" name="customer_email"
                         class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                </div>
                <div>
                  <label class="block text-sm font-medium text-gray-700 mb-2">Phone</label>
                  <input type="tel" name="customer_phone"
                         class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                </div>
              </div>
            </div>

            <!-- Additional Notes -->
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-2">Special Instructions (Optional)</label>
              <textarea name="notes" rows="3" placeholder="Any special requests or instructions for the driver..."
                        class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 resize-none"></textarea>
            </div>

            <!-- Route Summary & Submit -->
            <div class="border-t border-gray-200 pt-6">
              <div class="flex items-center justify-between">
                <div>
                  <p class="text-sm text-gray-600">Route</p>
                  <p class="font-medium text-gray-900" id="routeSummary">Select pickup and drop-off locations</p>
                </div>
                <button type="submit" id="submitBtn"
                        class="inline-flex items-center px-6 py-3 border border-transparent text-base font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed">
                  <span id="submitText">Continue to Payment</span>
                  <svg class="ml-2 -mr-1 w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7l5 5m0 0l-5 5m5-5H6"></path>
                  </svg>
                </button>
              </div>
            </div>
          </form>
        </div>
      </div>
    </div>
  </div>
</div>

<script>
// Enterprise-level JavaScript - Clean and Professional
(function() {
  'use strict';

  // Passenger count management
  window.changePax = function(delta) {
    const input = document.getElementById('paxInput');
    const current = parseInt(input.value) || 1;
    const max = parseInt(input.getAttribute('max')) || 10;
    const newValue = Math.max(1, Math.min(max, current + delta));
    input.value = newValue;
  };

  // Location type switching
  function setupLocationSwitching() {
    const pickupType = document.getElementById('pickupType');
    const dropType = document.getElementById('dropType');

    function togglePickupFields(type) {
      ['pickupAirport', 'pickupHotel', 'pickupApartment', 'pickupAddress'].forEach(id => {
        document.getElementById(id).classList.add('hidden');
      });
      document.getElementById('pickup' + type.charAt(0).toUpperCase() + type.slice(1)).classList.remove('hidden');
    }

    function toggleDropFields(type) {
      ['dropAirport', 'dropHotel', 'dropApartment', 'dropAddress'].forEach(id => {
        document.getElementById(id).classList.add('hidden');
      });
      document.getElementById('drop' + type.charAt(0).toUpperCase() + type.slice(1)).classList.remove('hidden');
    }

    pickupType.addEventListener('change', () => togglePickupFields(pickupType.value));
    dropType.addEventListener('change', () => toggleDropFields(dropType.value));

    // Initialize
    togglePickupFields(pickupType.value);
    toggleDropFields(dropType.value);
  }

  // Route summary updates
  function updateRouteSummary() {
    const pickupType = document.getElementById('pickupType').value;
    const dropType = document.getElementById('dropType').value;
    
    let from = 'Pickup';
    let to = 'Drop-off';

    // Get pickup location
    if (pickupType === 'airport') {
      const code = document.querySelector('[name="pickup_airport_code"]').value.trim();
      from = code || 'Airport';
    } else if (pickupType === 'hotel') {
      const name = document.querySelector('[name="pickup_hotel_name"]').value.trim();
      from = name || 'Hotel';
    } else if (pickupType === 'apartment') {
      const name = document.querySelector('[name="pickup_apartment_name"]').value.trim();
      from = name || 'Apartment';
    } else if (pickupType === 'address') {
      const addr = document.querySelector('[name="pickup_address"]').value.trim();
      from = addr || 'Address';
    }

    // Get drop-off location
    if (dropType === 'airport') {
      const code = document.querySelector('[name="drop_airport_code"]').value.trim();
      to = code || 'Airport';
    } else if (dropType === 'hotel') {
      const name = document.querySelector('[name="drop_hotel_name"]').value.trim();
      to = name || 'Hotel';
    } else if (dropType === 'apartment') {
      const name = document.querySelector('[name="drop_apartment_name"]').value.trim();
      to = name || 'Apartment';
    } else if (dropType === 'address') {
      const addr = document.querySelector('[name="drop_address"]').value.trim();
      to = addr || 'Address';
    }

    document.getElementById('routeSummary').textContent = `${from} → ${to}`;
    document.getElementById('fromHidden').value = from;
    document.getElementById('toHidden').value = to;
  }

  // Form validation
  function validateForm() {
    const requiredFields = [
      { name: 'date', message: 'Please select departure date' },
      { name: 'time', message: 'Please select departure time' },
      { name: 'customer_name', message: 'Please enter customer name' }
    ];

    for (const field of requiredFields) {
      const input = document.querySelector(`[name="${field.name}"]`);
      if (!input.value.trim()) {
        alert(field.message);
        input.focus();
        return false;
      }
    }

    // Validate location-specific fields
    const pickupType = document.getElementById('pickupType').value;
    const dropType = document.getElementById('dropType').value;

    const locationFields = {
      airport: '_airport_code',
      hotel: '_hotel_name', 
      apartment: '_apartment_name',
      address: '_address'
    };

    const pickupField = document.querySelector(`[name="pickup${locationFields[pickupType]}"]`);
    const dropField = document.querySelector(`[name="drop${locationFields[dropType]}"]`);

    if (pickupField && !pickupField.value.trim()) {
      alert('Please enter pickup location details');
      pickupField.focus();
      return false;
    }

    if (dropField && !dropField.value.trim()) {
      alert('Please enter drop-off location details');
      dropField.focus();
      return false;
    }

    return true;
  }

  // Form submission
  function setupFormSubmission() {
    const form = document.getElementById('bookingForm');
    const submitBtn = document.getElementById('submitBtn');
    const submitText = document.getElementById('submitText');

    form.addEventListener('submit', function(e) {
      // Run validation; if it fails, allow the alert then block submission
      if (!validateForm()) {
        e.preventDefault();
        return;
      }
      // Set loading state and allow native submission to proceed
      submitBtn.disabled = true;
      submitText.textContent = 'Processing...';
    });
  }

  // Initialize everything when DOM is ready
  document.addEventListener('DOMContentLoaded', function() {
    setupLocationSwitching();
    setupFormSubmission();
    
    // Add event listeners for route summary updates
    document.addEventListener('input', updateRouteSummary);
    document.addEventListener('change', updateRouteSummary);
    
    // Initial route summary
    updateRouteSummary();
  });
})();
</script>

</body>
</html>
