<?php
// app/Views/agent/taxi_bookings.php
// Modern Enterprise-Grade Design

$title = $title ?? 'Taxi Bookings';
$rows = $rows ?? [];

// Normalize pagination
$pager = $pager ?? null;
if (!$pager && isset($pagination)) {
  $pager = [
    'totalItems' => (int)($pagination['total'] ?? count($rows)),
    'currentPage' => (int)($pagination['page'] ?? 1),
    'perPage' => (int)($pagination['per_page'] ?? 25),
    'totalPages' => (int)($pagination['pages'] ?? 1),
  ];
}
if (!$pager) {
  $pager = [ 'totalItems' => (int)count($rows), 'currentPage' => 1, 'perPage' => 25, 'totalPages' => 1 ];
}

$csrf = $csrf ?? ($_SESSION['csrf'] ?? '');
$active = 'bookings';
require __DIR__ . '/partials/header.php';
?>

<!-- Modern Enterprise Header with Gradient Background -->
<div class="bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 min-h-screen">
  <div class="max-w-7xl mx-auto px-6 py-8">
    
    <!-- Premium Header Section -->
    <div class="bg-white rounded-2xl shadow-xl border border-slate-200/60 p-8 mb-8">
      <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-6">
        <div class="flex items-center space-x-4">
          <div class="w-16 h-16 bg-gradient-to-br from-blue-600 to-indigo-700 rounded-2xl flex items-center justify-center shadow-lg">
            <svg xmlns="http://www.w3.org/2000/svg" class="h-8 w-8 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
            </svg>
          </div>
          <div>
            <h1 class="text-4xl font-bold bg-gradient-to-r from-slate-800 to-slate-600 bg-clip-text text-transparent">
              Taxi Bookings
            </h1>
            <p class="text-slate-600 text-lg mt-1">Comprehensive booking management & analytics dashboard</p>
            <div class="flex items-center space-x-4 mt-3">
              <div class="flex items-center space-x-2 text-sm text-slate-500">
                <div class="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                <span>Live Updates</span>
              </div>
              <div class="text-sm text-slate-500">
                Last updated: <?= date('M j, Y \a\t g:i A') ?>
              </div>
            </div>
          </div>
        </div>
        
        <div class="flex flex-col sm:flex-row gap-3">
          <button class="btn bg-gradient-to-r from-emerald-600 to-green-600 hover:from-emerald-700 hover:to-green-700 text-white border-0 shadow-lg hover:shadow-xl transition-all duration-300">
            <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
            </svg>
            Export Report
          </button>
          <a href="/b2b/agent/taxi" class="btn bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white border-0 shadow-lg hover:shadow-xl transition-all duration-300">
            <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
            </svg>
            New Booking
          </a>
        </div>
      </div>
    </div>

    <!-- Advanced Analytics Cards -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
      <div class="bg-white rounded-xl shadow-lg border border-slate-200/60 p-6 hover:shadow-xl transition-all duration-300">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-slate-600 text-sm font-medium">Total Bookings</p>
            <p class="text-3xl font-bold text-slate-800 mt-1"><?= number_format($pager['totalItems']) ?></p>
            <p class="text-emerald-600 text-sm mt-1">
              <span class="inline-flex items-center">
                <svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M3.293 9.707a1 1 0 010-1.414l6-6a1 1 0 011.414 0l6 6a1 1 0 01-1.414 1.414L11 5.414V17a1 1 0 11-2 0V5.414L4.707 9.707a1 1 0 01-1.414 0z" clip-rule="evenodd"></path>
                </svg>
                +12% this month
              </span>
            </p>
          </div>
          <div class="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl flex items-center justify-center">
            <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path>
            </svg>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-xl shadow-lg border border-slate-200/60 p-6 hover:shadow-xl transition-all duration-300">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-slate-600 text-sm font-medium">Active Trips</p>
            <p class="text-3xl font-bold text-slate-800 mt-1">24</p>
            <p class="text-amber-600 text-sm mt-1">
              <span class="inline-flex items-center">
                <svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clip-rule="evenodd"></path>
                </svg>
                In progress
              </span>
            </p>
          </div>
          <div class="w-12 h-12 bg-gradient-to-br from-amber-500 to-orange-600 rounded-xl flex items-center justify-center">
            <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-xl shadow-lg border border-slate-200/60 p-6 hover:shadow-xl transition-all duration-300">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-slate-600 text-sm font-medium">Refunds Today</p>
            <p class="text-3xl font-bold text-slate-800 mt-1">฿<?= number_format($dailyRefunds['amount'] ?? 0, 2) ?></p>
            <p class="<?= ($dailyRefunds['is_positive'] ?? false) ? 'text-rose-600' : 'text-emerald-600' ?> text-sm mt-1">
              <span class="inline-flex items-center">
                <?php $arrowDown = !($dailyRefunds['is_positive'] ?? false); ?>
                <svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M3.293 9.707a1 1 0 010-1.414l6-6a1 1 0 011.414 0l6 6a1 1 0 01-1.414 1.414L11 5.414V17a1 1 0 11-2 0V5.414L4.707 9.707a1 1 0 01-1.414 0z" clip-rule="evenodd" <?= $arrowDown ? 'transform="rotate(180 10 10)"' : '' ?>></path>
                </svg>
                <?= isset($dailyRefunds['change_percent']) ? abs($dailyRefunds['change_percent']) . '% vs yesterday' : 'vs yesterday' ?>
              </span>
            </p>
            <p class="text-slate-500 text-xs mt-1">All-time refunds: ฿<?= number_format($totalRefunds ?? 0, 2) ?></p>
            <div class="mt-3">
              <a href="/b2b/agent/taxi/refunds" class="inline-flex items-center px-3 py-1.5 bg-rose-50 hover:bg-rose-100 text-rose-700 text-xs font-medium rounded-lg border border-rose-200 transition-all duration-200">
                <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                View Refund History
              </a>
            </div>
          </div>
          <div class="w-12 h-12 bg-gradient-to-br from-rose-500 to-red-600 rounded-xl flex items-center justify-center">
            <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0 2.08-.402 2.599-1"></path>
            </svg>
          </div>
        </div>
      </div>

      <div class="bg-white rounded-xl shadow-lg border border-slate-200/60 p-6 hover:shadow-xl transition-all duration-300">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-slate-600 text-sm font-medium">Wallet Balance</p>
            <p class="text-3xl font-bold text-slate-800 mt-1">฿<?= number_format($walletBalance['available'] ?? 0, 2) ?></p>
            <p class="text-slate-600 text-sm mt-1">
              <span class="inline-flex items-center">
                <svg class="w-3 h-3 mr-1 text-amber-500" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M4 4a2 2 0 012-2h4.586A2 2 0 0112 2.586L15.414 6A2 2 0 0116 7.414V16a2 2 0 01-2 2H6a2 2 0 01-2-2V4z" clip-rule="evenodd"></path>
                </svg>
                <?php if (($walletBalance['on_hold'] ?? 0) > 0): ?>
                  ฿<?= number_format($walletBalance['on_hold'], 2) ?> on hold
                <?php else: ?>
                  Available
                <?php endif; ?>
              </span>
            </p>
            <p class="text-slate-500 text-xs mt-1">Today spent: ฿<?= number_format($dailySpending['amount'] ?? 0, 2) ?></p>
            <div class="mt-3">
              <a href="/b2b/agent/wallet/history" class="inline-flex items-center px-3 py-1.5 bg-purple-50 hover:bg-purple-100 text-purple-700 text-xs font-medium rounded-lg border border-purple-200 transition-all duration-200">
                <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path>
                </svg>
                View Wallet History
              </a>
            </div>
          </div>
          <div class="w-12 h-12 bg-gradient-to-br from-purple-500 to-indigo-600 rounded-xl flex items-center justify-center">
            <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z"></path>
            </svg>
          </div>
        </div>
      </div>
    </div>

    <!-- Advanced Filter Panel -->
    <div class="bg-white rounded-2xl shadow-xl border border-slate-200/60 p-6 mb-8">
      <div class="flex items-center justify-between mb-6">
        <h3 class="text-xl font-semibold text-slate-800 flex items-center">
          <svg class="w-5 h-5 mr-2 text-slate-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2.586a1 1 0 01-.293.707l-6.414 6.414a1 1 0 00-.293.707V17l-4 4v-6.586a1 1 0 00-.293-.707L3.293 7.414A1 1 0 013 6.707V4z"></path>
          </svg>
          Advanced Filters
        </h3>
        <button id="resetFilters" class="text-slate-500 hover:text-slate-700 text-sm font-medium transition-colors duration-200">Reset All</button>
      </div>
      
      <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-6 gap-4">
        <div class="lg:col-span-2">
          <label class="block text-sm font-medium text-slate-700 mb-2">Search Bookings</label>
          <div class="relative">
            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
              <svg class="h-5 w-5 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
              </svg>
            </div>
            <input id="bookingSearch" type="text" class="block w-full pl-10 pr-3 py-3 border border-slate-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-200" placeholder="Search by booking code, customer, vehicle..." />
          </div>
        </div>

        <div>
          <label class="block text-sm font-medium text-slate-700 mb-2">Status</label>
          <select id="bookingStatus" class="block w-full px-3 py-3 border border-slate-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-200">
            <option value="">All Status</option>
            <option value="confirmed">✅ Confirmed</option>
            <option value="pending">⏳ Pending</option>
            <option value="cancelled">❌ Cancelled</option>
            <option value="completed">✅ Completed</option>
            <option value="refunded">💰 Refunded</option>
          </select>
        </div>

        <div>
          <label class="block text-sm font-medium text-slate-700 mb-2">Date From</label>
          <input id="bookingDateFrom" type="date" class="block w-full px-3 py-3 border border-slate-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-200" />
        </div>

        <div>
          <label class="block text-sm font-medium text-slate-700 mb-2">Date To</label>
          <input id="bookingDateTo" type="date" class="block w-full px-3 py-3 border border-slate-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-200" />
        </div>

        <div class="flex items-center space-x-3">
          <label class="text-sm font-medium text-slate-700">Show entries:</label>
          <select class="select select-bordered select-sm w-24 bg-white border-slate-300 focus:border-blue-500 focus:ring-2 focus:ring-blue-200" onchange="updateRowsPerPage(this.value)">
            <option value="10" <?= ($filters['per_page'] ?? 10) == 10 ? 'selected' : '' ?>>10</option>
            <option value="25" <?= ($filters['per_page'] ?? 10) == 25 ? 'selected' : '' ?>>25</option>
            <option value="50" <?= ($filters['per_page'] ?? 10) == 50 ? 'selected' : '' ?>>50</option>
            <option value="100" <?= ($filters['per_page'] ?? 10) == 100 ? 'selected' : '' ?>>100</option>
            <option value="200" <?= ($filters['per_page'] ?? 10) == 200 ? 'selected' : '' ?>>200</option>
          </select>
        </div>
      </div>

      <div class="flex flex-wrap gap-3 mt-6 pt-6 border-t border-slate-200">
        <button id="applyFilters" class="inline-flex items-center px-6 py-3 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white text-sm font-semibold rounded-lg shadow-md hover:shadow-lg transition-all duration-200">
          <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2.586a1 1 0 01-.293.707l-6.414 6.414a1 1 0 00-.293.707V17l-4 4v-6.586a1 1 0 00-.293-.707L3.293 7.414A1 1 0 013 6.707V4z"></path>
          </svg>
          Apply Filters
        </button>
        <button id="bookingExportCsv" class="inline-flex items-center px-4 py-2 bg-gradient-to-r from-slate-600 to-slate-700 hover:from-slate-700 hover:to-slate-800 text-white text-sm font-medium rounded-lg shadow-md hover:shadow-lg transition-all duration-200">
          <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
          </svg>
          Export CSV
        </button>
        <button class="inline-flex items-center px-4 py-2 bg-gradient-to-r from-green-600 to-emerald-600 hover:from-green-700 hover:to-emerald-700 text-white text-sm font-medium rounded-lg shadow-md hover:shadow-lg transition-all duration-200">
          <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 17v-2m3 2v-4m3 4v-6m2 10H7a4 4 0 01-4-4V5a4 4 0 014-4h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a4 4 0 01-4 4z"></path>
          </svg>
          Export Excel
        </button>
        <button class="inline-flex items-center px-4 py-2 bg-gradient-to-r from-purple-600 to-indigo-600 hover:from-purple-700 hover:to-indigo-700 text-white text-sm font-medium rounded-lg shadow-md hover:shadow-lg transition-all duration-200">
          <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 17h2a2 2 0 002-2v-4a2 2 0 00-2-2H5a2 2 0 00-2 2v4a2 2 0 002 2h2m2 4h6a2 2 0 002-2v-4a2 2 0 00-2-2H9a2 2 0 00-2 2v4a2 2 0 002 2zm8-12V5a2 2 0 00-2-2H9a2 2 0 00-2 2v4h10z"></path>
          </svg>
          Print Report
        </button>
      </div>
    </div>

    <!-- Premium Data Table -->
    <div class="bg-white rounded-2xl shadow-xl border border-slate-200/60 overflow-hidden">
      <div class="px-6 py-4 bg-gradient-to-r from-slate-50 to-slate-100 border-b border-slate-200">
        <div class="flex items-center justify-between">
          <h3 class="text-lg font-semibold text-slate-800 flex items-center">
            <svg class="w-5 h-5 mr-2 text-slate-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-3 7h3m-3 4h3m-6-4h.01M9 16h.01"></path>
            </svg>
            Booking Records
          </h3>
          <div class="flex items-center space-x-2 text-sm text-slate-600">
            <div class="flex items-center space-x-1">
              <div class="w-2 h-2 bg-green-500 rounded-full"></div>
              <span>Live Data</span>
            </div>
          </div>
        </div>
      </div>
      
      <div class="overflow-x-auto">
        <table class="w-full">
          <thead class="bg-gradient-to-r from-slate-100 to-slate-50">
            <tr class="border-b border-slate-200">
              <th class="px-6 py-4 text-left text-xs font-semibold text-slate-600 uppercase tracking-wider w-16">#</th>
              <th class="px-6 py-4 text-left text-xs font-semibold text-slate-600 uppercase tracking-wider">Booking Details</th>
              <th class="px-6 py-4 text-left text-xs font-semibold text-slate-600 uppercase tracking-wider">Trip Information</th>
              <th class="px-6 py-4 text-left text-xs font-semibold text-slate-600 uppercase tracking-wider">Vehicle & Vendor</th>
              <th class="px-6 py-4 text-left text-xs font-semibold text-slate-600 uppercase tracking-wider">Financial</th>
              <th class="px-6 py-4 text-center text-xs font-semibold text-slate-600 uppercase tracking-wider">Status</th>
              <th class="px-6 py-4 text-right text-xs font-semibold text-slate-600 uppercase tracking-wider">Actions</th>
            </tr>
          </thead>
          <tbody class="divide-y divide-slate-200">
            <?php foreach (($rows ?? []) as $idx => $r): 
              $status = strtolower($r['status'] ?? '');
              $isCompleted = !empty($r['completed_at']);
              $auditStatus = strtolower($r['audit_status'] ?? '');
              
              // Modern status styling
              $statusConfig = [
                'confirmed' => ['bg' => 'bg-blue-50 hover:bg-blue-100', 'badge' => 'bg-blue-100 text-blue-800', 'text' => $isCompleted ? 'Completed' : 'Confirmed'],
                'completed' => ['bg' => 'bg-emerald-50 hover:bg-emerald-100', 'badge' => 'bg-emerald-100 text-emerald-800', 'text' => 'Completed'],
                'cancelled' => ['bg' => 'bg-red-50 hover:bg-red-100', 'badge' => 'bg-red-100 text-red-800', 'text' => 'Cancelled'],
                'refunded' => ['bg' => 'bg-gray-50 hover:bg-gray-100', 'badge' => 'bg-gray-100 text-gray-800', 'text' => 'Refunded'],
                'pending' => ['bg' => 'bg-amber-50 hover:bg-amber-100', 'badge' => 'bg-amber-100 text-amber-800', 'text' => 'Pending']
              ];
              
              $config = $statusConfig[$status] ?? $statusConfig['pending'];
              if (in_array($auditStatus, ['picked_up','in_progress','on_the_way'])) {
                $config = ['bg' => 'bg-orange-50 hover:bg-orange-100', 'badge' => 'bg-orange-100 text-orange-800', 'text' => ucfirst(str_replace('_',' ', $auditStatus))];
              }
            ?>
              <tr class="<?= $config['bg'] ?> transition-colors duration-200">
                <td class="px-6 py-4 text-sm font-medium text-slate-900">
                  <?= ($idx + 1) + (($pager['currentPage'] - 1) * ($pager['perPage'])) ?>
                </td>
                
                <td class="px-6 py-4">
                  <div class="flex flex-col">
                    <a href="/b2b/agent/taxi/booking/view?id=<?= (int)($r['id'] ?? 0) ?>" class="text-lg font-semibold text-blue-600 hover:text-blue-800 transition-colors duration-200">
                      <?= htmlspecialchars($r['booking_code'] ?? 'N/A') ?>
                    </a>
                    <div class="text-sm text-slate-500 mt-1">ID: #<?= (int)($r['id'] ?? 0) ?></div>
                    <div class="text-xs text-slate-400 mt-1">
                      Created: <?= !empty($r['created_at']) ? date('M j, g:i A', strtotime($r['created_at'])) : '-' ?>
                    </div>
                  </div>
                </td>
                
                <td class="px-6 py-4">
                  <div class="flex flex-col space-y-1">
                    <div class="text-sm font-medium text-slate-900">
                      <?= !empty($r['trip_date']) ? date('M j, Y', strtotime($r['trip_date'])) : '-' ?>
                    </div>
                    <div class="text-sm text-slate-600">
                      🕒 <?= !empty($r['pickup_time']) ? date('g:i A', strtotime($r['pickup_time'])) : 'TBD' ?>
                    </div>
                    <div class="flex items-center space-x-2 mt-2">
                      <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-slate-100 text-slate-800">
                        👥 <?= (int)($r['pax'] ?? 1) ?> pax
                      </span>
                    </div>
                  </div>
                </td>
                
                <td class="px-6 py-4">
                  <div class="flex flex-col space-y-3">
                    <!-- Pickup Location -->
                    <div class="bg-slate-50 p-3 rounded-lg border border-slate-100">
                      <div class="flex items-start">
                        <div class="flex-shrink-0 mt-0.5">
                          <svg class="h-4 w-4 text-green-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                          </svg>
                        </div>
                        <div class="ml-2 flex-1 min-w-0">
                          <div class="text-xs font-semibold text-green-700 uppercase tracking-wider mb-1">Pickup Location</div>
                          <div class="text-sm font-medium text-slate-800 truncate"><?= htmlspecialchars($r['from_text'] ?? 'Not specified') ?></div>
                          
                          <?php if (!empty($r['pickup_landmark']) && $r['pickup_landmark'] !== 'N/A'): ?>
                          <div class="mt-1 flex items-center text-xs text-slate-500">
                            <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                            </svg>
                            <?= htmlspecialchars($r['pickup_landmark']) ?>
                          </div>
                          <?php endif; ?>
                          
                          <?php if (!empty($r['pickup_instructions'])): ?>
                          <div class="mt-1 text-xs text-slate-500 flex">
                            <svg class="w-3 h-3 mr-1 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                            <span class="truncate"><?= htmlspecialchars($r['pickup_instructions']) ?></span>
                          </div>
                          <?php endif; ?>
                        </div>
                      </div>
                    </div>
                    
                    <!-- Dropoff Location -->
                    <div class="bg-slate-50 p-3 rounded-lg border border-slate-100">
                      <div class="flex items-start">
                        <div class="flex-shrink-0 mt-0.5">
                          <svg class="h-4 w-4 text-blue-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                          </svg>
                        </div>
                        <div class="ml-2 flex-1 min-w-0">
                          <div class="text-xs font-semibold text-blue-700 uppercase tracking-wider mb-1">Dropoff Location</div>
                          <div class="text-sm font-medium text-slate-800 truncate"><?= htmlspecialchars($r['to_text'] ?? 'Not specified') ?></div>
                          
                          <?php if (!empty($r['dropoff_landmark']) && $r['dropoff_landmark'] !== 'N/A'): ?>
                          <div class="mt-1 flex items-center text-xs text-slate-500">
                            <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                            </svg>
                            <?= htmlspecialchars($r['dropoff_landmark']) ?>
                          </div>
                          <?php endif; ?>
                          
                          <?php if (!empty($r['dropoff_instructions'])): ?>
                          <div class="mt-1 text-xs text-slate-500 flex">
                            <svg class="w-3 h-3 mr-1 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                            <span class="truncate"><?= htmlspecialchars($r['dropoff_instructions']) ?></span>
                          </div>
                          <?php endif; ?>
                        </div>
                      </div>
                    </div>
                    
                    <!-- Vehicle & Trip Info -->
                    <div class="pt-2">
                      <div class="flex items-center justify-between">
                        <div class="text-sm font-medium text-slate-900">
                          <?= htmlspecialchars($r['ride_name'] ?? 'Standard Transfer') ?>
                        </div>
                    <?php if (!empty($r['vehicle_type'])): 
                      $vt = strtoupper((string)$r['vehicle_type']);
                      $vehicleColors = [
                        'SEDAN' => 'bg-blue-100 text-blue-800',
                        'SUV' => 'bg-purple-100 text-purple-800',
                        'VAN' => 'bg-green-100 text-green-800',
                        'BUS' => 'bg-orange-100 text-orange-800'
                      ];
                      $vehicleColor = 'bg-gray-100 text-gray-800';
                      foreach($vehicleColors as $type => $color) {
                        if (str_contains($vt, $type)) { $vehicleColor = $color; break; }
                      }
                    ?>
                      <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium <?= $vehicleColor ?>">
                        🚗 <?= htmlspecialchars($vt) ?>
                      </span>
                    <?php endif; ?>
                    <?php if (!empty($r['vendor_name'])): ?>
                      <div class="text-xs text-slate-500">
                        🏢 <?= htmlspecialchars($r['vendor_name']) ?>
                      </div>
                    <?php endif; ?>
                  </div>
                </td>
                
                <td class="px-6 py-4">
                  <div class="flex flex-col space-y-1">
                    <div class="text-lg font-bold text-slate-900">
                      <?= number_format((float)($r['amount_total'] ?? 0), 2) ?> <?= htmlspecialchars($r['currency'] ?? 'THB') ?>
                    </div>
                    <?php if (!empty($r['payment_method'])): ?>
                      <div class="text-xs text-slate-500 uppercase font-medium">
                        💳 <?= htmlspecialchars($r['payment_method']) ?>
                      </div>
                    <?php endif; ?>
                  </div>
                </td>
                
                <td class="px-6 py-4 text-center">
                  <div class="flex flex-col items-center space-y-2">
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium <?= $config['badge'] ?>">
                      <?= htmlspecialchars($config['text']) ?>
                    </span>
                    <?php if (!empty($r['audit_status'])): ?>
                      <div class="text-xs text-slate-500 flex items-center">
                        <svg class="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                          <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clip-rule="evenodd"></path>
                        </svg>
                        <?= htmlspecialchars($r['audit_status']) ?>
                      </div>
                    <?php endif; ?>
                  </div>
                </td>
                
                <td class="px-6 py-4">
                  <div class="flex justify-end space-x-2">
                    <a href="/b2b/agent/taxi/voucher?id=<?= (int)($r['id'] ?? 0) ?>" class="inline-flex items-center px-3 py-2 border border-slate-300 shadow-sm text-sm leading-4 font-medium rounded-lg text-slate-700 bg-white hover:bg-slate-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 transition-all duration-200">
                      <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
                      </svg>
                      View
                    </a>
                    <?php if (strtolower($r['status'] ?? '') !== 'cancelled'): ?>
                      <button onclick="showCancelModal(<?= (int)($r['id'] ?? 0) ?>)" class="inline-flex items-center px-3 py-2 border border-red-300 shadow-sm text-sm leading-4 font-medium rounded-lg text-red-700 bg-white hover:bg-red-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 transition-all duration-200">
                        <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                        Cancel
                      </button>
                    <?php endif; ?>
                  </div>
                </td>
              </tr>
            <?php endforeach; ?>
            
            <?php if (empty($rows)): ?>
              <tr>
                <td colspan="7" class="px-6 py-12 text-center">
                  <div class="flex flex-col items-center space-y-4">
                    <div class="w-16 h-16 bg-slate-100 rounded-full flex items-center justify-center">
                      <svg class="w-8 h-8 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path>
                      </svg>
                    </div>
                    <div class="text-lg font-medium text-slate-900">No bookings found</div>
                    <div class="text-sm text-slate-500">Try adjusting your search filters or create a new booking</div>
                    <a href="/b2b/agent/taxi" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-lg shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                      Create New Booking
                    </a>
                  </div>
                </td>
              </tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>

    <!-- Premium Pagination Footer -->
    <div class="mt-8 bg-white rounded-2xl shadow-lg border border-slate-200/60 px-6 py-4">
      <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
        <div class="flex items-center space-x-4">
          <div class="text-sm text-slate-600 pagination-info">
            Showing <span class="font-semibold text-slate-900"><?= ($pager['totalItems'] ? ((($pager['currentPage'] - 1) * $pager['perPage']) + 1) : 0) ?></span> to 
            <span class="font-semibold text-slate-900"><?= min($pager['currentPage'] * $pager['perPage'], $pager['totalItems']) ?></span> of 
            <span class="font-semibold text-slate-900"><?= number_format($pager['totalItems']) ?></span> results
          </div>
          <div class="text-xs text-slate-500">
            (<?= $pager['perPage'] ?> entries per page)
          </div>
        </div>
        
        <?php if (($pager['totalPages'] ?? 1) > 1): ?>
          <?php 
            $totalPages = $pager['totalPages'];
            $currentPage = $pager['currentPage'];
            $startPage = max(1, $currentPage - 2);
            $endPage = min($totalPages, $currentPage + 2);
          ?>
          <nav class="flex items-center space-x-1">
            <?php if ($currentPage > 1): ?>
              <a href="?page=<?= $currentPage - 1 ?>&<?= http_build_query(array_filter($_GET, fn($k) => $k !== 'page', ARRAY_FILTER_USE_KEY)) ?>" 
                 class="inline-flex items-center px-3 py-2 text-sm font-medium text-slate-500 bg-white border border-slate-300 rounded-lg hover:bg-slate-50 hover:text-slate-700 transition-colors duration-200">
                <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                </svg>
                Previous
              </a>
            <?php endif; ?>
            
            <div class="flex items-center space-x-1">
              <?php if ($startPage > 1): ?>
                <a href="?page=1&<?= http_build_query(array_filter($_GET, fn($k) => $k !== 'page', ARRAY_FILTER_USE_KEY)) ?>" 
                   class="inline-flex items-center px-3 py-2 text-sm font-medium text-slate-500 bg-white border border-slate-300 rounded-lg hover:bg-slate-50 hover:text-slate-700 transition-colors duration-200">1</a>
                <?php if ($startPage > 2): ?>
                  <span class="px-2 text-slate-400">...</span>
                <?php endif; ?>
              <?php endif; ?>
              
              <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                <a href="?page=<?= $i ?>&<?= http_build_query(array_filter($_GET, fn($k) => $k !== 'page', ARRAY_FILTER_USE_KEY)) ?>" 
                   class="inline-flex items-center px-3 py-2 text-sm font-medium rounded-lg transition-colors duration-200 <?= $i === $currentPage ? 'bg-blue-600 text-white border border-blue-600' : 'text-slate-500 bg-white border border-slate-300 hover:bg-slate-50 hover:text-slate-700' ?>">
                  <?= $i ?>
                </a>
              <?php endfor; ?>
              
              <?php if ($endPage < $totalPages): ?>
                <?php if ($endPage < $totalPages - 1): ?>
                  <span class="px-2 text-slate-400">...</span>
                <?php endif; ?>
                <a href="?page=<?= $totalPages ?>&<?= http_build_query(array_filter($_GET, fn($k) => $k !== 'page', ARRAY_FILTER_USE_KEY)) ?>" 
                   class="inline-flex items-center px-3 py-2 text-sm font-medium text-slate-500 bg-white border border-slate-300 rounded-lg hover:bg-slate-50 hover:text-slate-700 transition-colors duration-200"><?= $totalPages ?></a>
              <?php endif; ?>
            </div>
            
            <?php if ($currentPage < $totalPages): ?>
              <a href="?page=<?= $currentPage + 1 ?>&<?= http_build_query(array_filter($_GET, fn($k) => $k !== 'page', ARRAY_FILTER_USE_KEY)) ?>" 
                 class="inline-flex items-center px-3 py-2 text-sm font-medium text-slate-500 bg-white border border-slate-300 rounded-lg hover:bg-slate-50 hover:text-slate-700 transition-colors duration-200">
                Next
                <svg class="w-4 h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
              </a>
            <?php endif; ?>
          </nav>
        <?php endif; ?>
      </div>
    </div>

</div>

<!-- Cancel Modal -->
<dialog id="cancelBookingModal" class="modal">
  <div class="modal-box">
    <h3 class="font-bold text-lg mb-2">Cancel Booking</h3>
    <form action="#" method="post" class="space-y-3">
      <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
      <input type="hidden" name="id" id="cancelBookingId">
      <div>
        <label class="label">Reason for Cancellation</label>
        <select class="select select-bordered w-full" id="cancelReason" name="reason" required>
          <option value="">Select a reason</option>
          <option value="customer_request">Customer Request</option>
          <option value="no_show">No Show</option>
          <option value="taxi_unavailable">Taxi Unavailable</option>
          <option value="weather">Weather Conditions</option>
          <option value="other">Other Reason</option>
        </select>
      </div>
      <div id="otherReasonContainer" class="hidden">
        <label class="label">Please specify</label>
        <textarea class="textarea textarea-bordered w-full" id="otherReason" name="other_reason" rows="3"></textarea>
      </div>
      <div class="modal-action">
        <button type="button" class="btn" onclick="document.getElementById('cancelBookingModal').close()">Close</button>
        <button type="submit" class="btn btn-error">Confirm Cancellation</button>
      </div>
    </form>
  </div>
</dialog>

<script>
// Advanced Filters and table functionality
(function(){
  // Get all filter elements
  const search = document.getElementById('bookingSearch');
  const statusFilter = document.getElementById('bookingStatus');
  const dateFrom = document.getElementById('bookingDateFrom');
  const dateTo = document.getElementById('bookingDateTo');
  const rowsSel = document.querySelector('select[onchange*="updateRowsPerPage"]');
  const countEl = document.querySelector('.pagination-info');
  const resetBtn = document.getElementById('resetFilters');
  
  // Get table and rows
  const table = document.querySelector('table');
  if (!table) return;
  const rows = table.querySelectorAll('tbody tr');

  // Initialize date filters with current URL parameters
  const urlParams = new URLSearchParams(window.location.search);
  if (dateFrom && urlParams.get('from')) {
    dateFrom.value = urlParams.get('from');
  }
  if (dateTo && urlParams.get('to')) {
    dateTo.value = urlParams.get('to');
  }
  if (search && urlParams.get('q')) {
    search.value = urlParams.get('q');
  }
  if (statusFilter && urlParams.get('status')) {
    statusFilter.value = urlParams.get('status');
  }

  function parseDateFromCell(cell){
    const text = cell?.textContent?.trim() || '';
    const d = new Date(text);
    return isNaN(d.getTime()) ? null : d;
  }

  function filter(){
    if (!search) return;
    
    const q = (search.value || '').toLowerCase();
    const st = (statusFilter?.value || '').toLowerCase();
    const from = dateFrom?.value ? new Date(dateFrom.value) : null;
    const to = dateTo?.value ? new Date(dateTo.value) : null;
    if (from) from.setHours(0,0,0,0);
    if (to) to.setHours(23,59,59,999);

    let visible = 0;
    rows.forEach(row => {
      const cells = row.querySelectorAll('td');
      if (!cells.length) return;
      
      // Updated cell indices for new table structure
      const bookingCode = (cells[1]?.textContent || '').toLowerCase();
      const tripInfo = (cells[2]?.textContent || '').toLowerCase();
      const vehicleInfo = (cells[3]?.textContent || '').toLowerCase();
      const statusText = (cells[5]?.textContent || '').toLowerCase();
      
      // Parse date from trip information column
      const dateMatch = cells[2]?.textContent?.match(/\w{3}\s+\d{1,2},\s+\d{4}/);
      const d = dateMatch ? new Date(dateMatch[0]) : null;

      const matchQ = !q || bookingCode.includes(q) || tripInfo.includes(q) || vehicleInfo.includes(q);
      const matchS = !st || statusText.includes(st);
      const matchD = !from && !to ? true : (d ? (!from || d >= from) && (!to || d <= to) : true);

      const show = matchQ && matchS && matchD;
      row.style.display = show ? '' : 'none';
      if (show) visible++;
    });

    if (countEl) {
      const total = <?= (int)$pager['totalItems'] ?>;
      const start = total ? ((<?= (int)$pager['currentPage'] ?> - 1) * <?= (int)$pager['perPage'] ?>) + 1 : 0;
      const end = Math.min(<?= (int)$pager['currentPage'] ?> * <?= (int)$pager['perPage'] ?>, total);
      
      // Update pagination info with filter results
      const spans = countEl.querySelectorAll('span');
      if (spans.length >= 3) {
        spans[0].textContent = start;
        spans[1].textContent = Math.min(end, visible);
        spans[2].textContent = total.toLocaleString();
      }
    }
  }

  // Add debounced search for better performance
  let searchTimeout;
  search?.addEventListener('input', function() {
    clearTimeout(searchTimeout);
    searchTimeout = setTimeout(() => {
      updateURLParams();
      filter();
    }, 300);
  });
  
  statusFilter?.addEventListener('change', function() {
    updateURLParams();
    filter();
  });
  
  dateFrom?.addEventListener('change', function() {
    updateURLParams();
    filter();
  });
  
  dateTo?.addEventListener('change', function() {
    updateURLParams();
    filter();
  });
  
  rowsSel?.addEventListener('change', function(e) {
    const url = new URL(window.location);
    url.searchParams.set('per_page', e.target.value);
    url.searchParams.set('page', '1');
    window.location.href = url.toString();
  });

  // Function to update URL parameters without page reload
  function updateURLParams() {
    const url = new URL(window.location);
    
    if (search?.value) {
      url.searchParams.set('q', search.value);
    } else {
      url.searchParams.delete('q');
    }
    
    if (statusFilter?.value) {
      url.searchParams.set('status', statusFilter.value);
    } else {
      url.searchParams.delete('status');
    }
    
    if (dateFrom?.value) {
      url.searchParams.set('from', dateFrom.value);
    } else {
      url.searchParams.delete('from');
    }
    
    if (dateTo?.value) {
      url.searchParams.set('to', dateTo.value);
    } else {
      url.searchParams.delete('to');
    }
    
    url.searchParams.set('page', '1'); // Reset to first page when filtering
    window.history.pushState({}, '', url);
  }

    // Enhanced pagination info update
    function updatePaginationInfo() {
      const totalItems = <?= $pager['totalItems'] ?? 0 ?>;
      const currentPage = <?= $pager['currentPage'] ?? 1 ?>;
      const perPage = <?= $pager['perPage'] ?? 10 ?>;
      const totalPages = <?= $pager['totalPages'] ?? 1 ?>;
      
      const startItem = totalItems > 0 ? ((currentPage - 1) * perPage) + 1 : 0;
      const endItem = Math.min(currentPage * perPage, totalItems);
      
      // Update any pagination info displays
      const paginationInfo = document.querySelector('.pagination-info');
      if (paginationInfo) {
        paginationInfo.textContent = `Showing ${startItem} to ${endItem} of ${totalItems.toLocaleString()} results`;
      }
    }

    // Initialize pagination info on page load
    document.addEventListener('DOMContentLoaded', function() {
      updatePaginationInfo();
    });

  // Reset filters functionality
  resetBtn?.addEventListener('click', function() {
    if (search) search.value = '';
    if (statusFilter) statusFilter.value = '';
    if (dateFrom) dateFrom.value = '';
    if (dateTo) dateTo.value = '';
    
    // Update URL to remove filter parameters
    const url = new URL(window.location);
    url.searchParams.delete('q');
    url.searchParams.delete('status');
    url.searchParams.delete('from');
    url.searchParams.delete('to');
    url.searchParams.set('page', '1');
    window.history.pushState({}, '', url);
    
    filter();
  });

  // Apply Filters button functionality
  const applyBtn = document.getElementById('applyFilters');
  applyBtn?.addEventListener('click', function() {
    const url = new URL(window.location);
    
    if (search?.value) {
      url.searchParams.set('q', search.value);
    } else {
      url.searchParams.delete('q');
    }
    
    if (statusFilter?.value) {
      url.searchParams.set('status', statusFilter.value);
    } else {
      url.searchParams.delete('status');
    }
    
    if (dateFrom?.value) {
      url.searchParams.set('from', dateFrom.value);
    } else {
      url.searchParams.delete('from');
    }
    
    if (dateTo?.value) {
      url.searchParams.set('to', dateTo.value);
    } else {
      url.searchParams.delete('to');
    }
    
    url.searchParams.set('page', '1');
    window.location.href = url.toString();
  });

  // Initialize filters on page load
  filter();
})();

function showCancelModal(id){
  const dlg = document.getElementById('cancelBookingModal');
  const idInput = document.getElementById('cancelBookingId');
  if (idInput) idInput.value = id;
  dlg.showModal();
}

// Toggle other reason
(function(){
  const sel = document.getElementById('cancelReason');
  const box = document.getElementById('otherReasonContainer');
  if (!sel || !box) return;
  sel.addEventListener('change', () => {
    if (sel.value === 'other') { box.classList.remove('hidden'); }
    else { box.classList.add('hidden'); const t = document.getElementById('otherReason'); if (t) t.value=''; }
  });
})();
</script>

<?php require __DIR__ . '/partials/footer.php'; ?>
