<?php
// app/Views/agent/taxi_booking_view.php
// Tailwind + DaisyUI detailed view for a single taxi booking (Agent)

$title    = $title ?? 'Taxi Booking';
$booking  = $booking ?? [];
$events   = $events ?? [];
$payments = $payments ?? [];
$csrf     = $csrf ?? '';

require __DIR__ . '/partials/header.php';

$status = strtolower((string)($booking['status'] ?? ''));
$paymentStatus = strtolower((string)($booking['payment_status'] ?? ''));
$isCompleted = !empty($booking['completed_at']);

function badgeClass($status) {
  $status = strtolower((string)$status);
  return match ($status) {
    'confirmed' => 'badge-info',
    'completed' => 'badge-success',
    'cancelled' => 'badge-error',
    'refunded'  => 'badge-neutral',
    'pending', 'processing' => 'badge-warning',
    default => 'badge-ghost',
  };
}
?>

<div class="max-w-7xl mx-auto px-4 py-6">
  <!-- Breadcrumb and title -->
  <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-3 mb-6">
    <div>
      <div class="text-sm breadcrumbs">
        <ul>
          <li><a href="/b2b/agent">Dashboard</a></li>
          <li><a href="/b2b/agent/taxi/bookings">Taxi Bookings</a></li>
          <li>#<?= (int)($booking['id'] ?? 0) ?></li>
        </ul>
      </div>
      <h1 class="text-2xl font-bold flex items-center gap-2">
        <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-primary" viewBox="0 0 24 24" fill="currentColor"><path d="M18 10h-1V6a1 1 0 0 0-1-1H8a1 1 0 0 0-1 1v4H6a3 3 0 0 0-3 3v5a1 1 0 0 0 1 1h1a2 2 0 0 0 4 0h6a2 2 0 0 0 4 0h1a1 1 0 0 0 1-1v-5a3 3 0 0 0-3-3ZM9 7h6v3H9Z"/></svg>
        Taxi Booking #<?= (int)($booking['id'] ?? 0) ?>
      </h1>
      <div class="mt-1 flex items-center gap-2">
        <span class="badge <?= badgeClass($status) ?>"><?= ucfirst($status ?: 'N/A') ?></span>
        <span class="badge <?= badgeClass($paymentStatus) ?>"><?= ucfirst($paymentStatus ?: 'N/A') ?></span>
        <?php if ($isCompleted): ?>
          <span class="badge badge-success">Completed</span>
        <?php endif; ?>
      </div>
    </div>
    <div class="flex gap-2">
      <a href="/b2b/agent/taxi/bookings" class="btn btn-ghost">Back</a>
      <button class="btn btn-outline" onclick="window.print()">
        <svg xmlns="http://www.w3.org/2000/svg" class="h-4 w-4 mr-1" viewBox="0 0 24 24" fill="currentColor"><path d="M6 9V2h12v7H6Zm12 2a4 4 0 0 1 4 4v5h-4v2H6v-2H2v-5a4 4 0 0 1 4-4h12Zm-2 9v-5H8v5h8Z"/></svg>
        Print
      </button>
    </div>
  </div>

  <div class="grid grid-cols-1 lg:grid-cols-3 gap-5">
    <!-- Left: Main cards -->
    <div class="lg:col-span-2 space-y-5">
      <!-- Overview -->
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Booking Overview</h2>
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-2">
            <div class="space-y-3">
              <div>
                <div class="text-xs text-base-content/60">Booking Code</div>
                <div class="font-semibold"><?= htmlspecialchars((string)($booking['booking_code'] ?? '')) ?></div>
              </div>
              <div>
                <div class="text-xs text-base-content/60">Agent</div>
                <div class="font-semibold">#<?= (int)($booking['agent_id'] ?? 0) ?><?php if (!empty($booking['agent_name'])): ?> <span class="text-base-content/60">(<?= htmlspecialchars($booking['agent_name']) ?>)</span><?php endif; ?></div>
              </div>
              <div>
                <div class="text-xs text-base-content/60">Vendor</div>
                <div class="font-semibold">#<?= (int)($booking['vendor_id'] ?? 0) ?><?php if (!empty($booking['vendor_name'])): ?> <span class="text-base-content/60">(<?= htmlspecialchars((string)$booking['vendor_name']) ?>)</span><?php endif; ?></div>
              </div>
              <div>
                <div class="text-xs text-base-content/60">Passengers</div>
                <div class="font-semibold"><?= (int)($booking['pax'] ?? 1) ?></div>
              </div>
            </div>
            <div class="space-y-3">
              <div>
                <div class="text-xs text-base-content/60">Trip Date</div>
                <div class="font-semibold">
                  <?= !empty($booking['trip_date']) ? date('d M Y', strtotime((string)$booking['trip_date'])) : 'N/A' ?>
                  <?= !empty($booking['pickup_time']) ? '• ' . htmlspecialchars((string)$booking['pickup_time']) : '' ?>
                </div>
              </div>
              <div>
                <div class="text-xs text-base-content/60">Vehicle Type</div>
                <div class="font-semibold"><?= htmlspecialchars((string)($booking['vehicle_type'] ?? ($booking['ride_name'] ?? 'Standard'))) ?></div>
              </div>
              <div>
                <div class="text-xs text-base-content/60">Status</div>
                <div class="flex items-center gap-2">
                  <span class="badge <?= badgeClass($status) ?>"><?= ucfirst($status ?: 'N/A') ?></span>
                  <span class="badge <?= badgeClass($paymentStatus) ?>"><?= ucfirst($paymentStatus ?: 'N/A') ?></span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <!-- Customer Details -->
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Customer Details</h2>
          <div class="mt-2 grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <div class="text-xs text-base-content/60">Name</div>
              <div class="font-semibold"><?= htmlspecialchars((string)($booking['customer_name'] ?? 'Guest User')) ?></div>
            </div>
            <?php if (!empty($booking['customer_email'])): ?>
            <div>
              <div class="text-xs text-base-content/60">Email</div>
              <a class="link" href="mailto:<?= htmlspecialchars((string)$booking['customer_email']) ?>"><?= htmlspecialchars((string)$booking['customer_email']) ?></a>
            </div>
            <?php endif; ?>
            <?php if (!empty($booking['customer_phone'])): ?>
            <div>
              <div class="text-xs text-base-content/60">Phone</div>
              <a class="link" href="tel:<?= preg_replace('/[^0-9+]/', '', (string)$booking['customer_phone']) ?>"><?= htmlspecialchars((string)$booking['customer_phone']) ?></a>
            </div>
            <?php endif; ?>
          </div>
          <?php if (!empty($booking['customer_notes'])): ?>
          <div class="alert mt-4">
            <div>
              <span class="font-semibold mr-2">Customer Notes:</span>
              <span><?= nl2br(htmlspecialchars((string)$booking['customer_notes'])) ?></span>
            </div>
          </div>
          <?php endif; ?>
        </div>
      </div>

      <!-- Route & Pickup/Drop -->
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Route & Pickup/Drop</h2>
          <div class="mt-2 p-3 rounded-lg bg-base-200">
            <div class="flex items-center gap-3">
              <div class="text-center">
                <div class="badge badge-primary badge-outline">From</div>
              </div>
              <div class="font-semibold flex-1">
                <?= htmlspecialchars((string)($booking['from_text'] ?? 'N/A')) ?>
              </div>
              <div class="opacity-60">→</div>
              <div class="text-center">
                <div class="badge badge-primary badge-outline">To</div>
              </div>
              <div class="font-semibold flex-1">
                <?= htmlspecialchars((string)($booking['to_text'] ?? 'N/A')) ?>
              </div>
            </div>
          </div>

          <div class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-4">
            <!-- Pickup -->
            <div class="rounded-lg border p-4">
              <h3 class="font-semibold mb-2">Pickup Details</h3>
              <div class="space-y-2 text-sm">
                <?php if (!empty($booking['pickup_time'])): ?>
                <div><span class="opacity-60">Time:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['pickup_time']) ?></span></div>
                <?php endif; ?>
                <?php if (!empty($booking['pickup_airport_code'])): ?>
                <div><span class="opacity-60">Airport:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['pickup_airport_code']) ?></span></div>
                <?php endif; ?>
                <?php if (!empty($booking['pickup_hotel_name'])): ?>
                <div><span class="opacity-60">Hotel:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['pickup_hotel_name']) ?></span></div>
                <?php if (!empty($booking['pickup_hotel_address'])): ?><div class="text-xs text-base-content/60 ml-6"><?= htmlspecialchars((string)$booking['pickup_hotel_address']) ?></div><?php endif; ?>
                <?php endif; ?>
                <?php if (!empty($booking['pickup_apartment_name'])): ?>
                <div><span class="opacity-60">Apartment:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['pickup_apartment_name']) ?></span></div>
                <?php if (!empty($booking['pickup_apartment_address'])): ?><div class="text-xs text-base-content/60 ml-6"><?= htmlspecialchars((string)$booking['pickup_apartment_address']) ?></div><?php endif; ?>
                <?php endif; ?>
                <?php if (!empty($booking['pickup_address'])): ?>
                <div><span class="opacity-60">Address:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['pickup_address']) ?></span></div>
                <?php endif; ?>
                <?php if (!empty($booking['flight_no'])): ?>
                <div><span class="opacity-60">Flight No:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['flight_no']) ?></span></div>
                <?php endif; ?>
              </div>
            </div>
            <!-- Drop-off -->
            <div class="rounded-lg border p-4">
              <h3 class="font-semibold mb-2">Drop-off Details</h3>
              <div class="space-y-2 text-sm">
                <?php if (!empty($booking['drop_airport_code'])): ?>
                <div><span class="opacity-60">Airport:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['drop_airport_code']) ?></span></div>
                <?php endif; ?>
                <?php if (!empty($booking['drop_hotel_name'])): ?>
                <div><span class="opacity-60">Hotel:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['drop_hotel_name']) ?></span></div>
                <?php if (!empty($booking['drop_hotel_address'])): ?><div class="text-xs text-base-content/60 ml-6"><?= htmlspecialchars((string)$booking['drop_hotel_address']) ?></div><?php endif; ?>
                <?php endif; ?>
                <?php if (!empty($booking['drop_apartment_name'])): ?>
                <div><span class="opacity-60">Apartment:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['drop_apartment_name']) ?></span></div>
                <?php if (!empty($booking['drop_apartment_address'])): ?><div class="text-xs text-base-content/60 ml-6"><?= htmlspecialchars((string)$booking['drop_apartment_address']) ?></div><?php endif; ?>
                <?php endif; ?>
                <?php if (!empty($booking['drop_address'])): ?>
                <div><span class="opacity-60">Address:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['drop_address']) ?></span></div>
                <?php endif; ?>
                <?php if (!empty($booking['return_time'])): ?>
                <div><span class="opacity-60">Return Time:</span> <span class="font-medium"><?= htmlspecialchars((string)$booking['return_time']) ?></span></div>
                <?php endif; ?>
              </div>
            </div>
          </div>
        </div>
      </div>

      <!-- Itinerary (stops) -->
      <?php 
        $itJson = (string)($booking['itinerary_json'] ?? ''); 
        $stops = $itJson ? json_decode($itJson, true) : []; 
      ?>
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <div class="flex items-center justify-between">
            <h2 class="card-title">Journey Itinerary</h2>
            <span class="badge badge-outline"><?= count($stops ?? []) ?> <?= count($stops ?? []) === 1 ? 'Stop' : 'Stops' ?></span>
          </div>
          <?php if (is_array($stops) && count($stops) > 0): ?>
            <div class="mt-3 space-y-3">
              <?php foreach ($stops as $i => $s): 
                $stopType = strtolower($s['type'] ?? 'stop');
                $label = !empty($s['label']) ? $s['label'] : ('Stop ' . ($i + 1));
                $waitTime = !empty($s['wait_minutes']) ? (int)$s['wait_minutes'] : 0;
                $extraPrice = !empty($s['extra_price']) ? (float)$s['extra_price'] : 0;
              ?>
                <div class="rounded-xl border p-3 hover:shadow-sm transition">
                  <div class="flex items-start justify-between gap-3">
                    <div>
                      <div class="font-semibold flex items-center gap-2">
                        <?= htmlspecialchars($label) ?>
                        <?php if ($stopType === 'pickup'): ?><span class="badge badge-success badge-sm">Pickup</span><?php endif; ?>
                        <?php if ($stopType === 'dropoff' || $stopType === 'drop_off'): ?><span class="badge badge-error badge-sm">Drop-off</span><?php endif; ?>
                      </div>
                      <?php if (!empty($s['address'])): ?>
                        <div class="text-sm text-base-content/60 mt-1"><?= htmlspecialchars((string)$s['address']) ?></div>
                      <?php endif; ?>
                      <?php if (!empty($s['notes'])): ?>
                        <div class="text-xs text-base-content/60 mt-2 italic">“<?= nl2br(htmlspecialchars((string)$s['notes'])) ?>”</div>
                      <?php endif; ?>
                    </div>
                    <div class="text-right">
                      <?php if ($waitTime > 0): ?>
                        <div class="badge badge-warning badge-outline mb-1">Wait <?= $waitTime ?>m</div>
                      <?php endif; ?>
                      <?php if ($extraPrice > 0): ?>
                        <div class="text-success font-semibold">+<?= number_format($extraPrice, 2) ?> THB</div>
                      <?php endif; ?>
                    </div>
                  </div>
                  <?php if (!empty($s['contact_name']) || !empty($s['contact_phone'])): ?>
                    <div class="mt-2 pt-2 border-t text-xs text-base-content/60">
                      <div class="flex items-center gap-2">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-4 w-4" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4S14.21 4 12 4 8 5.79 8 8s1.79 4 4 4Zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4Z"/></svg>
                        <?php if (!empty($s['contact_name'])): ?><span><?= htmlspecialchars((string)$s['contact_name']) ?></span><?php endif; ?>
                        <?php if (!empty($s['contact_phone'])): ?><a class="link" href="tel:<?= preg_replace('/[^0-9+]/', '', (string)$s['contact_phone']) ?>"><?= htmlspecialchars((string)$s['contact_phone']) ?></a><?php endif; ?>
                      </div>
                    </div>
                  <?php endif; ?>
                </div>
              <?php endforeach; ?>
            </div>
          <?php else: ?>
            <div class="text-center py-10 opacity-70">
              <div class="text-4xl mb-2">🛣️</div>
              <div class="font-medium">No Stops Added</div>
              <div class="text-sm">This booking doesn't have any stops in the itinerary.</div>
            </div>
          <?php endif; ?>
        </div>
      </div>

      <!-- Audit Trail -->
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Audit Trail</h2>
          <?php if (!empty($events)): ?>
            <div class="overflow-x-auto">
              <table class="table">
                <thead><tr><th>#</th><th>Event</th><th>Note</th><th>When</th></tr></thead>
                <tbody>
                  <?php foreach ($events as $i => $ev): ?>
                    <tr>
                      <td><?= $i+1 ?></td>
                      <td class="font-medium"><?= htmlspecialchars((string)($ev['event_type'] ?? '')) ?></td>
                      <td class="text-sm opacity-80"><?= htmlspecialchars((string)($ev['note'] ?? '')) ?></td>
                      <td class="text-sm opacity-80"><?= htmlspecialchars((string)($ev['created_at'] ?? '')) ?></td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php else: ?>
            <div class="text-sm opacity-70">No audit events found.</div>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <!-- Right: Payment Summary -->
    <aside class="space-y-5">
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Payment Summary</h2>
          <div class="mt-2 space-y-2 text-sm">
            <div class="flex items-center justify-between">
              <span class="opacity-60">Amount</span>
              <span class="font-semibold"><?= number_format((float)($booking['amount_total'] ?? 0), 2) ?> <?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?></span>
            </div>
            <?php if (isset($booking['vendor_cost'])): ?>
              <div class="flex items-center justify-between">
                <span class="opacity-60">Vendor</span>
                <span class="font-medium"><?= number_format((float)$booking['vendor_cost'], 2) ?> <?= htmlspecialchars((string)($booking['vendor_currency'] ?? 'THB')) ?></span>
              </div>
            <?php endif; ?>
            <div class="flex items-center justify-between">
              <span class="opacity-60">Method</span>
              <span class="uppercase"><?= htmlspecialchars((string)($booking['payment_method'] ?? '')) ?></span>
            </div>
            <div class="flex items-center justify-between">
              <span class="opacity-60">Status</span>
              <span class="badge <?= badgeClass($paymentStatus) ?>"><?= ucfirst($paymentStatus ?: 'N/A') ?></span>
            </div>
          </div>
          <div class="divider"></div>
          <h3 class="font-semibold">Transactions</h3>
          <?php if (!empty($payments)): ?>
            <div class="overflow-x-auto">
              <table class="table table-sm">
                <thead><tr><th>#</th><th>Gateway</th><th>Amount</th><th>Status</th><th>At</th></tr></thead>
                <tbody>
                <?php foreach ($payments as $i => $p): ?>
                  <tr>
                    <td><?= $i+1 ?></td>
                    <td><?= htmlspecialchars((string)($p['gateway'] ?? '')) ?></td>
                    <td><?= number_format((float)($p['amount'] ?? 0), 2) ?> <?= htmlspecialchars((string)($p['currency'] ?? 'THB')) ?></td>
                    <td><span class="badge <?= badgeClass($p['status'] ?? '') ?>"><?= htmlspecialchars((string)($p['status'] ?? '')) ?></span></td>
                    <td class="text-xs opacity-70"><?= htmlspecialchars((string)($p['created_at'] ?? '')) ?></td>
                  </tr>
                <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php else: ?>
            <div class="text-sm opacity-70">No transactions yet.</div>
          <?php endif; ?>
        </div>
      </div>

      <?php if (strtolower((string)($booking['status'] ?? '')) !== 'cancelled' && !$isCompleted): ?>
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Actions</h2>
          <p class="text-sm opacity-70">To request cancellation or refund, contact support or your partner manager. Agent self-cancel can be added here if needed.</p>
          <!-- Placeholder buttons: wire to agent endpoints if required -->
          <div class="flex gap-2">
            <button class="btn btn-outline btn-error btn-sm" onclick="document.getElementById('cancelBookingModal').showModal()">Cancel Booking</button>
          </div>
        </div>
      </div>
      <?php endif; ?>
    </aside>
  </div>
</div>

<!-- Cancel modal (agent) -->
<dialog id="cancelBookingModal" class="modal">
  <div class="modal-box">
    <h3 class="font-bold text-lg mb-2">Cancel Booking</h3>
    <form action="#" method="post" class="space-y-3">
      <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
      <input type="hidden" name="id" value="<?= (int)($booking['id'] ?? 0) ?>">
      <div>
        <label class="label">Reason for Cancellation</label>
        <select class="select select-bordered w-full" name="reason" required>
          <option value="">Select a reason</option>
          <option value="customer_request">Customer Request</option>
          <option value="no_show">No Show</option>
          <option value="taxi_unavailable">Taxi Unavailable</option>
          <option value="weather">Weather Conditions</option>
          <option value="other">Other Reason</option>
        </select>
      </div>
      <div class="modal-action">
        <button type="button" class="btn" onclick="document.getElementById('cancelBookingModal').close()">Close</button>
        <button type="submit" class="btn btn-error">Submit Request</button>
      </div>
    </form>
  </div>
</dialog>

<?php require __DIR__ . '/partials/footer.php'; ?>
