<?php
// app/Views/agent/activity_booking_view.php
// Detailed view for a single activity booking (Agent) – matches Taxi style

$title    = $title ?? 'Activity Booking';
$booking  = $booking ?? [];
$events   = $events ?? [];
$payments = $payments ?? [];
$csrf     = $csrf ?? '';

require __DIR__ . '/partials/header.php';

$status = strtolower((string)($booking['status'] ?? ''));
$paymentStatus = strtolower((string)($booking['payment_status'] ?? ''));

function ab_badgeClass($status) {
  $status = strtolower((string)$status);
  return match ($status) {
    'confirmed' => 'badge-info',
    'completed' => 'badge-success',
    'cancelled' => 'badge-error',
    'refunded'  => 'badge-neutral',
    'pending', 'processing' => 'badge-warning',
    default => 'badge-ghost',
  };
}
?>

<div class="max-w-7xl mx-auto px-4 py-6">
  <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-3 mb-6">
    <div>
      <div class="text-sm breadcrumbs">
        <ul>
          <li><a href="/b2b/agent">Dashboard</a></li>
          <li><a href="/b2b/agent/activities/bookings">Activity Bookings</a></li>
          <li>#<?= (int)($booking['id'] ?? 0) ?></li>
        </ul>
      </div>
      <h1 class="text-2xl font-bold flex items-center gap-2">
        <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-primary" viewBox="0 0 24 24" fill="currentColor"><path d="M19 4h-4l-2-2H9L7 4H3v2h16V4ZM3 8v11a1 1 0 0 0 1 1h10l6-6V8H3Zm12 9v-3h3l-3 3Z"/></svg>
        Activity Booking #<?= (int)($booking['id'] ?? 0) ?>
      </h1>
      <div class="mt-1 flex items-center gap-2">
        <span class="badge <?= ab_badgeClass($status) ?>"><?= ucfirst($status ?: 'N/A') ?></span>
        <span class="badge <?= ab_badgeClass($paymentStatus) ?>"><?= ucfirst($paymentStatus ?: 'N/A') ?></span>
      </div>
    </div>
    <div class="flex gap-2">
      <a href="/b2b/agent/activities/bookings" class="btn btn-ghost">Back</a>
      <button class="btn btn-outline" onclick="window.print()">Print</button>
    </div>
  </div>

  <div class="grid grid-cols-1 lg:grid-cols-3 gap-5">
    <div class="lg:col-span-2 space-y-5">
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Booking Overview</h2>
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-2">
            <div class="space-y-3">
              <div>
                <div class="text-xs text-base-content/60">Booking Code</div>
                <div class="font-semibold"><?= htmlspecialchars((string)($booking['booking_code'] ?? '')) ?></div>
              </div>
              <div>
                <div class="text-xs text-base-content/60">Agent</div>
                <div class="font-semibold">#<?= (int)($booking['agent_id'] ?? 0) ?></div>
              </div>
              <div>
                <div class="text-xs text-base-content/60">Vendor</div>
                <div class="font-semibold">#<?= (int)($booking['vendor_id'] ?? 0) ?><?php if (!empty($booking['vendor_name'])): ?> <span class="text-base-content/60">(<?= htmlspecialchars((string)$booking['vendor_name']) ?>)</span><?php endif; ?></div>
              </div>
            </div>
            <div class="space-y-3">
              <div>
                <div class="text-xs text-base-content/60">Booking/Travel Date</div>
                <div class="font-semibold">
                  <?php $td = $booking['booking_date'] ?? $booking['travel_date'] ?? $booking['created_at'] ?? null; ?>
                  <?= $td ? date('d M Y', strtotime((string)$td)) : 'N/A' ?>
                </div>
              </div>
              <div>
                <div class="text-xs text-base-content/60">Status</div>
                <div class="flex items-center gap-2">
                  <span class="badge <?= ab_badgeClass($status) ?>"><?= ucfirst($status ?: 'N/A') ?></span>
                  <span class="badge <?= ab_badgeClass($paymentStatus) ?>"><?= ucfirst($paymentStatus ?: 'N/A') ?></span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Customer Details</h2>
          <div class="mt-2 grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <div class="text-xs text-base-content/60">Name</div>
              <div class="font-semibold"><?= htmlspecialchars((string)($booking['lead_name'] ?? $booking['customer_name'] ?? 'Guest User')) ?></div>
            </div>
            <?php if (!empty($booking['lead_email'] ?? $booking['customer_email'])): ?>
            <div>
              <div class="text-xs text-base-content/60">Email</div>
              <?php $em = (string)($booking['lead_email'] ?? $booking['customer_email']); ?>
              <a class="link" href="mailto:<?= htmlspecialchars($em) ?>"><?= htmlspecialchars($em) ?></a>
            </div>
            <?php endif; ?>
            <?php if (!empty($booking['lead_phone'] ?? $booking['customer_phone'])): ?>
            <div>
              <div class="text-xs text-base-content/60">Phone</div>
              <?php $ph = (string)($booking['lead_phone'] ?? $booking['customer_phone']); ?>
              <a class="link" href="tel:<?= preg_replace('/[^0-9+]/', '', $ph) ?>"><?= htmlspecialchars($ph) ?></a>
            </div>
            <?php endif; ?>
          </div>
        </div>
      </div>

      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Additional Info</h2>
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-2 text-sm">
            <div>
              <div class="text-xs text-base-content/60">Created At</div>
              <div class="font-medium"><?= htmlspecialchars((string)($booking['created_at'] ?? '')) ?></div>
            </div>
            <div>
              <div class="text-xs text-base-content/60">Currency</div>
              <div class="font-medium"><?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?></div>
            </div>
          </div>
        </div>
      </div>

      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Audit Trail</h2>
          <?php if (!empty($events)): ?>
            <div class="overflow-x-auto">
              <table class="table">
                <thead><tr><th>#</th><th>Event</th><th>Note</th><th>When</th></tr></thead>
                <tbody>
                  <?php foreach ($events as $i => $ev): ?>
                    <tr>
                      <td><?= $i+1 ?></td>
                      <td class="font-medium"><?= htmlspecialchars((string)($ev['event_type'] ?? '')) ?></td>
                      <td class="text-sm opacity-80"><?= htmlspecialchars((string)($ev['note'] ?? '')) ?></td>
                      <td class="text-sm opacity-80"><?= htmlspecialchars((string)($ev['created_at'] ?? '')) ?></td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php else: ?>
            <div class="text-sm opacity-70">No audit events found.</div>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <aside class="space-y-5">
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Payment Summary</h2>
          <div class="mt-2 space-y-2 text-sm">
            <div class="flex items-center justify-between">
              <span class="opacity-60">Amount</span>
              <span class="font-semibold"><?= number_format((float)($booking['amount_total'] ?? $booking['total_amount'] ?? 0), 2) ?> <?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?></span>
            </div>
            <div class="flex items-center justify-between">
              <span class="opacity-60">Status</span>
              <span class="badge <?= ab_badgeClass($paymentStatus) ?>"><?= ucfirst($paymentStatus ?: 'N/A') ?></span>
            </div>
          </div>
          <div class="divider"></div>
          <h3 class="font-semibold">Transactions</h3>
          <?php if (!empty($payments)): ?>
            <div class="overflow-x-auto">
              <table class="table table-sm">
                <thead><tr><th>#</th><th>Gateway</th><th>Amount</th><th>Status</th><th>At</th></tr></thead>
                <tbody>
                <?php foreach ($payments as $i => $p): ?>
                  <tr>
                    <td><?= $i+1 ?></td>
                    <td><?= htmlspecialchars((string)($p['gateway'] ?? '')) ?></td>
                    <td><?= number_format((float)($p['amount'] ?? 0), 2) ?> <?= htmlspecialchars((string)($p['currency'] ?? 'THB')) ?></td>
                    <td><span class="badge <?= ab_badgeClass($p['status'] ?? '') ?>"><?= htmlspecialchars((string)($p['status'] ?? '')) ?></span></td>
                    <td class="text-xs opacity-70"><?= htmlspecialchars((string)($p['created_at'] ?? '')) ?></td>
                  </tr>
                <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php else: ?>
            <div class="text-sm opacity-70">No transactions yet.</div>
          <?php endif; ?>
        </div>
      </div>

      <?php if (strtolower((string)($booking['status'] ?? '')) !== 'cancelled'): ?>
      <div class="card bg-base-100 shadow-sm">
        <div class="card-body">
          <h2 class="card-title">Actions</h2>
          <p class="text-sm opacity-70">To request cancellation or refund, contact support or your partner manager. Agent self-cancel can be added here if needed.</p>
          <div class="flex gap-2">
            <button class="btn btn-outline btn-error btn-sm" onclick="document.getElementById('cancelBookingModal').showModal()">Cancel Booking</button>
          </div>
        </div>
      </div>
      <?php endif; ?>
    </aside>
  </div>
</div>

<dialog id="cancelBookingModal" class="modal">
  <div class="modal-box">
    <h3 class="font-bold text-lg mb-2">Cancel Booking</h3>
    <form action="#" method="post" class="space-y-3">
      <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
      <input type="hidden" name="id" value="<?= (int)($booking['id'] ?? 0) ?>">
      <div>
        <label class="label">Reason for Cancellation</label>
        <select class="select select-bordered w-full" name="reason" required>
          <option value="">Select a reason</option>
          <option value="customer_request">Customer Request</option>
          <option value="vendor_unavailable">Vendor Unavailable</option>
          <option value="weather">Weather Conditions</option>
          <option value="other">Other Reason</option>
        </select>
      </div>
      <div class="modal-action">
        <button type="button" class="btn" onclick="document.getElementById('cancelBookingModal').close()">Close</button>
        <button type="submit" class="btn btn-error">Submit Request</button>
      </div>
    </form>
  </div>
</dialog>

<?php require __DIR__ . '/partials/footer.php'; ?>
