<?php
// app/Views/agent/activity.php
$title = $title ?? 'Activity';
$package = $package ?? [];
$variants = $variants ?? [];
$prices = $prices ?? [];
$showtimes = $showtimes ?? [];
$active = 'activities';
$csrf = $csrf ?? null;
require __DIR__ . '/partials/header.php';
?>

<!-- Activity Detail - Brand Styling -->
<div class="min-h-screen bg-gradient-to-br from-slate-50 via-white to-blue-50/30">
  <div class="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
    <div class="mb-4">
      <a href="/b2b/agent/activities" class="inline-flex items-center gap-2 text-blue-600 hover:text-blue-700 font-semibold transition-colors">
        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/></svg>
        Back to Activities
      </a>
    </div>

    <div class="rounded-2xl bg-gradient-to-r from-brand-600 to-blue-600 text-white shadow-lg overflow-hidden">
      <div class="grid grid-cols-1 md:grid-cols-3">
        <div class="md:col-span-1">
        <?php $thumb = trim((string)($package['thumbnail_path'] ?? '')); ?>
        <img src="<?= $thumb !== '' ? htmlspecialchars($thumb) : ('https://picsum.photos/seed/p'.urlencode((string)($package['id'] ?? rand(1,9999))).'/640/480') ?>"
             alt="Photo" class="w-full h-32 md:h-40 object-cover" loading="lazy">
        </div>
        <div class="md:col-span-2 p-4 lg:p-5">
          <h1 class="text-2xl font-bold text-white"><?= htmlspecialchars((string)($package['name'] ?? 'Activity')) ?></h1>
          <div class="text-xs text-blue-100/90 mt-0.5">
            <div class="flex items-center space-x-1">
              <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"/>
              </svg>
              <span><?= htmlspecialchars(trim(((string)($package['city'] ?? '')) . (isset($package['country']) ? ', ' . (string)$package['country'] : ''))) ?></span>
            </div>
          </div>
          <?php if (!empty($package['address_override'])): ?>
            <div class="text-xs mt-1 text-slate-700">Address: <?= htmlspecialchars((string)$package['address_override']) ?></div>
          <?php endif; ?>
          <div class="flex flex-wrap items-center gap-2 mt-3">
            <span class="px-2.5 py-1 text-xs font-medium rounded-full bg-white/10 backdrop-blur-sm border border-white/20">
              <svg class="w-3 h-3 inline mr-1 -mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-11a1 1 0 10-2 0v3.586L7.707 9.293a1 1 0 00-1.414 1.414l3 3a1 1 0 001.414 0l3-3a1 1 0 00-1.414-1.414L11 10.586V7z" clip-rule="evenodd"/>
              </svg>
              Mobile Ticket
            </span>
            <?php $agePol = strtolower((string)($package['age_policy'] ?? ($package['policy'] ?? ''))); if ($agePol !== ''): ?>
            <?php if ($agePol === 'adult_only'): ?>
              <span class="inline-flex items-center gap-1.5 px-2.5 py-0.5 rounded-full border border-rose-200 bg-rose-50 text-rose-700 font-semibold">
                <svg class="w-3.5 h-3.5" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true"><path d="M12 12a5 5 0 100-10 5 5 0 000 10zm-7 9a7 7 0 0114 0v1H5v-1z"/></svg>
                Adult only
              </span>
            <?php else: ?>
              <span class="inline-flex items-center gap-1.5 px-2.5 py-0.5 rounded-full border border-emerald-200 bg-emerald-50 text-emerald-700 font-semibold">
                <svg class="w-3.5 h-3.5" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true"><path d="M16 11c1.66 0 3-1.57 3-3.5S17.66 4 16 4s-3 1.57-3 3.5S14.34 11 16 11zm-8 0c1.66 0 3-1.57 3-3.5S9.66 4 8 4 5 5.57 5 7.5 6.34 11 8 11zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5C15 14.17 10.33 13 8 13zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.96 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z"/></svg>
                Family
              </span>
            <?php endif; ?>
            <?php endif; ?>
          </div>
        </div>
      </div>
    </div>

    <div class="mt-6 grid grid-cols-1 md:grid-cols-3 gap-6">
      <aside class="md:col-span-1">
        <form id="cartForm" class="rounded-2xl border border-slate-200/60 bg-white/95 backdrop-blur-sm p-6 shadow-lg" method="post" action="/b2b/agent/activity/checkout" onsubmit="return prepareBatchSubmit()">
        <input type="hidden" name="csrf_token" value="<?= htmlspecialchars((string)$csrf) ?>">
        <input type="hidden" name="package_id" value="<?= (int)($package['id'] ?? 0) ?>">
        <div class="space-y-3">
          <div>
            <label class="block text-xs font-semibold text-slate-700 mb-1.5 uppercase tracking-wider text-slate-500">Select Date</label>
            <div class="relative">
              <input type="text" 
                     id="bookingDate" 
                     name="booking_date" 
                     class="input input-bordered rounded-xl w-full focus:ring-2 focus:ring-brand-500 focus:border-brand-500 border-slate-300 hover:border-slate-400 transition-colors cursor-pointer"
                     placeholder="Select a date"
                     readonly
                     required>
              <div class="absolute right-3 top-1/2 -translate-y-1/2 text-slate-400">
                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                </svg>
              </div>
            </div>
          </div>
          <div id="timeBlock">
            <label class="block text-xs font-semibold text-slate-700 mb-1.5 uppercase tracking-wider text-slate-500">Select Time</label>
            <select id="timeSelect" 
                    name="show_time" 
                    class="select select-bordered rounded-xl w-full focus:ring-2 focus:ring-brand-500 focus:border-brand-500 border-slate-300 hover:border-slate-400 transition-colors"
                    required>
              <option value="">Select a time</option>
            </select>
          </div>
          <?php if (!empty($transportOptions)): ?>
          <div id="transportBlock" class="mt-4">
            <label class="block text-xs font-semibold text-slate-700 mb-1.5 uppercase tracking-wider text-slate-500">Optional Transport</label>
            <div class="space-y-2">
              <label class="flex items-center gap-3 p-3 rounded-xl border border-slate-200 hover:border-brand-300 bg-white cursor-pointer">
                <input type="radio" name="transport_option" value="" class="radio radio-primary" checked>
                <div class="flex-1">
                  <div class="text-sm font-medium text-slate-800">No Transport</div>
                  <div class="text-xs text-slate-500">You can arrange your own transport.</div>
                </div>
                <div class="text-sm font-semibold text-slate-700 whitespace-nowrap">0 THB</div>
              </label>
              <?php foreach ($transportOptions as $to): ?>
              <label class="flex items-center gap-3 p-3 rounded-xl border border-slate-200 hover:border-brand-300 bg-white cursor-pointer">
                <input type="radio" name="transport_option" value="<?= (int)$to['id'] ?>" data-price="<?= number_format((float)$to['price'],2,'.','') ?>" data-label="<?= htmlspecialchars((string)$to['vehicle_type']) ?>" class="radio radio-primary">
                <div class="flex-1">
                  <div class="text-sm font-medium text-slate-800"><?= htmlspecialchars((string)$to['vehicle_type']) ?></div>
                </div>
                <div class="text-sm font-semibold text-slate-700 whitespace-nowrap"><?= number_format((float)$to['price'], 2) ?> THB</div>
              </label>
              <?php endforeach; ?>
            </div>
            <input type="hidden" id="transport_option_id" name="transport_option_id" value="">
            <input type="hidden" id="transport_option_price" name="transport_option_price" value="0">
          </div>
          <?php endif; ?>
        </div>

        <div class="mt-6">
          <div class="text-sm font-medium text-slate-700 mb-3">Select Ticket Quantities</div>
          <div id="variantRows" class="space-y-4"></div>
        </div>

        <div class="mt-5 pt-4 border-t border-slate-200">
          <div class="space-y-2">
            <div class="flex items-center justify-between text-sm">
              <span class="text-slate-600 font-medium">Subtotal:</span>
              <span id="subtotalAmount" class="font-medium">0 THB</span>
            </div>
            <div id="discountAmount"></div>
            <div class="flex items-center justify-between text-sm pt-2 border-t border-slate-100">
              <span class="text-slate-700 font-semibold">Estimated total:</span>
              <span id="estTotal" class="text-lg font-bold text-brand-600">0 THB</span>
            </div>
          </div>
          <button type="submit" class="w-full mt-4 px-6 py-3.5 bg-gradient-to-r from-green-500 to-emerald-600 hover:from-green-600 hover:to-emerald-700 text-white font-semibold rounded-xl shadow-md hover:shadow-lg transition-all duration-300 transform hover:scale-[1.02] active:scale-100 flex items-center justify-center space-x-2">
            <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
            </svg>
            <span>Book Now</span>
          </button>
        </div>
        </form>
      </aside>
      <section class="md:col-span-2 space-y-4">
        <div class="rounded-2xl border border-slate-200/60 bg-white/95 backdrop-blur-sm p-6 shadow-sm">
          <div class="flex items-center justify-between mb-4">
            <h3 class="text-sm font-semibold text-slate-800 flex items-center">
              <svg class="w-4 h-4 mr-2 text-brand-500" fill="currentColor" viewBox="0 0 20 20">
                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
              </svg>
              Booking Summary
            </h3>
            <span class="px-2 py-1 text-xs font-medium rounded-full bg-brand-100 text-brand-800">
              <span id="itemCount">0</span> items
            </span>
          </div>
          <div id="summaryPanel" class="text-sm text-slate-700">
            <div class="text-center py-6 text-slate-400">
              <svg class="mx-auto h-12 w-12 text-slate-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/>
              </svg>
              <p class="mt-2 text-sm font-medium text-slate-500">No selections yet</p>
              <div class="text-sm text-slate-700 mt-2 space-y-1">
                <div class="flex items-start">
                  <span class="text-green-500 mr-2">✓</span>
                  <span>Free cancellation up to 72 hours before activity</span>
                </div>
                <div class="flex items-start">
                  <span class="text-amber-500 mr-2">!</span>
                  <span>50 THB cancellation fee per ticket between 24-72 hours</span>
                </div>
                <div class="flex items-start">
                  <span class="text-red-500 mr-2">✗</span>
                  <span>No cancellation within 24 hours of activity</span>
                </div>
                <div class="text-xs text-slate-500 mt-2">* No refunds for no-shows or late arrivals</div>
              </div>
            </div>
          </div>
        </div>
        <div class="rounded-2xl border border-slate-200/60 bg-white/95 backdrop-blur-sm p-6 shadow-sm">
          <h3 class="text-sm font-semibold text-slate-800 mb-3 flex items-center">
            <svg class="w-4 h-4 mr-2 text-brand-500" fill="currentColor" viewBox="0 0 20 20">
              <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h.01a1 1 0 100-2H10V9z" clip-rule="evenodd"/>
            </svg>
            About This Activity
          </h3>
          <div class="prose prose-sm max-w-none text-slate-600">
            <p>Experience <?= htmlspecialchars((string)($package['name'] ?? 'this activity')) ?> with instant confirmation and mobile ticket support. Your booking will be confirmed immediately and you'll receive an e-ticket directly to your email.</p>
            
            <div class="mt-4 space-y-3">
              <div class="flex items-start">
                <svg class="h-5 w-5 text-green-500 mr-2 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                </svg>
                <span class="text-sm">Free cancellation up to 72 hours before activity</span>
              </div>
              <div class="flex items-start">
                <svg class="h-5 w-5 text-amber-500 mr-2 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                </svg>
                <span class="text-sm">50 THB cancellation fee per ticket between 24-72 hours</span>
              </div>
              <div class="flex items-start">
                <svg class="h-5 w-5 text-red-500 mr-2 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                </svg>
                <span class="text-sm">No cancellation within 24 hours of activity</span>
              </div>
              <div class="flex items-start">
                <svg class="h-5 w-5 text-red-500 mr-2 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                </svg>
                <span class="text-sm">No refunds for no-shows or late arrivals</span>
              </div>
              <div class="flex items-start">
                <svg class="h-5 w-5 text-brand-500 mr-2 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                </svg>
                <span class="text-sm">Mobile ticket (no need to print)</span>
              </div>
              <div class="flex items-start">
                <svg class="h-5 w-5 text-brand-500 mr-2 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                  <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                </svg>
                <span class="text-sm">Instant confirmation</span>
              </div>
            </div>
          </div>
        </div>
      </section>
    </div>
  </div>
</div>

<!-- Debug output for pricing data -->
<script>
  // Package policy from PHP ('family' | 'adult_only' | '')
  const packagePolicy = '<?= strtolower((string)($package['age_policy'] ?? ($package['policy'] ?? ''))) ?>';
console.log('PHP Prices Data:', <?= json_encode($prices) ?>);
console.log('PHP Variants Data:', <?= json_encode($variants) ?>);
  // Prefill from session (set in checkout)
  const prefillGuest = <?= json_encode($_SESSION['activity_guest'] ?? []) ?>;
  const prefillCart = <?= json_encode($_SESSION['activity_cart'] ?? []) ?>;
  // Build a lookup from price_id -> {variant_id, pax_type}
  const priceIndex = (() => {
    const idx = {};
    const byVariant = <?= json_encode($prices) ?>;
    Object.entries(byVariant).forEach(([vid, list]) => {
      (list || []).forEach(row => {
        if (row && row.id) {
          idx[parseInt(row.id, 10)] = { variant_id: parseInt(vid, 10), pax_type: (row.pax_type || 'adult') };
        }
      });
    });
    return idx;
  })();
  const prefillQty = (() => {
    const map = {};
    (prefillCart || []).forEach(it => {
      const pid = parseInt(it.price_id || 0, 10); const qty = parseInt(it.qty || 0, 10);
      if (!pid || !qty) return;
      const meta = priceIndex[pid]; if (!meta) return;
      const key = meta.variant_id + '::' + (meta.pax_type || 'adult');
      map[key] = (map[key] || 0) + qty;
    });
    return map;
  })();
</script>

<!-- Include Pikaday CSS -->
<link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/pikaday/css/pikaday.css">

<!-- Include Pikaday JS -->
<script src="https://cdn.jsdelivr.net/npm/pikaday/pikaday.js"></script>
<script>
  // Initialize Pikaday
  // Set minimum date to tomorrow
  const tomorrow = new Date();
  tomorrow.setDate(tomorrow.getDate() + 1);
  
  const picker = new Pikaday({
    field: document.getElementById('bookingDate'),
    format: 'YYYY-MM-DD',
    minDate: tomorrow,
    setDefaultDate: false, // Don't set a default date
    showDaysInNextAndPreviousMonths: true,
    enableSelectionDaysInNextAndCurrentMonth: true,
    i18n: {
      previousMonth: 'Previous Month',
      nextMonth: 'Next Month',
      months: ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'],
      weekdays: ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'],
      weekdaysShort: ['Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa']
    },
    theme: 'light-theme',
    onSelect: function() {
      // You can add additional logic when a date is selected
      console.log('Selected date:', this.getMoment().format('YYYY-MM-DD'));
    }
  });
  
  // Add theme class to the calendar container
  document.addEventListener('DOMContentLoaded', function() {
    const container = document.querySelector('.pika-single');
    if (container) {
      container.classList.add('shadow-lg', 'rounded-xl', 'border', 'border-slate-200');
    }
    
    // Disable today's date in the calendar
    const today = new Date();
    const todayElements = document.querySelectorAll(`.pika-button[data-pika-year="${today.getFullYear()}"][data-pika-month="${today.getMonth()}"][data-pika-day="${today.getDate()}"]`);
    todayElements.forEach(el => {
      el.classList.add('is-disabled');
      el.style.textDecoration = 'line-through';
      el.style.opacity = '0.5';
      el.style.cursor = 'not-allowed';
    });
  });
</script>
<style>
  /* Custom Pikaday Theme */
  .pika-single {
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 0.75rem;
    z-index: 9999;
  }
  .pika-title {
    color: #1e293b;
  }
  .pika-label {
    font-weight: 500;
  }
  .pika-button {
    border-radius: 0.375rem;
    font-weight: 400;
  }
  .is-today .pika-button {
    color: #3b82f6;
    font-weight: 600;
  }
  .is-selected .pika-button {
    background: #3b82f6;
    box-shadow: 0 0 0 1px #3b82f6;
    font-weight: 500;
  }
  .pika-button:hover {
    background: #f1f5f9;
  }
  .pika-table th {
    color: #64748b;
    font-size: 0.75rem;
    font-weight: 500;
  }
</style>

<script>
  // Format price with thousands separators
  function formatPrice(amount) {
    if (amount === null || amount === undefined) return '0';
    // Convert to number to handle both string and number inputs
    const num = Number(amount);
    if (isNaN(num)) return '0';
    // Round to 2 decimal places and format with commas
    return num.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',');
  }

  // Update item count in the summary header
  function updateItemCount() {
    let totalItems = 0;
    variants.forEach(v => {
      const qa = parseInt(document.getElementById('qtyA-' + v.id)?.value || '0', 10);
      const qc = parseInt(document.getElementById('qtyC-' + v.id)?.value || '0', 10);
      totalItems += qa + qc;
    });
    document.getElementById('itemCount').textContent = totalItems;
  }

  // Debug output for variants and prices
  console.log('Processing variants with prices:', <?= json_encode($prices) ?>);
  
  const variants = <?= json_encode(array_map(function($v) use ($prices) {
    $variantPrices = $prices[$v['id']] ?? [];
    $basePrice = 0;
    $groupPrices = [];
    
    foreach ($variantPrices as $price) {
      if ($price['price_type'] === 'base' && $price['pax_type'] === 'adult') {
        $basePrice = (float)$price['agent_price'];
      } elseif ($price['price_type'] === 'group_tier') {
        $groupPrices[] = [
          'min_quantity' => (int)$price['min_quantity'],
          'price' => (float)$price['agent_price'],
          'pax_type' => $price['pax_type']
        ];
      }
    }
    
    return [
      'id' => (int)$v['id'],
      'name' => $v['name'],
      'notes' => $v['notes'] ?? '',
      'base_price' => $basePrice,
      'prices' => $groupPrices
    ];
  }, $variants)) ?>;
  const prices = <?= json_encode($prices) ?>; // by variant_id
  console.log('JavaScript prices object:', prices);
  const showtimes = <?= json_encode($showtimes) ?>; // by variant_id (0 for package-level)
  let hasShowTimes = false; // will be set in renderTimes()
  const transportOptions = <?= json_encode($transportOptions ?? []) ?>;

  function renderTimes() {
    // Use union of times across variants or package-level
    let times = showtimes[0] ? [...showtimes[0]] : [];
    for (const vid in showtimes) { if (vid !== '0') { (showtimes[vid]||[]).forEach(t => { if (!times.includes(t)) times.push(t); }); } }
    const sel = document.getElementById('timeSelect');
    const block = document.getElementById('timeBlock');
    sel.innerHTML = '';
    if (!times.length) {
      // Hide time block when there are no explicit showtimes
      if (block) block.style.display = 'none';
      if (sel) { sel.required = false; sel.innerHTML = '<option value="">Any time</option>'; sel.value = ''; }
      hasShowTimes = false;
      return;
    }
    // Show and populate
    if (block) block.style.display = '';
    if (sel) sel.required = true;
    hasShowTimes = true;
    times.forEach(t => {
      const opt = document.createElement('option');
      opt.value = t; opt.textContent = t; sel.appendChild(opt);
    });
  }

  function bestPriceForVariantPax(vid, qty, paxType) {
    const list = prices[vid] || [];
    if (!list.length) return null;
    
    // Get base price first
    const basePrice = list.find(p => p.pax_type === paxType && p.price_type === 'base');
    if (!basePrice) return null;
    
    // Find all applicable group tiers for this pax type
    const groupTiers = list
      .filter(p => p.pax_type === paxType && p.price_type === 'group_tier')
      .sort((a, b) => (parseInt(a.min_quantity || '0', 10) - parseInt(b.min_quantity || '0', 10)));
    
    // Find the best matching group tier
    let bestTier = null;
    for (const tier of groupTiers) {
      const minQty = parseInt(tier.min_quantity || '0', 10);
      if (qty >= minQty) {
        bestTier = tier;
      } else {
        break; // Tiers are sorted, so we can break early
      }
    }
    
    // Return the best tier if found, otherwise return base price
    const result = bestTier || basePrice;
    
    // Store the base price in the result for reference
    if (bestTier) {
      result.base_price = basePrice.agent_price || basePrice.vendor_cost;
      result.discount = result.base_price - (result.agent_price || result.vendor_cost);
    }
    
    return result;
  }

  function renderVariantRows() {
    const container = document.getElementById('variantRows');
    container.innerHTML = '';
    variants.forEach(v => {
      const vid = v.id;
      const list = prices[vid] || [];
      const row = document.createElement('div');
      row.className = 'bg-white p-4 rounded-xl border border-slate-200 shadow-sm hover:shadow transition-all';
      row.innerHTML = `
        <div class="flex items-start justify-between">
          <div class="flex-1">
            <div class="font-medium text-slate-800 mb-3">${v.name}</div>
            <div class="grid grid-cols-2 gap-4">
              <!-- Adults -->
              <div class="space-y-1">
                <div class="text-xs font-medium text-slate-500">Adults</div>
                <div class="flex items-center space-x-2">
                  <button type="button" class="w-9 h-9 flex items-center justify-center rounded-lg bg-slate-100 hover:bg-slate-200 text-slate-700 transition-colors" onclick="chgA(${vid}, -1)">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M20 12H4"/>
                    </svg>
                  </button>
                  <input id="qtyA-${vid}" type="number" min="0" value="0" class="w-16 text-center py-2 rounded-lg border border-slate-300">
                  <button type="button" class="w-9 h-9 flex items-center justify-center rounded-lg bg-brand-50 hover:bg-brand-100 text-brand-600 transition-colors" onclick="chgA(${vid}, 1)">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24" stroke-width="2">
                      <path stroke-linecap="round" stroke-linejoin="round" d="M12 4v16m8-8H4"/>
                    </svg>
                  </button>
                </div>
              </div>
              <!-- Children -->
              <div class="space-y-1" data-child-block="${vid}">
                <div class="text-xs font-medium text-slate-500">Children</div>
                <div class="flex items-center space-x-2">
                  <button type="button" class="w-9 h-9 flex items-center justify-center rounded-lg bg-slate-100 hover:bg-slate-200 text-slate-700 transition-colors" onclick="chgC(${vid}, -1)">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M20 12H4"/>
                    </svg>
                  </button>
                  <input id="qtyC-${vid}" type="number" min="0" value="0" class="w-16 text-center py-2 rounded-lg border border-slate-300">
                  <button type="button" class="w-9 h-9 flex items-center justify-center rounded-lg bg-brand-50 hover:bg-brand-100 text-brand-600 transition-colors" onclick="chgC(${vid}, 1)">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24" stroke-width="2">
                      <path stroke-linecap="round" stroke-linejoin="round" d="M12 4v16m8-8H4"/>
                    </svg>
                  </button>
                </div>
              </div>
            </div>
            <div class="mt-3 text-sm font-medium text-slate-700">
              Subtotal: <span id="sub-${vid}" class="text-brand-600">0 THB</span>
            </div>
          </div>
        </div>`;
      // Add price data attributes
      row.setAttribute('data-base-price', v.base_price);
      
      // Add hidden elements for group pricing
      if (v.prices && v.prices.length > 0) {
        v.prices.forEach(price => {
          const priceEl = document.createElement('div');
          priceEl.className = 'hidden';
          priceEl.setAttribute('data-variant-id', v.id);
          priceEl.setAttribute('data-price-type', 'group_tier');
          priceEl.setAttribute('data-pax-type', price.pax_type);
          priceEl.setAttribute('data-min-qty', price.min_quantity);
          priceEl.setAttribute('data-price', price.price);
          document.body.appendChild(priceEl);
        });
      }
      container.appendChild(row);
      // Hide children block when policy is adult_only
      if (packagePolicy === 'adult_only') {
        const childEl = row.querySelector(`[data-child-block="${vid}"]`);
        const cInput = row.querySelector(`#qtyC-${vid}`);
        if (childEl) childEl.style.display = 'none';
        if (cInput) { cInput.value = 0; }
      }
    });
    recalc();
  }
  function chgA(vid, delta) { 
    const bookingDate = document.getElementById('bookingDate').value;
    if (!bookingDate && delta > 0) {
      showNotification('Please select a booking date before adding tickets', 'Date Required');
      document.getElementById('bookingDate').focus();
      // Add a shake animation to the date picker
      const dateInput = document.getElementById('bookingDate');
      dateInput.classList.add('animate-shake');
      setTimeout(() => dateInput.classList.remove('animate-shake'), 500);
      return;
    }
    const el = document.getElementById('qtyA-' + vid); 
    if (!el) return;
    const v = Math.max(0, parseInt(el.value||'0',10) + delta); 
    el.value = v; 
    recalc(); 
  }
  
  function chgC(vid, delta) { 
    const bookingDate = document.getElementById('bookingDate').value;
    if (!bookingDate && delta > 0) {
      showNotification('Please select a booking date before adding tickets', 'Date Required');
      document.getElementById('bookingDate').focus();
      // Add a shake animation to the date picker
      const dateInput = document.getElementById('bookingDate');
      dateInput.classList.add('animate-shake');
      setTimeout(() => dateInput.classList.remove('animate-shake'), 500);
      return;
    }
    const el = document.getElementById('qtyC-' + vid); 
    if (!el) return;
    const v = Math.max(0, parseInt(el.value||'0',10) + delta); 
    el.value = v; 
    recalc(); 
  }

  // Get group pricing tiers for a variant
  function getGroupPricingTiers(variantId) {
    const tiers = [];
    const elements = document.querySelectorAll(`[data-variant-id="${variantId}"][data-price-type="group_tier"]`);
    
    elements.forEach(el => {
      const minQty = parseInt(el.getAttribute('data-min-qty')) || 0;
      const price = parseFloat(el.getAttribute('data-price')) || 0;
      const paxType = el.getAttribute('data-pax-type') || 'adult';
      
      if (minQty > 0 && price > 0) {
        tiers.push({
          minQty: minQty,
          price: price,
          type: paxType,
          variantId: variantId
        });
      }
    });
    
    // Sort by minQty in descending order (highest minimum quantity first)
    return tiers.sort((a, b) => b.minQty - a.minQty);
  }

  // Find the best matching group price tier
  function findBestGroupTier(variantId, qty, paxType) {
    console.log(`Finding best group tier for variant ${variantId}, qty ${qty}, type ${paxType}`);
    if (!qty || qty < 1) {
      console.log('No quantity or quantity < 1, returning null');
      return null;
    }
    
    const tiers = getGroupPricingTiers(variantId);
    console.log('Available tiers:', tiers);
    
    if (!tiers || tiers.length === 0) {
      console.log('No group pricing tiers found');
      return null;
    }
    
    // Filter by pax type and find the best tier where qty >= minQty
    const matchingTiers = tiers.filter(tier => {
      const matches = tier.type === paxType && qty >= tier.minQty;
      console.log(`Tier ${tier.minQty}+ (${tier.type}): ${matches ? 'MATCH' : 'no match'}`);
      return matches;
    });
    
    console.log('Matching tiers:', matchingTiers);
    
    // Return the tier with the highest min_quantity that's <= qty
    if (matchingTiers.length > 0) {
      const bestTier = matchingTiers.reduce((best, current) => 
        (current.minQty > (best?.minQty || 0)) ? current : best
      );
      console.log('Selected best tier:', bestTier);
      return bestTier;
    }
    
    console.log('No matching tiers found');
    return null;
  }

  function recalc() {
    console.log('--- Recalculating prices ---');
    // Update unit price labels and estimate
    let total = 0;
    let totalDiscount = 0;
    let totalOriginal = 0;
    let picked = [];
    let groupDiscountApplied = false;
    updateItemCount(); // Update the item count in the summary header
    
    // First pass: calculate quantities for group pricing
    const variantQuantities = {};
    variants.forEach(v => {
      const vid = v.id;
      const qa = parseInt(document.getElementById('qtyA-' + vid).value||'0',10);
      let qc = parseInt(document.getElementById('qtyC-' + vid).value||'0',10);
      if (packagePolicy === 'adult_only') { qc = 0; }
      variantQuantities[vid] = { adult: qa, child: qc, total: qa + qc };
    });
    
    // Second pass: calculate prices with group discounts (based on total qty per variant)
    variants.forEach(v => {
      const vid = v.id;
      const { adult: qa, child: qc, total: qTotal } = variantQuantities[vid] || { adult: 0, child: 0, total: 0 };

      // Resolve best price for adult and child using total qty (supports flat tiers)
      const prA = bestPriceForVariantPax(vid, qTotal, 'adult');
      const prC = bestPriceForVariantPax(vid, qTotal, 'child');

      const unitA = prA?._unit_price || 0;
      const unitC = prC?._unit_price || 0;
      const baseA = prA?._base_price || unitA;
      const baseC = prC?._base_price || unitC;

      const isGroupA = !!prA?._is_group;
      const isGroupC = !!prC?._is_group;

      const subA = unitA * qa;
      const subC = unitC * qc;
      const sub = subA + subC;

      const discountA = Math.max(0, (baseA - unitA)) * qa;
      const discountC = Math.max(0, (baseC - unitC)) * qc;
      const lineDiscount = discountA + discountC;

      total += sub;
      totalOriginal += (baseA * qa) + (baseC * qc);
      totalDiscount += lineDiscount;
      if (isGroupA || isGroupC) groupDiscountApplied = true;

      // Update display
      const subElement = document.getElementById('sub-' + vid);
      if (subElement) {
        subElement.textContent = `${Number(sub).toLocaleString()} THB`;
        if (lineDiscount > 0) {
          subElement.innerHTML += ` <span class="text-xs text-green-600">(Save ${Number(lineDiscount).toLocaleString()} THB)</span>`;
        }
      }
      
      // Add to picked items for summary
      if (qa > 0 && prA) {
        picked.push({ 
          vid, 
          name: v.name + ' • Adult', 
          qty: qa, 
          unit: unitA, 
          base: baseA,
          sub: subA,
          discount: discountA,
          tier: prA.price_type,
          isGroup: isGroupA
        });
      }
      if (qc > 0 && prC) {
        picked.push({ 
          vid, 
          name: v.name + ' • Child', 
          qty: qc, 
          unit: unitC, 
          base: baseC,
          sub: subC,
          discount: discountC,
          tier: prC.price_type,
          isGroup: isGroupC
        });
      }
    });
    // Update left card totals
    const subtotalEl = document.getElementById('subtotalAmount');
    if (subtotalEl) subtotalEl.textContent = `${Number(totalOriginal).toLocaleString()} THB`;
    const estEl = document.getElementById('estTotal');
    if (estEl) estEl.textContent = `${Number(total).toLocaleString()} THB`;
    const discEl = document.getElementById('discountAmount');
    if (discEl) {
      discEl.innerHTML = totalDiscount > 0 ? `<div class="flex items-center justify-between text-sm text-green-600"><span>Group Discount</span><span>-${Number(totalDiscount).toLocaleString()} THB</span></div>` : '';
    }

    // Render right-side summary
    const sp = document.getElementById('summaryPanel');
    if (!picked.length) { sp.textContent = 'No selections yet.'; return; }
    const showTime = document.getElementById('timeSelect').value || 'Any time';
    const bookingDate = document.getElementById('bookingDate').value || 'Not selected';
    const showTimeVal = document.getElementById('timeSelect') ? (document.getElementById('timeSelect').value || '') : '';
    const hasShowTimes = document.getElementById('timeSelect') && document.getElementById('timeSelect').options.length > 1;
    const showTimeLine = hasShowTimes ? `
            <div class="mt-1 text-xs text-slate-500 flex items-center">
              <svg class="w-3.5 h-3.5 mr-1.5 text-slate-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              Show time: ${showTimeVal}
            </div>` : '';

    let html = `
      <div class="bg-gradient-to-r from-blue-50 to-slate-50 p-4 rounded-xl border border-blue-100 shadow-sm mb-4">
        <div class="flex items-start space-x-3">
          <div class="bg-white p-2 rounded-lg shadow-sm border border-blue-100">
            <svg class="w-6 h-6 text-blue-500" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.8" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
            </svg>
          </div>
          <div class="flex-1">
            <div class="text-xs font-semibold text-blue-600 uppercase tracking-wider mb-1">Booking Date</div>
            <div class="text-lg font-bold text-slate-800">${bookingDate}</div>
            ${showTimeLine}
          </div>
        </div>
      </div>
    `;
    // Group items by variant and pax type
    const groupedItems = {};
    picked.forEach(it => {
      const key = `${it.vid}-${it.name.includes('Adult') ? 'adult' : 'child'}`;
      if (!groupedItems[key]) {
        groupedItems[key] = {
          ...it,
          totalQty: it.qty,
          totalSub: it.sub,
          totalDiscount: it.discount || 0
        };
      } else {
        groupedItems[key].totalQty += it.qty;
        groupedItems[key].totalSub += it.sub;
        groupedItems[key].totalDiscount += it.discount || 0;
      }
    });

    html += '<div class="space-y-2">';
    Object.values(groupedItems).forEach(it => {
      const isGroup = it.isGroup;
      const discount = it.totalDiscount;
      const originalSub = it.totalSub + discount;
      
      html += `
      <div class="bg-slate-50 rounded-lg p-3 border border-slate-100">
        <div class="flex items-center justify-between">
          <div>
            <div class="font-medium text-slate-800">${it.name.replace(' • Adult', '').replace(' • Child', '')} • ${it.name.includes('Adult') ? 'Adults' : 'Children'}</div>
            <div class="mt-1 text-sm text-slate-600">
              ${it.totalQty} × ${isGroup ? 
                `<span class="line-through text-slate-400">${Number(it.base).toLocaleString()}</span> 
                 <span class="text-green-600 font-medium">${Number(it.unit).toLocaleString()}</span> THB` : 
                `${Number(it.unit).toLocaleString()} THB`}
            </div>
          </div>
          <div class="text-right">
            <div class="font-semibold text-slate-800">${Number(it.totalSub).toLocaleString()} THB</div>
            ${isGroup ? `
              <div class="text-xs text-slate-500">
                <span class="line-through">${Number(originalSub).toLocaleString()}</span>
                <span class="ml-1 text-green-600 font-medium">Save ${Number(discount).toLocaleString()} THB</span>
              </div>` : ''}
          </div>
        </div>
      </div>`;
    });
    html += '</div>';
    // Add discount line if there are any discounts
    if (totalDiscount > 0) {
      html += `
      <div class="bg-green-50 border border-green-100 rounded-lg p-3 mt-2">
        <div class="flex items-center justify-between text-green-700">
          <div class="font-medium">Total Group Discount</div>
          <div class="font-bold">-${Number(totalDiscount).toLocaleString()} THB</div>
        </div>
      </div>`;
    }
    
    // Transport add-on
    let transport = { id: null, label: null, price: 0 };
    const pickedTransport = document.querySelector('input[name="transport_option"]:checked');
    if (pickedTransport && pickedTransport.value) {
      transport.id = parseInt(pickedTransport.value, 10);
      transport.price = parseFloat(pickedTransport.getAttribute('data-price') || '0');
      transport.label = pickedTransport.getAttribute('data-label') || 'Transport';
      total += transport.price;
      const hidId = document.getElementById('transport_option_id');
      const hidPrice = document.getElementById('transport_option_price');
      if (hidId) hidId.value = String(transport.id);
      if (hidPrice) hidPrice.value = String(transport.price);
    } else {
      const hidId = document.getElementById('transport_option_id');
      const hidPrice = document.getElementById('transport_option_price');
      if (hidId) hidId.value = '';
      if (hidPrice) hidPrice.value = '0';
    }

    // Add total amount
    html += `
      <div class="mt-4 pt-3 border-t border-slate-200">
        <div class="flex items-center justify-between">
          <div class="font-semibold text-slate-800">Total Amount</div>
          <div class="text-lg font-bold text-brand-600">${Number(total).toLocaleString()} THB</div>
        </div>
        ${totalDiscount > 0 ? 
          `<div class="text-xs text-slate-500 text-right mt-1">
            You save ${Number(totalDiscount).toLocaleString()} THB (${Math.round((totalDiscount / (total + totalDiscount)) * 100)}% off)
          </div>` : ''
        }
        ${transport.id ? `<div class="text-xs text-slate-600 text-right mt-1">Includes Transport (${transport.label}): +${Number(transport.price).toLocaleString()} THB</div>` : ''}
      </div>`;
    sp.innerHTML = html;
  }

  function prepareBatchSubmit() {
    const form = document.getElementById('cartForm');
    // remove previous items[]
    [...form.querySelectorAll('input[name^="items["]')].forEach(n => n.remove());
    let count = 0;
    const showTime = document.getElementById('timeSelect').value || '';
    variants.forEach(v => {
      const vid = v.id;
      const qa = parseInt(document.getElementById('qtyA-' + vid).value||'0',10);
      let qc = parseInt(document.getElementById('qtyC-' + vid).value||'0',10);
      if (packagePolicy === 'adult_only') { qc = 0; }
      const qTotal = qa + qc;
      const append = (name, val) => { const i = document.createElement('input'); i.type='hidden'; i.name=`items[${count}][${name}]`; i.value=val; form.appendChild(i); };
      if (qa > 0) {
        const bestA = bestPriceForVariantPax(vid, qTotal, 'adult'); if (!bestA) return;
        append('package_id', <?= (int)($package['id'] ?? 0) ?>);
        append('variant_id', vid);
        append('price_id', bestA.id);
        append('qty', qa);
        append('show_time', showTime);
        count++;
      }
      if (qc > 0) {
        const bestC = bestPriceForVariantPax(vid, qTotal, 'child'); if (!bestC) return;
        append('package_id', <?= (int)($package['id'] ?? 0) ?>);
        append('variant_id', vid);
        append('price_id', bestC.id);
        append('qty', qc);
        append('show_time', showTime);
        count++;
      }
    });
    // Add transport info fields already present as hidden inputs; nothing to do here
    if (count === 0) { alert('Please select at least one variant quantity.'); return false; }
    return true;
  }

  // init
  if (variants.length) { renderTimes(); renderVariantRows(); }
  // Apply prefill from session
  (function applyPrefill(){
    try {
      // Date
      const d = (prefillGuest && prefillGuest.date) ? String(prefillGuest.date) : '';
      if (d) {
        const di = document.getElementById('bookingDate'); if (di) { di.value = d; }
      }
      // Time
      const t = (prefillGuest && prefillGuest.show_time) ? String(prefillGuest.show_time) : '';
      const sel = document.getElementById('timeSelect');
      if (sel && t) {
        // Add option if missing
        let exists = false; [...sel.options].forEach(o => { if (o.value === t) exists = true; });
        if (!exists) { const opt = document.createElement('option'); opt.value = t; opt.textContent = t; sel.appendChild(opt); }
        sel.value = t;
      }
      // Transport
      const tid = (prefillGuest && prefillGuest.transport && prefillGuest.transport.option_id) ? String(prefillGuest.transport.option_id) : '';
      if (tid !== '') {
        const r = document.querySelector('input[name="transport_option"][value="'+tid+'"]');
        if (r) { r.checked = true; }
        const hidId = document.getElementById('transport_option_id'); if (hidId) hidId.value = tid;
      }
      // Quantities
      Object.entries(prefillQty).forEach(([k, q]) => {
        const [vidStr, pax] = k.split('::'); const vid = parseInt(vidStr, 10); const val = parseInt(q || 0, 10);
        if (!vid || !val) return;
        if (pax === 'child') { const el = document.getElementById('qtyC-' + vid); if (el) el.value = val; }
        else { const el = document.getElementById('qtyA-' + vid); if (el) el.value = val; }
      });
      recalc();
    } catch(e) { console.warn('prefill error', e); }
  })();
  // Transport change listener
  document.querySelectorAll('input[name="transport_option"]').forEach(el => {
    el.addEventListener('change', recalc);
  });

  // Helper to set quantity programmatically from summary (clears both adult/child)
  function setQty(vid, val){ const a = document.getElementById('qtyA-'+vid); const c=document.getElementById('qtyC-'+vid); if (a){a.value=Math.max(0,parseInt(val||'0',10));} if(c){c.value=Math.max(0,parseInt(val||'0',10));} recalc(); }

  // Price resolver for specific pax type
  function bestPriceForVariantPax(vid, qty, paxType) {
    // All prices for this variant
    const all = prices[vid] || [];
    if (!all.length) return null;

    // Base price resolution:
    // 1) exact pax base
    // 2) adult base fallback (so children can use adult base when only that exists)
    // 3) any base row
    let baseRow = all.find(p => p.price_type === 'base' && p.pax_type === paxType)
                || all.find(p => p.price_type === 'base' && p.pax_type === 'adult')
                || all.find(p => p.price_type === 'base');
    if (!baseRow) return null;

    // Candidate group tiers: match same pax OR flat tiers; choose the highest min_quantity <= qty
    const candidates = all
      .filter(p => p.price_type === 'group_tier' && (p.pax_type === paxType || p.pax_type === 'flat'))
      .sort((a,b) => (parseInt(a.min_quantity||'0',10) - parseInt(b.min_quantity||'0',10)));

    let bestTier = null;
    for (const t of candidates) {
      const minq = parseInt(t.min_quantity || '0', 10);
      if (qty >= minq) bestTier = t; else break;
    }

    // Attach helper fields for UI/discount computation
    const result = bestTier ? { ...bestTier } : { ...baseRow };
    // Agent portal should use agent_price first, then customer_price, then vendor_cost
    result._base_price = (baseRow.agent_price ?? baseRow.customer_price ?? baseRow.vendor_cost ?? 0);
    result._unit_price = (result.agent_price ?? result.customer_price ?? result.vendor_cost ?? 0);
    result._is_group = (bestTier != null);
    return result;
  }

  function baseUnitForPax(vid, pax) {
    const list = (prices[vid] || []).filter(p => (p.pax_type === pax) || (p.pax_type === 'flat') || (p.pax_type === null) || (p.pax_type === ''));
    const base = list.find(p => p.price_type === 'base');
    const unit = base ? (base.agent_price ?? base.customer_price ?? base.vendor_cost) : null;
    return unit;
  }
</script>

<?php require __DIR__ . '/partials/footer.php'; ?>
