<?php
// app/Views/admin/taxi_booking_view.php
$title = $title ?? 'Taxi Booking';
$booking = $booking ?? [];
$events = $events ?? [];
$payments = $payments ?? [];
$csrf = $csrf ?? '';

// Status badge colors
$statusBadgeClass = 'bg-light text-dark';
$status = strtolower((string)($booking['status'] ?? ''));
if (in_array($status, ['confirmed', 'completed'])) {
    $statusBadgeClass = 'bg-success text-white';
} elseif (in_array($status, ['cancelled', 'rejected'])) {
    $statusBadgeClass = 'bg-danger text-white';
} elseif (in_array($status, ['pending', 'processing'])) {
    $statusBadgeClass = 'bg-warning text-dark';
}

// Payment status badge
$paymentStatusBadgeClass = 'bg-light text-dark';
$paymentStatus = strtolower((string)($booking['payment_status'] ?? ''));
if ($paymentStatus === 'paid') {
    $paymentStatusBadgeClass = 'bg-success text-white';
} elseif ($paymentStatus === 'refunded') {
    $paymentStatusBadgeClass = 'bg-info text-dark';
} elseif (in_array($paymentStatus, ['failed', 'declined'])) {
    $paymentStatusBadgeClass = 'bg-danger text-white';
} elseif ($paymentStatus === 'pending') {
    $paymentStatusBadgeClass = 'bg-warning text-dark';
}

// Trip completion flag (vendor marked completed)
$isCompleted = !empty($booking['completed_at']);
$completedAtText = $isCompleted ? (date('M j, Y H:i', strtotime((string)$booking['completed_at']))) : '';
?>

<style>
  /* Trip Completed stamp styles */
  .stamp-badge {
    display: inline-flex;
    align-items: center;
    gap: .25rem;
    background: #198754; /* bootstrap success */
    color: #fff;
    padding: .35rem .6rem;
    border-radius: 999px;
    box-shadow: 0 1px 2px rgba(0,0,0,.12);
    font-size: .8rem;
    line-height: 1;
    white-space: nowrap;
  }
  .stamp-badge .stamp-text{ font-weight: 600; }
  .stamp-badge i{ font-size: .9rem; }
  /* Rubber stamp variant */
  .stamp-badge.rubber-stamp{
    padding: .55rem 1rem;
    font-size: .9rem;
    border: 2px solid #0f5132; /* darker success */
    border-radius: 9999px;
    background: linear-gradient(180deg, #27a969 0%, #198754 100%);
    box-shadow: inset 0 1px 0 rgba(255,255,255,.25), 0 2px 6px rgba(0,0,0,.15);
    letter-spacing: .2px;
  }
  .stamp-badge.rubber-stamp .stamp-text{ font-weight: 700; }
  .stamp-badge.rubber-stamp i{ font-size: 1rem; }
  /* Large circular rubber stamp */
  .round-stamp{
    display:inline-flex;
    flex-direction:column;
    align-items:center;
    justify-content:center;
    width: 190px;
    height: 190px;
    border-radius: 50%;
    border: 4px solid #198754; /* success */
    color:#14532d; /* deep green text */
    background: radial-gradient(ellipse at center, rgba(25,135,84,.16) 0%, rgba(25,135,84,.10) 60%, rgba(25,135,84,.05) 100%);
    box-shadow: 0 6px 14px rgba(0,0,0,.13), inset 0 2px 0 rgba(255,255,255,.25);
    text-align:center;
    position:relative;
    user-select:none;
    transform: rotate(-2deg);
  }
  .round-stamp:before{
    content:'';
    position:absolute; inset:12px;
    border: 3px dashed rgba(20,83,45,.55);
    border-radius:50%;
    pointer-events:none;
  }
  /* subtle paper/ink texture */
  .round-stamp:after{
    content:'';
    position:absolute; inset:0;
    border-radius:50%;
    background:
      radial-gradient(circle at 30% 30%, rgba(255,255,255,.08), transparent 40%),
      repeating-linear-gradient(135deg, rgba(0,0,0,.04) 0 2px, rgba(0,0,0,0) 2px 4px);
    mix-blend-mode: overlay;
    pointer-events:none;
  }
  .round-stamp .rs-main{
    font-weight: 900; text-transform:uppercase; letter-spacing:2.2px; font-size: 20px;
  }
  .round-stamp .rs-sub{
    margin-top:6px; font-size:12px; letter-spacing:1px; opacity:.85;
  }
  .round-stamp .rs-flag{ font-size:22px; color:#14532d; margin-bottom:6px; }
  .stamp-container{ margin-top: 0.9rem; margin-bottom: .25rem; display:flex; align-items:center; }
  /* Watermark placement inside right column */
  .stamp-bg{ position: relative; }
  .stamp-bg .content-foreground{ position: relative; z-index: 2; }
  .round-stamp.stamp-watermark{
    position: absolute;
    right: -6px;
    top: 6px;
    opacity: .92; /* visible but elegant */
    z-index: 1;
    pointer-events: none;
    transform: rotate(-3deg);
  }
  @media (min-width: 992px){ /* lg and up bigger */
    .round-stamp{ width: 220px; height:220px; }
    .round-stamp .rs-main{ font-size:24px; }
  }
  @media (prefers-color-scheme: dark){
    .stamp-badge{ box-shadow: 0 1px 2px rgba(0,0,0,.4); }
    .stamp-badge.rubber-stamp{ box-shadow: inset 0 1px 0 rgba(255,255,255,.06), 0 2px 8px rgba(0,0,0,.5); }
  }
</style>

<div class="container-fluid py-4">
    <!-- Header with Breadcrumb -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="d-flex align-items-center gap-3">
            <a href="/admin/booking/taxi" class="btn btn-outline-secondary btn-sm">
                <i class="bi bi-arrow-left me-1"></i> Back to List
            </a>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="bi bi-taxi-front-fill text-primary me-2"></i>Taxi Booking #<?= (int)($booking['id'] ?? 0) ?>
            </h1>
            <nav aria-label="breadcrumb" class="mt-2">
                <ol class="breadcrumb small">
                    <li class="breadcrumb-item"><a href="/admin">Dashboard</a></li>
                    <li class="breadcrumb-item"><a href="/admin/booking/taxi">Taxi Bookings</a></li>
                    <li class="breadcrumb-item active" aria-current="page">#<?= (int)($booking['id'] ?? 0) ?></li>
                </ol>
            </nav>
        </div>
        <div class="d-flex gap-2">
            <a href="#" class="btn btn-outline-primary btn-sm" onclick="window.print(); return false;">
                <i class="bi bi-printer me-1"></i> Print
            </a>
            <div class="d-flex align-items-center gap-2">
                <span class="badge <?= $statusBadgeClass ?> me-2"><?= ucfirst($status) ?></span>
                <?php if ($isCompleted): ?>
                  <span class="badge bg-success">
                    <i class="bi bi-check-circle-fill me-1"></i>Completed
                  </span>
                <?php endif; ?>
                <a href="/admin/booking/taxi/edit?id=<?= (int)($booking['id'] ?? 0) ?>" class="btn btn-sm btn-outline-primary" data-bs-toggle="tooltip" title="Edit Booking">
                  <i class="bi bi-pencil"></i> Edit
                </a>
            </div>
        </div>
    </div>

  <div class="row g-3">
    <div class="col-lg-8">
      <!-- Booking Overview Card -->
      <div class="card shadow-sm mb-4 border-0">
        <div class="card-header bg-light py-3">
          <h6 class="mb-0"><i class="bi bi-card-checklist text-primary me-2"></i>Booking Overview</h6>
        </div>
        <div class="card-body">
          <div class="row g-3">
            <div class="col-md-6">
              <div class="d-flex align-items-center mb-3">
                <div class="bg-light p-2 rounded me-3">
                  <i class="bi bi-ticket-detailed text-primary"></i>
                </div>
                <div>
                  <div class="text-muted small">Booking Code</div>
                  <div class="fw-semibold"><?= htmlspecialchars((string)($booking['booking_code'] ?? '')) ?></div>
                </div>
              </div>
              <div class="d-flex align-items-center mb-3">
                <div class="bg-light p-2 rounded me-3">
                  <i class="bi bi-person-badge text-primary"></i>
                </div>
                <div>
                  <div class="text-muted small">Agent</div>
                  <div class="fw-semibold">
                    #<?= (int)($booking['agent_id'] ?? 0) ?>
                    <?php if (!empty($booking['agent_name'])): ?>
                      <span class="text-muted small">(<?= htmlspecialchars($booking['agent_name']) ?>)</span>
                    <?php endif; ?>
                  </div>
                </div>
              </div>
              <div class="d-flex align-items-center mb-3">
                <div class="bg-light p-2 rounded me-3">
                  <i class="bi bi-building-gear text-primary"></i>
                </div>
                <div>
                  <div class="text-muted small">Vendor</div>
                  <div class="fw-semibold">
                    #<?= (int)($booking['vendor_id'] ?? 0) ?>
                    <?php if (!empty($booking['vendor_name'])): ?>
                      <span class="text-muted small">(<?= htmlspecialchars((string)$booking['vendor_name']) ?>)</span>
                    <?php endif; ?>
                  </div>
                </div>
              </div>
              <div class="d-flex align-items-center mb-3">
                <div class="bg-light p-2 rounded me-3">
                  <i class="bi bi-card-text text-primary"></i>
                </div>
                <div>
                  <div class="text-muted small">Booking For</div>
                  <div class="fw-semibold"><?= !empty($booking['ride_name']) ? htmlspecialchars((string)$booking['ride_name']) : 'N/A' ?></div>
                </div>
              </div>
              <div class="d-flex align-items-center">
                <div class="bg-light p-2 rounded me-3">
                  <i class="bi bi-people text-primary"></i>
                </div>
                <div>
                  <div class="text-muted small">Passengers</div>
                  <div class="fw-semibold"><?= (int)($booking['pax'] ?? 1) ?> <?= (int)($booking['pax'] ?? 1) === 1 ? 'Person' : 'People' ?></div>
                </div>
              </div>
            </div>
            <div class="col-md-6 stamp-bg">
              <div class="d-flex align-items-center mb-3">
                <div class="bg-light p-2 rounded me-3">
                  <i class="bi bi-calendar-event text-primary"></i>
                </div>
                <div>
                  <div class="text-muted small">Trip Date</div>
                  <div class="fw-semibold">
                    <?= !empty($booking['trip_date']) ? date('d M Y', strtotime($booking['trip_date'])) : 'N/A' ?>
                    <?= !empty($booking['pickup_time']) ? '• ' . htmlspecialchars($booking['pickup_time']) : '' ?>
                  </div>
                </div>
              </div>
              <div class="d-flex align-items-center mb-3">
                <div class="bg-light p-2 rounded me-3">
                  <i class="bi bi-car-front text-primary"></i>
                </div>
                <div>
                  <div class="text-muted small">Vehicle Type</div>
                  <div class="fw-semibold"><?php
                    $vehType = trim((string)($booking['vehicle_type'] ?? ''));
                    if ($vehType === '' && !empty($booking['ride_name'])) {
                      // Fallback to ride_name if vehicle_type missing
                      echo htmlspecialchars((string)$booking['ride_name']);
                    } else {
                      echo htmlspecialchars(ucfirst($vehType ?: 'Standard'));
                    }
                  ?></div>
                </div>
              </div>
              <div class="d-flex align-items-center">
                <div class="bg-light p-2 rounded me-3">
                  <i class="bi bi-shield-check text-primary"></i>
                </div>
                <div class="content-foreground">
                  <div class="text-muted small">Status</div>
                  <div>
                    <span class="badge <?= $statusBadgeClass ?> me-2">
                      <i class="bi <?= in_array($status, ['confirmed', 'completed']) ? 'bi-check-circle' : ($status === 'cancelled' ? 'bi-x-circle' : 'bi-hourglass') ?> me-1"></i>
                      <?= ucfirst(htmlspecialchars((string)($booking['status'] ?? 'N/A'))) ?>
                    </span>
                    <span class="badge <?= $paymentStatusBadgeClass ?>">
                      <i class="bi <?= $paymentStatus === 'paid' ? 'bi-credit-card' : ($paymentStatus === 'refunded' ? 'bi-arrow-counterclockwise' : 'bi-hourglass') ?> me-1"></i>
                      <?= ucfirst(htmlspecialchars((string)($booking['payment_status'] ?? 'N/A'))) ?>
                    </span>
                  </div>
                </div>
              </div>
              <?php if ($isCompleted): ?>
                <div class="round-stamp stamp-watermark" title="<?= htmlspecialchars((string)($booking['completed_at'] ?? '')) ?>" data-bs-toggle="tooltip">
                  <i class="bi bi-flag-fill rs-flag"></i>
                  <div class="rs-main">Completed</div>
                  <div class="rs-sub"><?= $completedAtText ?></div>
                </div>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>

      <!-- Customer Details Card -->
      <div class="card shadow-sm mb-4 border-0">
        <div class="card-header bg-light py-3">
          <h6 class="mb-0"><i class="bi bi-person-lines-fill text-primary me-2"></i>Customer Details</h6>
        </div>
        <div class="card-body">
          <div class="d-flex align-items-center mb-4">
            <div class="bg-light p-3 rounded-circle me-3">
              <i class="bi bi-person-circle fs-3 text-primary"></i>
            </div>
            <div>
              <h5 class="mb-1"><?= !empty($booking['customer_name']) ? htmlspecialchars($booking['customer_name']) : 'Guest User' ?></h5>
              <div class="d-flex flex-wrap gap-3 mt-2">
                <?php if (!empty($booking['customer_email'])): ?>
                  <a href="mailto:<?= htmlspecialchars($booking['customer_email']) ?>" class="text-decoration-none">
                    <span class="badge bg-light text-dark">
                      <i class="bi bi-envelope me-1"></i> <?= htmlspecialchars($booking['customer_email']) ?>
                    </span>
                  </a>
                <?php endif; ?>
                <?php if (!empty($booking['customer_phone'])): ?>
                  <a href="tel:<?= preg_replace('/[^0-9+]/', '', $booking['customer_phone']) ?>" class="text-decoration-none">
                    <span class="badge bg-light text-dark">
                      <i class="bi bi-telephone me-1"></i> <?= htmlspecialchars($booking['customer_phone']) ?>
                    </span>
                  </a>
                <?php endif; ?>
              </div>
            </div>
          </div>
          
          <?php if (!empty($booking['customer_notes'])): ?>
            <div class="alert alert-light mt-3">
              <div class="d-flex">
                <i class="bi bi-chat-square-text me-2 text-primary"></i>
                <div>
                  <div class="small text-muted mb-1">Customer Notes:</div>
                  <div class="fst-italic"><?= nl2br(htmlspecialchars($booking['customer_notes'])) ?></div>
                </div>
              </div>
            </div>
          <?php endif; ?>
        </div>
      </div>

      <!-- Route & Pickup/Drop Card -->
      <div class="card shadow-sm mb-4 border-0">
        <div class="card-header bg-light py-3">
          <h6 class="mb-0"><i class="bi bi-geo-alt-fill text-primary me-2"></i>Route & Pickup/Drop</h6>
        </div>
        <div class="card-body">
          <!-- Route Summary -->
          <div class="route-summary mb-4 p-3 bg-light rounded">
            <div class="d-flex align-items-center">
              <div class="text-center me-3">
                <div class="bg-primary bg-opacity-10 text-primary rounded-circle d-inline-flex align-items-center justify-content-center" style="width: 36px; height: 36px;">
                  <i class="bi bi-geo-alt"></i>
                </div>
                <div class="small text-muted mt-1">From</div>
              </div>
              <div class="flex-grow-1">
                <div class="fw-semibold"><?= htmlspecialchars((string)($booking['from_text'] ?? 'N/A')) ?></div>
              </div>
              <div class="mx-2 text-muted">
                <i class="bi bi-arrow-right"></i>
              </div>
              <div class="text-center ms-3">
                <div class="bg-primary bg-opacity-10 text-primary rounded-circle d-inline-flex align-items-center justify-content-center" style="width: 36px; height: 36px;">
                  <i class="bi bi-geo-alt"></i>
                </div>
                <div class="small text-muted mt-1">To</div>
              </div>
              <div class="flex-grow-1">
                <div class="fw-semibold"><?= htmlspecialchars((string)($booking['to_text'] ?? 'N/A')) ?></div>
              </div>
            </div>
          </div>

          <!-- Pickup & Drop Details -->
          <div class="row g-3">
            <!-- Pickup Details -->
            <div class="col-md-6">
              <div class="card h-100 border-0 shadow-sm">
                <div class="card-body">
                  <div class="d-flex align-items-center mb-3">
                    <div class="bg-primary bg-opacity-10 text-primary rounded-circle p-2 me-3">
                      <i class="bi bi-geo-alt"></i>
                    </div>
                    <h6 class="mb-0">Pickup Details</h6>
                  </div>
                  
                  <?php
                  $pickupType = strtolower((string)($booking['pickup_type'] ?? ''));
                  $pickupIcon = 'bi-geo-alt';
                  $pickupTitle = 'Location';
                  $pickupContent = '';
                  
                  if ($pickupType === 'airport') {
                      $pickupIcon = 'bi-airplane';
                      $pickupTitle = 'Airport';
                      $pickupContent = htmlspecialchars((string)($booking['pickup_airport_code'] ?? ''));
                  } elseif ($pickupType === 'hotel') {
                      $pickupIcon = 'bi-building';
                      $pickupTitle = 'Hotel';
                      $pickupContent = htmlspecialchars((string)($booking['pickup_hotel_name'] ?? ''));
                      if (!empty($booking['pickup_hotel_address'])) {
                          $pickupContent .= '<div class="small text-muted">' . htmlspecialchars((string)$booking['pickup_hotel_address']) . '</div>';
                      }
                  } elseif ($pickupType === 'apartment') {
                      $pickupIcon = 'bi-house';
                      $pickupTitle = 'Apartment';
                      $pickupContent = htmlspecialchars((string)($booking['pickup_apartment_name'] ?? ''));
                      if (!empty($booking['pickup_apartment_address'])) {
                          $pickupContent .= '<div class="small text-muted">' . htmlspecialchars((string)$booking['pickup_apartment_address']) . '</div>';
                      }
                  } elseif ($pickupType === 'address') {
                      $pickupIcon = 'bi-geo';
                      $pickupTitle = 'Address';
                      $pickupContent = htmlspecialchars((string)($booking['pickup_address'] ?? ''));
                  } else {
                      $pickupContent = 'Not specified';
                  }
                  ?>
                  
                  <div class="d-flex align-items-start mb-2">
                    <div class="me-2 text-muted">
                      <i class="bi <?= $pickupIcon ?>"></i>
                    </div>
                    <div>
                      <div class="small text-muted"><?= $pickupTitle ?></div>
                      <div class="fw-semibold"><?= $pickupContent ?: '—' ?></div>
                    </div>
                  </div>
                  
                  <?php if (!empty($booking['pickup_time'])): ?>
                  <div class="d-flex align-items-center mb-2">
                    <div class="me-2 text-muted">
                      <i class="bi bi-clock"></i>
                    </div>
                    <div>
                      <div class="small text-muted">Pickup Time</div>
                      <div class="fw-semibold"><?= htmlspecialchars($booking['pickup_time']) ?></div>
                    </div>
                  </div>
                  <?php endif; ?>
                  
                  <?php if (!empty($booking['flight_no'])): ?>
                  <div class="d-flex align-items-center">
                    <div class="me-2 text-muted">
                      <i class="bi bi-airplane"></i>
                    </div>
                    <div>
                      <div class="small text-muted">Flight Number</div>
                      <div class="fw-semibold"><?= htmlspecialchars($booking['flight_no']) ?></div>
                    </div>
                  </div>
                  <?php endif; ?>
                </div>
              </div>
            </div>
            
            <!-- Drop-off Details -->
            <div class="col-md-6">
              <div class="card h-100 border-0 shadow-sm">
                <div class="card-body">
                  <div class="d-flex align-items-center mb-3">
                    <div class="bg-primary bg-opacity-10 text-primary rounded-circle p-2 me-3">
                      <i class="bi bi-geo-alt"></i>
                    </div>
                    <h6 class="mb-0">Drop-off Details</h6>
                  </div>
                  
                  <?php
                  $dropType = strtolower((string)($booking['drop_type'] ?? ''));
                  $dropIcon = 'bi-geo-alt';
                  $dropTitle = 'Location';
                  $dropContent = '';
                  
                  if ($dropType === 'airport') {
                      $dropIcon = 'bi-airplane';
                      $dropTitle = 'Airport';
                      $dropContent = htmlspecialchars((string)($booking['drop_airport_code'] ?? ''));
                  } elseif ($dropType === 'hotel') {
                      $dropIcon = 'bi-building';
                      $dropTitle = 'Hotel';
                      $dropContent = htmlspecialchars((string)($booking['drop_hotel_name'] ?? ''));
                      if (!empty($booking['drop_hotel_address'])) {
                          $dropContent .= '<div class="small text-muted">' . htmlspecialchars((string)$booking['drop_hotel_address']) . '</div>';
                      }
                  } elseif ($dropType === 'apartment') {
                      $dropIcon = 'bi-house';
                      $dropTitle = 'Apartment';
                      $dropContent = htmlspecialchars((string)($booking['drop_apartment_name'] ?? ''));
                      if (!empty($booking['drop_apartment_address'])) {
                          $dropContent .= '<div class="small text-muted">' . htmlspecialchars((string)$booking['drop_apartment_address']) . '</div>';
                      }
                  } elseif ($dropType === 'address') {
                      $dropIcon = 'bi-geo';
                      $dropTitle = 'Address';
                      $dropContent = htmlspecialchars((string)($booking['drop_address'] ?? ''));
                  } else {
                      $dropContent = 'Not specified';
                  }
                  ?>
                  
                  <div class="d-flex align-items-start mb-3">
                    <div class="me-2 text-muted">
                      <i class="bi <?= $dropIcon ?>"></i>
                    </div>
                    <div>
                      <div class="small text-muted"><?= $dropTitle ?></div>
                      <div class="fw-semibold"><?= $dropContent ?: '—' ?></div>
                    </div>
                  </div>
                  
                  <?php if (!empty($booking['return_time'])): ?>
                  <div class="d-flex align-items-center">
                    <div class="me-2 text-muted">
                      <i class="bi bi-clock-history"></i>
                    </div>
                    <div>
                      <div class="small text-muted">Return Time</div>
                      <div class="fw-semibold"><?= htmlspecialchars($booking['return_time']) ?></div>
                    </div>
                  </div>
                  <?php endif; ?>
                </div>
              </div>
            </div>
          </div>
          
          <!-- Additional Notes -->
          <?php if (!empty($booking['special_instructions'])): ?>
          <div class="mt-4">
            <div class="alert alert-light">
              <div class="d-flex">
                <i class="bi bi-info-circle me-2 text-primary"></i>
                <div>
                  <div class="small text-muted mb-1">Special Instructions</div>
                  <div class="fst-italic"><?= nl2br(htmlspecialchars($booking['special_instructions'])) ?></div>
                </div>
              </div>
            </div>
          </div>
          <?php endif; ?>
        </div>
      </div>

      <!-- Stops/Itinerary Card -->
      <div class="card shadow-sm mb-4 border-0">
        <div class="card-header bg-light py-3">
          <div class="d-flex justify-content-between align-items-center">
            <h6 class="mb-0"><i class="bi bi-signpost-split-fill text-primary me-2"></i>Journey Itinerary</h6>
            <span class="badge bg-primary bg-opacity-10 text-primary">
              <?= count($stops ?? []) ?> <?= count($stops ?? []) === 1 ? 'Stop' : 'Stops' ?>
            </span>
          </div>
        </div>
        <div class="card-body">
          <?php 
          $itJson = (string)($booking['itinerary_json'] ?? ''); 
          $stops = $itJson ? json_decode($itJson, true) : []; 
          ?>
          
          <?php if (is_array($stops) && count($stops) > 0): ?>
            <div class="timeline">
              <?php foreach ($stops as $i => $s): 
                $isFirst = $i === 0;
                $isLast = $i === count($stops) - 1;
                $stopType = strtolower($s['type'] ?? 'stop');
                $icon = 'bi-geo-alt';
                $bgClass = 'bg-primary';
                
                if ($stopType === 'pickup') {
                  $icon = 'bi-geo-alt-fill';
                  $bgClass = 'bg-success';
                } elseif ($stopType === 'dropoff' || $stopType === 'drop_off') {
                  $icon = 'bi-geo-alt-fill';
                  $bgClass = 'bg-danger';
                } elseif ($stopType === 'wait') {
                  $icon = 'bi-clock';
                  $bgClass = 'bg-warning';
                } elseif ($stopType === 'break') {
                  $icon = 'bi-cup-hot';
                  $bgClass = 'bg-info';
                }
                
                $waitTime = !empty($s['wait_minutes']) ? (int)$s['wait_minutes'] : 0;
                $extraPrice = !empty($s['extra_price']) ? (float)$s['extra_price'] : 0;
              ?>
                <div class="timeline-item <?= $isLast ? 'timeline-item-last' : '' ?>">
                  <div class="timeline-badge <?= $bgClass ?> bg-opacity-10 text-primary d-flex align-items-center justify-content-center">
                    <i class="bi <?= $icon ?>"></i>
                  </div>
                  <div class="timeline-content card shadow-sm border-0">
                    <div class="card-body p-3">
                      <div class="d-flex justify-content-between align-items-start">
                        <div>
                          <h6 class="mb-1">
                            <?= !empty($s['label']) ? htmlspecialchars($s['label']) : ('Stop ' . ($i + 1)) ?>
                            <?php if ($stopType === 'pickup'): ?>
                              <span class="badge bg-success bg-opacity-10 text-success small ms-2">Pickup</span>
                            <?php elseif ($stopType === 'dropoff' || $stopType === 'drop_off'): ?>
                              <span class="badge bg-danger bg-opacity-10 text-danger small ms-2">Drop-off</span>
                            <?php endif; ?>
                          </h6>
                          <?php if (!empty($s['address'])): ?>
                            <p class="mb-1 text-muted small">
                              <i class="bi bi-geo-alt me-1"></i> 
                              <?= htmlspecialchars($s['address']) ?>
                            </p>
                          <?php endif; ?>
                        </div>
                        <div class="text-end">
                          <?php if ($waitTime > 0): ?>
                            <span class="badge bg-warning bg-opacity-10 text-warning mb-1" data-bs-toggle="tooltip" title="Wait time">
                              <i class="bi bi-clock-history me-1"></i> <?= $waitTime ?> min
                            </span>
                          <?php endif; ?>
                          <?php if ($extraPrice > 0): ?>
                            <div class="text-success fw-semibold">
                              +<?= number_format($extraPrice, 2) ?> THB
                            </div>
                          <?php endif; ?>
                        </div>
                      </div>
                      
                      <?php if (!empty($s['notes'])): ?>
                        <div class="alert alert-light p-2 mt-2 mb-0 small">
                          <i class="bi bi-chat-square-text me-1 text-muted"></i>
                          <?= nl2br(htmlspecialchars($s['notes'])) ?>
                        </div>
                      <?php endif; ?>
                      
                      <?php if (!empty($s['contact_name']) || !empty($s['contact_phone'])): ?>
                        <div class="d-flex align-items-center mt-2 pt-2 border-top small text-muted">
                          <i class="bi bi-person-lines me-2"></i>
                          <div>
                            <?php if (!empty($s['contact_name'])): ?>
                              <div class="fw-medium"><?= htmlspecialchars($s['contact_name']) ?></div>
                            <?php endif; ?>
                            <?php if (!empty($s['contact_phone'])): ?>
                              <div>
                                <a href="tel:<?= preg_replace('/[^0-9+]/', '', $s['contact_phone']) ?>" class="text-decoration-none">
                                  <i class="bi bi-telephone me-1"></i> <?= htmlspecialchars($s['contact_phone']) ?>
                                </a>
                              </div>
                            <?php endif; ?>
                          </div>
                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                </div>
              <?php endforeach; ?>
            </div>
            
            <style>
              .timeline {
                position: relative;
                padding-left: 2rem;
              }
              .timeline-item {
                position: relative;
                padding-bottom: 2rem;
              }
              .timeline-item:not(.timeline-item-last)::after {
                content: '';
                position: absolute;
                left: 1.5rem;
                top: 3.5rem;
                height: calc(100% - 1rem);
                width: 2px;
                background-color: #e9ecef;
                z-index: 1;
              }
              .timeline-badge {
                position: absolute;
                left: -2rem;
                top: 0;
                width: 3rem;
                height: 3rem;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 2;
                border: 2px solid #fff;
              }
              .timeline-content {
                margin-left: 1.5rem;
                transition: transform 0.2s, box-shadow 0.2s;
              }
              .timeline-content:hover {
                transform: translateY(-2px);
                box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.05) !important;
              }
              @media (max-width: 767.98px) {
                .timeline {
                  padding-left: 1.5rem;
                }
                .timeline-badge {
                  left: -1.5rem;
                  width: 2.5rem;
                  height: 2.5rem;
                }
                .timeline-content {
                  margin-left: 1rem;
                }
              }
            </style>
          <?php else: ?>
            <div class="text-center py-4">
              <div class="bg-light rounded-circle d-inline-flex align-items-center justify-content-center mb-3" style="width: 80px; height: 80px;">
                <i class="bi bi-signpost text-muted" style="font-size: 2rem;"></i>
              </div>
              <h6 class="text-muted mb-1">No Stops Added</h6>
              <p class="text-muted small mb-0">This booking doesn't have any stops in the itinerary.</p>
            </div>
          <?php endif; ?>
        </div>
      </div>

      <!-- Audit Trail Card -->
      <div class="card shadow-sm mb-4 border-0">
        <div class="card-header bg-light py-3">
          <div class="d-flex justify-content-between align-items-center">
            <h6 class="mb-0"><i class="bi bi-clock-history text-primary me-2"></i>Audit Trail</h6>
            <span class="badge bg-primary bg-opacity-10 text-primary">
              <?= count($events ?? []) ?> <?= count($events ?? []) === 1 ? 'Event' : 'Events' ?>
            </span>
          </div>
        </div>
        <div class="card-body p-0">
          <?php if (!empty($events)): ?>
            <div class="audit-trail">
              <?php 
              $prevDate = null;
              foreach ($events as $i => $ev): 
                $eventDate = !empty($ev['created_at']) ? date('Y-m-d', strtotime($ev['created_at'])) : '';
                $showDateHeader = $eventDate !== $prevDate;
                $isLast = $i === count($events) - 1;
                $prevDate = $eventDate;
                
                // Determine event icon and color
                $eventType = strtolower($ev['event_type'] ?? '');
                $eventIcon = 'bi-info-circle';
                $eventColor = 'primary';
                
                if (strpos($eventType, 'create') !== false) {
                  $eventIcon = 'bi-plus-circle';
                  $eventColor = 'success';
                } elseif (strpos($eventType, 'update') !== false) {
                  $eventIcon = 'bi-pencil-square';
                  $eventColor = 'info';
                } elseif (strpos($eventType, 'delete') !== false || strpos($eventType, 'cancel') !== false) {
                  $eventIcon = 'bi-trash';
                  $eventColor = 'danger';
                } elseif (strpos($eventType, 'payment') !== false) {
                  $eventIcon = 'bi-credit-card';
                  $eventColor = 'success';
                } elseif (strpos($eventType, 'status') !== false) {
                  $eventIcon = 'bi-arrow-repeat';
                  $eventColor = 'warning';
                }
                
                // Format the event time
                $eventTime = !empty($ev['created_at']) ? date('H:i', strtotime($ev['created_at'])) : '';
                $eventFullDate = !empty($ev['created_at']) ? date('M j, Y', strtotime($ev['created_at'])) : '';
                
                // Get user info if available
                $userName = 'System';
                $userInitials = 'S';
                $userId = (int)($ev['user_id'] ?? 0);
                
                if (!empty($ev['user_name'])) {
                  $userName = htmlspecialchars($ev['user_name']);
                  $nameParts = explode(' ', $userName);
                  $userInitials = strtoupper(substr($nameParts[0], 0, 1) . (isset($nameParts[1]) ? substr($nameParts[1], 0, 1) : ''));
                } elseif ($userId > 0) {
                  $userName = 'User #' . $userId;
                  $userInitials = 'U';
                }
                
                // Parse notes for better display
                $note = $ev['note'] ?? '';
                $noteLines = !empty($note) ? explode("\n", $note) : [];
                $mainNote = array_shift($noteLines);
                $additionalNotes = $noteLines;
              ?>
                <?php if ($showDateHeader): ?>
                  <div class="audit-date-header">
                    <span><?= $eventFullDate ?></span>
                  </div>
                <?php endif; ?>
                
                <div class="audit-item <?= $isLast ? 'audit-item-last' : '' ?>">
                  <div class="audit-item-icon bg-<?= $eventColor ?> bg-opacity-10 text-<?= $eventColor ?>">
                    <i class="bi <?= $eventIcon ?>"></i>
                  </div>
                  <div class="audit-item-content">
                    <div class="d-flex justify-content-between align-items-start">
                      <div class="d-flex align-items-center">
                        <div class="audit-user-avatar bg-light text-dark">
                          <?= $userInitials ?>
                        </div>
                        <div class="ms-2">
                          <div class="fw-semibold"><?= $userName ?></div>
                          <div class="text-muted small" title="<?= htmlspecialchars((string)($ev['created_at'] ?? '')) ?>" data-bs-toggle="tooltip">
                            <i class="bi bi-<?= $eventIcon ?> me-1"></i>
                            <?= ucfirst(htmlspecialchars($ev['event_type'] ?? 'Event')) ?>
                            <span class="ms-1">— <?= $eventFullDate ?> <?= $eventTime ?></span>
                          </div>
                        </div>
                      </div>
                      <div class="text-muted small"><?= $eventTime ?></div>
                    </div>
                    
                    <?php if (!empty($mainNote)): ?>
                      <div class="audit-note mt-2">
                        <div class="fw-medium"><?= nl2br(htmlspecialchars($mainNote)) ?></div>
                        <?php if (!empty($additionalNotes)): ?>
                          <div class="audit-additional-notes mt-2">
                            <?php foreach ($additionalNotes as $line): ?>
                              <?php if (trim($line) !== ''): ?>
                                <div class="small text-muted">• <?= htmlspecialchars(trim($line)) ?></div>
                              <?php endif; ?>
                            <?php endforeach; ?>
                          </div>
                        <?php endif; ?>
                      </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($ev['old_values']) || !empty($ev['new_values'])): ?>
                      <div class="audit-changes mt-2">
                        <?php 
                        $oldValues = !empty($ev['old_values']) ? (is_string($ev['old_values']) ? json_decode($ev['old_values'], true) : (array)$ev['old_values']) : [];
                        $newValues = !empty($ev['new_values']) ? (is_string($ev['new_values']) ? json_decode($ev['new_values'], true) : (array)$ev['new_values']) : [];
                        $allKeys = array_unique(array_merge(array_keys($oldValues), array_keys($newValues)));
                        
                        if (!empty($allKeys)): 
                        ?>
                          <div class="accordion" id="auditChanges<?= $i ?>">
                            <div class="accordion-item border-0">
                              <h2 class="accordion-header" id="heading<?= $i ?>">
                                <button class="accordion-button collapsed p-0 bg-transparent shadow-none" type="button" data-bs-toggle="collapse" data-bs-target="#collapse<?= $i ?>" aria-expanded="false" aria-controls="collapse<?= $i ?>">
                                  <span class="small">View Changes <i class="bi bi-chevron-down ms-1"></i></span>
                                </button>
                              </h2>
                              <div id="collapse<?= $i ?>" class="accordion-collapse collapse" aria-labelledby="heading<?= $i ?>" data-bs-parent="#auditChanges<?= $i ?>">
                                <div class="accordion-body p-0 pt-2">
                                  <div class="table-responsive">
                                    <table class="table table-sm table-borderless mb-0">
                                      <thead>
                                        <tr>
                                          <th class="small text-muted">Field</th>
                                          <th class="small text-muted">Old Value</th>
                                          <th class="small text-muted">New Value</th>
                                        </tr>
                                      </thead>
                                      <tbody>
                                        <?php foreach ($allKeys as $key): 
                                          $oldVal = $oldValues[$key] ?? '';
                                          $newVal = $newValues[$key] ?? '';
                                          
                                          // Format values for display
                                          if (is_array($oldVal)) $oldVal = json_encode($oldVal);
                                          if (is_array($newVal)) $newVal = json_encode($newVal);
                                          if ($oldVal === '') $oldVal = '<span class="text-muted">(empty)</span>';
                                          if ($newVal === '') $newVal = '<span class="text-muted">(empty)</span>';
                                          
                                          $isChanged = ($oldVal !== $newVal);
                                        ?>
                                          <tr class="<?= $isChanged ? 'table-warning' : '' ?>">
                                            <td class="small"><?= ucwords(str_replace('_', ' ', $key)) ?></td>
                                            <td class="small"><?= $isChanged ? $oldVal : $newVal ?></td>
                                            <td class="small"><?= $isChanged ? $newVal : '' ?></td>
                                          </tr>
                                        <?php endforeach; ?>
                                      </tbody>
                                    </table>
                                  </div>
                                </div>
                              </div>
                            </div>
                          </div>
                        <?php endif; ?>
                      </div>
                    <?php endif; ?>
                  </div>
                </div>
              <?php endforeach; ?>
            </div>
            
            <style>
              .audit-trail {
                position: relative;
                padding: 0 1rem 1rem 1rem;
              }
              .audit-date-header {
                position: relative;
                margin: 1.5rem 0 1rem 0;
                text-align: center;
              }
              .audit-date-header:first-child {
                margin-top: 0;
              }
              .audit-date-header span {
                display: inline-block;
                background-color: #f8f9fa;
                padding: 0.25rem 1rem;
                border-radius: 1rem;
                font-size: 0.75rem;
                font-weight: 500;
                color: #6c757d;
                border: 1px solid #e9ecef;
              }
              .audit-item {
                position: relative;
                padding-left: 2.5rem;
                margin-bottom: 1.5rem;
              }
              .audit-item:not(.audit-item-last)::before {
                content: '';
                position: absolute;
                left: 1.25rem;
                top: 2.5rem;
                height: calc(100% - 0.5rem);
                width: 2px;
                background-color: #e9ecef;
                z-index: 1;
              }
              .audit-item-icon {
                position: absolute;
                left: 0;
                top: 0;
                width: 2.5rem;
                height: 2.5rem;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 2;
                border: 2px solid #fff;
              }
              .audit-user-avatar {
                width: 2rem;
                height: 2rem;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-weight: 600;
                font-size: 0.75rem;
              }
              .audit-note {
                background-color: #f8f9fa;
                border-radius: 0.5rem;
                padding: 0.75rem;
                margin-top: 0.5rem;
                font-size: 0.875rem;
              }
              .audit-additional-notes {
                margin-top: 0.5rem;
                padding-left: 0.5rem;
                border-left: 2px solid #e9ecef;
              }
              .audit-changes {
                margin-top: 0.75rem;
              }
              .audit-changes .accordion-button::after {
                background-size: 0.75rem;
                margin-left: 0.5rem;
              }
              .audit-changes .accordion-button:not(.collapsed)::after {
                background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16' fill='%23212529'%3e%3cpath fill-rule='evenodd' d='M1.646 4.646a.5.5 0 0 1 .708 0L8 10.293l5.646-5.647a.5.5 0 0 1 .708.708l-6 6a.5.5 0 0 1-.708 0l-6-6a.5.5 0 0 1 0-.708z'/%3e%3c/svg%3e");
              }
              .audit-changes .accordion-button:focus {
                box-shadow: none;
              }
              .audit-changes .table {
                font-size: 0.75rem;
              }
              .audit-changes .table th {
                font-weight: 500;
                padding: 0.25rem 0.5rem;
              }
              .audit-changes .table td {
                padding: 0.25rem 0.5rem;
                vertical-align: middle;
              }
              @media (max-width: 767.98px) {
                .audit-item {
                  padding-left: 2rem;
                }
                .audit-item-icon {
                  width: 2rem;
                  height: 2rem;
                  font-size: 0.75rem;
                }
              }
            </style>
            
            <script>
              // Initialize tooltips for the audit trail
              document.addEventListener('DOMContentLoaded', function() {
                var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
                var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                  return new bootstrap.Tooltip(tooltipTriggerEl);
                });
              });
            </script>
          <?php else: ?>
            <div class="text-center py-5">
              <div class="bg-light rounded-circle d-inline-flex align-items-center justify-content-center mb-3" style="width: 80px; height: 80px;">
                <i class="bi bi-clock-history text-muted" style="font-size: 2rem;"></i>
              </div>
              <h6 class="text-muted mb-1">No Activity Yet</h6>
              <p class="text-muted small mb-0">There are no audit events recorded for this booking.</p>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <div class="col-lg-4">
      <!-- Payment Summary Card -->
      <div class="card shadow-sm mb-4 border-0">
        <div class="card-header bg-light py-3">
          <h6 class="mb-0"><i class="bi bi-credit-card text-primary me-2"></i>Payment Summary</h6>
        </div>
        <div class="card-body">
          <!-- Payment Status -->
          <div class="d-flex align-items-center mb-3">
            <div class="flex-shrink-0 me-3">
              <div class="bg-primary bg-opacity-10 text-primary p-3 rounded-circle">
                <i class="bi bi-credit-card fs-4"></i>
              </div>
            </div>
            <div class="flex-grow-1">
              <div class="d-flex justify-content-between align-items-center">
                <div>
                  <div class="text-muted small">Payment Status</div>
                  <div class="fw-semibold">
                    <?php 
                    $paymentStatus = strtolower((string)($booking['payment_status'] ?? ''));
                    $statusClass = 'badge bg-light text-dark';
                    
                    if ($paymentStatus === 'paid') {
                      $statusClass = 'badge bg-success bg-opacity-10 text-success';
                    } elseif (in_array($paymentStatus, ['refunded', 'partially_refunded'])) {
                      $statusClass = 'badge bg-info bg-opacity-10 text-info';
                    } elseif (in_array($paymentStatus, ['failed', 'declined', 'cancelled'])) {
                      $statusClass = 'badge bg-danger bg-opacity-10 text-danger';
                    } elseif ($paymentStatus === 'pending') {
                      $statusClass = 'badge bg-warning bg-opacity-10 text-warning';
                    }
                    ?>
                    <span class="<?= $statusClass ?>">
                      <?= ucfirst(htmlspecialchars($paymentStatus ?: 'Pending')) ?>
                    </span>
                  </div>
                </div>
                <div class="text-end">
                  <div class="h4 mb-0 fw-bold">
                    <?= number_format((float)($booking['amount_total'] ?? 0), 2) ?>
                  </div>
                  <div class="text-muted small">
                    <?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <!-- Payment Method -->
          <div class="border-top pt-3 mt-3">
            <div class="row g-2 small">
              <div class="col-6">
                <div class="text-muted">Method</div>
                <div class="fw-medium"><?= !empty($booking['payment_method']) ? htmlspecialchars($booking['payment_method']) : 'N/A' ?></div>
              </div>
              <?php if (!empty($booking['gateway_name'])): ?>
              <div class="col-6">
                <div class="text-muted">Gateway</div>
                <div class="fw-medium"><?= htmlspecialchars($booking['gateway_name']) ?></div>
              </div>
              <?php endif; ?>
              <?php if (!empty($booking['payment_txn_id'])): ?>
              <div class="col-12 mt-2">
                <div class="text-muted">Transaction ID</div>
                <div class="fw-medium text-truncate" style="max-width: 100%;" data-bs-toggle="tooltip" title="<?= htmlspecialchars($booking['payment_txn_id']) ?>">
                  <?= htmlspecialchars($booking['payment_txn_id']) ?>
                </div>
              </div>
              <?php endif; ?>
              <?php if (!empty($booking['paid_at'])): ?>
              <div class="col-12 mt-2">
                <div class="text-muted">Paid On</div>
                <div class="fw-medium"><?= date('M j, Y H:i', strtotime($booking['paid_at'])) ?></div>
              </div>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>

      <!-- Vendor Payout Card -->
      <div class="card shadow-sm mb-4 border-0">
        <div class="card-header bg-light py-3">
          <h6 class="mb-0"><i class="bi bi-cash-coin text-primary me-2"></i>Vendor Payout</h6>
        </div>
        <div class="card-body">
          <!-- Payout Status -->
          <div class="d-flex align-items-center mb-3">
            <div class="flex-shrink-0 me-3">
              <div class="bg-primary bg-opacity-10 text-primary p-3 rounded-circle">
                <i class="bi bi-wallet2 fs-4"></i>
              </div>
            </div>
            <div class="flex-grow-1">
              <div class="d-flex justify-content-between align-items-center">
                <div>
                  <div class="text-muted small">Payout Status</div>
                  <div class="fw-semibold">
                    <?php 
                    $payoutStatus = strtolower((string)($booking['vendor_pay_status'] ?? 'pending'));
                    $payoutStatusClass = 'badge bg-light text-dark';
                    // Treat 'cleared' as Paid for UI
                    $label = $payoutStatus === 'cleared' ? 'Paid' : ucfirst($payoutStatus);
                    if ($payoutStatus === 'cleared') {
                      $payoutStatusClass = 'badge bg-success bg-opacity-10 text-success';
                    } elseif ($payoutStatus === 'pending') {
                      $payoutStatusClass = 'badge bg-warning bg-opacity-10 text-warning';
                    } elseif (in_array($payoutStatus, ['cancelled', 'rejected'])) {
                      $payoutStatusClass = 'badge bg-danger bg-opacity-10 text-danger';
                    }
                    ?>
                    <span class="<?= $payoutStatusClass ?>">
                      <?= htmlspecialchars($label) ?>
                    </span>
                  </div>
                </div>
                <div class="text-end">
                  <div class="h4 mb-0 fw-bold">
                    <?= number_format((float)($booking['vendor_cost'] ?? ($booking['vendor_payout_amount'] ?? ($booking['amount_total'] ?? 0))), 2) ?>
                  </div>
                  <div class="text-muted small">
                    <?= htmlspecialchars((string)($booking['vendor_currency'] ?? ($booking['currency'] ?? 'THB'))) ?>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <?php 
            $isRefundedOrCancelled = (strtolower((string)($booking['payment_status'] ?? '')) === 'refunded') || (strtolower((string)($booking['status'] ?? '')) === 'cancelled');
            $isPaid = (strtolower((string)($booking['payment_status'] ?? '')) === 'paid');
          ?>
          <?php if (strtolower((string)($booking['vendor_pay_status'] ?? '')) === 'cleared'): ?>
            <!-- Payout Details (Paid) -->
            <div class="alert alert-success p-3 mb-3">
              <div class="d-flex align-items-center">
                <i class="bi bi-check-circle-fill me-2"></i>
                <div>
                  <div class="fw-semibold">Payout Processed</div>
                  <div class="small">Paid on <?= !empty($booking['vendor_paid_at']) ? date('M j, Y', strtotime($booking['vendor_paid_at'])) : 'N/A' ?></div>
                  
                  <?php if (!empty($booking['vendor_pay_method']) || !empty($booking['vendor_pay_txn_id']) || !empty($booking['vendor_invoice_no']) || !empty($booking['vendor_pay_note'])): ?>
                    <div class="mt-2 small">
                      <?php if (!empty($booking['vendor_pay_method'])): ?>
                        <div>
                          <span class="text-muted">Method:</span>
                          <span class="fw-semibold text-capitalize ms-1"><?php 
                            $m = (string)($booking['vendor_pay_method'] ?? '');
                            echo htmlspecialchars(str_replace('_', ' ', $m));
                          ?></span>
                        </div>
                      <?php endif; ?>
                      <?php if (!empty($booking['vendor_pay_txn_id'])): ?>
                        <div class="mt-1 d-flex align-items-center">
                          <span class="text-muted me-1">Txn:</span>
                          <code class="bg-light px-2 py-1 rounded"><?= htmlspecialchars((string)$booking['vendor_pay_txn_id']) ?></code>
                          <button type="button" class="btn btn-sm btn-outline-secondary ms-2 copy-txn" data-value="<?= htmlspecialchars((string)$booking['vendor_pay_txn_id']) ?>" data-bs-toggle="tooltip" title="Copy">
                            <i class="bi bi-clipboard"></i>
                          </button>
                        </div>
                      <?php endif; ?>
                      <?php if (!empty($booking['vendor_invoice_no'])): ?>
                        <div class="mt-1">
                          <span class="text-muted">Invoice #:</span>
                          <span class="fw-medium ms-1"><?= htmlspecialchars((string)$booking['vendor_invoice_no']) ?></span>
                        </div>
                      <?php endif; ?>
                      <?php if (!empty($booking['vendor_pay_note'])): ?>
                        <div class="mt-1">
                          <span class="text-muted">Note:</span>
                          <span class="fw-medium ms-1"><?= nl2br(htmlspecialchars((string)$booking['vendor_pay_note'])) ?></span>
                        </div>
                      <?php endif; ?>
                    </div>
                  <?php endif; ?>
                  
                  <?php if (!empty($booking['vendor_invoice_no'])): ?>
                    <div class="mt-1">
                      <span class="text-muted small">Invoice #:</span>
                      <span class="fw-medium ms-1"><?= htmlspecialchars($booking['vendor_invoice_no']) ?></span>
                    </div>
                  <?php endif; ?>
                </div>
              </div>
            </div>
          <?php else: ?>
            <!-- Payout Form (Not Paid) -->
            <?php if ($isRefundedOrCancelled || !$isPaid): ?>
              <div class="alert alert-warning">
                <i class="bi bi-exclamation-triangle-fill me-2"></i>
                Payout is disabled because this booking is <?= !$isPaid ? 'not paid yet' : (htmlspecialchars(strtolower((string)($booking['payment_status'] ?? '')) === 'refunded' ? 'refunded' : 'cancelled')) ?>.
              </div>
            <?php else: ?>
            <form id="vendorPayoutForm" method="post" action="/admin/vendor-payouts/mark-paid" class="vendor-payout-form">
              <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
              <input type="hidden" name="booking_ids[]" value="<?= (int)($booking['id'] ?? 0) ?>">
              <input type="hidden" name="redirect_to" value="<?= htmlspecialchars((string)($_SERVER['REQUEST_URI'] ?? '/admin/booking/taxi')) ?>">
              
              <div class="mb-3">
                <label class="form-label small text-muted mb-1">Payout Amount</label>
                <div class="input-group input-group-sm">
                  <span class="input-group-text"><?= htmlspecialchars((string)($booking['vendor_currency'] ?? ($booking['currency'] ?? 'THB'))) ?></span>
                  <input type="number" class="form-control" name="amount" value="<?= number_format((float)($booking['vendor_cost'] ?? ($booking['vendor_payout_amount'] ?? 0)), 2, '.', '') ?>" step="0.01" min="0" required>
                </div>
              </div>
              
              <div class="mb-3">
                <label class="form-label small text-muted mb-1">Payment Reference</label>
                <input type="text" name="payment_reference" class="form-control form-control-sm" placeholder="e.g. Bank Transfer #" required>
              </div>
              <div class="row g-2">
                <div class="col-sm-6">
                  <label class="form-label small text-muted mb-1">Method</label>
                  <select name="vendor_pay_method" class="form-select form-select-sm">
                    <option value="">Select method</option>
                    <option value="bank_transfer">Bank Transfer</option>
                    <option value="cash">Cash</option>
                    <option value="other">Other</option>
                  </select>
                </div>
                <div class="col-sm-6">
                  <label class="form-label small text-muted mb-1">Transaction ID</label>
                  <input type="text" name="vendor_pay_txn_id" class="form-control form-control-sm" placeholder="Txn / Ref ID">
                </div>
              </div>
              <div class="row g-2 mt-2">
                <div class="col-sm-6">
                  <label class="form-label small text-muted mb-1">Invoice #</label>
                  <input type="text" name="invoice_no" class="form-control form-control-sm" placeholder="Vendor invoice #">
                </div>
                <div class="col-sm-6">
                  <label class="form-label small text-muted mb-1">Note</label>
                  <input type="text" name="note" class="form-control form-control-sm" placeholder="Optional note to keep">
                </div>
              </div>
              
              <div class="mb-3">
                <label class="form-label small text-muted mb-1">Payment Date</label>
                <input type="date" name="paid_at" class="form-control form-control-sm" value="<?= date('Y-m-d') ?>" required>
              </div>
              
              <div class="d-grid">
                <button id="vendorPayoutSubmitBtn" type="button" class="btn btn-sm btn-success" data-bs-toggle="modal" data-bs-target="#vendorPayoutConfirmModal">
                  <i class="bi bi-check-circle me-1"></i> Mark as Paid
                </button>
              </div>
              
              <!-- Vendor Payout Confirm Modal -->
              <div class="modal fade" id="vendorPayoutConfirmModal" tabindex="-1" aria-labelledby="vendorPayoutConfirmModalLabel" aria-hidden="true">
                <div class="modal-dialog">
                  <div class="modal-content">
                    <div class="modal-header">
                      <h5 class="modal-title" id="vendorPayoutConfirmModalLabel">Confirm Vendor Payout</h5>
                      <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                      <div class="alert alert-info d-flex align-items-center" role="alert">
                        <i class="bi bi-info-circle me-2"></i>
                        Please review the details below before marking as paid.
                      </div>
                      <ul class="list-unstyled small mb-0">
                        <li><strong>Amount:</strong> <span id="vpAmtPreview"></span> <span id="vpCurPreview"></span></li>
                        <li><strong>Method:</strong> <span id="vpMethodPreview"></span></li>
                        <li><strong>Transaction ID:</strong> <span id="vpTxnPreview"></span></li>
                        <li><strong>Invoice #:</strong> <span id="vpInvPreview"></span></li>
                        <li><strong>Payment Date:</strong> <span id="vpDatePreview"></span></li>
                      </ul>
                    </div>
                    <div class="modal-footer">
                      <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
                      <button id="confirmVendorPayoutBtn" type="button" class="btn btn-success">
                        <i class="bi bi-check-circle me-1"></i> Confirm & Submit
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            </form>
            <?php endif; ?>
          <?php endif; ?>
          
          <!-- Commission & Profit -->
          <div class="border rounded p-3 mt-3">
            <div class="d-flex justify-content-between mb-2">
              <span class="text-muted small">Total Amount</span>
              <span class="fw-medium">
                <?= number_format((float)($booking['amount_total'] ?? 0), 2) ?> 
                <?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?>
              </span>
            </div>
            
            <div class="d-flex justify-content-between mb-2">
              <span class="text-muted small">Vendor Payout</span>
              <span class="fw-medium text-danger">
                <?php 
                  $vp = (float)($booking['vendor_cost'] ?? ($booking['vendor_payout_amount'] ?? 0));
                  $vpText = number_format($vp, 2);
                  $vpCur = htmlspecialchars((string)($booking['vendor_currency'] ?? ($booking['currency'] ?? 'THB')));
                  echo ($vp > 0 ? '-' : '') . $vpText . ' ' . $vpCur;
                ?>
              </span>
            </div>
            
            <div class="d-flex justify-content-between mb-2">
              <span class="text-muted small">
                Commission 
                <?php if (!empty($booking['commission_rate'])): ?>
                  (<?= number_format((float)$booking['commission_rate'], 2) ?>%)
                <?php endif; ?>
              </span>
              <span class="fw-medium">
                <?= number_format((float)($booking['commission_amount'] ?? 0), 2) ?> 
                <?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?>
              </span>
            </div>
            
            <hr class="my-2">
            
            <div class="d-flex justify-content-between fw-bold">
              <span>Estimated Profit</span>
              <span class="text-success">
                <?php 
                $profit = (float)($booking['amount_total'] ?? 0) - (float)($booking['vendor_cost'] ?? ($booking['vendor_payout_amount'] ?? 0));
                echo number_format($profit, 2) . ' ' . htmlspecialchars((string)($booking['currency'] ?? 'THB'));
                ?>
              </span>
            </div>
          </div>
        </div>
      </div>
      
      <!-- Quick Actions Card -->
      <div class="card shadow-sm mb-4 border-0">
        <div class="card-header bg-light py-3">
          <h6 class="mb-0"><i class="bi bi-lightning-charge text-primary me-2"></i>Quick Actions</h6>
        </div>
        <div class="card-body p-0">
          <div class="list-group list-group-flush">
            <a href="/admin/booking/taxi/edit/<?= (int)($booking['id'] ?? 0) ?>" class="list-group-item list-group-item-action border-0 py-3">
              <div class="d-flex align-items-center">
                <div class="bg-primary bg-opacity-10 text-primary p-2 rounded-circle me-3">
                  <i class="bi bi-pencil-square"></i>
                </div>
                <div>
                  <div class="fw-medium">Edit Booking</div>
                  <div class="small text-muted">Update booking details</div>
                </div>
                <div class="ms-auto text-muted">
                  <i class="bi bi-chevron-right"></i>
                </div>
              </div>
            </a>
            
            <a href="/admin/booking/taxi/send-confirmation/<?= (int)($booking['id'] ?? 0) ?>" class="list-group-item list-group-item-action border-0 py-3">
              <div class="d-flex align-items-center">
                <div class="bg-primary bg-opacity-10 text-primary p-2 rounded-circle me-3">
                  <i class="bi bi-envelope"></i>
                </div>
                <div>
                  <div class="fw-medium">Resend Confirmation</div>
                  <div class="small text-muted">Send booking confirmation email</div>
                </div>
                <div class="ms-auto text-muted">
                  <i class="bi bi-chevron-right"></i>
                </div>
              </div>
            </a>
            
            <a href="/admin/booking/taxi/voucher/<?= (int)($booking['id'] ?? 0) ?>" target="_blank" class="list-group-item list-group-item-action border-0 py-3">
              <div class="d-flex align-items-center">
                <div class="bg-primary bg-opacity-10 text-primary p-2 rounded-circle me-3">
                  <i class="bi bi-file-earmark-text"></i>
                </div>
                <div>
                  <div class="fw-medium">View Voucher</div>
                  <div class="small text-muted">Print or download booking voucher</div>
                </div>
                <div class="ms-auto text-muted">
                  <i class="bi bi-box-arrow-up-right"></i>
                </div>
              </div>
            </a>
          </div>
          
          <div class="p-3 border-top">
            <?php if (!$isCompleted): ?>
              <div class="d-grid gap-2">
                <!-- Cancel Booking Button -->
                <button type="button" class="btn btn-outline-danger btn-sm" data-bs-toggle="modal" data-bs-target="#cancelBookingModal" <?= ((string)($booking['status'] ?? '')==='cancelled'?'disabled':'') ?>>
                  <i class="bi bi-x-circle me-1"></i> Cancel Booking
                </button>
                
                <!-- Refund Buttons -->
                <div class="btn-group w-100" role="group">
                  <button type="button" class="btn btn-outline-warning btn-sm" data-bs-toggle="modal" data-bs-target="#refundWalletModal" <?= ((string)($booking['payment_method'] ?? '')!=='wallet' || (string)($booking['payment_status'] ?? '')!=='paid' ?'disabled':'') ?>>
                    <i class="bi bi-arrow-counterclockwise me-1"></i> Refund (Wallet)
                  </button>
                  <?php if (!empty($stripe_enabled)): ?>
                  <button type="button" class="btn btn-outline-danger btn-sm" data-bs-toggle="modal" data-bs-target="#refundStripeModal" <?= ((string)($booking['gateway_name'] ?? '')!=='stripe' || (string)($booking['payment_status'] ?? '')!=='paid' ?'disabled':'') ?>>
                    <i class="bi bi-credit-card me-1"></i> Refund (Stripe)
                  </button>
                  <?php endif; ?>
                </div>
              </div>
            <?php else: ?>
              <div class="alert alert-light border-0 text-muted mb-0">
                <i class="bi bi-flag-fill me-1 text-success"></i>
                Trip is completed (<?= $completedAtText ?>). Cancel and refund actions are disabled.
              </div>
            <?php endif; ?>
          </div>
        </div>
      </div>
      
      <!-- Cancel Booking Modal -->
      <div class="modal fade" id="cancelBookingModal" tabindex="-1" aria-labelledby="cancelBookingModalLabel" aria-hidden="true">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title" id="cancelBookingModalLabel">Cancel Booking</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="/admin/booking/taxi/cancel">
              <div class="modal-body">
                <p>Are you sure you want to cancel this booking? This action cannot be undone.</p>
                <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
                <input type="hidden" name="id" value="<?= (int)($booking['id'] ?? 0) ?>">
                <div class="mb-3">
                  <label for="cancelReason" class="form-label">Reason for cancellation</label>
                  <textarea class="form-control" id="cancelReason" name="reason" rows="3" required></textarea>
                </div>
                <div class="mb-3">
                  <label for="cancelPassword" class="form-label">Master Password</label>
                  <input type="password" class="form-control" id="cancelPassword" name="master_password" required>
                </div>
              </div>
              <div class="modal-footer">
                <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
                <button type="submit" class="btn btn-danger">Confirm Cancellation</button>
              </div>
            </form>
          </div>
        </div>
      </div>
      
      <!-- Refund Wallet Modal -->
      <div class="modal fade" id="refundWalletModal" tabindex="-1" aria-labelledby="refundWalletModalLabel" aria-hidden="true">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title" id="refundWalletModalLabel">Process Wallet Refund</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="walletRefundForm" method="post" action="/admin/booking/taxi/refund">
              <div class="modal-body">
                <p>Are you sure you want to process a refund to the customer's wallet?</p>
                <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
                <input type="hidden" name="id" value="<?= (int)($booking['id'] ?? 0) ?>">
                
                <div class="mb-3">
                  <label for="walletRefundAmount" class="form-label">Refund Amount</label>
                  <div class="input-group">
                    <span class="input-group-text"><?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?></span>
                    <?php 
                    $amount = number_format((float)($booking['amount_total'] ?? 0), 2, '.', '');
                    echo '<input type="number" ';
                    echo 'class="form-control" ';
                    echo 'id="walletRefundAmount" ';
                    echo 'name="amount" ';
                    echo 'value="' . htmlspecialchars($amount, ENT_QUOTES) . '" ';
                    echo 'step="0.01" ';
                    echo 'min="0" ';
                    echo 'max="' . htmlspecialchars($amount, ENT_QUOTES) . '" ';
                    echo 'required>';
                    ?>
                  </div>
                  <div class="form-text">Maximum refundable amount: <?= number_format((float)($booking['amount_total'] ?? 0), 2) ?> <?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?></div>
                </div>
                
                <div class="mb-3">
                  <label for="walletRefundReason" class="form-label">Reason for refund</label>
                  <textarea class="form-control" id="walletRefundReason" name="reason" rows="3" required></textarea>
                </div>
                
                <div class="mb-3">
                  <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="walletSendNotification" name="send_notification" checked>
                    <label class="form-check-label" for="walletSendNotification">
                      Send notification to customer
                    </label>
                  </div>
                </div>
                
                <div class="mb-3">
                  <label for="walletRefundPassword" class="form-label">Master Password</label>
                  <input type="password" class="form-control" id="walletRefundPassword" name="master_password" required>
                </div>
              </div>
              <div class="modal-footer">
                <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
                <button type="submit" class="btn btn-warning">Process Refund</button>
              </div>
            </form>
          </div>
        </div>
      </div>
      
      <?php if (!empty($stripe_enabled)): ?>
      <!-- Refund Stripe Modal -->
      <div class="modal fade" id="refundStripeModal" tabindex="-1" aria-labelledby="refundStripeModalLabel" aria-hidden="true">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title" id="refundStripeModalLabel">Process Stripe Refund</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="stripeRefundForm" method="post" action="/admin/booking/taxi/refund-gateway">
              <div class="modal-body">
                <p>Are you sure you want to process a refund through Stripe?</p>
                <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
                <input type="hidden" name="id" value="<?= (int)($booking['id'] ?? 0) ?>">
                
                <div class="alert alert-warning">
                  <i class="bi bi-exclamation-triangle-fill me-2"></i>
                  This will issue a refund to the original payment method. The customer will receive a notification from Stripe.
                </div>
                
                <div class="mb-3">
                  <label for="stripeRefundAmount" class="form-label">Refund Amount</label>
                  <div class="input-group">
                    <span class="input-group-text"><?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?></span>
                    <?php 
                    $stripeAmount = number_format((float)($booking['amount_total'] ?? 0), 2, '.', '');
                    ?>
                    <input type="number" 
                           class="form-control" 
                           id="stripeRefundAmount" 
                           name="amount" 
                           value="<?= htmlspecialchars($stripeAmount, ENT_QUOTES) ?>" 
                           step="0.01" 
                           min="0" 
                           max="<?= htmlspecialchars($stripeAmount, ENT_QUOTES) ?>" 
                           required>
                  </div>
                  <div class="form-text">Maximum refundable amount: <?= number_format((float)($booking['amount_total'] ?? 0), 2) ?> <?= htmlspecialchars((string)($booking['currency'] ?? 'THB')) ?></div>
                </div>
                
                <div class="mb-3">
                  <label for="stripeRefundReason" class="form-label">Reason for refund</label>
                  <select class="form-select" id="stripeRefundReason" name="reason" required>
                    <option value="">Select a reason</option>
                    <option value="duplicate">Duplicate</option>
                    <option value="fraudulent">Fraudulent</option>
                    <option value="requested_by_customer">Requested by customer</option>
                    <option value="other">Other</option>
                  </select>
                </div>
                
                <div class="mb-3">
                  <label for="stripeRefundNote" class="form-label">Note (Optional)</label>
                  <textarea class="form-control" id="stripeRefundNote" name="note" rows="2"></textarea>
                </div>
                
                <div class="mb-3">
                  <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="stripeSendNotification" name="send_notification" checked>
                    <label class="form-check-label" for="stripeSendNotification">
                      Send notification to customer
                    </label>
                  </div>
                </div>
                
                <div class="mb-3">
                  <label for="stripeRefundPassword" class="form-label">Master Password</label>
                  <input type="password" class="form-control" id="stripeRefundPassword" name="master_password" required>
                </div>
              </div>
              <div class="modal-footer">
                <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
                <button type="submit" class="btn btn-danger">Process Stripe Refund</button>
              </div>
            </form>
          </div>
        </div>
      </div>
      <?php endif; ?>
      
      <script>
        // Initialize tooltips
        document.addEventListener('DOMContentLoaded', function() {
          var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
          var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
          });
          
          // Initialize popovers
          var popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
          var popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
            return new bootstrap.Popover(popoverTriggerEl);
          });

          // Vendor payout confirm modal wiring
          var payoutForm = document.getElementById('vendorPayoutForm');
          var confirmModalEl = document.getElementById('vendorPayoutConfirmModal');
          if (payoutForm && confirmModalEl) {
            confirmModalEl.addEventListener('show.bs.modal', function () {
              var amt = payoutForm.querySelector('input[name="amount"]').value || '0.00';
              var cur = payoutForm.querySelector('.input-group-text')?.textContent?.trim() || '';
              var method = payoutForm.querySelector('select[name="vendor_pay_method"]').value || '-';
              var txn = payoutForm.querySelector('input[name="vendor_pay_txn_id"]').value || '-';
              var inv = payoutForm.querySelector('input[name="invoice_no"]').value || '-';
              var dt  = payoutForm.querySelector('input[name="paid_at"]').value || '-';

              document.getElementById('vpAmtPreview').textContent = amt;
              document.getElementById('vpCurPreview').textContent = cur;
              document.getElementById('vpMethodPreview').textContent = method || '-';
              document.getElementById('vpTxnPreview').textContent = txn || '-';
              document.getElementById('vpInvPreview').textContent = inv || '-';
              document.getElementById('vpDatePreview').textContent = dt || '-';
            });

            var confirmBtn = document.getElementById('confirmVendorPayoutBtn');
            if (confirmBtn) {
              confirmBtn.addEventListener('click', function() {
                // Minimal validation
                var amtInput = payoutForm.querySelector('input[name="amount"]');
                var methodSel = payoutForm.querySelector('select[name="vendor_pay_method"]');
                var paidAt    = payoutForm.querySelector('input[name="paid_at"]');
                var amtVal = parseFloat(amtInput.value || '0');
                if (!(amtVal >= 0) || !paidAt.value) {
                  amtInput.focus();
                  return;
                }
                // Prevent double submission
                if (confirmBtn.dataset.submitting === '1') return;
                confirmBtn.dataset.submitting = '1';
                confirmBtn.disabled = true;
                var oldHtml = confirmBtn.innerHTML;
                confirmBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-1" role="status" aria-hidden="true"></span>Submitting...';

                var m = bootstrap.Modal.getOrCreateInstance(confirmModalEl);
                m.hide();
                // Submit
                payoutForm.submit();
                // Fallback reset after 5s in case of navigation blocked
                setTimeout(function(){
                  confirmBtn.disabled = false;
                  confirmBtn.dataset.submitting = '';
                  confirmBtn.innerHTML = oldHtml;
                }, 5000);
              });
            }
          }
        });
      </script>

      <?php if (!empty($payments)): ?>
      <div id="paymentRecords" class="card shadow-sm border-0">
        <div class="card-header bg-light py-3">
          <h6 class="mb-0"><i class="bi bi-credit-card-2-back text-primary me-2"></i>Payment Records</h6>
        </div>
        <div class="card-body p-0">
          <div class="table-responsive">
            <table class="table table-hover align-middle mb-0">
              <thead class="table-light">
                <tr>
                  <th class="ps-3">#</th>
                  <th>Method</th>
                  <th>Gateway</th>
                  <th>Amount</th>
                  <th>Status</th>
                  <th>Transaction</th>
                  <th class="text-end pe-3">Date & Time</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($payments as $p): 
                  $statusClass = 'bg-light text-dark';
                  $status = strtolower((string)($p['status'] ?? ''));
                  
                  if (in_array($status, ['completed', 'succeeded', 'paid'])) {
                    $statusClass = 'bg-success bg-opacity-10 text-success';
                  } elseif (in_array($status, ['refunded', 'partially_refunded'])) {
                    $statusClass = 'bg-info bg-opacity-10 text-info';
                  } elseif (in_array($status, ['failed', 'declined', 'cancelled'])) {
                    $statusClass = 'bg-danger bg-opacity-10 text-danger';
                  } elseif (in_array($status, ['pending', 'processing'])) {
                    $statusClass = 'bg-warning bg-opacity-10 text-warning';
                  }
                ?>
                  <tr>
                    <td class="ps-3 fw-medium"><?= (int)$p['id'] ?></td>
                    <td>
                      <div class="d-flex align-items-center">
                        <div class="me-2">
                          <?php if (strpos(strtolower($p['method'] ?? ''), 'card') !== false): ?>
                            <i class="bi bi-credit-card text-muted"></i>
                          <?php elseif (strpos(strtolower($p['method'] ?? ''), 'wallet') !== false): ?>
                            <i class="bi bi-wallet2 text-muted"></i>
                          <?php else: ?>
                            <i class="bi bi-cash-coin text-muted"></i>
                          <?php endif; ?>
                        </div>
                        <div><?= htmlspecialchars(ucfirst((string)($p['method'] ?? 'N/A'))) ?></div>
                      </div>
                    </td>
                    <td><?= !empty($p['gateway_name']) ? htmlspecialchars($p['gateway_name']) : '<span class="text-muted">N/A</span>' ?></td>
                    <td class="fw-semibold">
                      <?= number_format((float)($p['amount'] ?? 0), 2) ?> 
                      <span class="text-muted"><?= htmlspecialchars((string)($p['currency'] ?? 'THB')) ?></span>
                    </td>
                    <td>
                      <span class="badge <?= $statusClass ?> small">
                        <?= ucfirst(htmlspecialchars($status)) ?>
                      </span>
                    </td>
                    <td>
                      <?php if (!empty($p['txn_id'])): ?>
                        <div class="d-flex align-items-center">
                          <span class="text-truncate d-inline-block" style="max-width: 100px;" data-bs-toggle="tooltip" title="<?= htmlspecialchars($p['txn_id']) ?>">
                            <?= htmlspecialchars($p['txn_id']) ?>
                          </span>
                          <button type="button" class="btn btn-link btn-sm p-0 ms-1" data-bs-toggle="tooltip" title="Copy to clipboard" onclick="copyToClipboard('<?= htmlspecialchars(addslashes($p['txn_id'])) ?>')">
                            <i class="bi bi-clipboard small"></i>
                          </button>
                        </div>
                      <?php else: ?>
                        <span class="text-muted small">N/A</span>
                      <?php endif; ?>
                    </td>
                    <td class="text-end pe-3">
                      <div class="d-flex flex-column">
                        <span class="small"><?= !empty($p['created_at']) ? date('M j, Y', strtotime($p['created_at'])) : '' ?></span>
                        <span class="text-muted small"><?= !empty($p['created_at']) ? date('H:i', strtotime($p['created_at'])) : '' ?></span>
                      </div>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        </div>
        <?php if (count($payments) > 5): ?>
          <div class="card-footer bg-light py-2">
            <div class="d-flex justify-content-between align-items-center">
              <div class="small text-muted">
                Showing <?= min(5, count($payments)) ?> of <?= count($payments) ?> records
              </div>
              <button type="button" class="btn btn-sm btn-outline-primary" data-bs-toggle="collapse" data-bs-target="#allPayments" aria-expanded="false" aria-controls="allPayments">
                Show all
              </button>
            </div>
          </div>
        <?php endif; ?>
      </div>
      
      <script>
        // Add copy to clipboard function
        function copyToClipboard(text) {
          navigator.clipboard.writeText(text).then(function() {
            // Show success tooltip
            var tooltip = new bootstrap.Tooltip(document.createElement('div'), {
              title: 'Copied!',
              trigger: 'manual',
              placement: 'top'
            });
            tooltip.show();
            
            // Hide after 1.5 seconds
            setTimeout(function() {
              tooltip.hide();
              tooltip.dispose();
            }, 1500);
          }).catch(function(err) {
            console.error('Could not copy text: ', err);
            showAlert('Failed to copy to clipboard', 'danger');
          });
        }
        
        // Show alert message
        function showAlert(message, type = 'info') {
          const alertHtml = `
            <div class="alert alert-${type} alert-dismissible fade show" role="alert">
              ${message}
              <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
          `;
          
          // Create a container for alerts if it doesn't exist
          let alertContainer = document.getElementById('alertContainer');
          if (!alertContainer) {
            alertContainer = document.createElement('div');
            alertContainer.id = 'alertContainer';
            alertContainer.className = 'position-fixed top-0 end-0 p-3';
            alertContainer.style.zIndex = '1100';
            document.body.appendChild(alertContainer);
          }
          
          // Add the new alert
          const alertElement = document.createElement('div');
          alertElement.innerHTML = alertHtml;
          alertContainer.appendChild(alertElement);
          
          // Auto-remove after 5 seconds
          setTimeout(() => {
            const bsAlert = new bootstrap.Alert(alertElement.querySelector('.alert'));
            bsAlert.close();
            setTimeout(() => alertElement.remove(), 150);
          }, 5000);
        }
        
        // Validate refund amount
        function validateRefundAmount(input, maxAmount) {
          const value = parseFloat(input.value) || 0;
          const errorElement = document.getElementById(input.id + 'Error');
          
          if (value <= 0) {
            if (errorElement) errorElement.textContent = 'Amount must be greater than 0';
            input.setCustomValidity('Amount must be greater than 0');
            return false;
          }
          
          if (value > maxAmount) {
            if (errorElement) errorElement.textContent = `Amount cannot exceed ${maxAmount.toFixed(2)}`;
            input.setCustomValidity(`Amount cannot exceed ${maxAmount.toFixed(2)}`);
            return false;
          }
          
          if (errorElement) errorElement.textContent = '';
          input.setCustomValidity('');
          return true;
        }
        
        // Initialize form validation and tooltips
        document.addEventListener('DOMContentLoaded', function() {
          // Initialize tooltips
          var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
          tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
          });
          
          // Wallet refund form validation
          const walletRefundForm = document.getElementById('walletRefundForm');
          if (walletRefundForm) {
            const walletAmountInput = document.getElementById('walletRefundAmount');
            const walletMaxAmount = parseFloat(walletAmountInput.getAttribute('max')) || 0;
            
            walletAmountInput.addEventListener('input', () => {
              validateRefundAmount(walletAmountInput, walletMaxAmount);
            });
            
            walletRefundForm.addEventListener('submit', (e) => {
              if (!validateRefundAmount(walletAmountInput, walletMaxAmount)) {
                e.preventDefault();
                showAlert('Please correct the errors in the form.', 'danger');
                return false;
              }
              
              // Show loading state
              const submitBtn = walletRefundForm.querySelector('button[type="submit"]');
              const originalText = submitBtn.innerHTML;
              submitBtn.disabled = true;
              submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Processing...';
              
              // Reset button state if form submission fails
              setTimeout(() => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
              }, 5000);
              
              return true;
            });
          }
          
          // Stripe refund form validation
          const stripeRefundForm = document.getElementById('stripeRefundForm');
          if (stripeRefundForm) {
            const stripeAmountInput = document.getElementById('stripeRefundAmount');
            const stripeMaxAmount = parseFloat(stripeAmountInput.getAttribute('max')) || 0;
            
            stripeAmountInput.addEventListener('input', () => {
              validateRefundAmount(stripeAmountInput, stripeMaxAmount);
            });
            
            stripeRefundForm.addEventListener('submit', (e) => {
              if (!validateRefundAmount(stripeAmountInput, stripeMaxAmount)) {
                e.preventDefault();
                showAlert('Please correct the errors in the form.', 'danger');
                return false;
              }
              
              // Show loading state
              const submitBtn = stripeRefundForm.querySelector('button[type="submit"]');
              const originalText = submitBtn.innerHTML;
              submitBtn.disabled = true;
              submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Processing...';
              
              // Reset button state if form submission fails
              setTimeout(() => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
              }, 5000);
              
              return true;
            });
          }
        });
      </script>
      <?php endif; ?>
    </div>
  </div>
</div>
