<?php
namespace App\Services;

use PDO;

class WalletService
{
    public function __construct(private PDO $pdo) {}

    public function getOrCreateWallet(int $userId): int
    {
        $stmt = $this->pdo->prepare('SELECT id FROM wallets WHERE user_id = :u LIMIT 1');
        $stmt->execute(['u' => $userId]);
        $w = $stmt->fetch();
        if ($w) return (int)$w['id'];
        $this->pdo->prepare('INSERT INTO wallets (user_id, balance) VALUES (:u, 0)')->execute(['u' => $userId]);
        return (int)$this->pdo->lastInsertId();
    }

    public function ledgerEntry(int $walletId, string $type, float $amount, string $method, string $status = 'pending', array $meta = []): int
    {
        $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w,:t,:a,:m,:s,:meta)')
            ->execute(['w'=>$walletId,'t'=>$type,'a'=>$amount,'m'=>$method,'s'=>$status,'meta'=>json_encode($meta)]);
        return (int)$this->pdo->lastInsertId();
    }

    public function approveLedger(int $ledgerId): bool
    {
        $this->pdo->beginTransaction();
        $row = $this->pdo->query('SELECT wallet_id, type, amount, status FROM wallet_ledger WHERE id='.(int)$ledgerId.' FOR UPDATE')->fetch();
        if (!$row || $row['status'] !== 'pending') { $this->pdo->rollBack(); return false; }
        $walletId = (int)$row['wallet_id'];
        $delta = ($row['type']==='credit' ? 1 : -1) * (float)$row['amount'];
        $this->pdo->prepare('UPDATE wallet_ledger SET status="approved" WHERE id=:id')->execute(['id'=>$ledgerId]);
        $this->pdo->prepare('UPDATE wallets SET balance = balance + :d WHERE id=:w')->execute(['d'=>$delta,'w'=>$walletId]);
        return $this->pdo->commit();
    }

    public function rejectLedger(int $ledgerId): bool
    {
        // Simply mark as rejected if currently pending.
        $stmt = $this->pdo->prepare('UPDATE wallet_ledger SET status="rejected" WHERE id = :id AND status = "pending"');
        $stmt->execute(['id' => $ledgerId]);
        return $stmt->rowCount() > 0;
    }

    public function getWalletIdByUser(int $userId): ?int
    {
        $stmt = $this->pdo->prepare('SELECT id FROM wallets WHERE user_id = :u LIMIT 1');
        $stmt->execute(['u' => $userId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['id'] : null;
    }

    // Create approved credit to a wallet (e.g., seeding treasury). Includes IP/meta logging.
    public function creditApproved(int $userId, float $amount, string $method, array $meta = []): bool
    {
        if ($amount <= 0) return false;
        $this->pdo->beginTransaction();
        try {
            $walletId = $this->getOrCreateWallet($userId);
            $metaJson = json_encode($meta);
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "credit", :a, :m, "approved", :meta)')
                ->execute(['w'=>$walletId,'a'=>$amount,'m'=>$method,'meta'=>$metaJson]);
            $this->pdo->prepare('UPDATE wallets SET balance = balance + :a WHERE id = :w')->execute(['a'=>$amount,'w'=>$walletId]);
            $this->pdo->commit();
            return true;
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            return false;
        }
    }

    // Atomic transfer between two user wallets with approved double-entry and IP/meta logging
    public function transferApproved(int $fromUserId, int $toUserId, float $amount, string $method, array $meta = []): bool
    {
        if ($amount <= 0) return false;
        if ($fromUserId === $toUserId) return false;
        $this->pdo->beginTransaction();
        try {
            $fromWallet = $this->getOrCreateWallet($fromUserId);
            $toWallet   = $this->getOrCreateWallet($toUserId);
            // lock source balance
            $row = $this->pdo->query('SELECT balance FROM wallets WHERE id='.(int)$fromWallet.' FOR UPDATE')->fetch(PDO::FETCH_ASSOC);
            if (!$row) { $this->pdo->rollBack(); return false; }
            if ((float)$row['balance'] < $amount) { $this->pdo->rollBack(); return false; }
            $metaJson = json_encode($meta);
            // sanitize method to allowed enum
            $m = ($method === 'stripe') ? 'stripe' : 'manual';
            // create ledger entries approved
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "debit", :a, :m, "approved", :meta)')
                ->execute(['w'=>$fromWallet,'a'=>$amount,'m'=>$m,'meta'=>$metaJson]);
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "credit", :a, :m, "approved", :meta)')
                ->execute(['w'=>$toWallet,'a'=>$amount,'m'=>$m,'meta'=>$metaJson]);
            // update balances
            $this->pdo->prepare('UPDATE wallets SET balance = balance - :a WHERE id=:w')->execute(['a'=>$amount,'w'=>$fromWallet]);
            $this->pdo->prepare('UPDATE wallets SET balance = balance + :a WHERE id=:w')->execute(['a'=>$amount,'w'=>$toWallet]);
            $this->pdo->commit();
            return true;
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            return false;
        }
    }

    // Reverse an approved transfer by moving funds back from recipient to sender with compensating entries.
    // Pre-conditions: amount > 0; sender != recipient; recipient has sufficient balance to debit.
    // Meta should include: ['flow'=>'reversal','reversal_of'=>originalId,'reason'=>..., ...]
    public function reverseTransferApproved(int $originalFromUserId, int $originalToUserId, float $amount, string $reason, array $extraMeta = []): bool
    {
        if ($amount <= 0) return false;
        if ($originalFromUserId === $originalToUserId) return false;
        $this->pdo->beginTransaction();
        try {
            $senderWalletId   = $this->getOrCreateWallet($originalFromUserId); // will receive back
            $recipientWalletId= $this->getOrCreateWallet($originalToUserId);   // will be debited
            // lock recipient balance to ensure funds exist to reverse
            $row = $this->pdo->query('SELECT balance FROM wallets WHERE id='.(int)$recipientWalletId.' FOR UPDATE')->fetch(PDO::FETCH_ASSOC);
            if (!$row) { $this->pdo->rollBack(); return false; }
            if ((float)$row['balance'] < $amount) { $this->pdo->rollBack(); return false; }
            $meta = array_merge([
                'flow' => 'reversal',
                'reason' => $reason,
                'reversed_at' => date('Y-m-d H:i:s'),
                'original_from_user_id' => $originalFromUserId,
                'original_to_user_id' => $originalToUserId,
            ], $extraMeta);
            $metaJson = json_encode($meta);
            // ledger: debit recipient, credit sender
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "debit", :a, "manual", "approved", :meta)')
                ->execute(['w'=>$recipientWalletId,'a'=>$amount,'meta'=>$metaJson]);
            $this->pdo->prepare('INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta) VALUES (:w, "credit", :a, "manual", "approved", :meta)')
                ->execute(['w'=>$senderWalletId,'a'=>$amount,'meta'=>$metaJson]);
            // update balances
            $this->pdo->prepare('UPDATE wallets SET balance = balance - :a WHERE id=:w')->execute(['a'=>$amount,'w'=>$recipientWalletId]);
            $this->pdo->prepare('UPDATE wallets SET balance = balance + :a WHERE id=:w')->execute(['a'=>$amount,'w'=>$senderWalletId]);
            $this->pdo->commit();
            return true;
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) $this->pdo->rollBack();
            return false;
        }
    }
}

