<?php

namespace App\Libraries;

class TaxiLogger {
    private static $logDir = __DIR__ . '/../../storage/logs/taxi_payments';
    private static $logFile;
    
    /**
     * Initialize the logger
     */
    private static function init() {
        if (!file_exists(self::$logDir)) {
            mkdir(self::$logDir, 0755, true);
        }
        self::$logFile = self::$logDir . '/' . date('Y-m-d') . '.log';
    }
    
    /**
     * Log a message with context
     * 
     * @param string $message The log message
     * @param array $context Additional context data
     * @param string $level Log level (info, error, debug)
     * @return void
     */
    public static function log($message, $context = [], $level = 'info') {
        self::init();
        
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? json_encode($context, JSON_PRETTY_PRINT) : '';
        
        $logMessage = "[$timestamp] [$level] $message" . PHP_EOL;
        if (!empty($contextStr)) {
            $logMessage .= "Context: " . $contextStr . PHP_EOL;
        }
        $logMessage .= str_repeat("-", 80) . PHP_EOL;
        
        // Write to log file
        file_put_contents(self::$logFile, $logMessage, FILE_APPEND);
        
        // Also log to PHP error log for errors
        if ($level === 'error') {
            error_log("[TaxiPayment] $message - " . $contextStr);
        }
    }
    
    /**
     * Log an error message
     */
    public static function error($message, $context = []) {
        self::log($message, $context, 'error');
    }
    
    /**
     * Log a debug message
     */
    public static function debug($message, $context = []) {
        self::log($message, $context, 'debug');
    }
    
    /**
     * Log an info message
     */
    public static function info($message, $context = []) {
        self::log($message, $context, 'info');
    }
}
