<?php
namespace App\Core\WhatsApp\Providers;

use App\Core\WhatsApp\Message;
use App\Core\WhatsApp\ProviderInterface;
use App\Core\WhatsApp\SendResult;

class MittosProvider implements ProviderInterface
{
    private string $apiBase;
    private string $sessionToken; // appended in URL for send_session
    private ?string $senderPhone;

    public function __construct(string $apiBase, string $sessionToken, ?string $senderPhone = null)
    {
        $this->apiBase = rtrim($apiBase ?: 'https://app.mittosapi.com', '/');
        $this->sessionToken = $sessionToken;
        $this->senderPhone = $senderPhone;
    }

    public function name(): string
    {
        return 'mittos';
    }

    public function send(Message $msg): SendResult
    {
        $endpoint = $this->apiBase . '/API_V2/Whatsapp/send_session/' . $this->sessionToken;
        $type = $msg->meta['type'] ?? $msg->type ?? 'text';
        $sender = $msg->from ?: ($this->senderPhone ?? '');
        // Mittos samples show digits with country code (no '+')
        $sender = preg_replace('/[^0-9]/', '', $sender);
        if ($type === 'quick_reply') {
            $payload = [
                'type' => 'quick_reply',
                'header' => (string)($msg->meta['header'] ?? ''),
                'body' => $msg->text,
                'footer' => (string)($msg->meta['footer'] ?? ''),
                'buttons' => array_values(array_filter((array)($msg->meta['buttons'] ?? []), fn($b)=>$b!=='')),
                'sender_phone' => $sender,
            ];
        } else { // default simple text
            $payload = [
                'type' => 'text',
                'message' => $msg->text,
                'sender_phone' => $sender,
            ];
            if (!empty($msg->meta['caption'])) {
                $payload['caption'] = (string)$msg->meta['caption'];
            }
        }

        $ch = curl_init($endpoint);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [ 'Content-Type: application/json' ],
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 20,
        ]);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE) ?: 0;
        $err = curl_error($ch);
        curl_close($ch);
        if ($resp === false) {
            return new SendResult(false, null, $err ?: 'HTTP error');
        }
        $data = json_decode($resp, true);
        $ok = ($code >= 200 && $code < 300) && (!isset($data['status']) || (int)$data['status'] === 200);
        $messageId = $data['messageId'] ?? ($data['id'] ?? null);
        if (!$ok) {
            $error = $data['statusText'] ?? ($data['message'] ?? $resp);
            return new SendResult(false, null, (string)$error);
        }
        return new SendResult(true, $messageId ?: null, null);
    }
}
