<?php
namespace App\Core\Sms;

class TwilioClient
{
    private string $sid;
    private string $token;
    private ?string $waFrom; // whatsapp:+123...
    private ?string $smsFrom; // optional E.164 like +123...

    public function __construct(?string $sid = null, ?string $token = null, ?string $waFrom = null, ?string $smsFrom = null)
    {
        $this->sid = $sid ?? (getenv('TWILIO_SID') ?: '');
        $this->token = $token ?? (getenv('TWILIO_TOKEN') ?: '');
        $this->waFrom = $waFrom ?? (getenv('TWILIO_WHATSAPP_FROM') ?: null);
        $this->smsFrom = $smsFrom ?? (getenv('TWILIO_SMS_FROM') ?: null);
    }

    public function enabled(): bool
    {
        return $this->sid !== '' && $this->token !== '';
    }

    /**
     * Send a WhatsApp text if configured, otherwise try SMS if configured.
     * $to can be a raw phone; we will prefix whatsapp: if sending WA.
     */
    public function send(string $to, string $body, array $options = []): array
    {
        if (!$this->enabled()) { return ['ok'=>false, 'status'=>0, 'message'=>'Twilio disabled']; }
        $prefer = strtolower((string)($options['prefer'] ?? 'whatsapp'));
        // Normalize phone
        $to = trim($to);
        if ($to === '') { return ['ok'=>false, 'status'=>0, 'message'=>'Missing recipient']; }

        // Try WhatsApp first
        if ($prefer === 'whatsapp' && !empty($this->waFrom)) {
            $res = $this->sendTwilioMessage('whatsapp:' . $this->waFrom, strpos($to, 'whatsapp:')===0 ? $to : ('whatsapp:' . $to), $body);
            if ($res['ok']) { return $res; }
        }
        // Fallback to SMS if available
        if (!empty($this->smsFrom)) {
            $res = $this->sendTwilioMessage($this->smsFrom, $to, $body);
            if ($res['ok']) { return $res; }
        }
        return ['ok'=>false, 'status'=>$res['status'] ?? 0, 'message'=>$res['message'] ?? 'Failed'];
    }

    private function sendTwilioMessage(string $from, string $to, string $body): array
    {
        $url = 'https://api.twilio.com/2010-04-01/Accounts/' . rawurlencode($this->sid) . '/Messages.json';
        $post = http_build_query(['From'=>$from, 'To'=>$to, 'Body'=>$body]);
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_USERPWD, $this->sid . ':' . $this->token);
        curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE) ?: 0;
        $err = curl_error($ch);
        curl_close($ch);
        $ok = ($code >= 200 && $code < 300);
        return ['ok'=>$ok, 'status'=>$code, 'message'=>$ok ? 'Sent' : ($resp ?: $err)];
    }
}
