<?php
namespace App\Core\Mail;

class SendGridClient
{
    private string $apiKey;

    public function __construct(?string $apiKey = null)
    {
        $key = $apiKey ?? (getenv('SENDGRID_API_KEY') ?: '');
        $this->apiKey = (string)$key;
    }

    public function enabled(): bool
    {
        return $this->apiKey !== '';
    }

    /**
     * Send an email using SendGrid v3 API. Minimal implementation.
     * @param string $to
     * @param string $subject
     * @param string $html
     * @param array $from ['address'=>..., 'name'=>...]
     * @param array $bcc list of addresses
     * @return array ['ok'=>bool,'message'=>string,'status'=>int]
     */
    public function sendHtml(string $to, string $subject, string $html, array $from, array $bcc = []): array
    {
        if (!$this->enabled()) {
            return ['ok'=>false,'message'=>'SENDGRID_API_KEY not set','status'=>0];
        }
        $payload = [
            'personalizations' => [[
                'to' => [[ 'email' => $to ]],
                'subject' => $subject,
            ]],
            'from' => [ 'email' => ($from['address'] ?? 'noreply@example.com'), 'name' => ($from['name'] ?? '') ],
            'content' => [[ 'type' => 'text/html', 'value' => $html ]],
        ];
        if (!empty($bcc)) {
            $payload['personalizations'][0]['bcc'] = array_map(fn($e)=>['email'=>$e], $bcc);
        }
        $ch = curl_init('https://api.sendgrid.com/v3/mail/send');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json',
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE) ?: 0;
        $err = curl_error($ch);
        curl_close($ch);
        $ok = ($code >= 200 && $code < 300);
        return ['ok'=>$ok, 'message'=>$ok ? 'Sent' : ($resp ?: $err), 'status'=>$code];
    }
}
