<?php
namespace App\Core\Mail\Providers;

use App\Core\Mail\Envelope;
use App\Core\Mail\ProviderInterface;
use App\Core\Mail\SendResult;
use Exception;

class ElasticEmailProvider implements ProviderInterface
{
    private array $config;
    private const API_URL = 'https://api.elasticemail.com/v2/email/send';

    public function __construct(array $config)
    {
        $this->config = $config;
    }

    public function send(Envelope $env): SendResult
    {
        $apiKey = $this->config['api_key'] ?? '';
        $fromEmail = ($env->fromEmail ?: ($this->config['from']['address'] ?? ''));
        $fromName  = ($env->fromName  ?: ($this->config['from']['name'] ?? ''));

        if ($apiKey === '' || $fromEmail === '') {
            return new SendResult(false, 'Elastic Email configuration is incomplete');
        }

        // Merge BCC from config copy_to with envelope bcc
        $bccConfig = array_filter(array_map('trim', explode(',', (string)($this->config['copy_to'] ?? ''))));
        $bccList = array_unique(array_filter(array_merge($env->bcc ?? [], $bccConfig)));
        $bccListStr = $bccList ? implode(';', $bccList) : '';

        $postData = [
            'apikey' => $apiKey,
            'from' => $fromEmail,
            'fromName' => $fromName,
            'to' => $env->to,
            'subject' => $env->subject,
            'bodyHtml' => $env->html,
            'isTransactional' => 'true',
        ];

        if (!empty($env->altText)) {
            $postData['bodyText'] = $env->altText;
        }
        if ($bccListStr !== '') {
            $postData['bcc'] = $bccListStr;
        }

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => self::API_URL,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($postData),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HEADER => false,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 15,
        ]);

        $response = curl_exec($ch);
        $error = curl_error($ch);
        $httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($error) {
            error_log('Elastic Email cURL Error: ' . $error);
            return new SendResult(false, 'cURL error: ' . $error, null, $httpCode);
        }

        $result = json_decode((string)$response, true);
        if (!is_array($result)) {
            return new SendResult(false, 'Invalid Elastic Email response', null, $httpCode);
        }

        $success = (bool)($result['success'] ?? false);
        if (!$success || $httpCode !== 200) {
            $msg = (string)($result['error'] ?? 'Elastic Email API error');
            $this->log('elasticemail', $env->to, $env->subject, $httpCode, $msg, (string)$response);
            error_log('Elastic Email API Error: ' . print_r($result, true));
            return new SendResult(false, $msg, null, $httpCode);
        }

        // v2 returns data with messageid(s)
        $messageId = null;
        if (!empty($result['data']['messageid'])) {
            $messageId = (string)$result['data']['messageid'];
        }
        $this->log('elasticemail', $env->to, $env->subject, $httpCode, 'Sent', (string)$response, $messageId);
        return new SendResult(true, 'Sent (ElasticEmail) messageId=' . ($messageId ?? 'n/a'), $messageId, $httpCode);
    }

    public function name(): string
    {
        return 'elasticemail';
    }

    private function log(string $provider, string $to, string $subject, int $status, string $message, string $rawResponse = '', ?string $messageId = null): void
    {
        try {
            $dir = defined('BASE_PATH') ? BASE_PATH . DIRECTORY_SEPARATOR . 'storage' . DIRECTORY_SEPARATOR . 'logs' : __DIR__ . '/../../../../storage/logs';
            if (!is_dir($dir)) {
                @mkdir($dir, 0777, true);
            }
            $file = rtrim($dir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'mail.log';
            $line = sprintf(
                "%s provider=%s to=%s status=%d msg=%s subject=%s messageId=%s\nraw=%s\n",
                date('Y-m-d H:i:s'),
                $provider,
                $to,
                $status,
                str_replace(["\r","\n"], ' ', $message),
                str_replace(["\r","\n"], ' ', $subject),
                $messageId ?? 'n/a',
                $rawResponse
            );
            @file_put_contents($file, $line, FILE_APPEND);
        } catch (\Throwable $_) {
            // ignore logging failures
        }
    }
}
