<?php
namespace App\Core\Mail;

class Mailer
{
    private array $cfg;

    public function __construct(array $config)
    {
        $this->cfg = $config;
    }

    private function profile(string $profile = null): array
    {
        if ($profile && isset($this->cfg['profiles'][$profile])) {
            return $this->cfg['profiles'][$profile];
        }
        return [
            'host' => $this->cfg['host'] ?? '',
            'port' => $this->cfg['port'] ?? 587,
            'username' => $this->cfg['username'] ?? '',
            'password' => $this->cfg['password'] ?? '',
            'from' => $this->cfg['from'] ?? ['address' => 'noreply@example.com', 'name' => ''],
            'encryption' => $this->cfg['encryption'] ?? 'tls',
            'copy_to' => $this->cfg['copy_to'] ?? '',
        ];
    }

    public function sendHtml(string $to, string $subject, string $html, string $altText = '', ?string $profile = null): array
    {
        $p = $this->profile($profile ?? '');
        $fromAddr = $p['from']['address'] ?? 'noreply@example.com';
        $fromName = $p['from']['name'] ?? '';

        // Generate AltBody if not provided
        if ($altText === '') {
            $altText = trim(strip_tags($html));
        }

        // Prefer PHPMailer if available
        if (class_exists('PHPMailer\\PHPMailer\\PHPMailer')) {
            return $this->sendWithPHPMailer($p, $to, $subject, $html, $altText, $fromAddr, $fromName);
        }

        // Fallback to mail()
        return $this->sendWithMailFunction($p, $to, $subject, $html, $altText, $fromAddr, $fromName);
    }

    private function sendWithPHPMailer(array $p, string $to, string $subject, string $html, string $altText, string $fromAddr, string $fromName): array
    {
        $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
        try {
            $mail->CharSet = 'UTF-8';
            $mail->isSMTP();
            $mail->Host = $p['host'] ?? '';
            $mail->SMTPAuth = true;
            $mail->Username = $p['username'] ?? '';
            $mail->Password = $p['password'] ?? '';
            $enc = strtolower((string)($p['encryption'] ?? 'tls'));
            if ($enc === 'ssl') $mail->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS;
            elseif ($enc === 'tls') $mail->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
            $mail->Port = (int)($p['port'] ?? 587);

            $mail->setFrom($fromAddr, $fromName);
            $mail->addAddress($to);
            if (!empty($p['copy_to'])) {
                foreach (explode(',', $p['copy_to']) as $bcc) {
                    $bcc = trim($bcc);
                    if ($bcc !== '') $mail->addBCC($bcc);
                }
            }
            // Return-Path / bounce address via Sender
            $bounce = 'bounce@' . $this->domainFromEmail($fromAddr);
            $mail->Sender = $bounce;

            $mail->Subject = $subject;
            $mail->isHTML(true);
            $mail->Body = $html;
            $mail->AltBody = $altText;

            // Deliverability-friendly headers
            $mail->addCustomHeader('List-Unsubscribe', '<mailto:unsubscribe@' . $this->domainFromEmail($fromAddr) . '>, <' . (rtrim(\APP_URL, '/') ?: 'https://example.com') . '/unsubscribe>');

            $mail->send();
            return ['ok' => true, 'message' => 'Sent'];
        } catch (\Throwable $e) {
            return ['ok' => false, 'message' => $e->getMessage()];
        }
    }

    private function sendWithMailFunction(array $p, string $to, string $subject, string $html, string $altText, string $fromAddr, string $fromName): array
    {
        $boundary = md5(uniqid((string)mt_rand(), true));
        $domain = $this->domainFromEmail($fromAddr);
        $headers = [];
        $headers[] = 'From: ' . ($fromName ? (sprintf('"%s" <%s>', addslashes($fromName), $fromAddr)) : $fromAddr);
        $headers[] = 'Reply-To: ' . $fromAddr;
        $headers[] = 'Return-Path: bounce@' . $domain;
        $headers[] = 'MIME-Version: 1.0';
        $headers[] = 'Content-Type: multipart/alternative; boundary="' . $boundary . '"';
        $headers[] = 'List-Unsubscribe: <mailto:unsubscribe@' . $domain . '>, <' . (rtrim(\APP_URL, '/') ?: 'https://example.com') . '/unsubscribe>';

        $body = '';
        $body .= '--' . $boundary . "\r\n";
        $body .= "Content-Type: text/plain; charset=UTF-8\r\n\r\n";
        $body .= $altText . "\r\n";
        $body .= '--' . $boundary . "\r\n";
        $body .= "Content-Type: text/html; charset=UTF-8\r\n\r\n";
        $body .= $html . "\r\n";
        $body .= '--' . $boundary . "--\r\n";

        $ok = @mail($to, $subject, $body, implode("\r\n", $headers));
        return ['ok' => (bool)$ok, 'message' => $ok ? 'Sent' : 'mail() failed'];
    }

    private function domainFromEmail(string $email): string
    {
        $parts = explode('@', $email);
        return isset($parts[1]) ? strtolower($parts[1]) : 'example.com';
    }
}
