<?php
namespace App\Core\Mail;

use App\Core\Mail\Providers\SmtpProvider;
use App\Core\Mail\Providers\SendGridProvider;
use App\Core\Mail\Providers\SesProvider;
use App\Core\Mail\Providers\ElasticEmailProvider;

class MailService
{
    private \PDO $pdo;

    public function __construct(\PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public function loadSettings(): array
    {
        $st = $this->pdo->query("SELECT * FROM mail_settings WHERE id=1");
        $row = $st? $st->fetch(\PDO::FETCH_ASSOC) : null;
        return $row ?: [];
    }

    public function activeProvider(): ProviderInterface
    {
        $cfg = $this->loadSettings();
        $envProvider = env('MAIL_ACTIVE_PROVIDER', '');
        $provider = $envProvider !== '' ? $envProvider : ($cfg['active_provider'] ?? 'smtp');
        // Build a common profile array
        $profile = [
            'from' => [
                'address' => ($cfg['from_email'] ?? '') ?: (env('MAIL_FROM_ADDRESS', 'noreply@example.com')),
                'name' => ($cfg['from_name'] ?? '') ?: (env('MAIL_FROM_NAME', '')),
            ],
            'copy_to' => $cfg['bcc_list'] ?? '',
        ];
        if ($provider === 'elasticemail') {
            $apiKey = $cfg['elasticemail_api_key'] ?? (env('ELASTICEMAIL_API_KEY', ''));
            if ($apiKey) {
                $elasticProfile = [
                    'api_key' => $apiKey,
                    'from' => $profile['from'],
                    'copy_to' => $profile['copy_to'] ?? '',
                ];
                return new ElasticEmailProvider($elasticProfile);
            }
        } elseif ($provider === 'ses') {
            // Prefer direct SES API if keys present; otherwise fallback to SMTP
            $accessKey = $cfg['ses_access_key'] ?? (env('SES_ACCESS_KEY_ID', ''));
            $secretKey = $cfg['ses_secret_key'] ?? (env('SES_SECRET_ACCESS_KEY', ''));
            $region = $cfg['ses_region'] ?? (env('SES_REGION', 'ap-southeast-1'));
            if ($accessKey && $secretKey) {
                $sesProfile = [
                    'access_key' => $accessKey,
                    'secret_key' => $secretKey,
                    'region' => $region,
                    'from' => $profile['from'],
                    'copy_to' => $profile['copy_to'] ?? '',
                ];
                return new SesProvider($sesProfile);
            }
            // Fallthrough to SMTP below when SES keys missing
        }
        if ($provider === 'sendgrid') {
            $profile['sendgrid_api_key'] = $cfg['sendgrid_api_key'] ?? '';
            return new SendGridProvider($profile);
        }
        // SMTP (generic or SES via SMTP creds)
        $smtp = [
            'host' => $cfg['smtp_host'] ?? env('MAIL_HOST', ''),
            'port' => (int)($cfg['smtp_port'] ?? (env('MAIL_PORT', 587))),
            'username' => $cfg['smtp_username'] ?? (env('MAIL_USERNAME', '')),
            'password' => $cfg['smtp_password'] ?? (env('MAIL_PASSWORD', '')),
            'from' => $profile['from'],
            'encryption' => $cfg['smtp_encryption'] ?? (env('MAIL_ENCRYPTION', 'tls')),
            'copy_to' => $profile['copy_to'] ?? '',
        ];
        return new SmtpProvider($smtp, $provider === 'ses' ? 'ses' : 'smtp');
    }

    public function send(Envelope $env): SendResult
    {
        // Suppressions check
        $email = strtolower(trim($env->to));
        if ($email !== '') {
            $st = $this->pdo->prepare("SELECT 1 FROM mail_suppressions WHERE email=:e LIMIT 1");
            $st->execute([':e'=>$email]);
            if ($st->fetchColumn()) {
                return new SendResult(false, 'suppressed recipient', null, 0);
            }
        }
        $provider = $this->activeProvider();
        return $provider->send($env);
    }

    public function enqueue(string $eventKey, string $audience, string $module, ?int $bookingId, string $toEmail, string $subject, string $html, string $locale = 'en'): bool
    {
        $sql = "INSERT INTO mail_queue (event_key,audience,module,booking_id,to_email,subject,html,locale,status,attempt_count,created_at)
                VALUES (:ev,:aud,:mod,:bid,:to,:sub,:html,:loc,'queued',0,NOW())";
        $st = $this->pdo->prepare($sql);
        return $st->execute([
            ':ev'=>$eventKey,
            ':aud'=>$audience,
            ':mod'=>$module,
            ':bid'=>$bookingId,
            ':to'=>$toEmail,
            ':sub'=>$subject,
            ':html'=>$html,
            ':loc'=>$locale,
        ]);
    }
}
