<?php
namespace App\Core\Auth;

use App\Core\Mail\MailService;
use PDO;
use PDOException;
use Exception;

class OTPService
{
    private PDO $pdo;
    private MailService $mailService;
    private int $otpExpiryMinutes = 10; // OTP expires in 10 minutes
    private int $otpLength = 6; // 6-digit OTP

    public function __construct(PDO $pdo, MailService $mailService)
    {
        $this->pdo = $pdo;
        $this->mailService = $mailService;
    }

    /**
     * Generate a new OTP and send it to the user's email
     */
    public function generateAndSendOTP(string $email, string $purpose = 'registration'): bool
    {
        $this->pdo->beginTransaction();
        
        try {
            // Clean up any existing OTPs for this email and purpose
            $this->cleanupOldOtps($email, $purpose);
            
            // Generate a new OTP
            $otp = $this->generateNumericOTP($this->otpLength);
            $expiresAt = (new \DateTime())->modify("+{$this->otpExpiryMinutes} minutes")->format('Y-m-d H:i:s');
            
            // Store the OTP in the database
            $stmt = $this->pdo->prepare(
                "INSERT INTO otp_verifications (email, otp, purpose, expires_at) 
                 VALUES (:email, :otp, :purpose, :expires_at)"
            );
            
            $stmt->execute([
                ':email' => $email,
                ':otp' => password_hash($otp, PASSWORD_DEFAULT),
                ':purpose' => $purpose,
                ':expires_at' => $expiresAt
            ]);
            
            // Send the OTP via email
            $this->sendOTPEmail($email, $otp, $purpose);
            
            $this->pdo->commit();
            return true;
            
        } catch (Exception $e) {
            $this->pdo->rollBack();
            error_log("OTP Generation Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Verify if the provided OTP is valid for the given email and purpose
     */
    public function verifyOTP(string $email, string $otp, string $purpose = 'registration'): bool
    {
        try {
            // Clean up expired OTPs first
            $this->cleanupOldOtps($email, $purpose);
            
            // Find a non-expired OTP for this email and purpose
            $stmt = $this->pdo->prepare(
                "SELECT id, otp FROM otp_verifications 
                 WHERE email = :email AND purpose = :purpose AND verified = 0 AND expires_at > NOW()
                 ORDER BY created_at DESC LIMIT 1"
            );
            
            $stmt->execute([':email' => $email, ':purpose' => $purpose]);
            $otpRecord = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$otpRecord || !password_verify($otp, $otpRecord['otp'])) {
                return false;
            }
            
            // Mark OTP as verified
            $updateStmt = $this->pdo->prepare(
                "UPDATE otp_verifications SET verified = 1, verified_at = NOW() WHERE id = :id"
            );
            $updateStmt->execute([':id' => $otpRecord['id']]);
            
            return true;
            
        } catch (Exception $e) {
            error_log("OTP Verification Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Clean up expired OTPs for an email and purpose
     */
    private function cleanupOldOtps(string $email, string $purpose): void
    {
        $stmt = $this->pdo->prepare(
            "DELETE FROM otp_verifications 
             WHERE (email = :email AND purpose = :purpose) 
             OR expires_at <= NOW()"
        );
        $stmt->execute([':email' => $email, ':purpose' => $purpose]);
    }
    
    /**
     * Generate a numeric OTP of specified length
     */
    private function generateNumericOTP(int $length = 6): string
    {
        $digits = '0123456789';
        $otp = '';
        
        for ($i = 0; $i < $length; $i++) {
            $otp .= $digits[random_int(0, 9)];
        }
        
        return $otp;
    }
    
    /**
     * Send OTP via email using the mail service
     */
    private function sendOTPEmail(string $email, string $otp, string $purpose): void
    {
        $subject = 'Your Verification Code';
        
        $body = "
            <h2>Your Verification Code</h2>
            <p>Hello,</p>
            <p>Your verification code for {$purpose} is: <strong>{$otp}</strong></p>
            <p>This code will expire in {$this->otpExpiryMinutes} minutes.</p>
            <p>If you didn't request this code, you can safely ignore this email.</p>
            <p>Best regards,<br>B2B Thai Hub Team</p>
        ";
        
        $this->mailService->activeProvider()->send($email, $subject, $body);
    }
}
