<?php
namespace App\Controllers;

use App\Core\Controller;

class WebhookController extends Controller
{
    // POST /webhooks/stripe
    public function stripe(): void
    {
        // Load from admin-configured gateway settings
        try {
            $pgCfg = require __DIR__ . '/../../config/payment_gateways.php';
        } catch (\Throwable $_) { $pgCfg = []; }
        $secret = (string)($pgCfg['stripe']['webhook_secret'] ?? '');
        if ($secret === '') {
            http_response_code(400);
            echo 'webhook_secret_missing';
            return;
        }

        $payload = file_get_contents('php://input');
        $sigHeader = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';

        try {
            $stripeSecret = (string)($pgCfg['stripe']['secret_key'] ?? '');
            \Stripe\Stripe::setApiKey($stripeSecret);
            $event = \Stripe\Webhook::constructEvent($payload, $sigHeader, $secret);
        } catch (\Throwable $e) {
            http_response_code(400);
            echo 'invalid_signature';
            return;
        }

        $type = $event['type'] ?? '';
        if ($type === 'checkout.session.completed') {
            $session = $event['data']['object'] ?? [];
            $meta = $session['metadata'] ?? [];
            $piId = isset($meta['pi_id']) ? (int)$meta['pi_id'] : 0;
            if ($piId > 0) {
                $this->handlePaymentIntentSucceeded($piId, $session);
            }
        }

        http_response_code(200);
        echo 'ok';
    }

    private function handlePaymentIntentSucceeded(int $piId, array $session): void
    {
        // Load payment_intent
        $stmt = $this->pdo->prepare('SELECT * FROM payment_intents WHERE id = :id LIMIT 1');
        $stmt->execute([':id' => $piId]);
        $pi = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$pi) { return; }
        if (($pi['status'] ?? '') === 'succeeded') { return; }

        $payload = json_decode((string)$pi['payload_json'], true);
        if (!is_array($payload)) { $payload = []; }
        $userId = (int)($pi['user_id'] ?? 0);
        $amount = (float)($pi['amount'] ?? 0);
        $currency = (string)($pi['currency'] ?? 'THB');

        // We'll prepare result payload after inserting bookings
        $providerPi = (string)($session['payment_intent'] ?? ($session['id'] ?? ''));

        // Finalize booking (minimal): insert into bookings and hotel_bookings if module=hotel
        $draft = $payload['draft'] ?? [];
        $guest = $payload['guest'] ?? [];
        $module = 'hotel';
        $hotelId = (int)($draft['hotel_id'] ?? 0);
        $checkin = (string)($draft['checkin'] ?? '');
        $checkout = (string)($draft['checkout'] ?? '');
        $roomsQty = (int)($draft['rooms_qty'] ?? 1);
        $items = is_array($draft['items'] ?? null) ? $draft['items'] : [];
        $customerName = (string)($guest['lead_guest']['name'] ?? 'Guest');
        $customerEmail = (string)($guest['lead_guest']['email'] ?? null);

        // Generate a booking_code
        $bookingCode = 'HT' . date('ymdHis') . substr(str_shuffle('ABCDEFGHJKMNPQRSTUVWXYZ23456789'), 0, 4);

        // Insert minimal generic booking (for listing)
        $bookingId = null;
        try {
            $details = [
                'customer_name' => $customerName,
                'lead_guest' => $guest['lead_guest'] ?? null,
            ];
            $stmtB = $this->pdo->prepare('INSERT INTO bookings (user_id, module, item_id, pax, price, status, payment_status, details_json, created_at) VALUES (:uid, :module, :item_id, :pax, :price, :status, :pay_status, :details, NOW())');
            $stmtB->execute([
                ':uid' => $userId,
                ':module' => $module,
                ':item_id' => $hotelId,
                ':pax' => 1,
                ':price' => $amount,
                ':status' => 'pending',
                ':pay_status' => 'paid',
                ':details' => json_encode($details),
            ]);
            $bookingId = (int)$this->pdo->lastInsertId();
        } catch (\Throwable $_) { /* ignore */ }

        // Insert hotel_bookings row for the first selected item (simplified)
        $hotelBookingId = null;
        try {
            $roomId = (int)($items[0]['room_id'] ?? 0);
            $ratePlanId = isset($items[0]['rate_plan_id']) ? (int)$items[0]['rate_plan_id'] : null;
            $guestsJson = json_encode($guest['rooms'] ?? []);
            // nights
            $nights = 1;
            try {
                $ci = new \DateTimeImmutable($checkin);
                $co = new \DateTimeImmutable($checkout);
                $nights = max(1, (int)$ci->diff($co)->days);
            } catch (\Throwable $_) {}

            $stmtH = $this->pdo->prepare('INSERT INTO hotel_bookings (hotel_id, vendor_id, booking_code, checkin, checkout, nights, room_id, rate_plan_id, rooms_qty, guests_json, customer_name, customer_email, status, total_price, currency, payment_status, created_at) VALUES (:hid, :vid, :code, :ci, :co, :n, :rid, :rpid, :qty, :guests, :cname, :cemail, "confirmed", :total, :cur, "paid", NOW())');
            $stmtH->execute([
                ':hid' => $hotelId,
                ':vid' => 0,
                ':code' => $bookingCode,
                ':ci' => $checkin,
                ':co' => $checkout,
                ':n' => $nights,
                ':rid' => $roomId,
                ':rpid' => $ratePlanId,
                ':qty' => $roomsQty,
                ':guests' => $guestsJson,
                ':cname' => $customerName,
                ':cemail' => $customerEmail,
                ':total' => $amount,
                ':cur' => $currency,
            ]);
            $hotelBookingId = (int)$this->pdo->lastInsertId();
        } catch (\Throwable $_) { /* ignore */ }

        // Load hotel info for notifications
        $hotelName = '';
        $hotelCity = '';
        try {
            $stH = $this->pdo->prepare('SELECT name, city FROM hotels WHERE id = :hid LIMIT 1');
            $stH->execute([':hid' => $hotelId]);
            $rowH = $stH->fetch(\PDO::FETCH_ASSOC) ?: [];
            $hotelName = (string)($rowH['name'] ?? '');
            $hotelCity = (string)($rowH['city'] ?? '');
        } catch (\Throwable $_) { /* ignore */ }

        // Now persist success status with combined result payload
        try {
            $result = [
                'session' => $session,
                'booking_id' => $bookingId,
                'hotel_booking_id' => $hotelBookingId,
                'booking_code' => $bookingCode,
                'hotel_name' => $hotelName,
                'hotel_city' => $hotelCity,
                'checkin' => $checkin,
                'checkout' => $checkout,
                'nights' => $nights,
            ];
            $resJson = json_encode($result);
            $stmt = $this->pdo->prepare('UPDATE payment_intents SET status = "succeeded", result_json = :res, processed_at = NOW(), provider_intent_id = :pid WHERE id = :id');
            $stmt->execute([':res' => $resJson, ':pid' => $providerPi, ':id' => $piId]);
        } catch (\Throwable $_) { /* ignore */ }

        // Send confirmation emails (client + vendor)
        try {
            $appCfg = require __DIR__ . '/../../config/app.php';
            $mailCfg = require __DIR__ . '/../../config/mail.php';
        } catch (\Throwable $_) { $appCfg = []; $mailCfg = []; }
        $appName = (string)($appCfg['name'] ?? 'BookMyThai');
        $voucherUrl = (rtrim((string)($appCfg['url'] ?? 'http://localhost:8000'), '/')) . '/b2b/agent/bookings/voucher?id=' . (int)$hotelBookingId;

        // Client email
        if (!empty($customerEmail)) {
            $subject = '['.$appName.'] Booking Confirmed - ' . $bookingCode;
            $html = '<p>Dear ' . htmlspecialchars($customerName ?: 'Guest') . ',</p>' .
                    '<p>Thank you for your payment. Your hotel booking is confirmed.</p>' .
                    '<p><strong>Booking Code:</strong> ' . htmlspecialchars($bookingCode) . '</p>' .
                    ($hotelName !== '' ? '<p><strong>Hotel:</strong> ' . htmlspecialchars($hotelName) . ($hotelCity!==''?' — '.htmlspecialchars($hotelCity):'') . '</p>' : '') .
                    '<p><strong>Check-in:</strong> ' . htmlspecialchars($checkin) . ' &nbsp; <strong>Check-out:</strong> ' . htmlspecialchars($checkout) . ' &nbsp; <strong>Nights:</strong> ' . (int)$nights . '</p>' .
                    '<p>You can view your voucher here: <a href="' . htmlspecialchars($voucherUrl) . '">View Voucher</a></p>' .
                    '<p>Regards,<br>' . htmlspecialchars($appName) . '</p>';
            $from = $mailCfg['profiles']['hotel']['from'] ?? ($mailCfg['from'] ?? ['address'=>'noreply@example.com','name'=>$appName]);
            $bccCsv = (string)($mailCfg['profiles']['hotel']['copy_to'] ?? ($mailCfg['copy_to'] ?? ''));
            $bcc = array_filter(array_map('trim', explode(',', $bccCsv)), fn($e)=>$e!=='');

            $sg = new \App\Core\Mail\SendGridClient(getenv('SENDGRID_API_KEY') ?: null);
            if ($sg->enabled()) {
                $sg->sendHtml($customerEmail, $subject, $html, $from, $bcc);
            } else {
                $mailer = new \App\Core\Mail\Mailer($mailCfg);
                $mailer->sendHtml($customerEmail, $subject, $html, strip_tags($html), 'hotel');
            }
        }

        // Vendor email (if available)
        try {
            $vendorEmail = '';
            $vendorPhone = '';
            // Attempt via hotels table
            $stV = $this->pdo->prepare('SELECT v.email, v.phone FROM hotels h LEFT JOIN vendors v ON v.id = h.vendor_id WHERE h.id = :hid LIMIT 1');
            $stV->execute([':hid' => $hotelId]);
            $rowV = $stV->fetch(\PDO::FETCH_ASSOC) ?: [];
            $vendorEmail = (string)($rowV['email'] ?? '');
            $vendorPhone = (string)($rowV['phone'] ?? '');
            if ($vendorEmail !== '') {
                $subjectV = '['.$appName.'] New Booking - ' . $bookingCode;
                $htmlV = '<p>Dear Partner,</p>' .
                         '<p>A new booking has been confirmed.</p>' .
                         '<p><strong>Code:</strong> ' . htmlspecialchars($bookingCode) . '</p>' .
                         ($hotelName !== '' ? '<p><strong>Hotel:</strong> ' . htmlspecialchars($hotelName) . ($hotelCity!==''?' — '.htmlspecialchars($hotelCity):'') . '</p>' : '') .
                         '<p><strong>Check-in:</strong> ' . htmlspecialchars($checkin) . ' &nbsp; <strong>Check-out:</strong> ' . htmlspecialchars($checkout) . ' &nbsp; <strong>Nights:</strong> ' . (int)$nights . '</p>' .
                         '<p><strong>Rooms:</strong> ' . (int)$roomsQty . '</p>' .
                         '<p>Please prepare for guest arrival.</p>';
                $from = $mailCfg['profiles']['hotel']['from'] ?? ($mailCfg['from'] ?? ['address'=>'noreply@example.com','name'=>$appName]);
                $sg = new \App\Core\Mail\SendGridClient(getenv('SENDGRID_API_KEY') ?: null);
                if ($sg->enabled()) { $sg->sendHtml($vendorEmail, $subjectV, $htmlV, $from); }
                else { (new \App\Core\Mail\Mailer($mailCfg))->sendHtml($vendorEmail, $subjectV, $htmlV, strip_tags($htmlV), 'hotel'); }
            }
        } catch (\Throwable $_) { /* ignore vendor mail errors */ }

        // Twilio WhatsApp/SMS notifications (client and vendor)
        try {
            $twilio = new \App\Core\Sms\TwilioClient();
            if ($twilio->enabled()) {
                // Client phone from guest lead payload if available
                $clientPhone = (string)($guest['lead_guest']['phone'] ?? '');
                if ($clientPhone !== '') {
                    $msg = 'Confirmed: ' . ($hotelName!==''?$hotelName.' ':'') . '(' . $bookingCode . ') ' .
                           'Check-in ' . $checkin . ', Nights ' . (int)$nights . '. Voucher: ' . $voucherUrl;
                    $twilio->send($clientPhone, $msg, ['prefer' => 'whatsapp']);
                }
                // Vendor phone if available
                if (!empty($vendorPhone)) {
                    $msgV = 'New booking ' . $bookingCode . ' — ' . ($hotelName!==''?$hotelName.' ':'') .
                            'Check-in ' . $checkin . ', Nights ' . (int)$nights . ', Rooms ' . (int)$roomsQty . '.';
                    $twilio->send($vendorPhone, $msgV, ['prefer' => 'whatsapp']);
                }
            }
        } catch (\Throwable $_) { /* ignore sms errors */ }
    }
}
