<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class VendorsController extends Controller
{
    private function ensureVendorSchema(): void
    {
        // Add extended columns if they don't exist (MySQL 8+ supports IF NOT EXISTS)
        $sql = [
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS company_name VARCHAR(150) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS module ENUM('hotel','activity','taxi','evisa','other') NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS country VARCHAR(100) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS city VARCHAR(100) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS address TEXT NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS whatsapp_number VARCHAR(30) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS calling_number VARCHAR(30) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS line_id VARCHAR(100) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS enquiry_email VARCHAR(150) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS confirm_email VARCHAR(150) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS vat_number VARCHAR(50) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS tat_number VARCHAR(50) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS bank_account VARCHAR(100) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS contact_person_name VARCHAR(150) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS contact_person_phone VARCHAR(50) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS contact_person_email VARCHAR(150) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS bank_name VARCHAR(150) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS bank_branch VARCHAR(150) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS bank_account_name VARCHAR(150) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS bank_swift VARCHAR(50) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS bank_iban VARCHAR(50) NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS is_adult_only TINYINT(1) NOT NULL DEFAULT 0",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS is_family_allowed TINYINT(1) NOT NULL DEFAULT 1",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS has_show_times TINYINT(1) NOT NULL DEFAULT 0",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS show_times JSON NULL",
            "ALTER TABLE vendors ADD COLUMN IF NOT EXISTS last_booking_cutoff TIME NULL"
        ];
        foreach ($sql as $q) {
            try { $this->pdo->exec($q); } catch (\Throwable $e) { /* ignore */ }
        }
    }

    public function create(): void
    {
        Auth::requireRole(['Admin']);
        $this->ensureVendorSchema();
        $csrf = Security::csrfToken();
        // Load existing vendors for display (full fields for prefill in Edit modal)
        try { $vendors = $this->pdo->query("SELECT id, name, company_name, module, country, city, address, contact_email, enquiry_email, confirm_email, vat_number, tat_number, bank_account, contact_person_name, contact_person_phone, contact_person_email, bank_name, bank_branch, bank_account_name, bank_swift, bank_iban, whatsapp_number, calling_number, line_id, is_adult_only, is_family_allowed, has_show_times, show_times, last_booking_cutoff FROM vendors ORDER BY name")->fetchAll(); }
        catch (\Throwable $e) { $vendors = []; }
        // Load modules list (from modules table, if present)
        try { $modules = $this->pdo->query("SELECT slug, name FROM modules ORDER BY name")->fetchAll(); }
        catch (\Throwable $e) { $modules = [ ['slug'=>'hotel','name'=>'Hotel'], ['slug'=>'activity','name'=>'Activity'], ['slug'=>'taxi','name'=>'Taxi'], ['slug'=>'evisa','name'=>'eVisa'] ]; }
        // Load locations for country/city selects
        try {
            $countries = $this->pdo->query("SELECT DISTINCT country FROM locations WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
            $cities = $this->pdo->query("SELECT country, city FROM locations WHERE city IS NOT NULL AND city<>'' ORDER BY country, city")->fetchAll();
        } catch (\Throwable $e) {
            $countries = [];
            $cities = [];
        }
        // Build login map by contact email -> {user_id, role, status}
        $loginsByEmail = [];
        try {
            $emails = array_values(array_unique(array_filter(array_map(fn($v)=> (string)($v['contact_email'] ?? ''), $vendors))));
            if (!empty($emails)) {
                // Prepare IN query safely
                $in = implode(',', array_fill(0, count($emails), '?'));
                $q = $this->pdo->prepare("SELECT id, email, role, status FROM users WHERE email IN ($in)");
                $q->execute($emails);
                foreach ($q->fetchAll(\PDO::FETCH_ASSOC) as $u) {
                    $loginsByEmail[strtolower((string)$u['email'])] = [
                        'user_id' => (int)$u['id'],
                        'role' => (string)$u['role'],
                        'status' => (string)$u['status'],
                    ];
                }
            }
        } catch (\Throwable $e) { $loginsByEmail = []; }

        $this->view('admin/vendors_create', [
            'csrf' => $csrf,
            'vendors' => $vendors,
            'modules' => $modules,
            'countries' => $countries,
            'cities' => $cities,
            'loginsByEmail' => $loginsByEmail,
        ]);
    }

    public function store(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        // No master password required for vendor login creation (mirrors customers create flow)
        $this->ensureVendorSchema();

        // Basic fields
        $name = trim((string)($_POST['name'] ?? ''));
        $company = trim((string)($_POST['company_name'] ?? ''));
        $module = trim((string)($_POST['module'] ?? ''));
        $country = trim((string)($_POST['country'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        $address = trim((string)($_POST['address'] ?? ''));
        $email = trim((string)($_POST['contact_email'] ?? ''));
        $enquiryEmail = trim((string)($_POST['enquiry_email'] ?? ''));
        $confirmEmail = trim((string)($_POST['confirm_email'] ?? ''));
        $vatNumber = trim((string)($_POST['vat_number'] ?? ''));
        $tatNumber = trim((string)($_POST['tat_number'] ?? ''));
        $bankAccount = trim((string)($_POST['bank_account'] ?? ''));
        $cpName = trim((string)($_POST['contact_person_name'] ?? ''));
        $cpPhone = trim((string)($_POST['contact_person_phone'] ?? ''));
        $cpEmail = trim((string)($_POST['contact_person_email'] ?? ''));
        $bankName = trim((string)($_POST['bank_name'] ?? ''));
        $bankBranch = trim((string)($_POST['bank_branch'] ?? ''));
        $bankAccName = trim((string)($_POST['bank_account_name'] ?? ''));
        $bankSwift = trim((string)($_POST['bank_swift'] ?? ''));
        $bankIban = trim((string)($_POST['bank_iban'] ?? ''));
        $whatsapp = trim((string)($_POST['whatsapp_number'] ?? ''));
        $calling = trim((string)($_POST['calling_number'] ?? ''));
        $lineId = trim((string)($_POST['line_id'] ?? ''));

        // Activity-specific
        $isAdultOnly = isset($_POST['is_adult_only']) ? 1 : 0;
        $isFamilyAllowed = isset($_POST['is_family_allowed']) ? 1 : 0;
        $hasShowTimes = isset($_POST['has_show_times']) ? 1 : 0;
        $showTimesRaw = trim((string)($_POST['show_times'] ?? ''));
        $cutoff = trim((string)($_POST['last_booking_cutoff'] ?? ''));
        $showTimesArr = [];
        if ($hasShowTimes && $showTimesRaw !== '') {
            // Accept comma-separated times like 09:00,13:30,17:45 or with AM/PM; normalize to HH:MM 24h
            $parts = preg_split('/\s*,\s*/', $showTimesRaw);
            foreach ($parts as $p) {
                $t = date_create($p);
                if ($t) { $showTimesArr[] = $t->format('H:i'); }
            }
            $showTimesArr = array_values(array_unique($showTimesArr));
        }

        $errors = [];
        if ($name === '') { $errors[] = 'Vendor name is required.'; }
        if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Contact email is invalid.'; }
        if ($enquiryEmail !== '' && !filter_var($enquiryEmail, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Enquiry email is invalid.'; }
        if ($confirmEmail !== '' && !filter_var($confirmEmail, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Confirmation email is invalid.'; }
        if ($module !== '' && !in_array($module, ['hotel','activity','taxi','evisa','other'], true)) { $errors[] = 'Invalid module selected.'; }
        if ($cutoff !== '') {
            $dt = date_create($cutoff);
            if (!$dt) { $errors[] = 'Invalid last booking cutoff time.'; }
            else { $cutoff = $dt->format('H:i:s'); }
        } else { $cutoff = null; }

        if (!empty($errors)) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old'] = $_POST;
            $this->redirect('/admin/vendors/create');
        }

        $stmt = $this->pdo->prepare('INSERT INTO vendors 
            (name, company_name, module, country, city, address, contact_email, enquiry_email, confirm_email, vat_number, tat_number, bank_account, contact_person_name, contact_person_phone, contact_person_email, bank_name, bank_branch, bank_account_name, bank_swift, bank_iban, whatsapp_number, calling_number, line_id, is_adult_only, is_family_allowed, has_show_times, show_times, last_booking_cutoff)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)');
        $ok = $stmt->execute([
            $name !== '' ? $name : null,
            $company !== '' ? $company : null,
            $module !== '' ? $module : null,
            $country !== '' ? $country : null,
            $city !== '' ? $city : null,
            $address !== '' ? $address : null,
            $email !== '' ? $email : null,
            $enquiryEmail !== '' ? $enquiryEmail : null,
            $confirmEmail !== '' ? $confirmEmail : null,
            $vatNumber !== '' ? $vatNumber : null,
            $tatNumber !== '' ? $tatNumber : null,
            $bankAccount !== '' ? $bankAccount : null,
            $cpName !== '' ? $cpName : null,
            $cpPhone !== '' ? $cpPhone : null,
            $cpEmail !== '' ? $cpEmail : null,
            $bankName !== '' ? $bankName : null,
            $bankBranch !== '' ? $bankBranch : null,
            $bankAccName !== '' ? $bankAccName : null,
            $bankSwift !== '' ? $bankSwift : null,
            $bankIban !== '' ? $bankIban : null,
            $whatsapp !== '' ? $whatsapp : null,
            $calling !== '' ? $calling : null,
            $lineId !== '' ? $lineId : null,
            (int)$isAdultOnly,
            (int)$isFamilyAllowed,
            (int)$hasShowTimes,
            !empty($showTimesArr) ? json_encode($showTimesArr) : null,
            $cutoff
        ]);

        $_SESSION['flash'] = $ok ? 'Vendor created successfully.' : 'Failed to create vendor.';
        if ($ok) {
            unset($_SESSION['old']);
        } else {
            $_SESSION['old'] = $_POST;
        }
        $this->redirect('/admin/vendors/create');
    }

    // POST /admin/vendors/reset-login
    public function resetLogin(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
        $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
        $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
        $email = trim((string)($_POST['email'] ?? ''));
        $newPassword = (string)($_POST['new_password'] ?? '');
        if ($email === '' || $newPassword === '' || strlen($newPassword) < 6) {
            $msg = 'Valid email and new password (min 6) are required.';
            if ($isAjax) { http_response_code(422); header('Content-Type: application/json'); echo json_encode(['ok'=>false,'error'=>$msg]); return; }
            $_SESSION['errors'] = [$msg];
            $this->redirect('/admin/vendors/create');
        }
        try {
            $st = $this->pdo->prepare('SELECT id FROM users WHERE email = :e LIMIT 1');
            $st->execute([':e'=>$email]);
            $uid = (int)($st->fetchColumn() ?: 0);
            if ($uid <= 0) {
                $msg = 'User not found for this email.';
                if ($isAjax) { http_response_code(404); header('Content-Type: application/json'); echo json_encode(['ok'=>false,'error'=>$msg]); return; }
                $_SESSION['errors'] = [$msg];
                $this->redirect('/admin/vendors/create');
            }
            $hash = password_hash($newPassword, PASSWORD_BCRYPT);
            try {
                $this->pdo->prepare('UPDATE users SET password = :p WHERE id = :id')->execute([':p'=>$hash, ':id'=>$uid]);
            } catch (\Throwable $e1) {
                $this->pdo->prepare('UPDATE users SET password_hash = :p WHERE id = :id')->execute([':p'=>$hash, ':id'=>$uid]);
            }
            if ($isAjax) { header('Content-Type: application/json'); echo json_encode(['ok'=>true,'message'=>'Password reset.']); return; }
            $_SESSION['flash'] = 'Password reset.';
        } catch (\Throwable $e) {
            $msg = 'Failed to reset password.';
            if ($isAjax) { http_response_code(500); header('Content-Type: application/json'); echo json_encode(['ok'=>false,'error'=>$msg]); return; }
            $_SESSION['errors'] = [$msg];
        }
        $this->redirect('/admin/vendors/create');
    }

    public function update(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $_SESSION['errors'] = ['Invalid vendor id.']; $this->redirect('/admin/vendors/create'); }

        // Basic and extended fields (mirrors store())
        $name = trim((string)($_POST['name'] ?? ''));
        $company = trim((string)($_POST['company_name'] ?? ''));
        $module = trim((string)($_POST['module'] ?? ''));
        $country = trim((string)($_POST['country'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        $address = trim((string)($_POST['address'] ?? ''));
        $email = trim((string)($_POST['contact_email'] ?? ''));
        $enquiryEmail = trim((string)($_POST['enquiry_email'] ?? ''));
        $confirmEmail = trim((string)($_POST['confirm_email'] ?? ''));
        $vatNumber = trim((string)($_POST['vat_number'] ?? ''));
        $tatNumber = trim((string)($_POST['tat_number'] ?? ''));
        $bankAccount = trim((string)($_POST['bank_account'] ?? ''));
        $cpName = trim((string)($_POST['contact_person_name'] ?? ''));
        $cpPhone = trim((string)($_POST['contact_person_phone'] ?? ''));
        $cpEmail = trim((string)($_POST['contact_person_email'] ?? ''));
        $bankName = trim((string)($_POST['bank_name'] ?? ''));
        $bankBranch = trim((string)($_POST['bank_branch'] ?? ''));
        $bankAccName = trim((string)($_POST['bank_account_name'] ?? ''));
        $bankSwift = trim((string)($_POST['bank_swift'] ?? ''));
        $bankIban = trim((string)($_POST['bank_iban'] ?? ''));
        $whatsapp = trim((string)($_POST['whatsapp_number'] ?? ''));
        $calling = trim((string)($_POST['calling_number'] ?? ''));
        $lineId = trim((string)($_POST['line_id'] ?? ''));

        $isAdultOnly = isset($_POST['is_adult_only']) ? 1 : 0;
        $isFamilyAllowed = isset($_POST['is_family_allowed']) ? 1 : 0;
        $hasShowTimes = isset($_POST['has_show_times']) ? 1 : 0;
        $showTimesRaw = trim((string)($_POST['show_times'] ?? ''));
        $cutoff = trim((string)($_POST['last_booking_cutoff'] ?? ''));
        $showTimesArr = [];
        if ($hasShowTimes && $showTimesRaw !== '') {
            $parts = preg_split('/\s*,\s*/', $showTimesRaw);
            foreach ($parts as $p) {
                $t = date_create($p);
                if ($t) { $showTimesArr[] = $t->format('H:i'); }
            }
            $showTimesArr = array_values(array_unique($showTimesArr));
        }

        $errors = [];
        if ($name === '') { $errors[] = 'Vendor name is required.'; }
        if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Contact email is invalid.'; }
        if ($enquiryEmail !== '' && !filter_var($enquiryEmail, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Enquiry email is invalid.'; }
        if ($confirmEmail !== '' && !filter_var($confirmEmail, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Confirmation email is invalid.'; }
        if ($module !== '' && !in_array($module, ['hotel','activity','taxi','evisa','other'], true)) { $errors[] = 'Invalid module selected.'; }
        if ($cutoff !== '') {
            $dt = date_create($cutoff);
            if (!$dt) { $errors[] = 'Invalid last booking cutoff time.'; }
            else { $cutoff = $dt->format('H:i:s'); }
        } else { $cutoff = null; }

        if (!empty($errors)) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old'] = $_POST;
            $this->redirect('/admin/vendors/create');
        }

        $stmt = $this->pdo->prepare('UPDATE vendors SET 
            name=?, company_name=?, module=?, country=?, city=?, address=?, contact_email=?, enquiry_email=?, confirm_email=?, vat_number=?, tat_number=?, bank_account=?, contact_person_name=?, contact_person_phone=?, contact_person_email=?, bank_name=?, bank_branch=?, bank_account_name=?, bank_swift=?, bank_iban=?, whatsapp_number=?, calling_number=?, line_id=?, is_adult_only=?, is_family_allowed=?, has_show_times=?, show_times=?, last_booking_cutoff=?
            WHERE id = ?');
        $ok = $stmt->execute([
            $name !== '' ? $name : null,
            $company !== '' ? $company : null,
            $module !== '' ? $module : null,
            $country !== '' ? $country : null,
            $city !== '' ? $city : null,
            $address !== '' ? $address : null,
            $email !== '' ? $email : null,
            $enquiryEmail !== '' ? $enquiryEmail : null,
            $confirmEmail !== '' ? $confirmEmail : null,
            $vatNumber !== '' ? $vatNumber : null,
            $tatNumber !== '' ? $tatNumber : null,
            $bankAccount !== '' ? $bankAccount : null,
            $cpName !== '' ? $cpName : null,
            $cpPhone !== '' ? $cpPhone : null,
            $cpEmail !== '' ? $cpEmail : null,
            $bankName !== '' ? $bankName : null,
            $bankBranch !== '' ? $bankBranch : null,
            $bankAccName !== '' ? $bankAccName : null,
            $bankSwift !== '' ? $bankSwift : null,
            $bankIban !== '' ? $bankIban : null,
            $whatsapp !== '' ? $whatsapp : null,
            $calling !== '' ? $calling : null,
            $lineId !== '' ? $lineId : null,
            (int)$isAdultOnly,
            (int)$isFamilyAllowed,
            (int)$hasShowTimes,
            !empty($showTimesArr) ? json_encode($showTimesArr) : null,
            $cutoff,
            $id
        ]);
        $_SESSION['flash'] = $ok ? 'Vendor updated.' : 'Failed to update vendor.';
        $this->redirect('/admin/vendors/create');
    }

    public function delete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $_SESSION['errors'] = ['Invalid vendor id.']; $this->redirect('/admin/vendors/create'); }
        $stmt = $this->pdo->prepare('DELETE FROM vendors WHERE id = ?');
        $ok = $stmt->execute([$id]);
        $_SESSION['flash'] = $ok ? 'Vendor deleted.' : 'Failed to delete vendor.';
        $this->redirect('/admin/vendors/create');
    }

    public function activityCreate(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        // Ensure vendor extended columns exist before we query vendors for country/city
        $this->ensureVendorSchema();

        // Filters
        $vendorId = (int)($_GET['vendor_id'] ?? 0);
        $selectedCountry = trim((string)($_GET['country'] ?? ''));
        $selectedCity = trim((string)($_GET['city'] ?? ''));

        // Load countries from locations table; fallback to vendors table if empty
        try {
            $countries = $this->pdo->query("SELECT DISTINCT country FROM locations WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
        } catch (\Throwable $e) { $countries = []; }
        if (!$countries || count($countries) === 0) {
            try {
                $countries = $this->pdo->query("SELECT DISTINCT country FROM vendors WHERE country IS NOT NULL AND country<>'' ORDER BY country")->fetchAll();
            } catch (\Throwable $e) { /* ignore */ }
        }

        // Load cities from locations table; fallback to vendors table if empty
        try {
            if ($selectedCountry !== '') {
                $stmt = $this->pdo->prepare("SELECT DISTINCT city FROM locations WHERE country=? AND city IS NOT NULL AND city<>'' ORDER BY city");
                $stmt->execute([$selectedCountry]);
                $cities = $stmt->fetchAll();
            } else {
                $cities = $this->pdo->query("SELECT DISTINCT city FROM locations WHERE city IS NOT NULL AND city<>'' ORDER BY city")->fetchAll();
            }
        } catch (\Throwable $e) { $cities = []; }
        if (!$cities || count($cities) === 0) {
            try {
                if ($selectedCountry !== '') {
                    $stmt = $this->pdo->prepare("SELECT DISTINCT city FROM vendors WHERE country=? AND city IS NOT NULL AND city<>'' ORDER BY city");
                    $stmt->execute([$selectedCountry]);
                    $cities = $stmt->fetchAll();
                } else {
                    $cities = $this->pdo->query("SELECT DISTINCT city FROM vendors WHERE city IS NOT NULL AND city<>'' ORDER BY city")->fetchAll();
                }
            } catch (\Throwable $e) { /* ignore */ }
        }

        // Vendors for selector (optionally filtered by country/city) – only Activity vendors
        try {
            if ($selectedCountry !== '' && $selectedCity !== '') {
                $stmtVendors = $this->pdo->prepare("SELECT id, name FROM vendors WHERE module='activity' AND country=? AND city=? ORDER BY name");
                $stmtVendors->execute([$selectedCountry, $selectedCity]);
                $vendors = $stmtVendors->fetchAll();
            } elseif ($selectedCountry !== '') {
                $stmtVendors = $this->pdo->prepare("SELECT id, name FROM vendors WHERE module='activity' AND country=? ORDER BY name");
                $stmtVendors->execute([$selectedCountry]);
                $vendors = $stmtVendors->fetchAll();
            } else {
                $vendors = $this->pdo->query("SELECT id, name FROM vendors WHERE module='activity' ORDER BY name")->fetchAll();
            }
        } catch (\Throwable $e) { $vendors = []; }

        $packages = [];
        $variantsByPkg = [];
        $pricesByVariant = [];
        $timesByPkg = [];
        $defaultRequiresTime = 1; // fallback
        $defaultAgePolicy = 'adult_only'; // fallback
        $defaultShowTimes = '';
        if ($vendorId > 0) {
            try {
                $stmt = $this->pdo->prepare("SELECT * FROM vendor_packages WHERE vendor_id=? ORDER BY id DESC");
                $stmt->execute([$vendorId]);
                $packages = $stmt->fetchAll();

                if ($packages) {
                    $pkgIds = array_column($packages, 'id');
                    // Variants
                    $in = implode(',', array_fill(0, count($pkgIds), '?'));
                    $stmtV = $this->pdo->prepare("SELECT * FROM vendor_package_variants WHERE package_id IN ($in) ORDER BY id DESC");
                    $stmtV->execute($pkgIds);
                    $variants = $stmtV->fetchAll();
                    foreach ($variants as $v) { $variantsByPkg[$v['package_id']][] = $v; }

                    // Prices
                    if (!empty($variants)) {
                        $variantIds = array_column($variants, 'id');
                        $in2 = implode(',', array_fill(0, count($variantIds), '?'));
                        $stmtP = $this->pdo->prepare("SELECT * FROM vendor_package_prices WHERE variant_id IN ($in2) ORDER BY price_type, min_quantity");
                        $stmtP->execute($variantIds);
                        $prices = $stmtP->fetchAll();
                        foreach ($prices as $p) { $pricesByVariant[$p['variant_id']][] = $p; }
                    }

                    // Showtimes overrides (package-level)
                    $stmtT = $this->pdo->prepare("SELECT * FROM vendor_package_showtimes WHERE package_id IN ($in) AND (variant_id IS NULL OR variant_id=0) AND active=1 ORDER BY time");
                    $stmtT->execute($pkgIds);
                    $times = $stmtT->fetchAll();
                    foreach ($times as $t) { $timesByPkg[$t['package_id']][] = $t['time']; }
                }

                // Compute defaults from historical packages
                if (!empty($packages)) {
                    $rtCounts = ['1' => 0, '0' => 0];
                    $apCounts = ['adult_only' => 0, 'family' => 0];
                    foreach ($packages as $p) {
                        $rt = isset($p['requires_show_time']) ? (string)(int)$p['requires_show_time'] : '1';
                        $ap = $p['age_policy'] ?? 'adult_only';
                        if (isset($rtCounts[$rt])) { $rtCounts[$rt]++; }
                        if (isset($apCounts[$ap])) { $apCounts[$ap]++; }
                    }
                    $defaultRequiresTime = ($rtCounts['1'] >= $rtCounts['0']) ? 1 : 0;
                    $defaultAgePolicy = ($apCounts['family'] > $apCounts['adult_only']) ? 'family' : 'adult_only';
                }

                // Compute default show times list from this vendor's packages (package-level showtimes)
                try {
                    $stmtST = $this->pdo->prepare("SELECT DISTINCT s.time FROM vendor_package_showtimes s JOIN vendor_packages p ON p.id=s.package_id WHERE p.vendor_id=? AND (s.variant_id IS NULL OR s.variant_id=0) ORDER BY s.time");
                    $stmtST->execute([$vendorId]);
                    $times = array_column($stmtST->fetchAll(), 'time');
                    if (!empty($times)) { $defaultShowTimes = implode(', ', $times); }
                } catch (\Throwable $e) { /* ignore */ }

                // Fallback: vendor.show_times JSON column if present
                if ($defaultShowTimes === '') {
                    // Try variant-level showtimes
                    try {
                        $stmtSTv = $this->pdo->prepare("SELECT DISTINCT s.time FROM vendor_package_showtimes s JOIN vendor_package_variants v ON v.id=s.variant_id JOIN vendor_packages p ON p.id=v.package_id WHERE p.vendor_id=? ORDER BY s.time");
                        $stmtSTv->execute([$vendorId]);
                        $timesV = array_column($stmtSTv->fetchAll(), 'time');
                        if (!empty($timesV)) { $defaultShowTimes = implode(', ', $timesV); }
                    } catch (\Throwable $e) { /* ignore */ }
                }

                // Fallback: use already loaded per-package times if available
                if ($defaultShowTimes === '' && !empty($timesByPkg)) {
                    $union = [];
                    foreach ($timesByPkg as $pid => $arr) {
                        foreach ($arr as $t) { $union[$t] = true; }
                    }
                    if (!empty($union)) { $defaultShowTimes = implode(', ', array_keys($union)); }
                }

                // Fallback: vendor.show_times JSON column if present
                if ($defaultShowTimes === '') {
                    try {
                        $stmtV = $this->pdo->prepare("SELECT show_times FROM vendors WHERE id=?");
                        $stmtV->execute([$vendorId]);
                        $json = $stmtV->fetchColumn();
                        if ($json !== false && $json !== null) {
                            $arr = json_decode($json, true);
                            if (is_array($arr) && !empty($arr)) { $defaultShowTimes = implode(', ', $arr); }
                        }
                    } catch (\Throwable $e) { /* ignore if column missing */ }
                }
            } catch (\Throwable $e) { /* ignore */ }
        }

        $this->view('admin/packages_index', compact(
            'csrf',
            'vendors',
            'vendorId',
            'packages',
            'variantsByPkg',
            'pricesByVariant',
            'timesByPkg',
            'countries',
            'cities',
            'selectedCountry',
            'selectedCity',
            'defaultRequiresTime',
            'defaultAgePolicy',
            'defaultShowTimes'
        ));
    }

    public function hotels(): void
    {
        Auth::requireRole(['Admin']);
        $rows = $this->pdo->query("SELECT h.id, h.name, h.city, h.country, v.name AS vendor FROM hotels h JOIN vendors v ON v.id=h.vendor_id ORDER BY h.id DESC LIMIT 100")->fetchAll();
        $this->view('admin/vendors_hotels', ['rows' => $rows]);
    }

    public function taxies(): void
    {
        Auth::requireRole(['Admin']);
        $rows = $this->pdo->query("SELECT t.id, t.name, t.route, v.name AS vendor FROM taxis t JOIN vendors v ON v.id=t.vendor_id ORDER BY t.id DESC LIMIT 100")->fetchAll();
        $this->view('admin/vendors_taxies', ['rows' => $rows]);
    }

    public function evisa(): void
    {
        Auth::requireRole(['Admin']);
        $rows = $this->pdo->query("SELECT e.id, e.country, e.base_price, v.name AS vendor FROM evisas e JOIN vendors v ON v.id=e.vendor_id ORDER BY e.id DESC LIMIT 100")->fetchAll();
        $this->view('admin/vendors_evisa', ['rows' => $rows]);
    }

    // POST /admin/vendors/create-login
    public function createLogin(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
        $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
        $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        $email = trim((string)($_POST['email'] ?? ''));
        $password = (string)($_POST['password'] ?? '');
        if ($vendorId <= 0 || $email === '' || $password === '') {
            $msg = 'Vendor id, email and password are required.';
            if ($isAjax) { http_response_code(422); header('Content-Type: application/json'); echo json_encode(['ok'=>false,'error'=>$msg]); return; }
            $_SESSION['errors'] = [$msg];
            $this->redirect('/admin/vendors/create');
        }
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $msg = 'Invalid email address.';
            if ($isAjax) { http_response_code(422); header('Content-Type: application/json'); echo json_encode(['ok'=>false,'error'=>$msg]); return; }
            $_SESSION['errors'] = [$msg];
            $this->redirect('/admin/vendors/create');
        }
        $name = '';
        try { $st=$this->pdo->prepare('SELECT name FROM vendors WHERE id=:id'); $st->execute([':id'=>$vendorId]); $name=(string)($st->fetchColumn()?:''); } catch (\Throwable $e) {}
        $hash = password_hash($password, PASSWORD_BCRYPT);
        try {
            // Discover users table columns to build compatible SQL
            $cols = [];
            try {
                $db = $this->pdo->query('SELECT DATABASE()')->fetchColumn();
                $qcols = $this->pdo->prepare("SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = :db AND TABLE_NAME = 'users'");
                $qcols->execute([':db' => $db]);
                $cols = array_map(fn($r)=> (string)$r['COLUMN_NAME'], $qcols->fetchAll(\PDO::FETCH_ASSOC) ?: []);
            } catch (\Throwable $e) { /* ignore, will try common variants */ }

            // Check if user exists
            $u = $this->pdo->prepare('SELECT id FROM users WHERE email=:e LIMIT 1');
            $u->execute([':e'=>$email]);
            $uid = (int)($u->fetchColumn() ?: 0);
            if ($uid > 0) {
                $did = false;
                if (in_array('password', $cols, true)) {
                    $up = $this->pdo->prepare("UPDATE users SET role='Vendor', status='Active', password=:p, name=COALESCE(name,:n) WHERE id=:id");
                    $up->execute([':p'=>$hash, ':n'=>$name!==''?$name:null, ':id'=>$uid]);
                    $did = true;
                } elseif (in_array('password_hash', $cols, true)) {
                    $up2 = $this->pdo->prepare("UPDATE users SET role='Vendor', status='Active', password_hash=:h, name=COALESCE(name,:n) WHERE id=:id");
                    $up2->execute([':h'=>$hash, ':n'=>$name!==''?$name:null, ':id'=>$uid]);
                    $did = true;
                }
                if (!$did) { throw new \RuntimeException('users table missing password columns'); }
            } else {
                // Build insert based on available columns (mirror employee/partner style: uses password column)
                $ok = false;
                if (in_array('password', $cols, true)) {
                    $sql = 'INSERT INTO users (name, email, role, status, password) VALUES (:n, :e, \'' . 'Vendor' . '\', \'' . 'Active' . '\', :p)';
                    $ins = $this->pdo->prepare($sql);
                    $ok = $ins->execute([':n'=>$name!==''?$name:null, ':e'=>$email, ':p'=>$hash]);
                } elseif (in_array('password_hash', $cols, true)) {
                    $sql = 'INSERT INTO users (name, email, role, status, password_hash) VALUES (:n, :e, \'' . 'Vendor' . '\', \'' . 'Active' . '\', :h)';
                    $ins = $this->pdo->prepare($sql);
                    $ok = $ins->execute([':n'=>$name!==''?$name:null, ':e'=>$email, ':h'=>$hash]);
                } else {
                    // As a last resort, try with timestamps and password column
                    $sql = 'INSERT INTO users (name, email, role, status, password, created_at, updated_at) VALUES (:n, :e, \'' . 'Vendor' . '\', \'' . 'Active' . '\', :p, NOW(), NOW())';
                    $ins = $this->pdo->prepare($sql);
                    try { $ok = $ins->execute([':n'=>$name!==''?$name:null, ':e'=>$email, ':p'=>$hash]); } catch (\Throwable $e4) { $ok = false; }
                }
                if (!$ok) { throw new \RuntimeException('insert_failed_users'); }
            }
            if ($isAjax) { header('Content-Type: application/json'); echo json_encode(['ok'=>true,'message'=>'Vendor login created/updated.']); return; }
            $_SESSION['flash'] = 'Vendor login created/updated.';
        } catch (\Throwable $e) {
            $msg = 'Failed to create vendor login.';
            $detail = $e->getMessage();
            $debug = (bool)(defined('APP_DEBUG') ? \APP_DEBUG : (function_exists('env') ? (env('APP_DEBUG','false')!=='false') : false));
            if ($isAjax) {
                http_response_code(500);
                header('Content-Type: application/json');
                $payload = ['ok'=>false,'error'=>$msg];
                if ($debug) { $payload['detail'] = $detail; }
                echo json_encode($payload);
                return;
            }
            $_SESSION['errors'] = [$msg . ($debug ? (" (".$detail.")") : '')];
        }
        $this->redirect('/admin/vendors/create');
    }
}
