<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;

class VendorPortalController extends Controller
{
    // GET /vendor/dashboard
    public function dashboard(): void
    {
        $vendor = $this->requireVendor();

        $stats = [
            'total_jobs' => 0,
            'pending_jobs' => 0,
            'completed_jobs' => 0,
            'revenue' => 0.0,
        ];

        $recentJobs = [];
        try {
            // Total jobs for this vendor (fallback to taxi.vendor_id for legacy rows)
            $q1 = $this->pdo->prepare('SELECT COUNT(*) FROM taxi_bookings b LEFT JOIN taxis t ON t.id=b.taxi_id WHERE COALESCE(b.vendor_id, t.vendor_id) = :vid');
            $q1->execute([':vid' => (int)$vendor['id']]);
            $stats['total_jobs'] = (int)$q1->fetchColumn();

            // Pending = jobs not completed yet (based on workflow columns)
            $q2 = $this->pdo->prepare("SELECT COUNT(*) FROM taxi_bookings b LEFT JOIN taxis t ON t.id=b.taxi_id WHERE COALESCE(b.vendor_id, t.vendor_id) = :vid AND (b.completed_at IS NULL)");
            $q2->execute([':vid' => (int)$vendor['id']]);
            $stats['pending_jobs'] = (int)$q2->fetchColumn();

            // Completed = completed_at set
            $q3 = $this->pdo->prepare('SELECT COUNT(*) FROM taxi_bookings b LEFT JOIN taxis t ON t.id=b.taxi_id WHERE COALESCE(b.vendor_id, t.vendor_id) = :vid AND b.completed_at IS NOT NULL');
            $q3->execute([':vid' => (int)$vendor['id']]);
            $stats['completed_jobs'] = (int)$q3->fetchColumn();

            // Vendor due = sum of paid bookings vendor_cost (fallback to amount_total if vendor_cost is NULL)
            $q4 = $this->pdo->prepare("SELECT COALESCE(SUM(COALESCE(b.vendor_cost, b.amount_total, 0)), 0) FROM taxi_bookings b LEFT JOIN taxis t ON t.id=b.taxi_id WHERE COALESCE(b.vendor_id, t.vendor_id) = :vid AND b.payment_status = 'paid'");
            $q4->execute([':vid' => (int)$vendor['id']]);
            $stats['revenue'] = (float)$q4->fetchColumn();

            // Recent jobs (try rich set; fallback to minimal if some columns are missing in DB)
            try {
                $q5 = $this->pdo->prepare('SELECT b.id, b.booking_code, b.trip_date, b.pickup_time, b.pax, b.status, b.payment_status, b.customer_name, b.from_text, b.to_text, b.amount_total, b.currency, b.vendor_cost, b.vendor_currency, b.created_at, t.name AS ride_name
                                           FROM taxi_bookings b LEFT JOIN taxis t ON t.id=b.taxi_id
                                           WHERE COALESCE(b.vendor_id, t.vendor_id) = :vid
                                           ORDER BY b.id DESC
                                           LIMIT 5');
                $q5->execute([':vid' => (int)$vendor['id']]);
                $recentJobs = $q5->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            } catch (\Throwable $eRich) {
                // Fallback: select minimal columns and alias placeholders for view compatibility
                $q5b = $this->pdo->prepare('SELECT b.id, b.booking_code, b.trip_date, b.pickup_time, b.pax, b.status, b.payment_status,
                                                    "" AS customer_name, "" AS from_text, "" AS to_text,
                                                    b.amount_total, b.currency, NULL AS vendor_cost, NULL AS vendor_currency,
                                                    b.created_at, t.name AS ride_name
                                           FROM taxi_bookings b LEFT JOIN taxis t ON t.id=b.taxi_id
                                           WHERE COALESCE(b.vendor_id, t.vendor_id) = :vid
                                           ORDER BY b.id DESC
                                           LIMIT 5');
                $q5b->execute([':vid' => (int)$vendor['id']]);
                $recentJobs = $q5b->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            }
        } catch (\Throwable $e) {
            error_log('Error loading vendor dashboard: '.$e->getMessage());
            $recentJobs = [];
        }

        $this->view('vendor/dashboard', [
            'title' => 'Vendor Dashboard',
            'vendor' => $vendor,
            'stats' => $stats,
            'recentJobs' => $recentJobs,
        ]);
    }
    
    private function requireVendor(): array
    {
        // Prefer a dedicated vendor session payload if available
        if (!empty($_SESSION['vendor']) && !empty($_SESSION['vendor']['id'])) {
            // validate that this vendor id exists; if not, rebuild below
            try {
                $chk = $this->pdo->prepare('SELECT id, name FROM vendors WHERE id = :id');
                $chk->execute([':id' => (int)$_SESSION['vendor']['id']]);
                if ($chk->fetch()) { return $_SESSION['vendor']; }
            } catch (\Throwable $e) { /* fallthrough to rebuild */ }
        }

        // Otherwise, derive from logged-in user with Vendor role
        $user = $_SESSION['user'] ?? null;
        if (!$user || ($user['role'] ?? '') !== 'Vendor' || empty($user['id'])) {
            $this->redirect('/vendor/login');
            exit;
        }
        // Ensure vendor_id/name present
        if (empty($user['vendor_id'])) {
            try {
                $st = $this->pdo->prepare('SELECT id, name FROM vendors WHERE contact_email = ? LIMIT 1');
                $st->execute([$user['email'] ?? '']);
                $v = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
                if ($v) {
                    $user['vendor_id'] = (int)$v['id'];
                    $user['vendor_name'] = (string)($v['name'] ?? '');
                    $_SESSION['user'] = $user;
                }
            } catch (\Throwable $e) {
                error_log('requireVendor lookup failed: '.$e->getMessage());
            }
        }
        // Normalize return: id should be vendor_id
        $vendor = [
            'id' => (int)($user['vendor_id'] ?? 0),
            'name' => (string)($user['vendor_name'] ?? ''),
            'email' => (string)($user['email'] ?? ''),
        ];
        if ($vendor['id'] <= 0) { $this->redirect('/vendor/login'); exit; }
        // Cache dedicated session for consistency
        $_SESSION['vendor'] = $vendor;
        return $vendor;
    }

    // GET /vendor/jobs
    public function jobs(): void
    {
        $vendor = $this->requireVendor();
        $date = isset($_GET['date']) ? (string)$_GET['date'] : '';
        $status = isset($_GET['status']) ? (string)$_GET['status'] : '';
        $where = ['COALESCE(b.vendor_id, t.vendor_id) = :vid'];
        $args = [':vid' => (int)$vendor['id']];
        if ($date !== '') { $where[] = 'b.trip_date = :dt'; $args[':dt'] = $date; }
        if ($status !== '') { $where[] = 'b.status = :st'; $args[':st'] = $status; }
        $whereSql = 'WHERE ' . implode(' AND ', $where);
        $rows = [];
        try {
            $sql = "SELECT b.id, b.booking_code, b.trip_date, b.pickup_time, b.pax, b.status, b.payment_status, b.from_text, b.to_text,
                           b.amount_total, b.currency, b.vendor_cost, b.vendor_currency,
                           t.name AS ride_name
                    FROM taxi_bookings b
                    LEFT JOIN taxis t ON t.id = b.taxi_id
                    $whereSql
                    ORDER BY b.trip_date ASC, b.pickup_time ASC, b.id DESC
                    LIMIT 300";
            $st = $this->pdo->prepare($sql);
            $st->execute($args);
            $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $rows = []; }
        $csrf = Security::csrfToken();
        $this->view('vendor/jobs', [
            'title' => 'My Jobs',
            'rows' => $rows,
            'filters' => ['date'=>$date,'status'=>$status],
            'csrf' => $csrf,
        ]);
    }

    // GET /vendor/job/view?id=
    public function jobView(): void
    {
        $vendor = $this->requireVendor();
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/vendor/jobs'); return; }
        $row = null; $events = [];
        try {
            $st = $this->pdo->prepare('SELECT b.*, t.name AS ride_name, t.vendor_id AS taxi_vendor_id FROM taxi_bookings b LEFT JOIN taxis t ON t.id=b.taxi_id WHERE b.id=:id AND COALESCE(b.vendor_id, t.vendor_id)=:vid');
            $st->execute([':id'=>$id, ':vid'=>(int)$vendor['id']]);
            $row = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $_) { $row = null; }
        if (!$row) { $this->redirect('/vendor/jobs'); return; }
        // Normalize: if booking.vendor_id is NULL but taxi maps to this vendor, set it now
        try {
            if (empty($row['vendor_id']) && !empty($row['taxi_vendor_id']) && (int)$row['taxi_vendor_id'] === (int)$vendor['id']) {
                $this->pdo->prepare('UPDATE taxi_bookings SET vendor_id = :vid, vendor_assigned_at = IFNULL(vendor_assigned_at, NOW()) WHERE id = :id')->execute([':vid'=>(int)$vendor['id'], ':id'=>$id]);
            }
        } catch (\Throwable $_) { /* ignore */ }
        try {
            $ev = $this->pdo->prepare('SELECT * FROM taxi_booking_events WHERE booking_id=:id ORDER BY id DESC');
            $ev->execute([':id'=>$id]);
            $events = $ev->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $events = []; }
        $csrf = Security::csrfToken();
        $this->view('vendor/job_view', [
            'title' => 'Job Detail',
            'booking' => $row,
            'events' => $events,
            'csrf' => $csrf,
        ]);
    }

    // POST /vendor/job/assign-driver
    public function assignDriver(): void
    {
        $vendor = $this->requireVendor();
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        $name = trim((string)($_POST['driver_name'] ?? ''));
        $phone = trim((string)($_POST['driver_phone'] ?? ''));
        $veh = trim((string)($_POST['vehicle_number'] ?? ''));
        if ($id <= 0 || $name === '' || $phone === '' || $veh === '') { $this->redirect('/vendor/jobs'); return; }
        try {
            $up = $this->pdo->prepare('UPDATE taxi_bookings SET driver_name=:n, driver_phone=:p, vehicle_number=:v, driver_assigned_at=NOW() WHERE id=:id AND vendor_id=:vid');
            $up->execute([':n'=>$name, ':p'=>$phone, ':v'=>$veh, ':id'=>$id, ':vid'=>(int)$vendor['id']]);
            $ev = $this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,\'driver_assigned\',:note,:data,NOW())');
            $ev->execute([':bid'=>$id, ':uid'=>(int)$vendor['id'], ':note'=>'Driver assigned by vendor', ':data'=>json_encode(['driver_name'=>$name,'driver_phone'=>$phone,'vehicle_number'=>$veh,'by'=>'vendor'])]);
        } catch (\Throwable $_) { /* ignore */ }
        $this->redirect('/vendor/job/view?id=' . $id);
    }

    // POST /vendor/job/status
    public function updateStatus(): void
    {
        $vendor = $this->requireVendor();
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        $status = (string)($_POST['status'] ?? '');
        $map = [
            'enroute' => 'enroute_at',
            'picked_up' => 'picked_up_at',
            'completed' => 'completed_at',
        ];
        if ($id <= 0 || !isset($map[$status])) { $this->redirect('/vendor/jobs'); return; }
        $col = $map[$status];
        try {
            $sql = "UPDATE taxi_bookings SET $col = NOW(), status = CASE WHEN :s='completed' THEN 'confirmed' ELSE status END WHERE id=:id AND vendor_id=:vid";
            $st = $this->pdo->prepare($sql);
            $st->execute([':s'=>$status, ':id'=>$id, ':vid'=>(int)$vendor['id']]);
            $ev = $this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,:type,:note,:data,NOW())');
            $ev->execute([':bid'=>$id, ':uid'=>(int)$vendor['id'], ':type'=>$status, ':note'=>'Status updated by vendor', ':data'=>json_encode(['by'=>'vendor'])]);
        } catch (\Throwable $_) { /* ignore */ }
        $this->redirect('/vendor/job/view?id=' . $id);
    }

    // POST /vendor/job/upload-pickup-photo
    public function uploadPickupPhoto(): void
    {
        $vendor = $this->requireVendor();
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0 || empty($_FILES['photo'])) { $this->redirect('/vendor/jobs'); return; }
        $f = $_FILES['photo'];
        if (($f['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) { $this->redirect('/vendor/job/view?id='.$id.'&err=upload'); return; }
        $tmp = (string)$f['tmp_name'];
        $name = (string)$f['name'];
        $size = (int)$f['size'];
        $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
        if (!in_array($ext, ['jpg','jpeg','png'], true) || $size > 6*1024*1024) { $this->redirect('/vendor/job/view?id='.$id.'&err=type'); return; }
        $dir = dirname(__DIR__,2) . '/public/assets/vendor/pickups';
        if (!is_dir($dir)) { @mkdir($dir, 0775, true); }
        $dest = $dir . '/' . 'bk_' . $id . '_' . bin2hex(random_bytes(6)) . '.' . $ext;
        if (!@move_uploaded_file($tmp, $dest)) { $this->redirect('/vendor/job/view?id='.$id.'&err=move'); return; }
        $rel = '/assets/vendor/pickups/' . basename($dest);
        try {
            $up = $this->pdo->prepare('UPDATE taxi_bookings SET pickup_photo_path=:p WHERE id=:id AND vendor_id=:vid');
            $up->execute([':p'=>$rel, ':id'=>$id, ':vid'=>(int)$vendor['id']]);
            $ev = $this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,\'pickup_photo\',:note,:data,NOW())');
            $ev->execute([':bid'=>$id, ':uid'=>(int)$vendor['id'], ':note'=>'Pickup photo uploaded', ':data'=>json_encode(['path'=>$rel,'by'=>'vendor'])]);
        } catch (\Throwable $_) { /* ignore */ }
        $this->redirect('/vendor/job/view?id=' . $id);
    }
}
