<?php
namespace App\Controllers;

use App\Core\Auth;
use App\Core\Controller;
use App\Core\Security;

class UsersController extends Controller
{
    /**
     * Retrieve KYC encryption key (base64-encoded 32 bytes) from environment.
     */
    private function getKycEncKey(): string
    {
        // Use global env() helper loaded by config/config.php
        $keyB64 = env('KYC_ENC_KEY', '');
        if ($keyB64 === '') { throw new \RuntimeException('KYC_ENC_KEY missing in environment'); }
        $key = base64_decode($keyB64, true);
        if ($key === false || strlen($key) !== 32) { throw new \RuntimeException('KYC_ENC_KEY must be base64-encoded 32 bytes'); }
        return $key;
    }

    /**
     * Encrypt a temporary uploaded file with AES-256-GCM (KEK) and store in private storage.
     * Returns relative encrypted file path under storage/ (e.g. storage/kyc/abc.enc)
     */
    private function encryptAndStoreUpload(string $tmpPath, string $originalName, string $subdir, string $prefix): string
    {
        if (!is_file($tmpPath)) { throw new \RuntimeException('Upload temp file missing'); }
        $kek = $this->getKycEncKey();
        $iv = random_bytes(12);
        $data = file_get_contents($tmpPath);
        if ($data === false) { throw new \RuntimeException('Failed reading upload'); }
        $tag = '';
        $cipher = openssl_encrypt($data, 'aes-256-gcm', $kek, OPENSSL_RAW_DATA, $iv, $tag);
        if ($cipher === false || $tag === '') { throw new \RuntimeException('Encryption failed'); }
        $root = dirname(__DIR__, 2);
        $dir = $root . DIRECTORY_SEPARATOR . 'storage' . DIRECTORY_SEPARATOR . $subdir . DIRECTORY_SEPARATOR;
        if (!is_dir($dir)) { @mkdir($dir, 0700, true); }
        $ts = time();
        $rand = bin2hex(random_bytes(8));
        $encBasename = $prefix . '_' . $ts . '_' . $rand . '.enc';
        $encAbs = $dir . $encBasename;
        $metaAbs = $dir . $prefix . '_' . $ts . '_' . $rand . '.meta.json';
        // Write encrypted payload
        if (file_put_contents($encAbs, $cipher) === false) { throw new \RuntimeException('Failed to write encrypted file'); }
        @chmod($encAbs, 0600);
        // Persist metadata (iv, tag, original name, mime guess, size, checksum)
        $mime = function_exists('mime_content_type') ? @mime_content_type($tmpPath) : null;
        $meta = [
            'iv' => base64_encode($iv),
            'tag' => base64_encode($tag),
            'original_name' => $originalName,
            'mime' => $mime,
            'size' => strlen($data),
            'sha256' => hash('sha256', $data),
            'algo' => 'aes-256-gcm',
        ];
        if (file_put_contents($metaAbs, json_encode($meta)) === false) { throw new \RuntimeException('Failed to write meta'); }
        @chmod($metaAbs, 0600);
        // Return relative path under storage
        $rel = 'storage/' . $subdir . '/' . $encBasename;
        return str_replace(['\\'], '/', $rel);
    }
    /**
     * Ensure essential tables for B2B Agent onboarding exist.
     * This is a lightweight safety net for dev/staging where migrations may not have run.
     */
    private function ensureAgentInfrastructure(): void
    {
        // users table is assumed to exist as app depends on it.
        // Create wallets (if not exists)
        $this->pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS wallets (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  balance DECIMAL(12,2) NOT NULL DEFAULT 0,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB
SQL);

        // Agent profile
        $this->pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS agent_profiles (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  company VARCHAR(150) NULL,
  phone VARCHAR(50) NULL,
  commission_percent DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  credit_limit DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  business_type ENUM('freelancer','sole_proprietor','partnership','company') NOT NULL DEFAULT 'freelancer',
  agency_name VARCHAR(150) NULL,
  profile_photo_path VARCHAR(255) NULL,
  title VARCHAR(20) NULL,
  country VARCHAR(100) NULL,
  state VARCHAR(100) NULL,
  city VARCHAR(100) NULL,
  pincode VARCHAR(20) NULL,
  address_line VARCHAR(255) NULL,
  gst_number VARCHAR(30) NULL,
  gst_company VARCHAR(150) NULL,
  iata_registered TINYINT(1) NOT NULL DEFAULT 0,
  iata_code VARCHAR(20) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NULL,
  UNIQUE KEY uniq_agent_profile (user_id),
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB
SQL);

        // Business documents
        $this->pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS agent_business_docs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  doc_type ENUM('selfie','registration_certificate','partnership_deed','incorporation_certificate','gst_certificate','address_proof') NOT NULL,
  file_path VARCHAR(255) NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB
SQL);

        // KYC table
        $this->pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS agent_kyc (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  id_type ENUM('passport','national_id','driving_license','other') NOT NULL,
  id_number VARCHAR(64) NOT NULL,
  doc_front_path VARCHAR(255) NOT NULL,
  doc_back_path VARCHAR(255) NULL,
  status ENUM('pending','approved','rejected') NOT NULL DEFAULT 'pending',
  remarks VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  verified_at TIMESTAMP NULL,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB
SQL);
    }

    public function index(): void
    {
        Auth::checkRole(['Admin']);
        $users = $this->pdo->query('SELECT id, name, email, role, status FROM users ORDER BY id DESC')->fetchAll();
        $this->view('users/index', ['title' => 'Users', 'users' => $users]);
    }

    public function create(): void
    {
        Auth::checkRole(['Admin']);
        $this->view('users/create', ['title' => 'Create User']);
    }

    public function store(): void
    {
        Auth::checkRole(['Admin']);
        if (!Security::verifyCsrf($_POST['csrf'] ?? '')) { http_response_code(400); echo 'CSRF'; return; }

        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = (string)($_POST['password'] ?? '');
        $role = (string)($_POST['role'] ?? 'Employee');
        $status = (string)($_POST['status'] ?? 'Active');
        // Optional agent fields
        $agentCompany = trim($_POST['agent_company'] ?? '');
        $agentPhone = trim($_POST['agent_phone'] ?? '');
        $agentCommission = is_numeric($_POST['agent_commission'] ?? null) ? (float)$_POST['agent_commission'] : null;
        $agentCreditLimit = is_numeric($_POST['agent_credit_limit'] ?? null) ? (float)$_POST['agent_credit_limit'] : null;
        // Business/Agency extras
        $businessType = (string)($_POST['business_type'] ?? 'freelancer');
        $agencyName = trim($_POST['agency_name'] ?? '');
        $title = trim($_POST['title'] ?? 'Mr');
        $country = trim($_POST['country'] ?? '');
        $state = trim($_POST['state'] ?? '');
        $city = trim($_POST['city'] ?? '');
        $pincode = trim($_POST['pincode'] ?? '');
        $addressLine = trim($_POST['address_line'] ?? '');
        $gstNumber = trim($_POST['gst_number'] ?? '');
        $gstCompany = trim($_POST['gst_company'] ?? '');
        $iataRegistered = ($_POST['iata_registered'] ?? '0') === '1' ? 1 : 0;
        $iataCode = trim($_POST['iata_code'] ?? '');
        // KYC fields
        $kycIdType = (string)($_POST['kyc_id_type'] ?? '');
        $kycIdNumber = trim($_POST['kyc_id_number'] ?? '');

        // Persist old inputs for repopulation on error
        $_SESSION['old'] = [
            'name' => $name,
            'email' => $email,
            // Optional agent extras (not stored yet in DB columns)
            'agent_company' => $agentCompany,
            'agent_phone' => $agentPhone,
            'agent_commission' => (string)($agentCommission ?? ''),
            'agent_credit_limit' => (string)($agentCreditLimit ?? ''),
            'business_type' => $businessType,
            'agency_name' => $agencyName,
            'title' => $title,
            'country' => $country,
            'state' => $state,
            'city' => $city,
            'pincode' => $pincode,
            'address_line' => $addressLine,
            'gst_number' => $gstNumber,
            'gst_company' => $gstCompany,
            'iata_registered' => (string)$iataRegistered,
            'iata_code' => $iataCode,
            // KYC
            'kyc_id_type' => $kycIdType,
            'kyc_id_number' => $kycIdNumber,
        ];

        // Load KYC config for dynamic server-side validations
        $kycCfg = require dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'kyc.php';
        $K = $kycCfg['agent'] ?? [];

        // Basic validations
        $errors = [];
        if ($name === '') { $errors[] = 'Name is required.'; }
        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Valid email is required.'; }
        if (strlen($password) < 6) { $errors[] = 'Password must be at least 6 characters.'; }
        $allowedRoles = ['Admin','Employee','Channel Partner','B2B Agent','Vendor','Customer'];
        if (!in_array($role, $allowedRoles, true)) { $errors[] = 'Invalid role selected.'; }

        // Unique email check
        if (!$errors) {
            $chk = $this->pdo->prepare('SELECT id FROM users WHERE email = :email LIMIT 1');
            $chk->execute(['email' => $email]);
            if ($chk->fetch()) { $errors[] = 'Email already exists.'; }
            // Note: Media (profile photo/business card) validation is handled below for B2B Agent with per-type overrides
        }

        // Additional validations for B2B Agent (KYC + numeric constraints)
        if ($role === 'B2B Agent') {
            $allowedBusiness = ['freelancer','sole_proprietor','partnership','company'];
            if (!in_array($businessType, $allowedBusiness, true)) { $errors[] = 'Invalid business type.'; }
            // Map business type to per_type key in config
            $typeKey = [
                'freelancer' => 'freelancer',
                'sole_proprietor' => 'sp',
                'partnership' => 'llp',
                'company' => 'co',
            ][$businessType] ?? 'freelancer';
            $PT = $K['per_type'][$typeKey] ?? [];

            // Effective settings: Government ID
            $govG = $K['gov_id'] ?? [];
            $govPT = $PT['gov_id'] ?? [];
            $govOverride = (bool)($govPT['override'] ?? false);
            $effGovEnabled = (bool)($govG['enabled'] ?? true);
            $effGovReqType = $govOverride ? (bool)($govPT['require_type'] ?? false) : (bool)($govG['require_type'] ?? true);
            $effGovReqNumber = $govOverride ? (bool)($govPT['require_number'] ?? false) : (bool)($govG['require_number'] ?? true);
            $effGovFrontEnabled = $govOverride ? (bool)($govPT['front_enabled'] ?? false) : (bool)($govG['require_front'] ?? true); // front_enabled vs require_front global: enable front if front is required globally
            $effGovReqFront = $govOverride ? (bool)($govPT['require_front'] ?? false) : (bool)($govG['require_front'] ?? true);
            $effGovBackEnabled = $govOverride ? (bool)($govPT['back_enabled'] ?? false) : (bool)($govG['back_enabled'] ?? true);
            $effGovReqBack = $govOverride ? (bool)($govPT['require_back'] ?? false) : (bool)($govG['require_back'] ?? false);

            // Effective settings: Registration
            $regG = $K['registration'] ?? [];
            $regPT = $PT['registration'] ?? [];
            $regOverride = (bool)($regPT['override'] ?? false);
            $effIataEnabled = $regOverride ? (bool)($regPT['iata_enabled'] ?? false) : (bool)($regG['iata_enabled'] ?? true);
            $effIataReqOnYes = $regOverride ? (bool)($regPT['iata_code_required_on_yes'] ?? false) : (bool)($regG['iata_code_required_on_yes'] ?? true);
            $effGstNumEnabled = $regOverride ? (bool)($regPT['gst_number']['enabled'] ?? false) : (bool)($regG['gst_number']['enabled'] ?? true);
            $effGstNumReq = $regOverride ? (bool)($regPT['gst_number']['required'] ?? false) : (bool)($regG['gst_number']['required'] ?? false);
            $effGstComEnabled = $regOverride ? (bool)($regPT['gst_company']['enabled'] ?? false) : (bool)($regG['gst_company']['enabled'] ?? true);
            $effGstComReq = $regOverride ? (bool)($regPT['gst_company']['required'] ?? false) : (bool)($regG['gst_company']['required'] ?? false);

            // Effective settings: Business Info
            $bizG = $K['business_info']['agency_name'] ?? [];
            $bizPT = ($PT['business_info']['agency_name'] ?? []);
            $bizOverride = (bool)(($PT['business_info']['override'] ?? false));
            $effAgencyEnabled = $bizOverride ? (bool)($bizPT['enabled'] ?? false) : (bool)($bizG['enabled'] ?? true);
            // Global key is required_non_freelancer; per-type uses 'required'
            $effAgencyRequired = $bizOverride ? (bool)($bizPT['required'] ?? false) : (bool)($K['business_info']['agency_name']['required_non_freelancer'] ?? false);

            // Effective settings: Media
            $mediaPT = $PT['media'] ?? [];
            $mediaOverride = (bool)($mediaPT['override'] ?? false);
            $effProfileEnabled = $mediaOverride ? (bool)($mediaPT['profile_photo']['enabled'] ?? false) : (bool)($K['profile_photo']['enabled'] ?? true);
            $effProfileRequired = $mediaOverride ? (bool)($mediaPT['profile_photo']['required'] ?? false) : (bool)($K['profile_photo']['required'] ?? true);
            $effBizCardEnabled = $mediaOverride ? (bool)($mediaPT['business_card']['enabled'] ?? false) : (bool)($K['business_card']['enabled'] ?? true);
            $effBizCardRequired = $mediaOverride ? (bool)($mediaPT['business_card']['required'] ?? false) : (bool)($K['business_card']['required'] ?? true);

            // Media validations (moved here for per-type overrides)
            $allowedImg = ['image/jpeg','image/png','image/webp','image/jpg'];
            $profileProvided = isset($_FILES['profile_photo']) && (($_FILES['profile_photo']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK);
            if ($effProfileEnabled && $effProfileRequired && !$profileProvided) { $errors[] = 'Profile photo is required.'; }
            if ($profileProvided) {
                if (($_FILES['profile_photo']['size'] ?? 0) > (3 * 1024 * 1024)) { $errors[] = 'Profile photo exceeds 3MB.'; }
                $pm = function_exists('mime_content_type') ? @mime_content_type($_FILES['profile_photo']['tmp_name']) : ($_FILES['profile_photo']['type'] ?? '');
                if ($pm && !in_array($pm, $allowedImg, true)) { $errors[] = 'Invalid profile photo type.'; }
            }
            $bizProvided = isset($_FILES['business_card']) && (($_FILES['business_card']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK);
            if ($effBizCardEnabled && $effBizCardRequired && !$bizProvided) { $errors[] = 'Business card is required.'; }
            if ($bizProvided) {
                if (($_FILES['business_card']['size'] ?? 0) > (3 * 1024 * 1024)) { $errors[] = 'Business card image exceeds 3MB.'; }
                $bm = function_exists('mime_content_type') ? @mime_content_type($_FILES['business_card']['tmp_name']) : ($_FILES['business_card']['type'] ?? '');
                if ($bm && !in_array($bm, $allowedImg, true)) { $errors[] = 'Invalid business card image type.'; }
            }
            // Address & business info
            if ($country === '') { $errors[] = 'Country is required.'; }
            if ($city === '') { $errors[] = 'City is required.'; }
            if ($addressLine === '') { $errors[] = 'Address is required.'; }
            // IATA requirement per effective KYC toggles
            if ($effIataEnabled && $effIataReqOnYes && $iataRegistered === 1 && $iataCode === '') { $errors[] = 'IATA code is required when IATA Registered is Yes.'; }
            // Commission and credit must be >= 0
            if ($agentCommission !== null && $agentCommission < 0) { $errors[] = 'Commission cannot be negative.'; }
            if ($agentCommission !== null && $agentCommission > 100) { $errors[] = 'Commission cannot exceed 100%.'; }
            if ($agentCreditLimit !== null && $agentCreditLimit < 0) { $errors[] = 'Credit limit cannot be negative.'; }

            // Government ID rules per effective toggles
            if ($effGovEnabled) {
                $allowedIdTypes = ['passport','national_id','driving_license','other'];
                if ($effGovReqType && !in_array($kycIdType, $allowedIdTypes, true)) { $errors[] = 'Valid KYC ID type is required.'; }
                if ($effGovReqNumber && $kycIdNumber === '') { $errors[] = 'KYC ID number is required.'; }
            }

            // File validations for KYC docs per toggles
            $maxSize = 5 * 1024 * 1024; // 5MB
            $allowedMimes = ['image/jpeg','image/png','image/webp','application/pdf','image/jpg'];
            $frontProvided = isset($_FILES['kyc_doc_front']) && (($_FILES['kyc_doc_front']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK);
            $backProvided = isset($_FILES['kyc_doc_back']) && (($_FILES['kyc_doc_back']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK);
            if ($effGovEnabled && $effGovReqFront && !$frontProvided) {
                $errors[] = 'KYC front document is required.';
            }
            if ($frontProvided) {
                if (($_FILES['kyc_doc_front']['size'] ?? 0) > $maxSize) { $errors[] = 'KYC front document exceeds 5MB.'; }
                $mime = function_exists('mime_content_type') ? @mime_content_type($_FILES['kyc_doc_front']['tmp_name']) : ($_FILES['kyc_doc_front']['type'] ?? '');
                if ($mime && !in_array($mime, $allowedMimes, true)) { $errors[] = 'Invalid front document type.'; }
            }
            if ($effGovEnabled && $effGovBackEnabled && $effGovReqBack && !$backProvided) {
                $errors[] = 'KYC back document is required.';
            }
            if ($backProvided) {
                if (($_FILES['kyc_doc_back']['size'] ?? 0) > $maxSize) { $errors[] = 'KYC back document exceeds 5MB.'; }
                $mimeB = function_exists('mime_content_type') ? @mime_content_type($_FILES['kyc_doc_back']['tmp_name']) : ($_FILES['kyc_doc_back']['type'] ?? '');
                if ($mimeB && !in_array($mimeB, $allowedMimes, true)) { $errors[] = 'Invalid back document type.'; }
            }

            // Business-type specific documents
            $imgMimes = ['image/jpeg','image/png','image/webp','image/jpg'];
            // Selfie enforcement per toggles
            $selfieEnabled = (bool)($K['selfie']['enabled'] ?? true);
            $selfieRequired = (bool)($K['selfie']['required'] ?? true);
            $selfieProvided = isset($_FILES['doc_selfie']) && (($_FILES['doc_selfie']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK);
            if ($selfieEnabled && $selfieRequired && !$selfieProvided) {
                $errors[] = 'Selfie is required.';
            }
            if ($selfieProvided) {
                if (($_FILES['doc_selfie']['size'] ?? 0) > (3 * 1024 * 1024)) { $errors[] = 'Selfie exceeds 3MB.'; }
                $sm = function_exists('mime_content_type') ? @mime_content_type($_FILES['doc_selfie']['tmp_name']) : ($_FILES['doc_selfie']['type'] ?? '');
                if ($sm && !in_array($sm, $imgMimes, true)) { $errors[] = 'Invalid selfie image type.'; }
            }

            // Address proof for freelancer per toggles
            $addrEnabled = (bool)($K['address_proof_freelancer']['enabled'] ?? true);
            $addrRequired = (bool)($K['address_proof_freelancer']['required'] ?? true);
            $addrProvided = isset($_FILES['doc_address_proof']) && (($_FILES['doc_address_proof']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK);
            if ($businessType === 'freelancer' && $addrEnabled && $addrRequired && !$addrProvided) {
                $errors[] = 'Address proof is required for Freelancer.';
            }
            if ($addrProvided) {
                if (($_FILES['doc_address_proof']['size'] ?? 0) > $maxSize) { $errors[] = 'Address proof exceeds 5MB.'; }
                $apm = function_exists('mime_content_type') ? @mime_content_type($_FILES['doc_address_proof']['tmp_name']) : ($_FILES['doc_address_proof']['type'] ?? '');
                if ($apm && !in_array($apm, $allowedMimes, true)) { $errors[] = 'Invalid address proof type.'; }
            }

            // Business-type docs per toggles
            $spEn = (bool)($K['reg_certificate_sp']['enabled'] ?? true);
            $spReq = (bool)($K['reg_certificate_sp']['required'] ?? false);
            if ($businessType === 'sole_proprietor' && $spEn) {
                if ($spReq && (!isset($_FILES['doc_registration_certificate']) || ($_FILES['doc_registration_certificate']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK)) {
                    $errors[] = 'Registration certificate is required for Sole Proprietor.';
                }
            }
            $ppEn = (bool)($K['partnership_deed_llp']['enabled'] ?? true);
            $ppReq = (bool)($K['partnership_deed_llp']['required'] ?? false);
            if ($businessType === 'partnership' && $ppEn) {
                if ($ppReq && (!isset($_FILES['doc_partnership_deed']) || ($_FILES['doc_partnership_deed']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK)) {
                    $errors[] = 'Partnership deed is required for Partnership/LLP.';
                }
            }
            $coEn = (bool)($K['incorp_cert_company']['enabled'] ?? true);
            $coReq = (bool)($K['incorp_cert_company']['required'] ?? false);
            if ($businessType === 'company' && $coEn) {
                if ($coReq && (!isset($_FILES['doc_incorporation_certificate']) || ($_FILES['doc_incorporation_certificate']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK)) {
                    $errors[] = 'Incorporation certificate is required for Company.';
                }
            }

            // Agency name requirement per effective toggles
            if ($effAgencyEnabled && $effAgencyRequired && $agencyName === '') {
                $errors[] = 'Agency Name is required for non-freelancer.';
            }

            // GST field requirements per effective toggles
            if ($effGstNumEnabled && $effGstNumReq && $gstNumber === '') { $errors[] = 'GST Number is required.'; }
            if ($effGstComEnabled && $effGstComReq && $gstCompany === '') { $errors[] = 'GST Company is required.'; }
        }

        if ($errors) {
            $_SESSION['flash'] = implode(' ', $errors);
            // Redirect back to the relevant create form
            $back = '/users/create';
            if ($role === 'B2B Agent') { $back = '/admin/management/agents/create'; }
            elseif ($role === 'Customer') { $back = '/admin/management/customers/create'; }
            header('Location: ' . $back);
            return;
        }

        // Safety net: ensure required tables exist (for dev/staging where schema may be partial)
        if ($role === 'B2B Agent') {
            // Run DDL outside of an explicit transaction to avoid implicit commit issues
            $this->ensureAgentInfrastructure();
        }
        // Use transaction to ensure atomic creation of user, profile, wallet, and KYC
        $this->pdo->beginTransaction();
        try {
            $stmt = $this->pdo->prepare('INSERT INTO users (name, email, role, password, status) VALUES (:name, :email, :role, :password, :status)');
            $stmt->execute([
                'name' => $name,
                'email' => $email,
                'role' => $role,
                'password' => password_hash($password, PASSWORD_BCRYPT),
                'status' => $status,
            ]);
            $newUserId = (int)$this->pdo->lastInsertId();

            // Auto-setup for B2B Agent
            if ($role === 'B2B Agent' && $newUserId > 0) {
                // Ensure a wallet exists for the agent
                $w = $this->pdo->prepare('INSERT INTO wallets (user_id, balance) VALUES (:uid, 0.00)');
                $w->execute(['uid' => $newUserId]);

                // Prepare for optional profile photo save
                $profilePhotoRel = null;
                $basePath = dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'public' . DIRECTORY_SEPARATOR;
                // Profile photo directory
                if (isset($_FILES['profile_photo']) && ($_FILES['profile_photo']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK) {
                    $avatarDir = $basePath . 'assets' . DIRECTORY_SEPARATOR . 'uploads' . DIRECTORY_SEPARATOR . 'agents' . DIRECTORY_SEPARATOR;
                    if (!is_dir($avatarDir)) { @mkdir($avatarDir, 0775, true); }
                    $tsAvatar = time();
                    $avatarExt = pathinfo($_FILES['profile_photo']['name'] ?? 'img', PATHINFO_EXTENSION) ?: 'jpg';
                    $profilePhotoRel = 'assets/uploads/agents/agent_' . $newUserId . '_' . $tsAvatar . '.' . $avatarExt;
                    $profilePhotoAbs = $basePath . str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $profilePhotoRel);
                    if (!move_uploaded_file($_FILES['profile_photo']['tmp_name'], $profilePhotoAbs)) {
                        throw new \RuntimeException('Failed to save profile photo.');
                    }
                }

                // Create agent profile (extended)
                $ap = $this->pdo->prepare('INSERT INTO agent_profiles (user_id, company, phone, commission_percent, credit_limit, business_type, agency_name, profile_photo_path, title, country, state, city, pincode, address_line, gst_number, gst_company, iata_registered, iata_code) VALUES (:uid, :company, :phone, :commission, :credit, :business_type, :agency_name, :photo, :title, :country, :state, :city, :pincode, :address_line, :gst_number, :gst_company, :iata_registered, :iata_code)');
                $ap->execute([
                    'uid' => $newUserId,
                    'company' => $agentCompany !== '' ? $agentCompany : null,
                    'phone' => $agentPhone !== '' ? $agentPhone : null,
                    'commission' => $agentCommission !== null ? $agentCommission : 0,
                    'credit' => $agentCreditLimit !== null ? $agentCreditLimit : 0,
                    'business_type' => $businessType,
                    'agency_name' => $agencyName !== '' ? $agencyName : null,
                    'photo' => $profilePhotoRel,
                    'title' => $title !== '' ? $title : null,
                    'country' => $country !== '' ? $country : null,
                    'state' => $state !== '' ? $state : null,
                    'city' => $city !== '' ? $city : null,
                    'pincode' => $pincode !== '' ? $pincode : null,
                    'address_line' => $addressLine !== '' ? $addressLine : null,
                    'gst_number' => $gstNumber !== '' ? $gstNumber : null,
                    'gst_company' => $gstCompany !== '' ? $gstCompany : null,
                    'iata_registered' => $iataRegistered,
                    'iata_code' => $iataCode !== '' ? $iataCode : null,
                ]);

                // Handle KYC file encryption + storage (private)
                $frontRel = $this->encryptAndStoreUpload(
                    $_FILES['kyc_doc_front']['tmp_name'],
                    $_FILES['kyc_doc_front']['name'] ?? 'front',
                    'kyc',
                    'agent_' . $newUserId . '_front'
                );
                $backRel = null;
                if (isset($_FILES['kyc_doc_back']) && ($_FILES['kyc_doc_back']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK) {
                    $backRel = $this->encryptAndStoreUpload(
                        $_FILES['kyc_doc_back']['tmp_name'],
                        $_FILES['kyc_doc_back']['name'] ?? 'back',
                        'kyc',
                        'agent_' . $newUserId . '_back'
                    );
                }

                // Insert KYC row
                $kyc = $this->pdo->prepare('INSERT INTO agent_kyc (user_id, id_type, id_number, doc_front_path, doc_back_path, status) VALUES (:uid, :type, :number, :front, :back, :status)');
                $kyc->execute([
                    'uid' => $newUserId,
                    'type' => $kycIdType,
                    'number' => $kycIdNumber,
                    'front' => $frontRel,
                    'back' => $backRel,
                    'status' => 'pending',
                ]);

                // Handle business docs upload (encrypted, private)
                $insDoc = $this->pdo->prepare('INSERT INTO agent_business_docs (user_id, doc_type, file_path) VALUES (:uid, :type, :path)');
                $saveDoc = function(string $field, string $type) use ($newUserId, $insDoc) {
                    if (!isset($_FILES[$field]) || ($_FILES[$field]['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) { return; }
                    $rel = $this->encryptAndStoreUpload(
                        $_FILES[$field]['tmp_name'],
                        $_FILES[$field]['name'] ?? $type,
                        'business_docs',
                        'agent_' . $newUserId . '_' . $type
                    );
                    $insDoc->execute(['uid' => $newUserId, 'type' => $type, 'path' => $rel]);
                };
                $saveDoc('doc_selfie', 'selfie');
                // Also persist Business Card if provided (field name from view: 'business_card')
                if (isset($_FILES['business_card']) && (($_FILES['business_card']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK)) {
                    $rel = $this->encryptAndStoreUpload(
                        $_FILES['business_card']['tmp_name'],
                        $_FILES['business_card']['name'] ?? 'business_card',
                        'business_docs',
                        'agent_' . $newUserId . '_business_card'
                    );
                    $insDoc->execute(['uid' => $newUserId, 'type' => 'business_card', 'path' => $rel]);
                }
                $saveDoc('doc_registration_certificate', 'registration_certificate');
                $saveDoc('doc_partnership_deed', 'partnership_deed');
                $saveDoc('doc_incorporation_certificate', 'incorporation_certificate');
                $saveDoc('doc_gst_certificate', 'gst_certificate');
                $saveDoc('doc_address_proof', 'address_proof');
            }

            $this->pdo->commit();
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) {
                $this->pdo->rollBack();
            }
            $_SESSION['flash'] = 'Could not create user: ' . $e->getMessage();
            $back = ($role === 'B2B Agent') ? '/admin/management/agents/create' : (($role === 'Customer') ? '/admin/management/customers/create' : '/users/create');
            header('Location: ' . $back);
            return;
        }
        unset($_SESSION['old']);
        $_SESSION['flash'] = 'User created successfully.';
        header('Location: /users');
    }

    public function edit(): void
    {
        Auth::checkRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0);
        $stmt = $this->pdo->prepare('SELECT id, name, email, role, status FROM users WHERE id = :id');
        $stmt->execute(['id' => $id]);
        $user = $stmt->fetch();
        $this->view('users/edit', ['title' => 'Edit User', 'user' => $user]);
    }

    public function update(): void
    {
        Auth::checkRole(['Admin']);
        if (!Security::verifyCsrf($_POST['csrf'] ?? '')) die('CSRF');
        $id = (int)($_POST['id'] ?? 0);
        $stmt = $this->pdo->prepare('UPDATE users SET name=:name, email=:email, role=:role, status=:status WHERE id=:id');
        $stmt->execute([
            'id' => $id,
            'name' => $_POST['name'] ?? '',
            'email' => $_POST['email'] ?? '',
            'role' => $_POST['role'] ?? 'Employee',
            'status' => $_POST['status'] ?? 'Active',
        ]);
        header('Location: /users');
    }

    public function delete(): void
    {
        Auth::checkRole(['Admin']);
        if (!Security::verifyCsrf($_POST['csrf'] ?? '')) die('CSRF');
        $id = (int)($_POST['id'] ?? 0);
        $stmt = $this->pdo->prepare('DELETE FROM users WHERE id=:id');
        $stmt->execute(['id' => $id]);
        header('Location: /users');
    }
}
