<?php
namespace App\Controllers;

use App\Core\Controller;

class TaxiController extends Controller
{
    // GET /taxi/verify?code=...
    public function verify(): void
    {
        // Prefer opaque signed token to prevent easy enumeration
        $token = trim((string)($_GET['t'] ?? ''));
        $code = trim((string)($_GET['code'] ?? ''));
        $code = preg_replace('/[^A-Za-z0-9\-]/', '', $code);

        // If token provided, decode and verify signature
        if ($token !== '') {
            $decodedCode = $this->decodeVerifyToken($token);
            if ($decodedCode !== null) { $code = $decodedCode; }
        }
        $row = null;
        if ($code !== '') {
            try {
                $st = $this->pdo->prepare('SELECT b.id, b.booking_code, b.status, b.payment_status, b.pax, b.trip_date, b.from_text, b.to_text, b.customer_name, b.created_at, t.vehicle_type, t.name AS ride_name
                                            FROM taxi_bookings b JOIN taxis t ON t.id = b.taxi_id
                                            WHERE b.booking_code = :code LIMIT 1');
                $st->execute([':code' => $code]);
                $row = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
            } catch (\Throwable $_) { $row = null; }
        }

        $this->view('agent/taxi_verify', [
            'title' => 'Taxi Booking Verification',
            'result' => $row,
            'code' => $code,
            'token' => $this->encodeVerifyToken($code),
        ]);
    }

    // Build a signed token from booking code: base64url(code).'.'.base64url(HMAC)
    private function encodeVerifyToken(?string $code): ?string
    {
        if (!$code) return null;
        $secret = (string)($_ENV['VERIFY_SECRET'] ?? ($_ENV['APP_KEY'] ?? 'dev-secret-change')); // set VERIFY_SECRET in .env for production
        $payload = rtrim(strtr(base64_encode($code), '+/', '-_'), '=');
        $sig = hash_hmac('sha256', $code, $secret, true);
        $sigB64 = rtrim(strtr(base64_encode($sig), '+/', '-_'), '=');
        return $payload . '.' . $sigB64;
    }

    // Validate token and return booking code if valid; else null
    private function decodeVerifyToken(string $token): ?string
    {
        $parts = explode('.', $token, 2);
        if (count($parts) !== 2) return null;
        [$p, $s] = $parts;
        $code = base64_decode(strtr($p, '-_', '+/'));
        if ($code === false || $code === '') return null;
        $secret = (string)($_ENV['VERIFY_SECRET'] ?? ($_ENV['APP_KEY'] ?? 'dev-secret-change'));
        $expected = hash_hmac('sha256', $code, $secret, true);
        $sig = base64_decode(strtr($s, '-_', '+/'));
        if ($sig === false) return null;
        if (!hash_equals($expected, $sig)) return null;
        return preg_replace('/[^A-Za-z0-9\-]/', '', $code);
    }
}
