<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;

class TaxiBookingsAdminController extends Controller
{
    // GET /admin/booking/taxi
    public function index(): void
    {
        // Pagination
        $perPage = (int)($_GET['per_page'] ?? 20);
        $perPage = max(10, min(100, $perPage)); // Clamp between 10 and 100
        $currentPage = max(1, (int)($_GET['page'] ?? 1));
        $offset = ($currentPage - 1) * $perPage;

        // Filters
        $status = isset($_GET['status']) ? (string)$_GET['status'] : '';
        $payStatus = isset($_GET['payment_status']) ? (string)$_GET['payment_status'] : '';
        $date = isset($_GET['date']) ? (string)$_GET['date'] : '';
        $vendorId = isset($_GET['vendor_id']) ? (int)$_GET['vendor_id'] : 0;

        // Build where conditions
        $where = [];
        $args = [];
        if ($status !== '') { $where[] = 'b.status = :st'; $args[':st'] = $status; }
        if ($payStatus !== '') { $where[] = 'b.payment_status = :ps'; $args[':ps'] = $payStatus; }
        if ($date !== '') { $where[] = 'b.trip_date = :dt'; $args[':dt'] = $date; }
        if ($vendorId > 0) { $where[] = 'b.vendor_id = :vid'; $args[':vid'] = $vendorId; }
        $whereSql = $where ? ('WHERE '.implode(' AND ', $where)) : '';

        // Get total count for pagination
        $totalBookings = 0;
        try {
            $countSql = "SELECT COUNT(*) FROM taxi_bookings b $whereSql";
            $st = $this->pdo->prepare($countSql);
            $st->execute($args);
            $totalBookings = (int)$st->fetchColumn();
        } catch (\Throwable $e) { 
            error_log("Error counting taxi bookings: " . $e->getMessage());
            $totalBookings = 0;
        }

        // Calculate total pages
        $totalPages = $totalBookings > 0 ? (int)ceil($totalBookings / $perPage) : 1;
        $currentPage = min($currentPage, $totalPages);
        $offset = ($currentPage - 1) * $perPage;

        // Get paginated results
        $rows = [];
        try {
            $sql = "SELECT b.id, b.booking_code, b.agent_id, b.taxi_id, t.name AS ride_name, 
                           b.trip_date, b.pax, b.amount_total, b.currency, b.status, 
                           b.payment_status, b.payment_method, b.gateway_name, b.payment_txn_id, 
                           b.created_at, b.customer_name, b.customer_phone, b.customer_email,
                           b.from_text, b.to_text, b.notes, b.vendor_cost, b.vendor_currency,
                           b.vendor_pay_status, b.vendor_paid_at, v.name AS vendor_name
                    FROM taxi_bookings b
                    LEFT JOIN taxis t ON t.id = b.taxi_id
                    LEFT JOIN vendors v ON v.id = b.vendor_id
                    $whereSql
                    ORDER BY b.id DESC 
                    LIMIT :limit OFFSET :offset";
            
            $st = $this->pdo->prepare($sql);
            foreach ($args as $k => $v) {
                $st->bindValue($k, $v);
            }
            $st->bindValue(':limit', $perPage, \PDO::PARAM_INT);
            $st->bindValue(':offset', $offset, \PDO::PARAM_INT);
            $st->execute();
            $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $e) { 
            error_log("Error fetching taxi bookings: " . $e->getMessage());
            $rows = []; 
        }

        // Get vendors for filter
        $vendors = [];
        try {
            $vendorStmt = $this->pdo->query("SELECT id, name FROM vendors ORDER BY name");
            $vendors = $vendorStmt->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $e) {
            error_log("Error fetching vendors: " . $e->getMessage());
            $vendors = [];
        }

        // Get stats for dashboard cards
        $stats = [
            'total_bookings' => 0,
            'total_revenue' => 0,
            'pending_bookings' => 0,
            'completed_bookings' => 0
        ];
        
        try {
            // Get total bookings count
            $stats['total_bookings'] = (int)$this->pdo->query("SELECT COUNT(*) FROM taxi_bookings")->fetchColumn();
            
            // Get total revenue
            $revenue = $this->pdo->query("SELECT COALESCE(SUM(amount_total), 0) FROM taxi_bookings WHERE payment_status = 'paid'")->fetchColumn();
            $stats['total_revenue'] = (float)$revenue;
            
            // Get pending bookings count
            $stats['pending_bookings'] = (int)$this->pdo->query("SELECT COUNT(*) FROM taxi_bookings WHERE status IN ('pending', 'confirmed')")->fetchColumn();
            
            // Get completed bookings count
            $stats['completed_bookings'] = (int)$this->pdo->query("SELECT COUNT(*) FROM taxi_bookings WHERE status = 'completed'")->fetchColumn();
            
        } catch (\Throwable $e) {
            error_log("Error fetching stats: " . $e->getMessage());
        }

        $csrf = Security::csrfToken();
        $this->view('admin/taxi_bookings_index', [
            'title' => 'Taxi Bookings',
            'rows' => $rows,
            'vendors' => $vendors,
            'filters' => [
                'status' => $status,
                'payment_status' => $payStatus,
                'date' => $date,
                'vendor_id' => $vendorId,
                'per_page' => $perPage,
                'page' => $currentPage
            ],
            'currentPage' => $currentPage,
            'perPage' => $perPage,
            'totalBookings' => $totalBookings,
            'totalPages' => $totalPages,
            'stats' => $stats,
            'csrf' => $csrf,
        ]);
    }

    // GET /admin/booking/taxi/view?id=
    public function view(): void
    {
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/booking/taxi'); return; }
        $row = null; $events = []; $payments = [];
        try {
            $st = $this->pdo->prepare('SELECT b.*, t.name AS ride_name FROM taxi_bookings b LEFT JOIN taxis t ON t.id=b.taxi_id WHERE b.id=:id');
            $st->execute([':id'=>$id]);
            $row = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $_) { $row = null; }
        if (!$row) { $this->redirect('/admin/booking/taxi'); return; }
        try {
            $ev = $this->pdo->prepare('SELECT * FROM taxi_booking_events WHERE booking_id = :id ORDER BY id DESC');
            $ev->execute([':id'=>$id]);
            $events = $ev->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $events = []; }
        try {
            $pv = $this->pdo->prepare('SELECT * FROM taxi_payments WHERE booking_id = :id ORDER BY id DESC');
            $pv->execute([':id'=>$id]);
            $payments = $pv->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $payments = []; }

        $csrf = Security::csrfToken();
        $pgCfg = @require __DIR__ . '/../../config/payment_gateways.php';
        $stripeEnabled = (bool)($pgCfg['stripe']['enabled'] ?? false);
        $this->view('admin/taxi_booking_view', [
            'title' => 'Taxi Booking',
            'booking' => $row,
            'events' => $events,
            'payments' => $payments,
            'csrf' => $csrf,
            'stripe_enabled' => $stripeEnabled,
        ]);
    }

    // POST /admin/booking/taxi/cancel
    public function cancel(): void
    {
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/booking/taxi'); return; }
        try {
            $this->pdo->prepare("UPDATE taxi_bookings SET status='cancelled', updated_at=NOW() WHERE id=:id")
                ->execute([':id'=>$id]);
            $ev = $this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,:type,:note,:data,NOW())');
            $ev->execute([
                ':bid'=>$id,
                ':uid'=>(int)($_SESSION['admin']['id'] ?? 0),
                ':type'=>'cancelled',
                ':note'=>'Booking cancelled by admin',
                ':data'=>json_encode(['ip'=>($_SERVER['REMOTE_ADDR'] ?? '')]),
            ]);
        } catch (\Throwable $_) { /* ignore */ }
        $this->redirect('/admin/booking/taxi/view?id=' . $id);
    }

    // POST /admin/booking/taxi/refund
    public function refund(): void
    {
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/booking/taxi'); return; }
        $row = null;
        try {
            $st = $this->pdo->prepare('SELECT * FROM taxi_bookings WHERE id=:id');
            $st->execute([':id'=>$id]);
            $row = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $_) { $row = null; }
        if (!$row) { $this->redirect('/admin/booking/taxi'); return; }
        $amount = (float)($row['amount_total'] ?? 0);
        $method = (string)($row['payment_method'] ?? '');
        if ($amount <= 0 || $method !== 'wallet') { $this->redirect('/admin/booking/taxi/view?id=' . $id . '&err=refund_method'); return; }
        try {
            $agentId = (int)($row['agent_id'] ?? 0);
            $ws = new \App\Services\WalletService($this->pdo);
            $walletId = $ws->getOrCreateWallet($agentId);
            $ledgerId = $ws->ledgerEntry($walletId, 'credit', $amount, 'taxi_refund', 'pending', ['booking_id'=>$id]);
            $ok = $ws->approveLedger($ledgerId);
            if ($ok) {
                $this->pdo->prepare("UPDATE taxi_bookings SET status='refunded', payment_status='refunded', updated_at=NOW() WHERE id=:id")
                    ->execute([':id'=>$id]);
                $p = $this->pdo->prepare('INSERT INTO taxi_payments (booking_id, method, gateway_name, amount, currency, status, txn_id, created_at, updated_at) VALUES (:bid,:m,:g,:amt,:cur,\'captured\',:tx,NOW(),NOW())');
                $p->execute([':bid'=>$id, ':m'=>'wallet_refund', ':g'=>'wallet', ':amt'=>$amount, ':cur'=> (string)($row['currency'] ?? 'THB'), ':tx'=>(string)$ledgerId]);
                $ev = $this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,:type,:note,:data,NOW())');
                $ev->execute([':bid'=>$id, ':uid'=>(int)($_SESSION['admin']['id'] ?? 0), ':type'=>'refunded', ':note'=>'Wallet refund executed', ':data'=>json_encode(['ledger_id'=>$ledgerId,'ip'=>($_SERVER['REMOTE_ADDR'] ?? '')])]);
            }
        } catch (\Throwable $_) { /* ignore */ }
        $this->redirect('/admin/booking/taxi/view?id=' . $id);
    }

    // POST /admin/booking/taxi/refund-gateway
    public function refundGateway(): void
    {
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/booking/taxi'); return; }
        // Load booking
        $row = null;
        try {
            $st = $this->pdo->prepare('SELECT * FROM taxi_bookings WHERE id=:id');
            $st->execute([':id'=>$id]);
            $row = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $_) { $row = null; }
        if (!$row) { $this->redirect('/admin/booking/taxi'); return; }
        if ((string)($row['gateway_name'] ?? '') !== 'stripe' || (string)($row['payment_status'] ?? '') !== 'paid') {
            $this->redirect('/admin/booking/taxi/view?id=' . $id . '&err=refund_gateway'); return;
        }
        // Load Stripe config
        $cfg = @require __DIR__ . '/../../config/payment_gateways.php';
        $secret = (string)($cfg['stripe']['secret_key'] ?? '');
        if ($secret === '') { $this->redirect('/admin/booking/taxi/view?id=' . $id . '&err=stripe_cfg'); return; }
        $txn = (string)($row['payment_txn_id'] ?? '');
        if ($txn === '') { $this->redirect('/admin/booking/taxi/view?id=' . $id . '&err=txn_missing'); return; }
        try {
            $stripe = new \Stripe\StripeClient($secret);
            $refund = $stripe->refunds->create([ 'payment_intent' => $txn ]);
            // Update booking and log
            $this->pdo->prepare("UPDATE taxi_bookings SET status='refunded', payment_status='refunded', updated_at=NOW() WHERE id=:id")
                ->execute([':id'=>$id]);
            $p = $this->pdo->prepare('INSERT INTO taxi_payments (booking_id, method, gateway_name, amount, currency, status, provider_ref, raw_event, created_at, updated_at) VALUES (:bid, "gateway_refund", "stripe", 0, :cur, "captured", :ref, :raw, NOW(), NOW())');
            $p->execute([':bid'=>$id, ':cur'=>(string)($row['currency'] ?? 'THB'), ':ref'=>(string)($refund->id ?? ''), ':raw'=>json_encode($refund)]);
            $ev = $this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,:type,:note,:data,NOW())');
            $ev->execute([':bid'=>$id, ':uid'=>(int)($_SESSION['admin']['id'] ?? 0), ':type'=>'refunded', ':note'=>'Stripe refund executed', ':data'=>json_encode(['refund_id'=>$refund->id ?? null,'ip'=>($_SERVER['REMOTE_ADDR'] ?? '')])]);
        } catch (\Throwable $_) { /* ignore */ }
        $this->redirect('/admin/booking/taxi/view?id=' . $id);
    }
}
