<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class SettingsController extends Controller
{
    public function kyc(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = isset($_SESSION['settings_kyc_unlocked']) && $_SESSION['settings_kyc_unlocked'] === true;
        $allowManual = (env('ALLOW_MANUAL_UPLOADS', 'true') !== 'false');
        // Load existing KYC flags from env (with defaults)
        $flags = [
            'KYC_ENABLE_PROFILE_PHOTO' => env('KYC_ENABLE_PROFILE_PHOTO', 'true'),
            'KYC_REQUIRE_PROFILE_PHOTO' => env('KYC_REQUIRE_PROFILE_PHOTO', 'true'),
            'KYC_ENABLE_BUSINESS_CARD' => env('KYC_ENABLE_BUSINESS_CARD', 'true'),
            'KYC_REQUIRE_BUSINESS_CARD' => env('KYC_REQUIRE_BUSINESS_CARD', 'true'),
            'KYC_ENABLE_SELFIE' => env('KYC_ENABLE_SELFIE', 'true'),
            'KYC_REQUIRE_SELFIE' => env('KYC_REQUIRE_SELFIE', 'true'),
            'KYC_ENABLE_REG_CERT_SP' => env('KYC_ENABLE_REG_CERT_SP', 'true'),
            'KYC_REQUIRE_REG_CERT_SP' => env('KYC_REQUIRE_REG_CERT_SP', 'false'),
            'KYC_ENABLE_PARTNERSHIP_DEED_LLP' => env('KYC_ENABLE_PARTNERSHIP_DEED_LLP', 'true'),
            'KYC_REQUIRE_PARTNERSHIP_DEED_LLP' => env('KYC_REQUIRE_PARTNERSHIP_DEED_LLP', 'false'),
            'KYC_ENABLE_INCORP_CERT_COMPANY' => env('KYC_ENABLE_INCORP_CERT_COMPANY', 'true'),
            'KYC_REQUIRE_INCORP_CERT_COMPANY' => env('KYC_REQUIRE_INCORP_CERT_COMPANY', 'false'),
            'KYC_ENABLE_ADDRESS_PROOF_FREELANCER' => env('KYC_ENABLE_ADDRESS_PROOF_FREELANCER', 'true'),
            'KYC_REQUIRE_ADDRESS_PROOF_FREELANCER' => env('KYC_REQUIRE_ADDRESS_PROOF_FREELANCER', 'true'),

            // Gov ID
            'KYC_ENABLE_GOV_ID_SECTION' => env('KYC_ENABLE_GOV_ID_SECTION', 'true'),
            'KYC_REQUIRE_GOV_ID_TYPE' => env('KYC_REQUIRE_GOV_ID_TYPE', 'true'),
            'KYC_REQUIRE_GOV_ID_NUMBER' => env('KYC_REQUIRE_GOV_ID_NUMBER', 'true'),
            'KYC_REQUIRE_GOV_ID_FRONT' => env('KYC_REQUIRE_GOV_ID_FRONT', 'true'),
            'KYC_ENABLE_GOV_ID_BACK' => env('KYC_ENABLE_GOV_ID_BACK', 'true'),
            'KYC_REQUIRE_GOV_ID_BACK' => env('KYC_REQUIRE_GOV_ID_BACK', 'false'),

            // Registration
            'KYC_ENABLE_IATA_BLOCK' => env('KYC_ENABLE_IATA_BLOCK', 'true'),
            'KYC_REQUIRE_IATA_CODE_ON_YES' => env('KYC_REQUIRE_IATA_CODE_ON_YES', 'true'),
            'KYC_ENABLE_GST_NUMBER' => env('KYC_ENABLE_GST_NUMBER', 'true'),
            'KYC_REQUIRE_GST_NUMBER' => env('KYC_REQUIRE_GST_NUMBER', 'false'),
            'KYC_ENABLE_GST_COMPANY' => env('KYC_ENABLE_GST_COMPANY', 'true'),
            'KYC_REQUIRE_GST_COMPANY' => env('KYC_REQUIRE_GST_COMPANY', 'false'),

            // Business info
            'KYC_ENABLE_AGENCY_NAME' => env('KYC_ENABLE_AGENCY_NAME', 'true'),
            'KYC_REQUIRE_AGENCY_NAME_NON_FREELANCER' => env('KYC_REQUIRE_AGENCY_NAME_NON_FREELANCER', 'false'),

            // Financials
            'KYC_ENABLE_COMMISSION' => env('KYC_ENABLE_COMMISSION', 'true'),
            'KYC_DEFAULT_COMMISSION' => env('KYC_DEFAULT_COMMISSION', '0'),
            'KYC_ENABLE_CREDIT_LIMIT' => env('KYC_ENABLE_CREDIT_LIMIT', 'true'),
            'KYC_DEFAULT_CREDIT_LIMIT' => env('KYC_DEFAULT_CREDIT_LIMIT', '0'),
        ];
        // Per-Business-Type overrides (default to 'false')
        $btKeys = [
            // Freelancer
            'BT_FREELANCER_OVERRIDE_GOVID','BT_FREELANCER_REQUIRE_ID_TYPE','BT_FREELANCER_REQUIRE_ID_NUMBER','BT_FREELANCER_ENABLE_ID_FRONT','BT_FREELANCER_REQUIRE_ID_FRONT','BT_FREELANCER_ENABLE_ID_BACK','BT_FREELANCER_REQUIRE_ID_BACK',
            'BT_FREELANCER_OVERRIDE_REG','BT_FREELANCER_IATA_ENABLE','BT_FREELANCER_IATA_REQUIRE_CODE','BT_FREELANCER_GST_NUMBER_ENABLE','BT_FREELANCER_GST_NUMBER_REQUIRE','BT_FREELANCER_GST_COMPANY_ENABLE','BT_FREELANCER_GST_COMPANY_REQUIRE',
            'BT_FREELANCER_OVERRIDE_BIZINFO','BT_FREELANCER_AGENCY_NAME_ENABLE','BT_FREELANCER_AGENCY_NAME_REQUIRE',
            'BT_FREELANCER_OVERRIDE_MEDIA','BT_FREELANCER_PROFILE_PHOTO_ENABLE','BT_FREELANCER_PROFILE_PHOTO_REQUIRE','BT_FREELANCER_BUSINESS_CARD_ENABLE','BT_FREELANCER_BUSINESS_CARD_REQUIRE',
            // Docs
            'BT_FREELANCER_OVERRIDE_DOCS','BT_FREELANCER_SELFIE_ENABLE','BT_FREELANCER_SELFIE_REQUIRE','BT_FREELANCER_ADDRESS_PROOF_ENABLE','BT_FREELANCER_ADDRESS_PROOF_REQUIRE',

            // Sole Proprietor
            'BT_SP_OVERRIDE_GOVID','BT_SP_REQUIRE_ID_TYPE','BT_SP_REQUIRE_ID_NUMBER','BT_SP_ENABLE_ID_FRONT','BT_SP_REQUIRE_ID_FRONT','BT_SP_ENABLE_ID_BACK','BT_SP_REQUIRE_ID_BACK',
            'BT_SP_OVERRIDE_REG','BT_SP_IATA_ENABLE','BT_SP_IATA_REQUIRE_CODE','BT_SP_GST_NUMBER_ENABLE','BT_SP_GST_NUMBER_REQUIRE','BT_SP_GST_COMPANY_ENABLE','BT_SP_GST_COMPANY_REQUIRE',
            'BT_SP_OVERRIDE_BIZINFO','BT_SP_AGENCY_NAME_ENABLE','BT_SP_AGENCY_NAME_REQUIRE',
            'BT_SP_OVERRIDE_MEDIA','BT_SP_PROFILE_PHOTO_ENABLE','BT_SP_PROFILE_PHOTO_REQUIRE','BT_SP_BUSINESS_CARD_ENABLE','BT_SP_BUSINESS_CARD_REQUIRE',
            // Docs
            'BT_SP_OVERRIDE_DOCS','BT_SP_SELFIE_ENABLE','BT_SP_SELFIE_REQUIRE','BT_SP_ADDRESS_PROOF_ENABLE','BT_SP_ADDRESS_PROOF_REQUIRE',

            // Partnership / LLP
            'BT_LLP_OVERRIDE_GOVID','BT_LLP_REQUIRE_ID_TYPE','BT_LLP_REQUIRE_ID_NUMBER','BT_LLP_ENABLE_ID_FRONT','BT_LLP_REQUIRE_ID_FRONT','BT_LLP_ENABLE_ID_BACK','BT_LLP_REQUIRE_ID_BACK',
            'BT_LLP_OVERRIDE_REG','BT_LLP_IATA_ENABLE','BT_LLP_IATA_REQUIRE_CODE','BT_LLP_GST_NUMBER_ENABLE','BT_LLP_GST_NUMBER_REQUIRE','BT_LLP_GST_COMPANY_ENABLE','BT_LLP_GST_COMPANY_REQUIRE',
            'BT_LLP_OVERRIDE_BIZINFO','BT_LLP_AGENCY_NAME_ENABLE','BT_LLP_AGENCY_NAME_REQUIRE',
            'BT_LLP_OVERRIDE_MEDIA','BT_LLP_PROFILE_PHOTO_ENABLE','BT_LLP_PROFILE_PHOTO_REQUIRE','BT_LLP_BUSINESS_CARD_ENABLE','BT_LLP_BUSINESS_CARD_REQUIRE',
            // Docs
            'BT_LLP_OVERRIDE_DOCS','BT_LLP_SELFIE_ENABLE','BT_LLP_SELFIE_REQUIRE','BT_LLP_ADDRESS_PROOF_ENABLE','BT_LLP_ADDRESS_PROOF_REQUIRE',

            // Company
            'BT_CO_OVERRIDE_GOVID','BT_CO_REQUIRE_ID_TYPE','BT_CO_REQUIRE_ID_NUMBER','BT_CO_ENABLE_ID_FRONT','BT_CO_REQUIRE_ID_FRONT','BT_CO_ENABLE_ID_BACK','BT_CO_REQUIRE_ID_BACK',
            'BT_CO_OVERRIDE_REG','BT_CO_IATA_ENABLE','BT_CO_IATA_REQUIRE_CODE','BT_CO_GST_NUMBER_ENABLE','BT_CO_GST_NUMBER_REQUIRE','BT_CO_GST_COMPANY_ENABLE','BT_CO_GST_COMPANY_REQUIRE',
            'BT_CO_OVERRIDE_BIZINFO','BT_CO_AGENCY_NAME_ENABLE','BT_CO_AGENCY_NAME_REQUIRE',
            'BT_CO_OVERRIDE_MEDIA','BT_CO_PROFILE_PHOTO_ENABLE','BT_CO_PROFILE_PHOTO_REQUIRE','BT_CO_BUSINESS_CARD_ENABLE','BT_CO_BUSINESS_CARD_REQUIRE',
            // Docs
            'BT_CO_OVERRIDE_DOCS','BT_CO_SELFIE_ENABLE','BT_CO_SELFIE_REQUIRE','BT_CO_ADDRESS_PROOF_ENABLE','BT_CO_ADDRESS_PROOF_REQUIRE',
        ];
        foreach ($btKeys as $bk) {
            $flags[$bk] = env($bk, 'false');
        }
        $this->view('admin/settings_kyc', compact('csrf', 'allowManual', 'flags', 'unlocked'));
    }

    /**
     * Unlock KYC settings using master password
     */
    public function kycUnlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $_SESSION['settings_kyc_unlocked'] = true;
        $this->redirect('/admin/settings/kyc');
    }

    public function kycUpdate(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['settings_kyc_unlocked'])) {
            $_SESSION['errors'][] = 'Please unlock this page with the update password before saving.';
            $this->redirect('/admin/settings/kyc');
            return;
        }
        // Optional: require master password to change critical settings
        // Security::requireMasterPassword();

        $allowManual = isset($_POST['allow_manual_uploads']) && $_POST['allow_manual_uploads'] === '1';
        $this->writeEnvValue('ALLOW_MANUAL_UPLOADS', $allowManual ? 'true' : 'false');

        // List of KYC keys to persist
        $keys = [
            'KYC_ENABLE_PROFILE_PHOTO',
            'KYC_REQUIRE_PROFILE_PHOTO',
            'KYC_ENABLE_BUSINESS_CARD',
            'KYC_REQUIRE_BUSINESS_CARD',
            'KYC_ENABLE_SELFIE',
            'KYC_REQUIRE_SELFIE',
            'KYC_ENABLE_REG_CERT_SP',
            'KYC_REQUIRE_REG_CERT_SP',
            'KYC_ENABLE_PARTNERSHIP_DEED_LLP',
            'KYC_REQUIRE_PARTNERSHIP_DEED_LLP',
            'KYC_ENABLE_INCORP_CERT_COMPANY',
            'KYC_REQUIRE_INCORP_CERT_COMPANY',
            'KYC_ENABLE_ADDRESS_PROOF_FREELANCER',
            'KYC_REQUIRE_ADDRESS_PROOF_FREELANCER',

            // Gov ID
            'KYC_ENABLE_GOV_ID_SECTION',
            'KYC_REQUIRE_GOV_ID_TYPE',
            'KYC_REQUIRE_GOV_ID_NUMBER',
            'KYC_REQUIRE_GOV_ID_FRONT',
            'KYC_ENABLE_GOV_ID_BACK',
            'KYC_REQUIRE_GOV_ID_BACK',

            // Registration
            'KYC_ENABLE_IATA_BLOCK',
            'KYC_REQUIRE_IATA_CODE_ON_YES',
            'KYC_ENABLE_GST_NUMBER',
            'KYC_REQUIRE_GST_NUMBER',
            'KYC_ENABLE_GST_COMPANY',
            'KYC_REQUIRE_GST_COMPANY',

            // Business info
            'KYC_ENABLE_AGENCY_NAME',
            'KYC_REQUIRE_AGENCY_NAME_NON_FREELANCER',

            // Financials (bool switches)
            'KYC_ENABLE_COMMISSION',
            'KYC_ENABLE_CREDIT_LIMIT',
        ];
        // Persist per-business-type overrides
        $keys = array_merge($keys, [
            // Freelancer
            'BT_FREELANCER_OVERRIDE_GOVID','BT_FREELANCER_REQUIRE_ID_TYPE','BT_FREELANCER_REQUIRE_ID_NUMBER','BT_FREELANCER_ENABLE_ID_FRONT','BT_FREELANCER_REQUIRE_ID_FRONT','BT_FREELANCER_ENABLE_ID_BACK','BT_FREELANCER_REQUIRE_ID_BACK',
            'BT_FREELANCER_OVERRIDE_REG','BT_FREELANCER_IATA_ENABLE','BT_FREELANCER_IATA_REQUIRE_CODE','BT_FREELANCER_GST_NUMBER_ENABLE','BT_FREELANCER_GST_NUMBER_REQUIRE','BT_FREELANCER_GST_COMPANY_ENABLE','BT_FREELANCER_GST_COMPANY_REQUIRE',
            'BT_FREELANCER_OVERRIDE_BIZINFO','BT_FREELANCER_AGENCY_NAME_ENABLE','BT_FREELANCER_AGENCY_NAME_REQUIRE',
            'BT_FREELANCER_OVERRIDE_MEDIA','BT_FREELANCER_PROFILE_PHOTO_ENABLE','BT_FREELANCER_PROFILE_PHOTO_REQUIRE','BT_FREELANCER_BUSINESS_CARD_ENABLE','BT_FREELANCER_BUSINESS_CARD_REQUIRE',
            'BT_FREELANCER_OVERRIDE_DOCS','BT_FREELANCER_SELFIE_ENABLE','BT_FREELANCER_SELFIE_REQUIRE','BT_FREELANCER_ADDRESS_PROOF_ENABLE','BT_FREELANCER_ADDRESS_PROOF_REQUIRE',

            // Sole Proprietor
            'BT_SP_OVERRIDE_GOVID','BT_SP_REQUIRE_ID_TYPE','BT_SP_REQUIRE_ID_NUMBER','BT_SP_ENABLE_ID_FRONT','BT_SP_REQUIRE_ID_FRONT','BT_SP_ENABLE_ID_BACK','BT_SP_REQUIRE_ID_BACK',
            'BT_SP_OVERRIDE_REG','BT_SP_IATA_ENABLE','BT_SP_IATA_REQUIRE_CODE','BT_SP_GST_NUMBER_ENABLE','BT_SP_GST_NUMBER_REQUIRE','BT_SP_GST_COMPANY_ENABLE','BT_SP_GST_COMPANY_REQUIRE',
            'BT_SP_OVERRIDE_BIZINFO','BT_SP_AGENCY_NAME_ENABLE','BT_SP_AGENCY_NAME_REQUIRE',
            'BT_SP_OVERRIDE_MEDIA','BT_SP_PROFILE_PHOTO_ENABLE','BT_SP_PROFILE_PHOTO_REQUIRE','BT_SP_BUSINESS_CARD_ENABLE','BT_SP_BUSINESS_CARD_REQUIRE',
            'BT_SP_OVERRIDE_DOCS','BT_SP_SELFIE_ENABLE','BT_SP_SELFIE_REQUIRE','BT_SP_ADDRESS_PROOF_ENABLE','BT_SP_ADDRESS_PROOF_REQUIRE',

            // Partnership / LLP
            'BT_LLP_OVERRIDE_GOVID','BT_LLP_REQUIRE_ID_TYPE','BT_LLP_REQUIRE_ID_NUMBER','BT_LLP_ENABLE_ID_FRONT','BT_LLP_REQUIRE_ID_FRONT','BT_LLP_ENABLE_ID_BACK','BT_LLP_REQUIRE_ID_BACK',
            'BT_LLP_OVERRIDE_REG','BT_LLP_IATA_ENABLE','BT_LLP_IATA_REQUIRE_CODE','BT_LLP_GST_NUMBER_ENABLE','BT_LLP_GST_NUMBER_REQUIRE','BT_LLP_GST_COMPANY_ENABLE','BT_LLP_GST_COMPANY_REQUIRE',
            'BT_LLP_OVERRIDE_BIZINFO','BT_LLP_AGENCY_NAME_ENABLE','BT_LLP_AGENCY_NAME_REQUIRE',
            'BT_LLP_OVERRIDE_MEDIA','BT_LLP_PROFILE_PHOTO_ENABLE','BT_LLP_PROFILE_PHOTO_REQUIRE','BT_LLP_BUSINESS_CARD_ENABLE','BT_LLP_BUSINESS_CARD_REQUIRE',
            'BT_LLP_OVERRIDE_DOCS','BT_LLP_SELFIE_ENABLE','BT_LLP_SELFIE_REQUIRE','BT_LLP_ADDRESS_PROOF_ENABLE','BT_LLP_ADDRESS_PROOF_REQUIRE',

            // Company
            'BT_CO_OVERRIDE_GOVID','BT_CO_REQUIRE_ID_TYPE','BT_CO_REQUIRE_ID_NUMBER','BT_CO_ENABLE_ID_FRONT','BT_CO_REQUIRE_ID_FRONT','BT_CO_ENABLE_ID_BACK','BT_CO_REQUIRE_ID_BACK',
            'BT_CO_OVERRIDE_REG','BT_CO_IATA_ENABLE','BT_CO_IATA_REQUIRE_CODE','BT_CO_GST_NUMBER_ENABLE','BT_CO_GST_NUMBER_REQUIRE','BT_CO_GST_COMPANY_ENABLE','BT_CO_GST_COMPANY_REQUIRE',
            'BT_CO_OVERRIDE_BIZINFO','BT_CO_AGENCY_NAME_ENABLE','BT_CO_AGENCY_NAME_REQUIRE',
            'BT_CO_OVERRIDE_MEDIA','BT_CO_PROFILE_PHOTO_ENABLE','BT_CO_PROFILE_PHOTO_REQUIRE','BT_CO_BUSINESS_CARD_ENABLE','BT_CO_BUSINESS_CARD_REQUIRE',
            'BT_CO_OVERRIDE_DOCS','BT_CO_SELFIE_ENABLE','BT_CO_SELFIE_REQUIRE','BT_CO_ADDRESS_PROOF_ENABLE','BT_CO_ADDRESS_PROOF_REQUIRE',
        ]);
        foreach ($keys as $k) {
            $val = isset($_POST[$k]) && $_POST[$k] === '1' ? 'true' : 'false';
            $this->writeEnvValue($k, $val);
        }

        // Numeric defaults
        if (isset($_POST['KYC_DEFAULT_COMMISSION'])) {
            $this->writeEnvValue('KYC_DEFAULT_COMMISSION', trim((string)$_POST['KYC_DEFAULT_COMMISSION']));
        }
        if (isset($_POST['KYC_DEFAULT_CREDIT_LIMIT'])) {
            $this->writeEnvValue('KYC_DEFAULT_CREDIT_LIMIT', trim((string)$_POST['KYC_DEFAULT_CREDIT_LIMIT']));
        }
        $_SESSION['flash'] = 'Settings updated.';
        $this->redirect('/admin/settings/kyc');
    }

    public function smtp(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = isset($_SESSION['settings_smtp_unlocked']) && $_SESSION['settings_smtp_unlocked'] === true;
        // Load current values from env for all profiles
        $profiles = [
            'default' => [
                'prefix' => 'MAIL',
                'label' => 'Default',
            ],
            'hotel' => [
                'prefix' => 'MAIL_HOTEL',
                'label' => 'Hotel',
            ],
            'activity' => [
                'prefix' => 'MAIL_ACTIVITY',
                'label' => 'Activity',
            ],
            'taxi' => [
                'prefix' => 'MAIL_TAXI',
                'label' => 'Taxi',
            ],
            'visa' => [
                'prefix' => 'MAIL_VISA',
                'label' => 'Visa',
            ],
        ];
        foreach ($profiles as $key => $meta) {
            $p = $meta['prefix'];
            $profiles[$key]['values'] = [
                'HOST' => env($p . '_HOST', ''),
                'PORT' => env($p . '_PORT', '587'),
                'USERNAME' => env($p . '_USERNAME', ''),
                // Do not prefill password
                'PASSWORD' => '',
                'ENCRYPTION' => env($p . '_ENCRYPTION', 'tls'),
                'FROM_ADDRESS' => env($p . '_FROM_ADDRESS', 'noreply@example.com'),
                'FROM_NAME' => env($p . '_FROM_NAME', 'B2B Travel'),
                // BCC copy list (comma-separated)
                'COPY_TO' => env($p . '_COPY_TO', env('MAIL_COPY_TO','')),
            ];
        }
        // Provider API settings (used by new mail system or external SDKs)
        $providers = [
            'SENDGRID_API_KEY' => env('SENDGRID_API_KEY', ''),
            'SES_REGION' => env('SES_REGION', ''),
            'SES_ACCESS_KEY_ID' => env('SES_ACCESS_KEY_ID', ''),
            // Do not prefill secret
            'SES_SECRET_ACCESS_KEY' => '',
            'ELASTICEMAIL_API_KEY' => env('ELASTICEMAIL_API_KEY', ''),
        ];
        $activeProvider = env('MAIL_ACTIVE_PROVIDER', 'smtp'); // smtp | sendgrid | ses | elasticemail
        $this->view('admin/settings_smtp', compact('csrf', 'profiles', 'providers', 'activeProvider', 'unlocked'));
    }

    /**
     * Unlock SMTP settings using master password
     */
    public function smtpUnlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $_SESSION['settings_smtp_unlocked'] = true;
        $this->redirect('/admin/settings/smtp');
    }

    /**
     * POST /admin/settings/smtp/test-elastic-direct
     * Sends a test email using Elastic Email REST API directly (cURL) and shows raw response.
     */
    public function smtpTestElasticDirect(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $to = trim((string)($_POST['to'] ?? ''));
        $fromEmail = trim((string)($_POST['from_email'] ?? ''));
        $fromName  = trim((string)($_POST['from_name'] ?? ''));
        if ($to === '') { $_SESSION['errors'] = ['Enter a recipient email']; $this->redirect('/admin/settings/smtp'); return; }

        $apiKey = env('ELASTICEMAIL_API_KEY', '');
        if ($apiKey === '') { $_SESSION['errors'] = ['ELASTICEMAIL_API_KEY is empty. Save it in Providers (API).']; $this->redirect('/admin/settings/smtp'); return; }

        $fromAddr = $fromEmail !== '' ? $fromEmail : env('MAIL_FROM_ADDRESS', 'noreply@example.com');
        $fromNm = $fromName !== '' ? $fromName : env('MAIL_FROM_NAME', 'B2B Travel');
        $subject = 'Elastic Email Direct API Test';
        $bodyHtml = '<p>This is a direct Elastic Email API test from B2B Travel.</p>';
        $bodyText = 'This is a direct Elastic Email API test from B2B Travel.';

        $post = [
            'apikey' => $apiKey,
            'from' => $fromAddr,
            'fromName' => $fromNm,
            'to' => $to,
            'subject' => $subject,
            'bodyHtml' => $bodyHtml,
            'bodyText' => $bodyText,
            'isTransactional' => 'true',
        ];

        $url = 'https://api.elasticemail.com/v2/email/send';
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($post),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HEADER => false,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 20,
        ]);
        $response = curl_exec($ch);
        $error = curl_error($ch);
        $http = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $raw = "HTTP $http\n";
        if ($error) { $raw .= "cURL error: $error\n"; }
        $raw .= (string)$response;
        $_SESSION['raw_mail_test'] = $raw;
        if ($http === 200) { $_SESSION['flash'] = 'Elastic Email direct API call attempted. See Raw Response below.'; }
        else { $_SESSION['errors'][] = 'Elastic Email direct API call returned HTTP ' . $http . '. See Raw Response below.'; }
        $this->redirect('/admin/settings/smtp');
    }

    public function smtpUpdate(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['settings_smtp_unlocked'])) {
            $_SESSION['errors'][] = 'Please unlock this page with the update password before saving.';
            $this->redirect('/admin/settings/smtp');
            return;
        }

        $profiles = ['MAIL','MAIL_HOTEL','MAIL_ACTIVITY','MAIL_TAXI','MAIL_VISA'];
        $fields = ['HOST','PORT','USERNAME','PASSWORD','ENCRYPTION','FROM_ADDRESS','FROM_NAME','COPY_TO'];
        foreach ($profiles as $prefix) {
            foreach ($fields as $f) {
                $envKey = $prefix . '_' . $f;
                if (!isset($_POST[$envKey])) continue;
                $val = trim((string)$_POST[$envKey]);
                if ($f === 'PASSWORD' && $val === '') continue; // keep existing password if left blank
                $this->writeEnvValue($envKey, $val);
            }
        }
        // Provider API settings
        if (isset($_POST['SENDGRID_API_KEY'])) {
            $sg = trim((string)$_POST['SENDGRID_API_KEY']);
            if ($sg !== '') { $this->writeEnvValue('SENDGRID_API_KEY', $sg); }
        }
        if (isset($_POST['ELASTICEMAIL_API_KEY'])) {
            $ee = trim((string)$_POST['ELASTICEMAIL_API_KEY']);
            if ($ee !== '') { $this->writeEnvValue('ELASTICEMAIL_API_KEY', $ee); }
        }
        if (isset($_POST['SES_REGION'])) {
            $this->writeEnvValue('SES_REGION', trim((string)$_POST['SES_REGION']));
        }
        if (isset($_POST['SES_ACCESS_KEY_ID'])) {
            $ak = trim((string)$_POST['SES_ACCESS_KEY_ID']);
            if ($ak !== '') { $this->writeEnvValue('SES_ACCESS_KEY_ID', $ak); }
        }
        if (isset($_POST['SES_SECRET_ACCESS_KEY'])) {
            $sk = trim((string)$_POST['SES_SECRET_ACCESS_KEY']);
            if ($sk !== '') { $this->writeEnvValue('SES_SECRET_ACCESS_KEY', $sk); }
        }
        // Active provider toggle
        if (isset($_POST['MAIL_ACTIVE_PROVIDER'])) {
            $ap = trim((string)$_POST['MAIL_ACTIVE_PROVIDER']);
            if (!in_array($ap, ['smtp','sendgrid','ses','elasticemail'], true)) { $ap = 'smtp'; }
            $this->writeEnvValue('MAIL_ACTIVE_PROVIDER', $ap);
        }
        $_SESSION['flash'] = 'SMTP settings saved.';
        $this->redirect('/admin/settings/smtp');
    }

    /**
     * POST /admin/settings/smtp/test-api
     * Sends a test email using the selected provider in Providers (API) tab.
     * For now, implements SendGrid API if selected; otherwise falls back to SMTP default profile.
     */
    public function smtpTestApi(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $to = trim((string)($_POST['to'] ?? ''));
        $fromEmail = trim((string)($_POST['from_email'] ?? ''));
        $fromName  = trim((string)($_POST['from_name'] ?? ''));
        if ($to === '') { $_SESSION['errors'] = ['Enter a recipient email']; $this->redirect('/admin/settings/smtp'); return; }

        $active = env('MAIL_ACTIVE_PROVIDER', 'smtp');
        try {
            if ($active === 'ses') {
                $fromAddr = $fromEmail !== '' ? $fromEmail : env('MAIL_FROM_ADDRESS', 'noreply@example.com');
                $fromNm = $fromName !== '' ? $fromName : env('MAIL_FROM_NAME', 'B2B Travel');
                $region = env('SES_REGION', 'ap-southeast-1');
                $key = env('SES_ACCESS_KEY_ID', '');
                $secret = env('SES_SECRET_ACCESS_KEY', '');
                try {
                    $client = new \Aws\Ses\SesClient([
                        'version' => 'latest',
                        'region' => $region,
                        'credentials' => ['key' => $key, 'secret' => $secret],
                    ]);
                } catch (\Throwable $_) {
                    // Fallback for older aws-sdk-php releases that don't alias 'latest'
                    $client = new \Aws\Ses\SesClient([
                        'version' => '2010-12-01',
                        'region' => $region,
                        'credentials' => ['key' => $key, 'secret' => $secret],
                    ]);
                }
                $result = $client->sendEmail([
                    'Destination' => ['ToAddresses' => [$to]],
                    'Message' => [
                        'Body' => ['Html' => ['Charset' => 'UTF-8', 'Data' => '<p>This is an Amazon SES API test from B2B Travel.</p>']],
                        'Subject' => ['Charset' => 'UTF-8', 'Data' => 'Amazon SES API Test'],
                    ],
                    'Source' => ($fromNm !== '' ? sprintf('%s <%s>', $fromNm, $fromAddr) : $fromAddr),
                ]);
                $_SESSION['flash'] = 'SES API test sent. MessageId: ' . (string)($result['MessageId'] ?? '');
                $this->redirect('/admin/settings/smtp'); return;
            }
            if ($active === 'sendgrid') {
                $fromAddr = $fromEmail !== '' ? $fromEmail : env('MAIL_FROM_ADDRESS', 'noreply@example.com');
                $fromNm = $fromName !== '' ? $fromName : env('MAIL_FROM_NAME', 'B2B Travel');
                $client = new \App\Core\Mail\SendGridClient(env('SENDGRID_API_KEY', ''));
                $res = $client->sendHtml($to, 'SendGrid API Test', '<p>This is a SendGrid API test from B2B Travel.</p>', ['address'=>$fromAddr,'name'=>$fromNm]);
                if (!empty($res['ok'])) { $_SESSION['flash'] = 'SendGrid API test sent.'; }
                else { $_SESSION['errors'] = ['SendGrid API test failed: ' . (string)($res['message'] ?? 'unknown')]; }
                $this->redirect('/admin/settings/smtp'); return;
            }
            if ($active === 'elasticemail') {
                $fromAddr = $fromEmail !== '' ? $fromEmail : env('MAIL_FROM_ADDRESS', 'noreply@example.com');
                $fromNm = $fromName !== '' ? $fromName : env('MAIL_FROM_NAME', 'B2B Travel');
                $ms = new \App\Core\Mail\MailService($this->pdo);
                $env = new \App\Core\Mail\Envelope($to, 'Elastic Email API Test', '<p>This is an Elastic Email API test from B2B Travel.</p>', '', $fromAddr, $fromNm);
                $res = $ms->send($env);
                if (!empty($res->ok)) { $_SESSION['flash'] = 'Elastic Email API test sent.'; }
                else { $_SESSION['errors'] = ['Elastic Email API test failed: ' . ($res->message ?? 'unknown')]; }
                $this->redirect('/admin/settings/smtp'); return;
            }
            // SMTP fallback using default profile
            $baseCfg = require __DIR__.'/../../config/mail.php';
            if ($fromEmail !== '' || $fromName !== '') {
                $baseCfg['from'] = [
                    'address' => $fromEmail !== '' ? $fromEmail : ($baseCfg['from']['address'] ?? 'noreply@example.com'),
                    'name' => $fromName !== '' ? $fromName : ($baseCfg['from']['name'] ?? 'B2B Travel'),
                ];
            }
            $mailer = new \App\Core\Mail\Mailer($baseCfg);
            $r = $mailer->sendHtml($to, 'SMTP Test', '<p>This is an SMTP test from B2B Travel.</p>');
            if (!empty($r['ok'])) { $_SESSION['flash'] = 'SMTP test sent.'; }
            else { $_SESSION['errors'] = ['SMTP test failed: ' . (string)($r['message'] ?? 'unknown')]; }
        } catch (\Throwable $e) {
            $_SESSION['errors'] = ['Test send error: ' . $e->getMessage()];
        }
        $this->redirect('/admin/settings/smtp');
    }

    private function writeEnvValue(string $key, string $value): void
    {
        $envPath = BASE_PATH . DIRECTORY_SEPARATOR . '.env';
        $lines = file_exists($envPath)
            ? file($envPath, FILE_IGNORE_NEW_LINES)
            : [];
        $found = false;
        foreach ($lines as $i => $line) {
            $trim = trim($line);
            if ($trim === '' || str_starts_with($trim, '#')) continue;
            [$k] = array_pad(explode('=', $line, 2), 2, '');
            if (trim($k) === $key) {
                $lines[$i] = $key . '=' . $value;
                $found = true;
                break;
            }
        }
        if (!$found) {
            $lines[] = $key . '=' . $value;
        }
        file_put_contents($envPath, implode(PHP_EOL, $lines) . PHP_EOL);
    }
}
