<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;
use App\Core\Mail\MailService;
use App\Core\Mail\Envelope;
use DateTimeImmutable;
use DateInterval;

class PasswordResetController extends Controller
{
    protected $mailService;
    
    public function __construct(\PDO $pdo)
    {
        parent::__construct($pdo);
        $this->mailService = new MailService($this->pdo);
    }

    // Show the forgot password form
    public function showForgotPasswordForm()
    {
        $this->view('agent/auth/forgot_password', [
            'title' => 'Forgot Password',
            'csrf' => Security::csrfToken(),
            'csrf_token' => Security::csrfToken(),
        ]);
    }

    // Process the forgot password form
    public function sendResetLink()
    {
        if (!Security::verifyCsrf((string)($_POST['csrf_token'] ?? ''))) {
            $_SESSION['forgot_password_error'] = 'Invalid CSRF token. Please try again.';
            $this->redirect('/b2b/agent/forgot-password');
            return;
        }

        $email = filter_var(trim((string)($_POST['email'] ?? '')), FILTER_VALIDATE_EMAIL);
        $email = $email ? strtolower($email) : null;
        
        if (!$email) {
            $_SESSION['forgot_password_error'] = 'Please enter a valid email address.';
            $this->redirect('/b2b/agent/forgot-password');
            return;
        }

        // --- Basic throttling (DB-backed table + minimal session fallback) ---
        $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        try {
            // Create throttle table if missing
            $this->pdo->exec(
                "CREATE TABLE IF NOT EXISTS password_reset_throttle (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    email VARCHAR(255) NOT NULL,
                    ip VARCHAR(45) NOT NULL,
                    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    INDEX idx_email_created (email, created_at),
                    INDEX idx_ip_created (ip, created_at)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci"
            );
            // Limits
            $stEmailCnt = $this->pdo->prepare('SELECT COUNT(*) FROM password_reset_throttle WHERE LOWER(email)=LOWER(:email) AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)');
            $stEmailCnt->execute([':email' => (string)$email]);
            $emailCount = (int)$stEmailCnt->fetchColumn();
            if ($emailCount >= 3) {
                $_SESSION['forgot_password_success'] = 'If your email is registered, you will receive a password reset link shortly.';
                $this->redirect('/b2b/agent/forgot-password');
                return;
            }
            $stIpCnt = $this->pdo->prepare('SELECT COUNT(*) FROM password_reset_throttle WHERE ip = :ip AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)');
            $stIpCnt->execute([':ip' => $ip]);
            $ipCount = (int)$stIpCnt->fetchColumn();
            if ($ipCount >= 10) {
                $_SESSION['forgot_password_success'] = 'If your email is registered, you will receive a password reset link shortly.';
                $this->redirect('/b2b/agent/forgot-password');
                return;
            }
        } catch (\Throwable $__) {
            // Fail open if DB unavailable; minimal session-based throttle fallback
            $now = time();
            $key = 'forgot_pw_ip_req';
            if (!isset($_SESSION[$key]) || !is_array($_SESSION[$key])) { $_SESSION[$key] = []; }
            $_SESSION[$key] = array_filter($_SESSION[$key], function($ts) use ($now) { return ($now - (int)$ts) < 3600; });
            if (count($_SESSION[$key]) >= 10) {
                $_SESSION['forgot_password_success'] = 'If your email is registered, you will receive a password reset link shortly.';
                $this->redirect('/b2b/agent/forgot-password');
                return;
            }
            $_SESSION[$key][] = $now;
        }

        // Find user by email
        $user = $this->pdo->prepare('SELECT id, name, email FROM users WHERE LOWER(email) = LOWER(:email) AND role = "B2B Agent" LIMIT 1');
        $user->execute([':email' => (string)$email]);
        $user = $user->fetch(\PDO::FETCH_ASSOC);

        if (!$user) {
            // For security, don't reveal if the email exists or not
            $_SESSION['forgot_password_success'] = 'If your email is registered, you will receive a password reset link shortly.';
            $this->redirect('/b2b/agent/forgot-password');
            return;
        }

        // Generate a secure token
        $token = bin2hex(random_bytes(32));
        $expiresAt = (new DateTimeImmutable())->add(new DateInterval('PT1H')); // 1 hour from now (for email text only)

        // Store the token in the database using DB time to avoid timezone drift
        $stmt = $this->pdo->prepare('INSERT INTO password_resets (email, token, created_at, expires_at) VALUES (:email, :token, NOW(), DATE_ADD(NOW(), INTERVAL 1 HOUR))');
        $stmt->execute([
            ':email' => (string)$email,
            ':token' => $token,
        ]);

        // Record throttle after token creation (as requested)
        try {
            $insThr = $this->pdo->prepare('INSERT INTO password_reset_throttle (email, ip, created_at) VALUES (:email, :ip, NOW())');
            $insThr->execute([':email' => (string)$email, ':ip' => $ip]);
        } catch (\Throwable $__) {
            // ignore
        }

        // Send the password reset email
        $resetLink = 'http://' . $_SERVER['HTTP_HOST'] . '/b2b/agent/reset-password?token=' . urlencode($token) . '&email=' . urlencode($email);
        $html = $this->renderViewToString('emails/agent/password_reset', [
            'name' => $user['name'],
            'resetLink' => $resetLink,
            'expiresAt' => $expiresAt,
        ]);
        $subject = 'Password Reset Request';
        $cfg = $this->mailService->loadSettings();
        // Prefer .env over DB to allow quick overrides
      //  $fromEmail = (env('MAIL_FROM_ADDRESS', '')) ?: ($cfg['from_email'] ?? '');
        $fromName = (env('MAIL_FROM_NAME', '')) ?: ($cfg['from_name'] ?? '');
        // Safety: never allow example.com as sender
        $fromEmail = 'admin@bookmythai.com';
        if ($fromName === '') {
            $fromName = 'BookMyThai';
        }
        // Fallback to known working sender if not configured
        if ($fromEmail === '') { $fromEmail = 'admin@bookmythai.com'; }
        if ($fromName === '') { $fromName = 'BookMyThai'; }
        error_log('[PasswordReset] using fromEmail='.$fromEmail.' fromName='.$fromName);
        $env = new Envelope($email, $subject, $html, '', $fromEmail, $fromName);
        $sendRes = $this->mailService->send($env);
        if (!$sendRes->ok) {
            error_log('[PasswordReset] send reset link failed: '.$sendRes->message.' status='.$sendRes->status);
        }

        $_SESSION['forgot_password_success'] = 'A password reset link has been sent to your email address. The link will expire in 1 hour.';
        $this->redirect('/b2b/agent/forgot-password');
    }

    // Show the reset password form
    public function showResetForm()
    {
        $token = $_GET['token'] ?? '';
        $email = strtolower(trim((string)($_GET['email'] ?? '')));

        if (empty($token) || empty($email)) {
            $_SESSION['reset_password_error'] = 'Invalid password reset link.';
            $this->redirect('/b2b/agent/forgot-password');
            return;
        }

        // Verify the token
        $reset = $this->pdo->prepare('SELECT * FROM password_resets WHERE LOWER(email) = LOWER(:email) AND token = :token AND used_at IS NULL AND expires_at > NOW() ORDER BY created_at DESC LIMIT 1');
        $reset->execute([':email' => (string)$email, ':token' => $token]);
        $reset = $reset->fetch(\PDO::FETCH_ASSOC);

        if (!$reset) {
            $_SESSION['forgot_password_error'] = 'Invalid or expired password reset link. Please request a new one.';
            $this->redirect('/b2b/agent/forgot-password');
            return;
        }

        $this->view('agent/auth/reset_password', [
            'title' => 'Reset Password',
            'token' => $token,
            'email' => $email,
            'csrf' => Security::csrfToken(),
        ]);
    }

    // Process the password reset form
    public function reset()
    {
        if (!Security::verifyCsrf((string)($_POST['csrf_token'] ?? ''))) {
            $_SESSION['reset_password_error'] = 'Invalid CSRF token. Please try again.';
            $this->redirectBack();
            return;
        }

        $token = $_POST['token'] ?? '';
        $email = strtolower(trim((string)($_POST['email'] ?? '')));
        $password = $_POST['password'] ?? '';
        $passwordConfirm = $_POST['password_confirmation'] ?? '';

        // Basic validation
        if (empty($token) || empty($email) || empty($password) || empty($passwordConfirm)) {
            $_SESSION['reset_password_error'] = 'All fields are required.';
            $this->redirectBack();
            return;
        }

        if ($password !== $passwordConfirm) {
            $_SESSION['reset_password_error'] = 'Passwords do not match.';
            $this->redirectBack();
            return;
        }

        if (strlen($password) < 8) {
            $_SESSION['reset_password_error'] = 'Password must be at least 8 characters long.';
            $this->redirectBack();
            return;
        }

        // Verify the token
        $reset = $this->pdo->prepare('SELECT * FROM password_resets WHERE LOWER(email) = LOWER(:email) AND token = :token AND used_at IS NULL AND expires_at > NOW() ORDER BY created_at DESC LIMIT 1 FOR UPDATE');
        $reset->execute([':email' => (string)$email, ':token' => $token]);
        $reset = $reset->fetch(\PDO::FETCH_ASSOC);

        if (!$reset) {
            $_SESSION['forgot_password_error'] = 'Invalid or expired password reset link. Please request a new one.';
            $this->redirect('/b2b/agent/forgot-password');
            return;
        }

        // Update the user's password
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $this->pdo->prepare('UPDATE users SET password = :password WHERE email = :email');
        $stmt->execute([':password' => $hashedPassword, ':email' => $email]);

        // Mark the token as used
        $this->pdo->prepare('UPDATE password_resets SET used_at = NOW() WHERE id = :id')
                 ->execute([':id' => $reset['id']]);

        // Get user details for email
        $user = $this->pdo->prepare('SELECT id, name, email FROM users WHERE email = :email LIMIT 1');
        $user->execute([':email' => $email]);
        $user = $user->fetch(\PDO::FETCH_ASSOC);

        // Send confirmation email
        $html2 = $this->renderViewToString('emails/agent/password_updated', [
            'name' => $user['name'],
            'loginUrl' => 'http://' . $_SERVER['HTTP_HOST'] . '/b2b/agent/login',
        ]);
        $subject2 = 'Your Password Has Been Reset';
        $env2 = new Envelope($email, $subject2, $html2, '', $fromEmail ?? null, $fromName ?? null);
        $sendRes2 = $this->mailService->send($env2);
        if (!$sendRes2->ok) {
            error_log('[PasswordReset] send confirmation failed: '.$sendRes2->message.' status='.$sendRes2->status);
        }

        // Flash message for login screen
        $_SESSION['agent_login_success'] = 'Your password has been changed. Please log in with your new password.';
        // Backward-compat key (if any other view reads this)
        $_SESSION['login_success'] = 'Your password has been changed. Please log in with your new password.';
        $this->redirect('/b2b/agent/login');
    }

    // Helper: redirect back to the previous page or to a sensible default
    private function redirectBack(): void
    {
        $back = $_SERVER['HTTP_REFERER'] ?? '/b2b/agent/forgot-password';
        $this->redirect($back);
    }

    // Helper: render a view file into a string (for email HTML bodies)
    private function renderViewToString(string $template, array $data = []): string
    {
        $templatePath = dirname(__DIR__) . '/Views/' . $template . '.php';
        if (!file_exists($templatePath)) {
            return '';
        }
        if (!empty($data)) {
            extract($data, EXTR_OVERWRITE);
        }
        ob_start();
        include $templatePath;
        return (string)ob_get_clean();
    }
}
