<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class ManagementController extends Controller
{
    public function agentCreate(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $allowManualUploads = (env('ALLOW_MANUAL_UPLOADS', 'true') !== 'false');
        // Load KYC config
        $kyc = require BASE_PATH . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'kyc.php';
        $agentKyc = $kyc['agent'] ?? [];

        // Compute effective KYC flags for the currently selected/default business type
        // so visibility matches per-type overrides immediately on initial render.
        $bt = $_SESSION['old']['business_type'] ?? 'freelancer';
        $map = [
            'freelancer' => 'freelancer',
            'sole_proprietor' => 'sp',
            'partnership' => 'llp',
            'company' => 'co',
        ];
        $btKey = $map[$bt] ?? 'freelancer';
        $perType = $agentKyc['per_type'][$btKey] ?? [];

        // Override Government ID flags when an override is defined for the selected business type
        if (!empty($perType['gov_id']) && !empty($perType['gov_id']['override'])) {
            $gov = $agentKyc['gov_id'] ?? [];
            $pt = $perType['gov_id'];
            // If front/back are not enabled in per-type, treat section as disabled.
            $enabled = !empty($pt['front_enabled']) || !empty($pt['back_enabled']) || !empty($pt['require_type']) || !empty($pt['require_number']);
            $gov['enabled'] = $enabled;
            $gov['require_type'] = !empty($pt['require_type']);
            $gov['require_number'] = !empty($pt['require_number']);
            $gov['require_front'] = !empty($pt['require_front']);
            $gov['back_enabled'] = !empty($pt['back_enabled']);
            $gov['require_back'] = !empty($pt['require_back']);
            // If per-type does not enable front explicitly, do not require front
            if (empty($pt['front_enabled'])) {
                $gov['require_front'] = false;
            }
            $agentKyc['gov_id'] = $gov;
        }

        // Override Registration flags (IATA/GST) per business type if override is enabled
        if (!empty($perType['registration']) && !empty($perType['registration']['override'])) {
            $reg = $agentKyc['registration'] ?? [];
            $pr = $perType['registration'];
            $reg['iata_enabled'] = !empty($pr['iata_enabled']);
            $reg['iata_code_required_on_yes'] = !empty($pr['iata_code_required_on_yes']);
            $reg['gst_number'] = [
                'enabled' => !empty($pr['gst_number']['enabled'] ?? false),
                'required' => !empty($pr['gst_number']['required'] ?? false),
            ];
            $reg['gst_company'] = [
                'enabled' => !empty($pr['gst_company']['enabled'] ?? false),
                'required' => !empty($pr['gst_company']['required'] ?? false),
            ];
            $agentKyc['registration'] = $reg;

            // Ensure the per-type registration document block is visible for the selected type
            if ($btKey === 'sp') {
                $agentKyc['reg_certificate_sp'] = $agentKyc['reg_certificate_sp'] ?? [];
                $agentKyc['reg_certificate_sp']['enabled'] = true;
            } elseif ($btKey === 'llp') {
                $agentKyc['partnership_deed_llp'] = $agentKyc['partnership_deed_llp'] ?? [];
                $agentKyc['partnership_deed_llp']['enabled'] = true;
            } elseif ($btKey === 'co') {
                $agentKyc['incorp_cert_company'] = $agentKyc['incorp_cert_company'] ?? [];
                $agentKyc['incorp_cert_company']['enabled'] = true;
            }
        }

        // Business Documents per-type override (Selfie and Address Proof) via env BT_* keys
        // These keys are not yet in config/kyc.php, so we read from env directly when OVERRIDE_DOCS is on.
        $prefixMap = [
            'freelancer' => 'BT_FREELANCER',
            'sp' => 'BT_SP',
            'llp' => 'BT_LLP',
            'co' => 'BT_CO',
        ];
        $pfx = $prefixMap[$btKey] ?? 'BT_FREELANCER';
        $isTrue = function($val){ return ($val === true) || ($val === 1) || ($val === '1') || ($val === 'true'); };
        if ($isTrue(env($pfx . '_OVERRIDE_DOCS', 'false'))) {
            // Selfie applies to all business types
            $selfieEn = $isTrue(env($pfx . '_SELFIE_ENABLE', 'false'));
            $selfieReq = $isTrue(env($pfx . '_SELFIE_REQUIRE', 'false'));
            $agentKyc['selfie'] = [
                'enabled' => $selfieEn,
                'required' => $selfieReq && $selfieEn,
            ];
            // Address proof currently relevant for Freelancer block in the view
            if ($btKey === 'freelancer') {
                $addrEn = $isTrue(env('BT_FREELANCER_ADDRESS_PROOF_ENABLE', 'false'));
                $addrReq = $isTrue(env('BT_FREELANCER_ADDRESS_PROOF_REQUIRE', 'false'));
                $agentKyc['address_proof_freelancer'] = [
                    'enabled' => $addrEn,
                    'required' => $addrReq && $addrEn,
                ];
            }
        }

        $this->view('admin/management_agents_create', compact('csrf','allowManualUploads','agentKyc'));
    }

    public function partnerCreate(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $this->view('admin/management_partners_create', compact('csrf'));
    }

    private function ensureChannelPartnerSchema(): void
    {
        $sql = [
            // Core tables are in schema.sql but ensure presence in runtime
            "CREATE TABLE IF NOT EXISTS channel_partners (\n  id INT AUTO_INCREMENT PRIMARY KEY,\n  name VARCHAR(150) NOT NULL,\n  contact_email VARCHAR(150) NULL,\n  contact_person VARCHAR(150) NULL,\n  contact_mobile VARCHAR(50) NULL,\n  contact_whatsapp VARCHAR(50) NULL,\n  contact_line VARCHAR(100) NULL,\n  address_line VARCHAR(255) NULL,\n  city VARCHAR(120) NULL,\n  state VARCHAR(120) NULL,\n  pincode VARCHAR(30) NULL,\n  country VARCHAR(120) NULL,\n  status ENUM('Active','Inactive') NOT NULL DEFAULT 'Active',\n  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n) ENGINE=InnoDB",
            "CREATE TABLE IF NOT EXISTS channel_partner_wallet (\n  id INT AUTO_INCREMENT PRIMARY KEY,\n  partner_id INT NOT NULL,\n  balance DECIMAL(12,2) NOT NULL DEFAULT 0,\n  FOREIGN KEY (partner_id) REFERENCES channel_partners(id) ON DELETE CASCADE\n) ENGINE=InnoDB",
            "CREATE TABLE IF NOT EXISTS channel_partner_wallet_ledger (\n  id INT AUTO_INCREMENT PRIMARY KEY,\n  wallet_id INT NOT NULL,\n  type ENUM('credit','debit') NOT NULL,\n  amount DECIMAL(12,2) NOT NULL,\n  method ENUM('manual','transfer') NOT NULL,\n  status ENUM('pending','approved','rejected') NOT NULL DEFAULT 'pending',\n  meta JSON NULL,\n  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n  FOREIGN KEY (wallet_id) REFERENCES channel_partner_wallet(id) ON DELETE CASCADE\n) ENGINE=InnoDB",
            // Settings holder (created lean; columns added via ALTERs below to avoid mismatches across envs)
            "CREATE TABLE IF NOT EXISTS channel_partner_settings (\n  partner_id INT PRIMARY KEY,\n  FOREIGN KEY (partner_id) REFERENCES channel_partners(id) ON DELETE CASCADE\n) ENGINE=InnoDB",
            // Mapping CP -> Agents they manage (for future agent creation by CP)
            "CREATE TABLE IF NOT EXISTS channel_partner_agents (\n  id INT AUTO_INCREMENT PRIMARY KEY,\n  partner_id INT NOT NULL,\n  agent_user_id INT NOT NULL,\n  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n  UNIQUE KEY uniq_partner_agent (partner_id, agent_user_id),\n  FOREIGN KEY (partner_id) REFERENCES channel_partners(id) ON DELETE CASCADE,\n  FOREIGN KEY (agent_user_id) REFERENCES users(id) ON DELETE CASCADE\n) ENGINE=InnoDB"
        ];
        foreach ($sql as $q) { try { $this->pdo->exec($q); } catch (\Throwable $e) { /* ignore */ } }

        // Attempt to add columns for contact details if table already existed
        $alter = [
            "ALTER TABLE channel_partners ADD COLUMN contact_person VARCHAR(150) NULL",
            "ALTER TABLE channel_partners ADD COLUMN contact_mobile VARCHAR(50) NULL",
            "ALTER TABLE channel_partners ADD COLUMN contact_whatsapp VARCHAR(50) NULL",
            "ALTER TABLE channel_partners ADD COLUMN contact_line VARCHAR(100) NULL",
            "ALTER TABLE channel_partners ADD COLUMN address_line VARCHAR(255) NULL",
            "ALTER TABLE channel_partners ADD COLUMN city VARCHAR(120) NULL",
            "ALTER TABLE channel_partners ADD COLUMN state VARCHAR(120) NULL",
            "ALTER TABLE channel_partners ADD COLUMN pincode VARCHAR(30) NULL",
            "ALTER TABLE channel_partners ADD COLUMN country VARCHAR(120) NULL",
            // Secure transaction password for partner actions
            "ALTER TABLE channel_partners ADD COLUMN transaction_password_hash VARCHAR(255) NULL",
            // Encrypted-at-rest storage for 6-digit TX PIN (AES-256-GCM)
            "ALTER TABLE channel_partners ADD COLUMN tx_pin_enc VARBINARY(255) NULL",
            "ALTER TABLE channel_partners ADD COLUMN tx_pin_iv VARBINARY(16) NULL",
            "ALTER TABLE channel_partners ADD COLUMN tx_pin_tag VARBINARY(16) NULL",
            // Ensure settings columns exist (granular controls)
            "ALTER TABLE channel_partner_settings ADD COLUMN allow_wallet_transfer TINYINT(1) NOT NULL DEFAULT 1",
            "ALTER TABLE channel_partner_settings ADD COLUMN require_tx_password TINYINT(1) NOT NULL DEFAULT 1",
            "ALTER TABLE channel_partner_settings ADD COLUMN wallet_transfer_daily_limit DECIMAL(12,2) NULL",
            "ALTER TABLE channel_partner_settings ADD COLUMN allow_agent_creation TINYINT(1) NOT NULL DEFAULT 1",
            "ALTER TABLE channel_partner_settings ADD COLUMN agent_creation_limit INT NULL",
            "ALTER TABLE channel_partner_settings ADD COLUMN allow_agent_password_change TINYINT(1) NOT NULL DEFAULT 0",
            "ALTER TABLE channel_partner_settings ADD COLUMN api_enabled TINYINT(1) NOT NULL DEFAULT 0",
            "ALTER TABLE channel_partner_settings ADD COLUMN allow_extra_margin TINYINT(1) NOT NULL DEFAULT 0",
            "ALTER TABLE channel_partner_settings ADD COLUMN max_margin_percent DECIMAL(5,2) NULL",
            "ALTER TABLE channel_partner_settings ADD COLUMN max_flat_markup DECIMAL(10,2) NULL",
            "ALTER TABLE channel_partner_settings ADD COLUMN created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP",
            "ALTER TABLE channel_partner_settings ADD COLUMN updated_at TIMESTAMP NULL",
            // Ensure unique constraint compatible with INSERT ... ON DUPLICATE KEY
            "ALTER TABLE channel_partner_settings ADD UNIQUE KEY uniq_partner (partner_id)"
        ];
        foreach ($alter as $q) { try { $this->pdo->exec($q); } catch (\Throwable $e) { /* likely exists, ignore */ } }
    }

    // Encrypt/decrypt helpers using AES-256-GCM
    private function getTxKey(): string
    {
        $keyB64 = (string)\App\Core\Env::get('TX_PIN_KEY', '');
        $keyB64 = trim($keyB64);
        // Strip surrounding quotes if present
        if ((str_starts_with($keyB64, '"') && str_ends_with($keyB64, '"')) || (str_starts_with($keyB64, "'") && str_ends_with($keyB64, "'"))) {
            $keyB64 = substr($keyB64, 1, -1);
        }
        if ($keyB64 === '') { throw new \RuntimeException('TX_PIN_KEY missing in .env'); }
        $key = base64_decode($keyB64, true);
        if ($key === false || strlen($key) !== 32) { throw new \RuntimeException('TX_PIN_KEY must be 32-byte base64'); }
        return $key;
    }

    private function encryptTxPin(string $pin, ?string $keyOverride = null): array
    {
        $key = $keyOverride !== null ? $keyOverride : $this->getTxKey();
        $methods = openssl_get_cipher_methods(true) ?: [];
        if (in_array('aes-256-gcm', $methods, true)) {
            // Preferred: GCM with 12-byte IV and 16-byte tag
            $iv = random_bytes(12);
            $tag = '';
            $enc = openssl_encrypt($pin, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag, 'cp_txpin');
            if ($enc === false) { throw new \RuntimeException('Failed to encrypt TX PIN (GCM)'); }
            return [$enc, $iv, $tag];
        }
        // Fallback: CBC + HMAC-SHA256 (tag truncated to 16 bytes to fit column)
        $iv = random_bytes(16);
        $enc = openssl_encrypt($pin, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $iv);
        if ($enc === false) { throw new \RuntimeException('Failed to encrypt TX PIN (CBC)'); }
        $fullTag = hash_hmac('sha256', $iv . $enc, $key, true); // 32 bytes
        $tag = substr($fullTag, 0, 16); // store 16 bytes
        return [$enc, $iv, $tag];
    }

    private function decryptTxPin(string $enc, string $iv, string $tag, ?string $keyOverride = null): string
    {
        $key = $keyOverride !== null ? $keyOverride : $this->getTxKey();
        // Detect mode by IV length: 12 => GCM, else CBC-HMAC fallback
        if (strlen($iv) === 12) {
            $plain = openssl_decrypt($enc, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag, 'cp_txpin');
            if ($plain === false) { throw new \RuntimeException('Failed to decrypt TX PIN (GCM)'); }
            return $plain;
        }
        // CBC + HMAC verify
        $calc = substr(hash_hmac('sha256', $iv . $enc, $key, true), 0, 16);
        if (!hash_equals($calc, $tag)) { throw new \RuntimeException('TX PIN integrity check failed'); }
        $plain = openssl_decrypt($enc, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $iv);
        if ($plain === false) { throw new \RuntimeException('Failed to decrypt TX PIN (CBC)'); }
        return $plain;
    }

    public function partnerSettings(): void
    {
        Auth::requireRole(['Admin']);
        $this->ensureChannelPartnerSchema();
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { http_response_code(400); echo json_encode(['error' => 'Invalid partner id']); return; }
        try {
            // Base partner and tx password presence
            $sp = $this->pdo->prepare('SELECT id, name, contact_email, transaction_password_hash FROM channel_partners WHERE id = ? LIMIT 1');
            $sp->execute([$id]);
            $partner = $sp->fetch(\PDO::FETCH_ASSOC);
            if (!$partner) { http_response_code(404); echo json_encode(['error' => 'Partner not found']); return; }
            $hasTx = !empty($partner['transaction_password_hash']);
            // Settings row
            $ss = $this->pdo->prepare('SELECT allow_wallet_transfer, require_tx_password, wallet_transfer_daily_limit, allow_agent_creation, agent_creation_limit, allow_agent_password_change, api_enabled, allow_extra_margin, max_margin_percent, max_flat_markup FROM channel_partner_settings WHERE partner_id = ? LIMIT 1');
            $ss->execute([$id]);
            $s = $ss->fetch(\PDO::FETCH_ASSOC) ?: [];
            $defaults = [
                'allow_wallet_transfer' => 1,
                'require_tx_password' => 1,
                'wallet_transfer_daily_limit' => null,
                'allow_agent_creation' => 1,
                'agent_creation_limit' => null,
                'allow_agent_password_change' => 0,
                'api_enabled' => 0,
                'allow_extra_margin' => 0,
                'max_margin_percent' => null,
                'max_flat_markup' => null,
            ];
            $out = array_merge($defaults, $s);
            $out['partner_id'] = $id;
            $out['tx_password_set'] = $hasTx ? 1 : 0;
            header('Content-Type: application/json');
            echo json_encode($out);
        } catch (\Throwable $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to load settings']);
        }
    }

    public function partnerSettingsSave(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        // Require master password to confirm sensitive changes
        try { Security::requireMasterPassword(); } catch (\Throwable $e) { /* allow if not configured */ }

        $this->ensureChannelPartnerSchema();
        $id = (int)($_POST['id'] ?? 0);
        $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
        $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
        $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
        if ($id <= 0) {
            if ($isAjax) { http_response_code(400); header('Content-Type: application/json'); echo json_encode(['ok'=>false,'error'=>'Invalid partner.']); return; }
            $_SESSION['flash'] = 'Invalid partner.'; $this->redirect('/admin/management/partners');
        }

        // Toggles
        $b = function($key){ $v = $_POST[$key] ?? null; return ($v === '1' || $v === 1 || $v === 'on' || $v === 'true'); };
        $allowWallet = $b('allow_wallet_transfer') ? 1 : 0;
        $requireTx = $b('require_tx_password') ? 1 : 0;
        $allowAgentCreate = $b('allow_agent_creation') ? 1 : 0;
        $allowAgentPw = $b('allow_agent_password_change') ? 1 : 0;
        $apiEnabled = $b('api_enabled') ? 1 : 0;
        $allowExtraMargin = $b('allow_extra_margin') ? 1 : 0;

        // Numeric limits
        $dailyLimit = $_POST['wallet_transfer_daily_limit'] ?? '';
        $dailyLimit = ($dailyLimit === '' || !is_numeric($dailyLimit)) ? null : (float)$dailyLimit;
        $agentLimit = $_POST['agent_creation_limit'] ?? '';
        $agentLimit = ($agentLimit === '' || !is_numeric($agentLimit)) ? null : (int)$agentLimit;
        $maxMarginPct = $_POST['max_margin_percent'] ?? '';
        $maxMarginPct = ($maxMarginPct === '' || !is_numeric($maxMarginPct)) ? null : (float)$maxMarginPct;
        $maxFlat = $_POST['max_flat_markup'] ?? '';
        $maxFlat = ($maxFlat === '' || !is_numeric($maxFlat)) ? null : (float)$maxFlat;

        // Optional: set/update TX password
        $txPassword = (string)($_POST['transaction_password'] ?? '');

        // Basic validations
        $errs = [];
        if ($maxMarginPct !== null && ($maxMarginPct < 0 || $maxMarginPct > 100)) { $errs[] = 'Max margin percent must be between 0 and 100.'; }
        if ($agentLimit !== null && $agentLimit < 0) { $errs[] = 'Agent creation limit cannot be negative.'; }
        if ($dailyLimit !== null && $dailyLimit < 0) { $errs[] = 'Daily transfer limit cannot be negative.'; }
        // If TX password provided, enforce exactly 6 digits
        if ($txPassword !== '' && !preg_match('/^\d{6}$/', $txPassword)) {
            $errs[] = 'Transaction PIN must be exactly 6 digits.';
        }
        if (!empty($errs)) {
            if ($isAjax) { http_response_code(422); header('Content-Type: application/json'); echo json_encode(['ok'=>false,'errors'=>$errs]); return; }
            $_SESSION['errors'] = $errs; $this->redirect('/admin/management/partners');
        }

        try {
            $this->pdo->beginTransaction();
            // Upsert settings
            $ins = $this->pdo->prepare("INSERT INTO channel_partner_settings (partner_id, allow_wallet_transfer, require_tx_password, wallet_transfer_daily_limit, allow_agent_creation, agent_creation_limit, allow_agent_password_change, api_enabled, allow_extra_margin, max_margin_percent, max_flat_markup) VALUES (?,?,?,?,?,?,?,?,?,?,?) ON DUPLICATE KEY UPDATE allow_wallet_transfer=VALUES(allow_wallet_transfer), require_tx_password=VALUES(require_tx_password), wallet_transfer_daily_limit=VALUES(wallet_transfer_daily_limit), allow_agent_creation=VALUES(allow_agent_creation), agent_creation_limit=VALUES(agent_creation_limit), allow_agent_password_change=VALUES(allow_agent_password_change), api_enabled=VALUES(api_enabled), allow_extra_margin=VALUES(allow_extra_margin), max_margin_percent=VALUES(max_margin_percent), max_flat_markup=VALUES(max_flat_markup), updated_at=NOW()");
            $ins->execute([$id, $allowWallet, $requireTx, $dailyLimit, $allowAgentCreate, $agentLimit, $allowAgentPw, $apiEnabled, $allowExtraMargin, $maxMarginPct, $maxFlat]);

            // Update TX password if provided
            if ($txPassword !== '') {
                $hash = password_hash($txPassword, PASSWORD_BCRYPT);
                $up = $this->pdo->prepare('UPDATE channel_partners SET transaction_password_hash = ? WHERE id = ?');
                $up->execute([$hash, $id]);

                // Store encrypted-at-rest PIN for optional future reveal (admin-gated)
                $encWarn = false;
                $encErrorMsg = null;
                $debugKeyLen = null;
                // Optional debug override: base64 key passed in request (only when explicitly allowed via env)
                $txKeyOverride = null;
                $allowOverride = in_array(strtolower((string)\App\Core\Env::get('ALLOW_TX_KEY_OVERRIDE', 'false')), ['1','true','yes'], true);
                if ($allowOverride && isset($_POST['tx_key_override']) && $_POST['tx_key_override'] !== '') {
                    $b64 = trim((string)$_POST['tx_key_override']);
                    $raw = base64_decode($b64, true);
                    if ($raw !== false && strlen($raw) === 32) { $txKeyOverride = $raw; }
                }
                try {
                    [$enc, $iv, $tag] = $this->encryptTxPin($txPassword, $txKeyOverride);
                    $up2 = $this->pdo->prepare('UPDATE channel_partners SET tx_pin_enc = ?, tx_pin_iv = ?, tx_pin_tag = ? WHERE id = ?');
                    $up2->execute([$enc, $iv, $tag, $id]);
                } catch (\Throwable $e) {
                    // If encryption fails, keep hash only; don't block settings save
                    $encWarn = true;
                    $encErrorMsg = $e->getMessage();
                    // Lightweight diagnostic: what key length does the process see?
                    try {
                        $raw = base64_decode((string)\App\Core\Env::get('TX_PIN_KEY', ''), true);
                        if ($raw !== false) { $debugKeyLen = strlen($raw); }
                    } catch (\Throwable $e2) { /* ignore */ }
                }
                if ($encWarn) {
                    $_SESSION['flash'] = 'Settings saved. Warning: TX PIN stored but not viewable (encryption key missing or invalid).';
                }
            }

            $this->pdo->commit();
            if (empty($_SESSION['flash'])) { $_SESSION['flash'] = 'Partner settings saved.'; }
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) { $this->pdo->rollBack(); }
            if ($isAjax) {
                http_response_code(500);
                header('Content-Type: application/json');
                $resp = ['ok'=>false,'error'=>'Failed to save settings.'];
                // Provide diagnostic detail in dev/admin context
                $resp['detail'] = $e->getMessage();
                echo json_encode($resp);
                return;
            }
            $_SESSION['flash'] = 'Failed to save settings.';
        }
        if ($isAjax) {
            $msg = $_SESSION['flash'] ?? 'Partner settings saved.';
            $warning = (strpos($msg, 'Warning: TX PIN stored') !== false);
            header('Content-Type: application/json');
            $resp = ['ok'=>true,'message'=>$msg,'warning'=>$warning];
            // Include diagnostics only when warning is true to help debugging locally
            if ($warning ?? false) {
                if (isset($encErrorMsg) && $encErrorMsg) { $resp['enc_error'] = $encErrorMsg; }
                if (isset($debugKeyLen)) { $resp['tx_key_len'] = $debugKeyLen; }
            }
            echo json_encode($resp);
            unset($_SESSION['flash']);
            return;
        }
        $this->redirect('/admin/management/partners');
    }

    public function partnerSettingsViewPin(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $this->ensureChannelPartnerSchema();
        header('Content-Type: application/json');

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { http_response_code(400); echo json_encode(['error' => 'Invalid partner id']); return; }
        try {
            $q = $this->pdo->prepare('SELECT tx_pin_enc, tx_pin_iv, tx_pin_tag FROM channel_partners WHERE id = ? LIMIT 1');
            $q->execute([$id]);
            $row = $q->fetch(\PDO::FETCH_ASSOC);
            if (!$row || empty($row['tx_pin_enc']) || empty($row['tx_pin_iv']) || empty($row['tx_pin_tag'])) {
                http_response_code(404); echo json_encode(['error' => 'PIN not set']); return;
            }
            // Optional debug override: base64 key passed in request (only when allowed via env)
            $txKeyOverride = null;
            $allowOverride = in_array(strtolower((string)\App\Core\Env::get('ALLOW_TX_KEY_OVERRIDE', 'false')), ['1','true','yes'], true);
            if ($allowOverride && isset($_POST['tx_key_override']) && $_POST['tx_key_override'] !== '') {
                $b64 = trim((string)$_POST['tx_key_override']);
                $raw = base64_decode($b64, true);
                if ($raw !== false && strlen($raw) === 32) { $txKeyOverride = $raw; }
            }
            $pin = $this->decryptTxPin($row['tx_pin_enc'], $row['tx_pin_iv'], $row['tx_pin_tag'], $txKeyOverride);
            // Basic guard: ensure format
            if (!preg_match('/^\\d{6}$/', $pin)) { http_response_code(500); echo json_encode(['error' => 'Corrupt PIN']); return; }
            echo json_encode(['pin' => $pin]);
        } catch (\Throwable $e) {
            http_response_code(500);
            $msg = $e->getMessage();
            if (strpos($msg, 'TX_PIN_KEY') !== false) {
                echo json_encode(['error' => 'Encryption key missing or invalid.']);
            } else {
                echo json_encode(['error' => 'Failed to read PIN']);
            }
        }
    }

    public function agentGenerateApiCredentials(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        // Allow operation even if master password is not configured
        try { Security::requireMasterPassword(); } catch (\Throwable $e) { /* allow if not configured */ }
        $this->ensureAgentSettingsSchema();
        header('Content-Type: application/json');

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { http_response_code(400); echo json_encode(['error'=>'Invalid agent id']); return; }
        try {
            $chk = $this->pdo->prepare("SELECT id FROM users WHERE id = ? AND role = 'B2B Agent' LIMIT 1");
            $chk->execute([$id]);
            if (!$chk->fetch(\PDO::FETCH_ASSOC)) { http_response_code(404); echo json_encode(['error'=>'Agent not found']); return; }

            $q = $this->pdo->prepare('SELECT api_enabled FROM agent_settings WHERE user_id = ? LIMIT 1');
            $q->execute([$id]);
            $row = $q->fetch(\PDO::FETCH_ASSOC);
            if (!$row || (int)$row['api_enabled'] !== 1) { http_response_code(422); echo json_encode(['error'=>'Enable API access first.']); return; }

            $apiKey = 'ag_' . bin2hex(random_bytes(16));
            $apiSecret = bin2hex(random_bytes(32));
            $secretHash = password_hash($apiSecret, PASSWORD_BCRYPT);

            $up = $this->pdo->prepare('UPDATE agent_settings SET api_key = ?, api_secret_hash = ?, updated_at = NOW() WHERE user_id = ?');
            $up->execute([$apiKey, $secretHash, $id]);

            echo json_encode(['ok'=>true,'api_key'=>$apiKey,'api_secret'=>$apiSecret]);
        } catch (\Throwable $e) {
            http_response_code(500);
            echo json_encode(['error'=>'Failed to generate credentials']);
        }
    }

    public function partnerStore(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        // Optional: protect with master password similar to VendorsController
        try { Security::requireMasterPassword(); } catch (\Throwable $e) { /* allow if not configured */ }

        $this->ensureChannelPartnerSchema();

        $name = trim((string)($_POST['name'] ?? ''));
        $email = trim((string)($_POST['email'] ?? ''));
        $contactPerson = trim((string)($_POST['contact_person'] ?? ''));
        $contactMobile = trim((string)($_POST['contact_mobile'] ?? ''));
        $contactWhatsApp = trim((string)($_POST['contact_whatsapp'] ?? ''));
        $contactLine = trim((string)($_POST['contact_line'] ?? ''));
        $password = (string)($_POST['password'] ?? '');
        $addressLine = trim((string)($_POST['address_line'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        $state = trim((string)($_POST['state'] ?? ''));
        $pincode = trim((string)($_POST['pincode'] ?? ''));
        $country = trim((string)($_POST['country'] ?? ''));
        $defaultMargin = trim((string)($_POST['default_margin_percent'] ?? ''));
        $fallbackShare = trim((string)($_POST['fallback_share_percent'] ?? ''));
        $notes = trim((string)($_POST['notes'] ?? ''));

        $errors = [];
        if ($name === '') { $errors[] = 'Partner name is required.'; }
        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Valid email is required.'; }
        if ($password === '' || strlen($password) < 6) { $errors[] = 'Password must be at least 6 characters.'; }
        if ($addressLine === '') { $errors[] = 'Address is required.'; }
        if ($city === '') { $errors[] = 'City is required.'; }
        if ($country === '') { $errors[] = 'Country is required.'; }
        if ($defaultMargin !== '' && !is_numeric($defaultMargin)) { $errors[] = 'Default margin must be a number.'; }
        if ($fallbackShare !== '' && !is_numeric($fallbackShare)) { $errors[] = 'Fallback share must be a number.'; }
        if ($defaultMargin !== '' && is_numeric($defaultMargin) && (float)$defaultMargin > 30) { $errors[] = 'Default margin cannot exceed 30%.'; }
        if ($fallbackShare !== '' && is_numeric($fallbackShare) && (float)$fallbackShare > 30) { $errors[] = 'Fallback share cannot exceed 30%.'; }

        // Check duplicate email in users table
        try {
            $stmt = $this->pdo->prepare('SELECT id FROM users WHERE email = ? LIMIT 1');
            $stmt->execute([$email]);
            if ($stmt->fetch()) { $errors[] = 'Email already in use.'; }
        } catch (\Throwable $e) { $errors[] = 'Database error while checking email.'; }

        if (!empty($errors)) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old'] = $_POST;
            $this->redirect('/admin/management/partners/create');
        }

        try {
            $this->pdo->beginTransaction();

            // 1) Create login user with role Channel Partner
            $hash = password_hash($password, PASSWORD_BCRYPT);
            $stmtU = $this->pdo->prepare('INSERT INTO users (name, email, password, role, status) VALUES (?,?,?,?,?)');
            $okU = $stmtU->execute([$name, $email, $hash, 'Channel Partner', 'Active']);
            if (!$okU) { throw new \RuntimeException('Failed to create user'); }
            $userId = (int)$this->pdo->lastInsertId();

            // 2) Create channel partner row
            $stmtP = $this->pdo->prepare('INSERT INTO channel_partners (name, contact_email, contact_person, contact_mobile, contact_whatsapp, contact_line, address_line, city, state, pincode, country, status) VALUES (?,?,?,?,?,?,?,?,?,?,?,?)');
            $okP = $stmtP->execute([
                $name,
                $email,
                $contactPerson !== '' ? $contactPerson : null,
                $contactMobile !== '' ? $contactMobile : null,
                $contactWhatsApp !== '' ? $contactWhatsApp : null,
                $contactLine !== '' ? $contactLine : null,
                $addressLine,
                $city,
                $state !== '' ? $state : null,
                $pincode !== '' ? $pincode : null,
                $country,
                'Active'
            ]);
            if (!$okP) { throw new \RuntimeException('Failed to create partner'); }
            $partnerId = (int)$this->pdo->lastInsertId();

            // 3) Wallet for partner
            $stmtW = $this->pdo->prepare('INSERT INTO channel_partner_wallet (partner_id, balance) VALUES (?, 0)');
            $stmtW->execute([$partnerId]);

            // 4) Settings for margins and fallback
            $stmtS = $this->pdo->prepare('INSERT INTO channel_partner_settings (partner_id, default_margin_percent, fallback_share_percent, notes) VALUES (?,?,?,?)');
            $dm = ($defaultMargin === '') ? null : (float)$defaultMargin;
            $fs = ($fallbackShare === '') ? null : (float)$fallbackShare;
            $stmtS->execute([$partnerId, $dm, $fs, $notes !== '' ? $notes : null]);

            $this->pdo->commit();
            $_SESSION['flash'] = 'Channel Partner created.';
            unset($_SESSION['old']);
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) { $this->pdo->rollBack(); }
            $_SESSION['errors'] = ['Failed to create Channel Partner.', $e->getMessage()];
            $_SESSION['old'] = $_POST;
        }

        $this->redirect('/admin/management/partners/create');
    }

    public function employeeCreate(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $this->view('admin/management_employees_create', compact('csrf'));
    }

    public function customerCreate(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $this->view('admin/management_customers_create', compact('csrf'));
    }

    public function agentIndex(): void
    {
        Auth::requireRole(['Admin']);
        // List users with role B2B Agent, include linked Channel Partner (if any)
        $stmt = $this->pdo->prepare("SELECT u.id, u.name, u.email, u.status,
                                             ap.company, ap.phone, ap.city, ap.country, ap.commission_percent,
                                             cp.name AS partner_name, cp.id AS partner_id,
                                             w.balance AS wallet_balance,
                                             CASE WHEN w.id IS NOT NULL THEN 1 ELSE 0 END AS has_wallet
                                      FROM users u
                                      LEFT JOIN agent_profiles ap ON ap.user_id = u.id
                                      LEFT JOIN channel_partner_agents cpa ON cpa.agent_user_id = u.id
                                      LEFT JOIN channel_partners cp ON cp.id = cpa.partner_id
                                      LEFT JOIN wallets w ON w.user_id = u.id
                                      WHERE u.role = 'B2B Agent' ORDER BY u.id DESC");
        $stmt->execute();
        $agents = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        $csrf = Security::csrfToken();
        $this->view('admin/management_agents_index', compact('agents','csrf'));
    }

    public function partnerIndex(): void
    {
        Auth::requireRole(['Admin']);
        $this->ensureChannelPartnerSchema();
        $stmt = $this->pdo->prepare("SELECT cp.id, cp.name, cp.contact_email, cp.contact_person, cp.contact_mobile, cp.contact_whatsapp, cp.contact_line, cp.city, cp.country, cp.status, cp.created_at,
                                            COALESCE(cpw.balance, 0) AS wallet_balance
                                     FROM channel_partners cp
                                     LEFT JOIN channel_partner_wallet cpw ON cpw.partner_id = cp.id
                                     ORDER BY cp.id DESC");
        $stmt->execute();
        $partners = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        $csrf = Security::csrfToken();
        $this->view('admin/management_partners_index', compact('partners','csrf'));
    }

    public function agentView(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/management/agents'); }
        // Agent core + profile
        $stmt = $this->pdo->prepare("SELECT u.id, u.name, u.email, u.status,
                                            ap.company, ap.phone, ap.city, ap.country, ap.commission_percent,
                                            COALESCE(w.balance, 0) AS wallet_balance
                                     FROM users u
                                     LEFT JOIN agent_profiles ap ON ap.user_id = u.id
                                     LEFT JOIN wallets w ON w.user_id = u.id
                                     WHERE u.id = ? AND u.role = 'B2B Agent' LIMIT 1");
        $stmt->execute([$id]);
        $agent = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$agent) { $this->redirect('/admin/management/agents'); }
        // Partner link
        $stmt2 = $this->pdo->prepare("SELECT cp.id, cp.name FROM channel_partner_agents cpa LEFT JOIN channel_partners cp ON cp.id = cpa.partner_id WHERE cpa.agent_user_id = ? LIMIT 1");
        try { $stmt2->execute([$id]); $partner = $stmt2->fetch(\PDO::FETCH_ASSOC) ?: null; } catch (\Throwable $e) { $partner = null; }
        $this->view('admin/management_agents_view', compact('agent','partner'));
    }

    public function agentToggleStatus(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/management/agents'); }
        // flip status between Active/Inactive
        $stmt = $this->pdo->prepare("SELECT status FROM users WHERE id = ? AND role = 'B2B Agent' LIMIT 1");
        $stmt->execute([$id]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        if ($row) {
            $next = ($row['status'] === 'Active') ? 'Inactive' : 'Active';
            $up = $this->pdo->prepare("UPDATE users SET status = ? WHERE id = ?");
            $up->execute([$next, $id]);
            $_SESSION['flash'] = 'Agent status updated to ' . $next . '.';
        }
        $this->redirect('/admin/management/agents');
    }

    // POST /admin/management/agents/create_wallet
    public function createWallet(): void
    {
        // Set up logging
        $logFile = __DIR__ . '/../../logs/wallet_creation.log';
        $logMessage = function($message) use ($logFile) {
            $timestamp = date('Y-m-d H:i:s');
            $logEntry = "[$timestamp] $message" . PHP_EOL;
            file_put_contents($logFile, $logEntry, FILE_APPEND);
        };
        
        $logMessage("=== Starting wallet creation process ===");
        $logMessage("Request data: " . json_encode($_POST));
        
        Auth::requireRole(['Admin']);
        
        // Check if this is an AJAX request
        $isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                  strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
                  
        $logMessage("Is AJAX request: " . ($isAjax ? 'Yes' : 'No'));
        
        try {
            $agentId = (int)($_POST['agent_id'] ?? 0);
            $logMessage("Processing agent ID: $agentId");
            
            if ($agentId <= 0) {
                $errorMsg = 'Invalid agent ID provided';
                $logMessage("ERROR: $errorMsg");
                throw new \Exception($errorMsg);
            }
            
            // Verify agent exists and is an agent
            $logMessage("Verifying agent exists and is a B2B Agent");
            $stmt = $this->pdo->prepare("SELECT id, name, email FROM users WHERE id = ? AND role = 'B2B Agent' LIMIT 1");
            $stmt->execute([$agentId]);
            $agent = $stmt->fetch(\PDO::FETCH_ASSOC);
            
            if (!$agent) {
                $errorMsg = "Agent not found or not a B2B Agent (ID: $agentId)";
                $logMessage("ERROR: $errorMsg");
                throw new \Exception('Agent not found');
            }
            
            $logMessage("Agent found: {$agent['name']} ({$agent['email']})");
            
            // Check if wallet already exists
            $logMessage("Checking if wallet already exists for agent ID: $agentId");
            $walletCheck = $this->pdo->prepare("SELECT id, balance FROM wallets WHERE user_id = ? LIMIT 1");
            $walletCheck->execute([$agentId]);
            
            if ($existingWallet = $walletCheck->fetch(\PDO::FETCH_ASSOC)) {
                $errorMsg = "Wallet already exists for this agent (Wallet ID: {$existingWallet['id']}, Balance: {$existingWallet['balance']})";
                $logMessage("ERROR: $errorMsg");
                throw new \Exception('Wallet already exists for this agent');
            }
            
            $logMessage("No existing wallet found, proceeding with creation");
            
            // Create wallet with initial balance of 0
            $logMessage("Starting database transaction for wallet creation");
            $this->pdo->beginTransaction();
            
            try {
                // Insert wallet
                $logMessage("Executing wallet creation query");
                $walletStmt = $this->pdo->prepare(
                    "INSERT INTO wallets (user_id, balance) " . 
                    "VALUES (?, 0.00)"
                );
                $walletStmt->execute([$agentId]);
                $walletId = $this->pdo->lastInsertId();
                $logMessage("Wallet created successfully. Wallet ID: $walletId");
            } catch (\PDOException $e) {
                $errorMsg = "Database error during wallet creation: " . $e->getMessage();
                $logMessage("ERROR: $errorMsg");
                $this->pdo->rollBack();
                throw new \Exception('Failed to create wallet. Please try again.');
            }
            
            try {
                // Log the wallet creation in wallet_ledger
                $logMessage("Logging wallet creation in wallet_ledger");
                
                // Get the current balance (should be 0.00 for new wallet)
                $balanceStmt = $this->pdo->prepare("SELECT balance FROM wallets WHERE id = ?");
                $balanceStmt->execute([$walletId]);
                $wallet = $balanceStmt->fetch(\PDO::FETCH_ASSOC);
                $currentBalance = $wallet ? (float)$wallet['balance'] : 0.00;
                
                $txStmt = $this->pdo->prepare(
                    "INSERT INTO wallet_ledger (wallet_id, type, amount, method, status, meta, balance_before, balance_after, created_at) " .
                    "VALUES (?, 'credit', 0.00, 'manual', 'approved', ?, 0.00, ?, NOW())"
                );
                
                $metadata = [
                    'admin_id' => $_SESSION['user_id'] ?? null,
                    'action' => 'wallet_created',
                    'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
                    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null,
                    'timestamp' => date('c')
                ];
                
                $logMessage("Wallet ledger metadata: " . json_encode($metadata));
                
                $txStmt->execute([
                    $walletId,  // wallet_id
                    json_encode($metadata),
                    $currentBalance  // balance_after
                ]);
                
                $logMessage("Wallet ledger entry created successfully");
                $this->pdo->commit();
                $logMessage("Transaction committed successfully");
                
            } catch (\Exception $e) {
                $this->pdo->rollBack();
                $errorMsg = "Error logging wallet creation: " . $e->getMessage();
                $logMessage("ERROR: $errorMsg");
                throw new \Exception('Wallet was created but could not be logged. Please contact support.');
            }
            
            // Log success
            $logMessage("Wallet creation process completed successfully for agent ID: $agentId, Wallet ID: $walletId");
            
            if ($isAjax) {
                $response = [
                    'success' => true,
                    'message' => 'Wallet created successfully',
                    'wallet_id' => $walletId
                ];
                $logMessage("Sending success response: " . json_encode($response));
                
                header('Content-Type: application/json');
                echo json_encode($response);
                return;
            }
            
            $_SESSION['flash'] = 'Wallet created successfully';
            $logMessage("Redirecting to agents list");
            
        } catch (\Throwable $e) {
            // Log detailed error information
            $errorDetails = [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'agent_id' => $agentId ?? 'unknown',
                'admin_id' => $_SESSION['user_id'] ?? 'unknown',
                'timestamp' => date('c')
            ];
            
            $logMessage("ERROR DETAILS: " . json_encode($errorDetails, JSON_PRETTY_PRINT));
            
            if ($this->pdo->inTransaction()) {
                $logMessage("Rolling back transaction due to error");
                $this->pdo->rollBack();
            }
            
            $errorMessage = 'Error: ' . $e->getMessage();
            $logMessage("Error occurred: $errorMessage");
            
            if ($isAjax) {
                $response = [
                    'success' => false,
                    'message' => $e->getMessage()
                ];
                $logMessage("Sending error response: " . json_encode($response));
                
                http_response_code(400);
                header('Content-Type: application/json');
                echo json_encode($response);
                return;
            }
            
            $_SESSION['flash'] = $errorMessage;
        }
        
        if (!$isAjax) {
            $this->redirect('/admin/management/agents');
        }
    }

    // Ensure Agent Settings schema (runtime safe)
    private function ensureAgentSettingsSchema(): void
    {
        // Agent settings holder
        $sql = [
            "CREATE TABLE IF NOT EXISTS agent_settings (\n  user_id INT PRIMARY KEY,\n  allow_wallet_transfer TINYINT(1) NOT NULL DEFAULT 0,\n  require_tx_password TINYINT(1) NOT NULL DEFAULT 1,\n  wallet_transfer_daily_limit DECIMAL(12,2) NULL,\n  api_enabled TINYINT(1) NOT NULL DEFAULT 0,\n  allow_extra_margin TINYINT(1) NOT NULL DEFAULT 0,\n  max_margin_percent DECIMAL(5,2) NULL,\n  max_flat_markup DECIMAL(10,2) NULL,\n  wallet_pay_enabled TINYINT(1) NOT NULL DEFAULT 1,\n  gateway_booking_enabled TINYINT(1) NOT NULL DEFAULT 1,\n  api_key VARCHAR(64) NULL,\n  api_secret_hash VARCHAR(255) NULL,\n  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n  updated_at TIMESTAMP NULL,\n  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE\n) ENGINE=InnoDB"
        ];
        foreach ($sql as $q) { try { $this->pdo->exec($q); } catch (\Throwable $e) { /* ignore */ } }
        // Backfill new columns if table already existed
        try { $this->pdo->exec("ALTER TABLE agent_settings ADD COLUMN wallet_pay_enabled TINYINT(1) NOT NULL DEFAULT 1"); } catch (\Throwable $e) { /* likely exists */ }
        try { $this->pdo->exec("ALTER TABLE agent_settings ADD COLUMN gateway_booking_enabled TINYINT(1) NOT NULL DEFAULT 1"); } catch (\Throwable $e) { /* likely exists */ }
        try { $this->pdo->exec("ALTER TABLE agent_settings ADD COLUMN api_key VARCHAR(64) NULL"); } catch (\Throwable $e) { /* likely exists */ }
        try { $this->pdo->exec("ALTER TABLE agent_settings ADD COLUMN api_secret_hash VARCHAR(255) NULL"); } catch (\Throwable $e) { /* likely exists */ }
        // TX password/PIN columns on users
        $alter = [
            "ALTER TABLE users ADD COLUMN transaction_password_hash VARCHAR(255) NULL",
            "ALTER TABLE users ADD COLUMN tx_pin_enc VARBINARY(255) NULL",
            "ALTER TABLE users ADD COLUMN tx_pin_iv VARBINARY(16) NULL",
            "ALTER TABLE users ADD COLUMN tx_pin_tag VARBINARY(16) NULL",
        ];
        foreach ($alter as $q) { try { $this->pdo->exec($q); } catch (\Throwable $e) { /* likely exists */ } }
    }

    // ===== Agent Settings (full controls) =====
    public function agentSettings(): void
    {
        Auth::requireRole(['Admin']);
        $this->ensureAgentSettingsSchema();
        header('Content-Type: application/json');
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { http_response_code(400); echo json_encode(['error' => 'Invalid agent id']); return; }
        try {
            // Ensure agent exists
            $chk = $this->pdo->prepare("SELECT id FROM users WHERE id = ? AND role = 'B2B Agent' LIMIT 1");
            $chk->execute([$id]);
            if (!$chk->fetch(\PDO::FETCH_ASSOC)) { http_response_code(404); echo json_encode(['error'=>'Agent not found']); return; }
            $q = $this->pdo->prepare('SELECT commission_percent FROM agent_profiles WHERE user_id = ? LIMIT 1');
            $q->execute([$id]);
            $row = $q->fetch(\PDO::FETCH_ASSOC) ?: null;
            // Settings row
            $ss = $this->pdo->prepare('SELECT allow_wallet_transfer, require_tx_password, wallet_transfer_daily_limit, api_enabled, allow_extra_margin, max_margin_percent, max_flat_markup, wallet_pay_enabled, gateway_booking_enabled, api_key FROM agent_settings WHERE user_id = ? LIMIT 1');
            $ss->execute([$id]);
            $s = $ss->fetch(\PDO::FETCH_ASSOC) ?: [];
            $defaults = [
                'allow_wallet_transfer' => 0,
                'require_tx_password' => 1,
                'wallet_transfer_daily_limit' => null,
                'api_enabled' => 0,
                'allow_extra_margin' => 0,
                'max_margin_percent' => null,
                'max_flat_markup' => null,
                'wallet_pay_enabled' => 1,
                'gateway_booking_enabled' => 1,
                'api_key' => null,
            ];
            // TX password presence
            $tp = $this->pdo->prepare('SELECT transaction_password_hash FROM users WHERE id = ? LIMIT 1');
            $tp->execute([$id]);
            $tu = $tp->fetch(\PDO::FETCH_ASSOC) ?: [];
            $hasTx = !empty($tu['transaction_password_hash']);
            $resp = array_merge($defaults, $s);
            $resp['commission_percent'] = isset($row['commission_percent']) ? (float)$row['commission_percent'] : null;
            $resp['tx_password_set'] = $hasTx ? 1 : 0;
            echo json_encode($resp);
        } catch (\Throwable $e) {
            http_response_code(500); echo json_encode(['error' => 'Failed to load settings']);
        }
    }

    public function agentSettingsSave(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        // Protect with master password for sensitive changes
        try { Security::requireMasterPassword(); } catch (\Throwable $e) { /* allow if not configured */ }

        $id = (int)($_POST['id'] ?? 0);
        $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
        $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
        $isAjax = (stripos($accept, 'application/json') !== false) || (strcasecmp($xhr, 'XMLHttpRequest') === 0);
        if ($id <= 0) {
            if ($isAjax) { http_response_code(400); echo json_encode(['ok'=>false,'error'=>'Invalid agent id']); return; }
            $this->redirect('/admin/management/agents');
        }
        $this->ensureAgentSettingsSchema();
        // Commission percent
        $commission = trim((string)($_POST['commission_percent'] ?? ''));
        $cp = ($commission === '' || !is_numeric($commission)) ? null : (float)$commission;
        // agent_profiles.commission_percent is NOT NULL in schema; avoid inserting NULL
        $cpIns = ($cp === null) ? 0.0 : $cp;
        // Toggles
        $b = function($key){ $v = $_POST[$key] ?? null; return ($v === '1' || $v === 1 || $v === 'on' || $v === 'true'); };
        $allowWallet = $b('allow_wallet_transfer') ? 1 : 0;
        $requireTx = $b('require_tx_password') ? 1 : 0;
        $apiEnabled = $b('api_enabled') ? 1 : 0;
        $allowExtraMargin = $b('allow_extra_margin') ? 1 : 0;
        $walletPayEnabled = $b('wallet_pay_enabled') ? 1 : 0;
        $gatewayBooking = $b('gateway_booking_enabled') ? 1 : 0;
        // Numerics
        $dailyLimit = $_POST['wallet_transfer_daily_limit'] ?? '';
        $dailyLimit = ($dailyLimit === '' || !is_numeric($dailyLimit)) ? null : (float)$dailyLimit;
        $maxMarginPct = $_POST['max_margin_percent'] ?? '';
        $maxMarginPct = ($maxMarginPct === '' || !is_numeric($maxMarginPct)) ? null : (float)$maxMarginPct;
        $maxFlat = $_POST['max_flat_markup'] ?? '';
        $maxFlat = ($maxFlat === '' || !is_numeric($maxFlat)) ? null : (float)$maxFlat;
        // Enforce mutual exclusivity between percent and flat per requested rule
        $markupType = (string)($_POST['markup_type'] ?? ''); // 'percent' | 'flat'
        if ($markupType === 'percent') { $maxFlat = null; }
        elseif ($markupType === 'flat') { $maxMarginPct = null; }
        else {
            // If both provided, prefer percent; else keep whichever is set
            if ($maxMarginPct !== null && $maxFlat !== null) { $maxFlat = null; }
        }
        // Optional TX PIN (6 digits)
        $txPassword = (string)($_POST['transaction_password'] ?? '');

        try {
            // Ensure agent exists
            $chk = $this->pdo->prepare("SELECT id FROM users WHERE id = ? AND role = 'B2B Agent' LIMIT 1");
            $chk->execute([$id]);
            if (!$chk->fetch(\PDO::FETCH_ASSOC)) { throw new \RuntimeException('Agent not found'); }
            // Validate inputs
            $errs = [];
            if ($maxMarginPct !== null && ($maxMarginPct < 0 || $maxMarginPct > 100)) { $errs[] = 'Max margin percent must be between 0 and 100.'; }
            if ($dailyLimit !== null && $dailyLimit < 0) { $errs[] = 'Daily transfer limit cannot be negative.'; }
            if ($txPassword !== '' && !preg_match('/^\d{6}$/', $txPassword)) { $errs[] = 'Transaction PIN must be exactly 6 digits.'; }
            if (!empty($errs)) {
                if ($isAjax) { http_response_code(422); echo json_encode(['ok'=>false,'errors'=>$errs]); return; }
                $_SESSION['errors'] = $errs; $this->redirect('/admin/management/agents');
            }

            $this->pdo->beginTransaction();
            // Upsert commission in agent_profiles
            $ins = $this->pdo->prepare("INSERT INTO agent_profiles (user_id, commission_percent) VALUES (?, ?) ON DUPLICATE KEY UPDATE commission_percent = VALUES(commission_percent)");
            $ins->execute([$id, $cpIns]);
            // Upsert advanced settings
            $ins2 = $this->pdo->prepare("INSERT INTO agent_settings (user_id, allow_wallet_transfer, require_tx_password, wallet_transfer_daily_limit, api_enabled, allow_extra_margin, max_margin_percent, max_flat_markup, wallet_pay_enabled, gateway_booking_enabled) VALUES (?,?,?,?,?,?,?,?,?,?) ON DUPLICATE KEY UPDATE allow_wallet_transfer=VALUES(allow_wallet_transfer), require_tx_password=VALUES(require_tx_password), wallet_transfer_daily_limit=VALUES(wallet_transfer_daily_limit), api_enabled=VALUES(api_enabled), allow_extra_margin=VALUES(allow_extra_margin), max_margin_percent=VALUES(max_margin_percent), max_flat_markup=VALUES(max_flat_markup), wallet_pay_enabled=VALUES(wallet_pay_enabled), gateway_booking_enabled=VALUES(gateway_booking_enabled), updated_at=NOW()");
            $ins2->execute([$id, $allowWallet, $requireTx, $dailyLimit, $apiEnabled, $allowExtraMargin, $maxMarginPct, $maxFlat, $walletPayEnabled, $gatewayBooking]);
            // Update TX password/PIN if provided
            if ($txPassword !== '') {
                $hash = password_hash($txPassword, PASSWORD_BCRYPT);
                $up = $this->pdo->prepare('UPDATE users SET transaction_password_hash = ? WHERE id = ?');
                $up->execute([$hash, $id]);
                // Encrypt and store PIN for optional view
                try {
                    [$enc, $iv, $tag] = $this->encryptTxPin($txPassword);
                    $up2 = $this->pdo->prepare('UPDATE users SET tx_pin_enc = ?, tx_pin_iv = ?, tx_pin_tag = ? WHERE id = ?');
                    $up2->execute([$enc, $iv, $tag, $id]);
                } catch (\Throwable $e) {
                    // Keep hash only; do not block
                }
            }
            $this->pdo->commit();
            if ($isAjax) { echo json_encode(['ok'=>true,'message'=>'Saved']); return; }
            $_SESSION['flash'] = 'Agent settings saved.';
        } catch (\Throwable $e) {
            if ($this->pdo->inTransaction()) { $this->pdo->rollBack(); }
            if ($isAjax) { http_response_code(500); echo json_encode(['ok'=>false,'error'=>'Failed to save settings']); return; }
            $_SESSION['flash'] = 'Failed to save agent settings.';
        }
        $this->redirect('/admin/management/agents');
    }

    public function agentSettingsViewPin(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $this->ensureAgentSettingsSchema();
        header('Content-Type: application/json');

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { http_response_code(400); echo json_encode(['error' => 'Invalid agent id']); return; }
        try {
            $q = $this->pdo->prepare("SELECT tx_pin_enc, tx_pin_iv, tx_pin_tag FROM users WHERE id = ? AND role = 'B2B Agent' LIMIT 1");
            $q->execute([$id]);
            $row = $q->fetch(\PDO::FETCH_ASSOC);
            if (!$row || empty($row['tx_pin_enc']) || empty($row['tx_pin_iv']) || empty($row['tx_pin_tag'])) {
                http_response_code(404); echo json_encode(['error' => 'PIN not set']); return;
            }
            $pin = $this->decryptTxPin($row['tx_pin_enc'], $row['tx_pin_iv'], $row['tx_pin_tag']);
            if (!preg_match('/^\\d{6}$/', $pin)) { http_response_code(500); echo json_encode(['error' => 'Corrupt PIN']); return; }
            echo json_encode(['pin' => $pin]);
        } catch (\Throwable $e) {
            http_response_code(500);
            $msg = $e->getMessage();
            if (strpos($msg, 'TX_PIN_KEY') !== false) {
                echo json_encode(['error' => 'Encryption key missing or invalid.']);
            } else {
                echo json_encode(['error' => 'Failed to read PIN']);
            }
        }
    }

    public function agentResetPassword(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        // Require master password for safety
        try { Security::requireMasterPassword(); } catch (\Throwable $e) { /* allow if not configured */ }

        $id = (int)($_POST['id'] ?? 0);
        $newPassword = (string)($_POST['new_password'] ?? '');
        if ($id <= 0) { $this->redirect('/admin/management/agents'); }
        if ($newPassword === '' || strlen($newPassword) < 6) {
            $_SESSION['flash'] = 'Password must be at least 6 characters.';
            $this->redirect('/admin/management/agents');
        }
        try {
            $u = $this->pdo->prepare("SELECT id FROM users WHERE id = ? AND role = 'B2B Agent' LIMIT 1");
            $u->execute([$id]);
            $user = $u->fetch(\PDO::FETCH_ASSOC);
            if (!$user) { $_SESSION['flash'] = 'Agent not found.'; $this->redirect('/admin/management/agents'); }
            $hash = password_hash($newPassword, PASSWORD_BCRYPT);
            $up = $this->pdo->prepare('UPDATE users SET password = ? WHERE id = ?');
            $up->execute([$hash, $id]);
            $_SESSION['flash'] = 'Agent password updated.';
        } catch (\Throwable $e) {
            $_SESSION['flash'] = 'Failed to update password.';
        }
        $this->redirect('/admin/management/agents');
    }

    public function partnerView(): void
    {
        Auth::requireRole(['Admin']);
        $this->ensureChannelPartnerSchema();
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/management/partners'); }
        $stmt = $this->pdo->prepare("SELECT cp.*, COALESCE(cpw.balance,0) AS wallet_balance
                                  FROM channel_partners cp
                                  LEFT JOIN channel_partner_wallet cpw ON cpw.partner_id = cp.id
                                  WHERE cp.id = ? LIMIT 1");
        $stmt->execute([$id]);
        $partner = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$partner) { $this->redirect('/admin/management/partners'); }
        $this->view('admin/management_partners_view', compact('partner'));
    }

    public function partnerWallet(): void
    {
        Auth::requireRole(['Admin']);
        $this->ensureChannelPartnerSchema();
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/management/partners'); }
        // Partner + wallet id/balance
        $stmt = $this->pdo->prepare("SELECT cp.*, cpw.id AS wallet_id, COALESCE(cpw.balance,0) AS wallet_balance
                                     FROM channel_partners cp
                                     LEFT JOIN channel_partner_wallet cpw ON cpw.partner_id = cp.id
                                     WHERE cp.id = ? LIMIT 1");
        $stmt->execute([$id]);
        $partner = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$partner) { $this->redirect('/admin/management/partners'); }
        $walletId = (int)($partner['wallet_id'] ?? 0);
        if ($walletId <= 0) { $this->view('admin/management_partners_wallet', ['partner' => $partner, 'entries' => []]); return; }

        // Ledger entries for this partner wallet
        $stmtL = $this->pdo->prepare("SELECT l.id, l.type, l.amount, l.status, l.meta, l.created_at
                                      FROM channel_partner_wallet_ledger l
                                      WHERE l.wallet_id = ?
                                      ORDER BY l.created_at DESC, l.id DESC");
        $stmtL->execute([$walletId]);
        $rows = $stmtL->fetchAll(\PDO::FETCH_ASSOC) ?: [];

        $entries = [];
        foreach ($rows as $row) {
            $meta = [];
            if (!empty($row['meta'])) {
                try { $meta = json_decode((string)$row['meta'], true) ?: []; } catch (\Throwable $e) { $meta = []; }
            }
            $flow = (string)($meta['flow'] ?? '');
            $amount = (float)$row['amount'];
            $status = (string)$row['status'];
            $createdAt = (string)$row['created_at'];
            $note = trim((string)($meta['reason'] ?? $meta['note'] ?? $meta['memo'] ?? ''));

            $source = '';
            $target = '';

            if ($flow === 'admin_to_partner') {
                $source = 'Admin';
                $target = $partner['name'] ?? 'Partner';
            } elseif ($flow === 'reverse_admin_to_partner') {
                // Partner returns funds to Admin
                $source = $partner['name'] ?? 'Partner';
                $target = 'Admin';
            } elseif ($flow === 'partner_to_agent') {
                $source = $partner['name'] ?? 'Partner';
                // agent name
                $agentId = (int)($meta['to_user_id'] ?? $meta['agent_user_id'] ?? 0);
                if ($agentId > 0) {
                    try {
                        $su = $this->pdo->prepare('SELECT name, email FROM users WHERE id = ? LIMIT 1');
                        $su->execute([$agentId]);
                        if ($u = $su->fetch(\PDO::FETCH_ASSOC)) { $target = ($u['name'] ?? 'Agent') . ' (' . ($u['email'] ?? '') . ')'; }
                    } catch (\Throwable $e) { /* ignore */ }
                }
                if ($target === '') { $target = 'Agent'; }
            } elseif ($flow === 'reverse_partner_to_agent') {
                // Agent returns funds to Partner
                $target = $partner['name'] ?? 'Partner';
                $agentId = (int)($meta['agent_user_id'] ?? $meta['from_user_id'] ?? 0);
                if ($agentId > 0) {
                    try {
                        $su = $this->pdo->prepare('SELECT name, email FROM users WHERE id = ? LIMIT 1');
                        $su->execute([$agentId]);
                        if ($u = $su->fetch(\PDO::FETCH_ASSOC)) { $source = ($u['name'] ?? 'Agent') . ' (' . ($u['email'] ?? '') . ')'; }
                    } catch (\Throwable $e) { /* ignore */ }
                }
                if ($source === '') { $source = 'Agent'; }
            } else {
                // Unknown/seed/manual
                $source = ($row['type'] === 'credit') ? 'System' : ($partner['name'] ?? 'Partner');
                $target = ($row['type'] === 'credit') ? ($partner['name'] ?? 'Partner') : 'System';
                if ($flow === '') { $flow = 'seed'; }
            }

            $entries[] = [
                'created_at' => $createdAt,
                'flow' => $flow,
                'source' => $source,
                'target' => $target,
                'amount' => $amount,
                'status' => $status,
                'note' => $note,
            ];
        }

        $this->view('admin/management_partners_wallet', [
            'partner' => $partner,
            'entries' => $entries,
        ]);
    }

    public function partnerAgents(): void
    {
        Auth::requireRole(['Admin']);
        $this->ensureChannelPartnerSchema();
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/management/partners'); }
        // Load partner basic
        $stmt = $this->pdo->prepare("SELECT cp.*, COALESCE(cpw.balance,0) AS wallet_balance
                                     FROM channel_partners cp
                                     LEFT JOIN channel_partner_wallet cpw ON cpw.partner_id = cp.id
                                     WHERE cp.id = ? LIMIT 1");
        $stmt->execute([$id]);
        $partner = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$partner) { $this->redirect('/admin/management/partners'); }

        // Agents mapped to this partner
        try {
            $sa = $this->pdo->prepare("SELECT u.id, u.name, u.email, u.status, COALESCE(w.balance,0) AS wallet_balance
                                       FROM channel_partner_agents cpa
                                       JOIN users u ON u.id = cpa.agent_user_id
                                       LEFT JOIN wallets w ON w.user_id = u.id
                                       WHERE cpa.partner_id = ?
                                       ORDER BY u.id DESC");
            $sa->execute([$id]);
            $agents = $sa->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $e) {
            $agents = [];
        }

        $csrf = Security::csrfToken();
        $this->view('admin/management_partners_agents', compact('partner','agents','csrf'));
    }

    public function partnerEdit(): void
    {
        Auth::requireRole(['Admin']);
        $this->ensureChannelPartnerSchema();
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/management/partners'); }
        $stmt = $this->pdo->prepare("SELECT * FROM channel_partners WHERE id = ? LIMIT 1");
        $stmt->execute([$id]);
        $partner = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$partner) { $this->redirect('/admin/management/partners'); }
        $csrf = Security::csrfToken();
        $this->view('admin/management_partners_edit', compact('partner','csrf'));
    }

    public function partnerImpersonate(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        // Require master password for safety
        Security::requireMasterPassword();

        $partnerId = (int)($_POST['id'] ?? 0);
        if ($partnerId <= 0) { $this->redirect('/admin/management/partners'); }
        $this->ensureChannelPartnerSchema();

        // Fetch partner and deduce user via email & role
        $stmt = $this->pdo->prepare('SELECT id, name, contact_email FROM channel_partners WHERE id = ? LIMIT 1');
        $stmt->execute([$partnerId]);
        $partner = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$partner) { $_SESSION['flash'] = 'Partner not found'; $this->redirect('/admin/management/partners'); }

        $email = (string)($partner['contact_email'] ?? '');
        $su = $this->pdo->prepare("SELECT id, name, email, role, status FROM users WHERE email = ? AND role = 'Channel Partner' LIMIT 1");
        $su->execute([$email]);
        $u = $su->fetch(\PDO::FETCH_ASSOC);
        if (!$u) { $_SESSION['flash'] = 'No Channel Partner user linked to this email'; $this->redirect('/admin/management/partners'); }
        if (($u['status'] ?? '') !== 'Active') { $_SESSION['flash'] = 'User is not active'; $this->redirect('/admin/management/partners'); }

        // Save current admin session to restore later
        if (empty($_SESSION['impersonator'])) {
            $_SESSION['impersonator'] = $_SESSION['user'] ?? null;
        }
        $_SESSION['impersonating'] = true;
        $_SESSION['impersonating_as_partner_id'] = $partnerId;
        $_SESSION['user'] = [
            'id' => (int)$u['id'],
            'name' => (string)$u['name'],
            'email' => (string)$u['email'],
            'role' => 'Channel Partner',
        ];
        $_SESSION['flash'] = 'Now logged in as Channel Partner: ' . ($partner['name'] ?? '');
        // Redirect to partner dashboard/home if exists, else to root
        $this->redirect('/');
    }

    public function stopImpersonation(): void
    {
        // Allow both Admin and impersonated sessions to access this endpoint
        if (!empty($_SESSION['impersonator'])) {
            $_SESSION['user'] = $_SESSION['impersonator'];
        }
        unset($_SESSION['impersonating'], $_SESSION['impersonator'], $_SESSION['impersonating_as_partner_id']);
        $_SESSION['flash'] = 'Impersonation ended. You are now back as Admin.';
        $this->redirect('/admin');
    }

    public function partnerResetPassword(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $partnerId = (int)($_POST['id'] ?? 0);
        $newPassword = (string)($_POST['new_password'] ?? '');
        if ($partnerId <= 0) { $this->redirect('/admin/management/partners'); }
        if ($newPassword === '' || strlen($newPassword) < 6) {
            $_SESSION['flash'] = 'Password must be at least 6 characters.';
            $this->redirect('/admin/management/partners');
        }

        $this->ensureChannelPartnerSchema();
        try {
            // Find partner and linked user by email with role Channel Partner
            $sp = $this->pdo->prepare('SELECT name, contact_email FROM channel_partners WHERE id = ? LIMIT 1');
            $sp->execute([$partnerId]);
            $partner = $sp->fetch(\PDO::FETCH_ASSOC);
            if (!$partner) { $_SESSION['flash'] = 'Partner not found'; $this->redirect('/admin/management/partners'); }
            $email = (string)($partner['contact_email'] ?? '');
            $su = $this->pdo->prepare("SELECT id FROM users WHERE email = ? AND role = 'Channel Partner' LIMIT 1");
            $su->execute([$email]);
            $user = $su->fetch(\PDO::FETCH_ASSOC);
            if (!$user) { $_SESSION['flash'] = 'Linked Channel Partner user not found'; $this->redirect('/admin/management/partners'); }

            $hash = password_hash($newPassword, PASSWORD_BCRYPT);
            $up = $this->pdo->prepare('UPDATE users SET password = ? WHERE id = ?');
            $up->execute([$hash, (int)$user['id']]);
            $_SESSION['flash'] = 'Password updated for ' . ($partner['name'] ?? 'Partner') . '.';
        } catch (\Throwable $e) {
            $_SESSION['flash'] = 'Failed to update password.';
        }
        $this->redirect('/admin/management/partners');
    }

    public function partnerUpdate(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $this->ensureChannelPartnerSchema();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/management/partners'); }
        $name = trim((string)($_POST['name'] ?? ''));
        $email = trim((string)($_POST['contact_email'] ?? ''));
        $contactPerson = trim((string)($_POST['contact_person'] ?? ''));
        $contactMobile = trim((string)($_POST['contact_mobile'] ?? ''));
        $contactWhatsApp = trim((string)($_POST['contact_whatsapp'] ?? ''));
        $contactLine = trim((string)($_POST['contact_line'] ?? ''));
        $addressLine = trim((string)($_POST['address_line'] ?? ''));
        $city = trim((string)($_POST['city'] ?? ''));
        $state = trim((string)($_POST['state'] ?? ''));
        $pincode = trim((string)($_POST['pincode'] ?? ''));
        $country = trim((string)($_POST['country'] ?? ''));
        $errors = [];
        if ($name === '') { $errors[] = 'Partner name is required.'; }
        if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) { $errors[] = 'Valid email is required.'; }
        if ($addressLine === '') { $errors[] = 'Address is required.'; }
        if ($city === '') { $errors[] = 'City is required.'; }
        if ($country === '') { $errors[] = 'Country is required.'; }
        if (!empty($errors)) { $_SESSION['errors'] = $errors; $this->redirect('/admin/management/partners/edit?id=' . $id); }
        $stmt = $this->pdo->prepare("UPDATE channel_partners SET name=?, contact_email=?, contact_person=?, contact_mobile=?, contact_whatsapp=?, contact_line=?, address_line=?, city=?, state=?, pincode=?, country=? WHERE id=?");
        $stmt->execute([
            $name,
            $email !== '' ? $email : null,
            $contactPerson !== '' ? $contactPerson : null,
            $contactMobile !== '' ? $contactMobile : null,
            $contactWhatsApp !== '' ? $contactWhatsApp : null,
            $contactLine !== '' ? $contactLine : null,
            $addressLine,
            $city,
            $state !== '' ? $state : null,
            $pincode !== '' ? $pincode : null,
            $country,
            $id
        ]);
        $_SESSION['flash'] = 'Channel Partner updated.';
        $this->redirect('/admin/management/partners');
    }

    public function partnerToggleStatus(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $this->ensureChannelPartnerSchema();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/management/partners'); }
        $stmt = $this->pdo->prepare("SELECT status FROM channel_partners WHERE id = ? LIMIT 1");
        $stmt->execute([$id]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        if ($row) {
            $next = ($row['status'] === 'Active') ? 'Inactive' : 'Active';
            $up = $this->pdo->prepare("UPDATE channel_partners SET status = ? WHERE id = ?");
            $up->execute([$next, $id]);
            $_SESSION['flash'] = 'Channel Partner status updated to ' . $next . '.';
        }
        $this->redirect('/admin/management/partners');
    }
}
