<?php
namespace App\Controllers;

use App\Core\Auth;
use App\Core\Controller;
use App\Core\Security;

class KycController extends Controller
{
    /**
     * Load master encryption key for KYC docs from env (base64-encoded 32 bytes).
     */
    private function getKycEncKey(): string
    {
        // Use global env() helper which reads .env
        $keyB64 = env('KYC_ENC_KEY', '');
        if ($keyB64 === '') { throw new \RuntimeException('KYC_ENC_KEY missing'); }
        $key = base64_decode($keyB64, true);
        if ($key === false || strlen($key) !== 32) { throw new \RuntimeException('Invalid KYC_ENC_KEY'); }
        return $key;
    }

    /**
     * Decrypt and stream a private stored document to the browser.
     * $storageRel must start with 'storage/'.
     */
    private function streamDecrypted(string $storageRel, ?string $downloadName = null, ?string $fallbackMime = 'application/octet-stream'): void
    {
        // Normalize and validate path: allow optional leading slash
        $storageRel = ltrim($storageRel, "/\\");
        if (strpos($storageRel, 'storage/') !== 0) { http_response_code(400); echo 'Bad path'; return; }
        $root = dirname(__DIR__, 2);
        $abs = $root . DIRECTORY_SEPARATOR . str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $storageRel);
        // Ensure resolved path remains under storage directory
        $storageRoot = $root . DIRECTORY_SEPARATOR . 'storage' . DIRECTORY_SEPARATOR;
        $absReal = realpath(dirname($abs));
        if ($absReal === false || strpos($absReal . DIRECTORY_SEPARATOR, $storageRoot) !== 0) { http_response_code(400); echo 'Bad path'; return; }
        if (!is_file($abs)) { http_response_code(404); echo 'Not found'; return; }
        // meta alongside: prefer <name>.meta.json, fallback to legacy <name>.enc.meta.json
        $metaAbs = preg_replace('/\.enc$/', '.meta.json', $abs);
        if (!is_file($metaAbs)) {
            $altMeta = $abs . '.meta.json';
            if (is_file($altMeta)) { $metaAbs = $altMeta; } else { http_response_code(404); echo 'Meta missing'; return; }
        }
        $meta = json_decode((string)file_get_contents($metaAbs), true) ?: [];
        $iv = base64_decode($meta['iv'] ?? '', true);
        $tag = base64_decode($meta['tag'] ?? '', true);
        if (!$iv || !$tag) { http_response_code(500); echo 'Meta invalid'; return; }
        $cipher = file_get_contents($abs);
        if ($cipher === false) { http_response_code(500); echo 'Read error'; return; }
        $plain = openssl_decrypt($cipher, 'aes-256-gcm', $this->getKycEncKey(), OPENSSL_RAW_DATA, $iv, $tag);
        if ($plain === false) { http_response_code(500); echo 'Decrypt error'; return; }
        $mime = $meta['mime'] ?? $fallbackMime ?? 'application/octet-stream';
        $name = $downloadName ?: ($meta['original_name'] ?? 'document');
        $download = isset($_GET['download']) && (string)$_GET['download'] === '1';
        header('X-Content-Type-Options: nosniff');
        header('Content-Type: ' . $mime);
        header('Content-Disposition: ' . ($download ? 'attachment' : 'inline') . '; filename="' . addslashes($name) . '"');
        header('Content-Length: ' . strlen($plain));
        $this->logAccess('stream', [
            'path' => $storageRel,
            'download' => $download,
            'mime' => $mime,
            'name' => $name,
        ]);
        echo $plain;
        exit;
    }
    public function index(): void
    {
        Auth::requireRole(['Admin']);
        // List all KYC entries, pending first, include agent country
        $stmt = $this->pdo->query("SELECT k.id, k.user_id, u.name, u.email, ap.country, k.id_type, k.id_number, k.status, k.created_at
                                   FROM agent_kyc k
                                   JOIN users u ON u.id = k.user_id
                                   LEFT JOIN agent_profiles ap ON ap.user_id = k.user_id
                                   ORDER BY FIELD(k.status,'pending','rejected','approved'), k.created_at DESC");
        $kycs = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        $this->view('admin/kyc_index', compact('kycs'));
    }

    public function show(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { http_response_code(404); echo 'KYC not found'; return; }
        // Load KYC plus user and profile details
        $stmt = $this->pdo->prepare("SELECT k.*, u.name, u.email,
                                            ap.company, ap.phone, ap.business_type, ap.agency_name,
                                            ap.country, ap.state, ap.city, ap.pincode, ap.address_line,
                                            ap.gst_number, ap.gst_company, ap.iata_registered, ap.iata_code
                                     FROM agent_kyc k
                                     JOIN users u ON u.id = k.user_id
                                     LEFT JOIN agent_profiles ap ON ap.user_id = k.user_id
                                     WHERE k.id = :id");
        $stmt->execute(['id' => $id]);
        $kyc = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$kyc) { http_response_code(404); echo 'KYC not found'; return; }
        // Load business documents for this user (if any)
        $docs = [];
        if (!empty($kyc['user_id'])) {
            $d = $this->pdo->prepare("SELECT id, doc_type, file_path, created_at FROM agent_business_docs WHERE user_id = :uid ORDER BY id DESC");
            $d->execute(['uid' => (int)$kyc['user_id']]);
            $docs = $d->fetchAll(\PDO::FETCH_ASSOC);
        }
        $csrf = Security::csrfToken();
        $this->view('admin/kyc_show', compact('kyc','csrf','docs'));
    }

    public function approve(): void
    {
        Auth::requireRole(['Admin']);
        if (!Security::verifyCsrf($_POST['csrf'] ?? '')) { http_response_code(400); echo 'CSRF'; return; }
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/kyc'); }
        $stmt = $this->pdo->prepare("UPDATE agent_kyc SET status = 'approved', remarks = NULL, verified_at = NOW() WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $_SESSION['flash'] = 'KYC approved.';
        $this->redirect('/admin/kyc');
    }

    public function reject(): void
    {
        Auth::requireRole(['Admin']);
        if (!Security::verifyCsrf($_POST['csrf'] ?? '')) { http_response_code(400); echo 'CSRF'; return; }
        $id = (int)($_POST['id'] ?? 0);
        $remarks = trim($_POST['remarks'] ?? '');
        if ($id <= 0) { $this->redirect('/admin/kyc'); }
        $stmt = $this->pdo->prepare("UPDATE agent_kyc SET status = 'rejected', remarks = :remarks WHERE id = :id");
        $stmt->execute(['id' => $id, 'remarks' => $remarks !== '' ? $remarks : null]);
        $_SESSION['flash'] = 'KYC rejected.';
        $this->redirect('/admin/kyc');
    }

    // Secure streaming endpoints
    public function file(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0);
        $side = ($_GET['side'] ?? 'front') === 'back' ? 'back' : 'front';
        if ($id <= 0) { http_response_code(404); echo 'Not found'; return; }
        $col = $side === 'front' ? 'doc_front_path' : 'doc_back_path';
        $stmt = $this->pdo->prepare("SELECT {$col} AS path FROM agent_kyc WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$row || empty($row['path'])) { http_response_code(404); echo 'File not found'; return; }
        $this->logAccess('lookup', ['type' => 'kyc', 'kyc_id' => $id, 'side' => $side, 'status' => 'found']);
        $this->streamDecrypted($row['path'], $side . '.document');
    }

    public function doc(): void
    {
        Auth::requireRole(['Admin']);
        $docId = (int)($_GET['id'] ?? 0);
        if ($docId <= 0) { http_response_code(404); echo 'Not found'; return; }
        $stmt = $this->pdo->prepare("SELECT doc_type, file_path FROM agent_business_docs WHERE id = :id");
        $stmt->execute(['id' => $docId]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$row || empty($row['file_path'])) { http_response_code(404); echo 'File not found'; return; }
        $name = ($row['doc_type'] ?? 'document') . '.file';
        $this->logAccess('lookup', ['type' => 'business_doc', 'doc_id' => $docId, 'status' => 'found']);
        $this->streamDecrypted($row['file_path'], $name);
    }

    /**
     * Append a structured log line for auditing access to encrypted files.
     */
    private function logAccess(string $action, array $context = []): void
    {
        try {
            $root = dirname(__DIR__, 2);
            $dir = $root . DIRECTORY_SEPARATOR . 'storage' . DIRECTORY_SEPARATOR . 'logs';
            if (!is_dir($dir)) { @mkdir($dir, 0700, true); }
            $file = $dir . DIRECTORY_SEPARATOR . 'kyc_access.log';
            $user = $_SESSION['user'] ?? null;
            $entry = [
                'ts' => gmdate('c'),
                'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
                'endpoint' => $_SERVER['REQUEST_URI'] ?? null,
                'user_id' => $user['id'] ?? null,
                'role' => $user['role'] ?? null,
                'action' => $action,
                'context' => $context,
            ];
            @file_put_contents($file, json_encode($entry, JSON_UNESCAPED_SLASHES) . "\n", FILE_APPEND | LOCK_EX);
            @chmod($file, 0600);
        } catch (\Throwable $e) {
            // Swallow logging errors to not impact streaming
        }
    }
}
