<?php
namespace App\Controllers;

use App\Core\Auth;
use App\Core\Controller;
use App\Core\Security;
use PDO;

class HotelsAdminController extends Controller
{
    public function index(): void
    {
        Auth::requireRole(['Admin']);
        $q = trim($_GET['q'] ?? '');
        $country = trim($_GET['country'] ?? '');
        $city = trim($_GET['city'] ?? '');
        $sql = "SELECT h.*, v.name AS vendor,
                (SELECT file_path FROM hotel_images hi
                 WHERE hi.hotel_id = h.id
                 ORDER BY COALESCE(sort_order, 9999), id DESC
                 LIMIT 1) AS thumb
                FROM hotels h JOIN vendors v ON v.id = h.vendor_id WHERE 1";
        $params = [];
        if ($q !== '') { $sql .= " AND (h.name LIKE :q OR v.name LIKE :q)"; $params['q'] = "%$q%"; }
        if ($country !== '') { $sql .= " AND h.country = :country"; $params['country'] = $country; }
        if ($city !== '') { $sql .= " AND h.city = :city"; $params['city'] = $city; }
        $sql .= " ORDER BY h.id DESC";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $this->view('admin/hotels_index', [
            'title' => 'Hotels',
            'rows' => $rows,
            'csrf' => Security::csrfToken(),
            'q' => $q, 'country' => $country, 'city' => $city,
        ]);
    }

    public function create(): void
    {
        Auth::requireRole(['Admin']);
        // Vendors: prefer module='hotel'; fallback gracefully if column not present
        try {
            $vendors = $this->pdo->query("SELECT id, name, country, city FROM vendors WHERE module='hotel' ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
        } catch (\Throwable $e) {
            try { $vendors = $this->pdo->query("SELECT id, name, country, city FROM vendors ORDER BY name")->fetchAll(PDO::FETCH_ASSOC); }
            catch (\Throwable $e2) { $vendors = $this->pdo->query('SELECT id, name FROM vendors ORDER BY name')->fetchAll(PDO::FETCH_ASSOC); }
        }
        // Load countries and cities from locations
        $countries = $this->pdo->query("SELECT DISTINCT country FROM locations WHERE country <> '' ORDER BY country")->fetchAll(PDO::FETCH_COLUMN);
        $citiesStmt = $this->pdo->query("SELECT country, city FROM locations WHERE city IS NOT NULL AND city <> '' ORDER BY country, city");
        $cities = $citiesStmt->fetchAll(PDO::FETCH_ASSOC);
        $this->view('admin/hotels_form', [
            'title' => 'Add Hotel',
            'vendors' => $vendors,
            'hotel' => null,
            'csrf' => Security::csrfToken(),
            'rooms' => [],
            'rate_plans' => [],
            'tab' => $_GET['tab'] ?? 'details',
            'countries' => $countries,
            'cities' => $cities,
        ]);
    }

    public function store(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $stars = (int)($_POST['stars'] ?? 0);
        if ($stars < 0) $stars = 0; if ($stars > 5) $stars = 5;
        // New audience flags with safe defaults when not provided by form yet
        $visible_customer = isset($_POST['visible_customer']) ? ((($_POST['visible_customer']) ? 1 : 0)) : 1;
        $visible_agent = isset($_POST['visible_agent']) ? ((($_POST['visible_agent']) ? 1 : 0)) : 1;
        $featured_customer = isset($_POST['featured_customer']) ? ((($_POST['featured_customer']) ? 1 : 0)) : 0;
        $featured_agent = isset($_POST['featured_agent']) ? ((($_POST['featured_agent']) ? 1 : 0)) : 0;
        // Single featured label stored as agent label; used for both audiences
        $allowedLabels = ['top_choice','recommended','trusted_partner','best_value'];
        $agentLabel = trim($_POST['featured_agent_label'] ?? '');
        if ($featured_agent !== 1) { $agentLabel = null; }
        else if ($agentLabel === '' || !in_array($agentLabel, $allowedLabels, true)) { $agentLabel = 'recommended'; }
        try {
            // Prefer inserting with stars and audience flag columns if they exist
            $stmt = $this->pdo->prepare('INSERT INTO hotels (vendor_id, name, city, country, base_price, stars, visible_customer, visible_agent, featured_customer, featured_agent, featured_agent_label) VALUES (:vendor_id, :name, :city, :country, :base_price, :stars, :visible_customer, :visible_agent, :featured_customer, :featured_agent, :featured_agent_label)');
            $stmt->execute([
                'vendor_id' => (int)($_POST['vendor_id'] ?? 0),
                'name' => trim($_POST['name'] ?? ''),
                'city' => trim($_POST['city'] ?? ''),
                'country' => trim($_POST['country'] ?? ''),
                'base_price' => (float)($_POST['base_price'] ?? 0),
                'stars' => $stars,
                'visible_customer' => $visible_customer,
                'visible_agent' => $visible_agent,
                'featured_customer' => $featured_customer,
                'featured_agent' => $featured_agent,
                'featured_agent_label' => $agentLabel,
            ]);
        } catch (\Throwable $e) {
            try {
                // Fallback for environments with stars but without audience flags yet
                $stmt = $this->pdo->prepare('INSERT INTO hotels (vendor_id, name, city, country, base_price, stars) VALUES (:vendor_id, :name, :city, :country, :base_price, :stars)');
                $stmt->execute([
                    'vendor_id' => (int)($_POST['vendor_id'] ?? 0),
                    'name' => trim($_POST['name'] ?? ''),
                    'city' => trim($_POST['city'] ?? ''),
                    'country' => trim($_POST['country'] ?? ''),
                    'base_price' => (float)($_POST['base_price'] ?? 0),
                    'stars' => $stars,
                ]);
            } catch (\Throwable $e2) {
                // Fallback for environments without the stars column
                $stmt = $this->pdo->prepare('INSERT INTO hotels (vendor_id, name, city, country, base_price) VALUES (:vendor_id, :name, :city, :country, :base_price)');
                $stmt->execute([
                    'vendor_id' => (int)($_POST['vendor_id'] ?? 0),
                    'name' => trim($_POST['name'] ?? ''),
                    'city' => trim($_POST['city'] ?? ''),
                    'country' => trim($_POST['country'] ?? ''),
                    'base_price' => (float)($_POST['base_price'] ?? 0),
                ]);
            }
        }
        $_SESSION['flash'] = 'Hotel created.';
        $this->redirect('/admin/hotels');
    }

    public function edit(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0);
        $stmt = $this->pdo->prepare('SELECT * FROM hotels WHERE id = :id');
        $stmt->execute(['id' => $id]);
        $hotel = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$hotel) { $_SESSION['errors'][] = 'Hotel not found.'; $this->redirect('/admin/hotels'); return; }
        // Vendors: prefer module='hotel'; fallback gracefully if column not present
        try {
            $vendors = $this->pdo->query("SELECT id, name, country, city FROM vendors WHERE module='hotel' ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
        } catch (\Throwable $e) {
            try { $vendors = $this->pdo->query("SELECT id, name, country, city FROM vendors ORDER BY name")->fetchAll(PDO::FETCH_ASSOC); }
            catch (\Throwable $e2) { $vendors = $this->pdo->query('SELECT id, name FROM vendors ORDER BY name')->fetchAll(PDO::FETCH_ASSOC); }
        }
        // Load rooms and rate plans
        $rooms = $this->pdo->prepare('SELECT * FROM hotel_rooms WHERE hotel_id = :hid ORDER BY id DESC');
        $rooms->execute(['hid' => $id]);
        $ratePlans = $this->pdo->prepare('SELECT * FROM rate_plans WHERE hotel_id = :hid ORDER BY id DESC');
        $ratePlans->execute(['hid' => $id]);
        // Load policies
        $policy = $this->pdo->prepare('SELECT * FROM hotel_policies WHERE hotel_id = :hid LIMIT 1');
        $policy->execute(['hid' => $id]);
        $policyRow = $policy->fetch(PDO::FETCH_ASSOC) ?: null;

        // Load images
        $imgStmt = $this->pdo->prepare('SELECT * FROM hotel_images WHERE hotel_id = :hid ORDER BY COALESCE(sort_order, 9999), id DESC');
        $imgStmt->execute(['hid' => $id]);
        $images = $imgStmt->fetchAll(PDO::FETCH_ASSOC);

        // Optional: calendar data if requested
        $cal = null; $selRoom = null; $selRatePlan = null; $ym = null;
        $tabSel = $_GET['tab'] ?? 'details';
        if ($tabSel === 'calendar') {
            $selRoom = (int)($_GET['room_id'] ?? 0);
            $selRatePlan = (int)($_GET['rate_plan_id'] ?? 0);
            $ym = $_GET['ym'] ?? '';
            if ($selRoom > 0 && $ym && preg_match('/^\d{4}-\d{2}$/', $ym)) {
                [$year, $month] = array_map('intval', explode('-', $ym));
                $firstDay = sprintf('%04d-%02d-01', $year, $month);
                $daysInMonth = (int)date('t', strtotime($firstDay));
                $lastDay = sprintf('%04d-%02d-%02d', $year, $month, $daysInMonth);
                // Fetch allocations
                $allocStmt = $this->pdo->prepare('SELECT * FROM room_allocations WHERE room_id=:rid AND date BETWEEN :d1 AND :d2');
                $allocStmt->execute(['rid' => $selRoom, 'd1' => $firstDay, 'd2' => $lastDay]);
                $alloc = [];
                foreach ($allocStmt->fetchAll(PDO::FETCH_ASSOC) as $a) { $alloc[$a['date']] = $a; }
                // Fetch prices if rate plan selected
                $prices = [];
                if ($selRatePlan > 0) {
                    $priceStmt = $this->pdo->prepare('SELECT * FROM room_prices WHERE room_id=:rid AND rate_plan_id=:rpid AND date BETWEEN :d1 AND :d2');
                    $priceStmt->execute(['rid' => $selRoom, 'rpid' => $selRatePlan, 'd1' => $firstDay, 'd2' => $lastDay]);
                    foreach ($priceStmt->fetchAll(PDO::FETCH_ASSOC) as $p) { $prices[$p['date']] = $p; }
                }
                $cal = [];
                for ($day = 1; $day <= $daysInMonth; $day++) {
                    $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
                    $a = $alloc[$date] ?? null;
                    $p = $prices[$date] ?? null;
                    $cal[] = [
                        'date' => $date,
                        'allotment' => $a['allotment'] ?? '',
                        'min_stay' => $a['min_stay'] ?? '',
                        'max_stay' => $a['max_stay'] ?? '',
                        'closed' => isset($a['closed']) ? (int)$a['closed'] : 0,
                        'cta' => isset($a['cta']) ? (int)$a['cta'] : 0,
                        'ctd' => isset($a['ctd']) ? (int)$a['ctd'] : 0,
                        'price' => $p['price'] ?? '',
                        'vendor_cost' => $p['vendor_cost'] ?? '',
                        'agent_price' => $p['agent_price'] ?? ($p['price'] ?? ''),
                        'customer_price' => $p['customer_price'] ?? '',
                        'tax_included' => isset($p['tax_included']) ? (int)$p['tax_included'] : 1,
                    ];
                }
            }
        }
        // Load countries and cities from locations for dependent selects
        try { $countries = $this->pdo->query("SELECT DISTINCT country FROM locations WHERE country <> '' ORDER BY country")->fetchAll(PDO::FETCH_COLUMN); }
        catch (\Throwable $e) { $countries = []; }
        try { $cities = $this->pdo->query("SELECT country, city FROM locations WHERE city IS NOT NULL AND city <> '' ORDER BY country, city")->fetchAll(PDO::FETCH_ASSOC); }
        catch (\Throwable $e) { $cities = []; }

        $this->view('admin/hotels_form', [
            'title' => 'Edit Hotel',
            'vendors' => $vendors,
            'hotel' => $hotel,
            'csrf' => Security::csrfToken(),
            'rooms' => $rooms->fetchAll(PDO::FETCH_ASSOC),
            'rate_plans' => $ratePlans->fetchAll(PDO::FETCH_ASSOC),
            'tab' => $tabSel,
            'policy' => $policyRow,
            'images' => $images,
            'cal' => $cal,
            'cal_room_id' => $selRoom,
            'cal_rate_plan_id' => $selRatePlan,
            'cal_ym' => $ym,
            'countries' => $countries,
            'cities' => $cities,
        ]);
    }

    public function update(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $stars = (int)($_POST['stars'] ?? 0);
        if ($stars < 0) $stars = 0; if ($stars > 5) $stars = 5;
        // New audience flags with safe defaults when not provided by form yet
        $visible_customer = isset($_POST['visible_customer']) ? ((($_POST['visible_customer']) ? 1 : 0)) : 1;
        $visible_agent = isset($_POST['visible_agent']) ? ((($_POST['visible_agent']) ? 1 : 0)) : 1;
        $featured_customer = isset($_POST['featured_customer']) ? ((($_POST['featured_customer']) ? 1 : 0)) : 0;
        $featured_agent = isset($_POST['featured_agent']) ? ((($_POST['featured_agent']) ? 1 : 0)) : 0;
        // Single featured label stored as agent label; used for both audiences
        $allowedLabels = ['top_choice','recommended','trusted_partner','best_value'];
        $agentLabel = trim($_POST['featured_agent_label'] ?? '');
        if ($featured_agent !== 1) { $agentLabel = null; }
        else if ($agentLabel === '' || !in_array($agentLabel, $allowedLabels, true)) { $agentLabel = 'recommended'; }
        try {
            $stmt = $this->pdo->prepare('UPDATE hotels SET vendor_id=:vendor_id, name=:name, city=:city, country=:country, base_price=:base_price, stars=:stars, visible_customer=:visible_customer, visible_agent=:visible_agent, featured_customer=:featured_customer, featured_agent=:featured_agent, featured_agent_label=:featured_agent_label WHERE id=:id');
            $stmt->execute([
                'id' => (int)($_POST['id'] ?? 0),
                'vendor_id' => (int)($_POST['vendor_id'] ?? 0),
                'name' => trim($_POST['name'] ?? ''),
                'city' => trim($_POST['city'] ?? ''),
                'country' => trim($_POST['country'] ?? ''),
                'base_price' => (float)($_POST['base_price'] ?? 0),
                'stars' => $stars,
                'visible_customer' => $visible_customer,
                'visible_agent' => $visible_agent,
                'featured_customer' => $featured_customer,
                'featured_agent' => $featured_agent,
                'featured_agent_label' => $agentLabel,
            ]);
        } catch (\Throwable $e) {
            try {
                $stmt = $this->pdo->prepare('UPDATE hotels SET vendor_id=:vendor_id, name=:name, city=:city, country=:country, base_price=:base_price, stars=:stars WHERE id=:id');
                $stmt->execute([
                    'id' => (int)($_POST['id'] ?? 0),
                    'vendor_id' => (int)($_POST['vendor_id'] ?? 0),
                    'name' => trim($_POST['name'] ?? ''),
                    'city' => trim($_POST['city'] ?? ''),
                    'country' => trim($_POST['country'] ?? ''),
                    'base_price' => (float)($_POST['base_price'] ?? 0),
                    'stars' => $stars,
                ]);
            } catch (\Throwable $e2) {
                $stmt = $this->pdo->prepare('UPDATE hotels SET vendor_id=:vendor_id, name=:name, city=:city, country=:country, base_price=:base_price WHERE id=:id');
                $stmt->execute([
                    'id' => (int)($_POST['id'] ?? 0),
                    'vendor_id' => (int)($_POST['vendor_id'] ?? 0),
                    'name' => trim($_POST['name'] ?? ''),
                    'city' => trim($_POST['city'] ?? ''),
                    'country' => trim($_POST['country'] ?? ''),
                    'base_price' => (float)($_POST['base_price'] ?? 0),
                ]);
            }
        }
        $_SESSION['flash'] = 'Hotel updated.';
        $this->redirect('/admin/hotels');
    }

    public function delete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        if (empty($_SESSION['admin_hotels_unlocked'])) {
            $_SESSION['errors'][] = 'Unlock hotels with the update password first.';
            $this->redirect('/admin/hotels');
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $stmt = $this->pdo->prepare('DELETE FROM hotels WHERE id=:id');
            $stmt->execute(['id' => $id]);
            $_SESSION['flash'] = 'Hotel deleted.';
        }
        $this->redirect('/admin/hotels');
    }

    public function enable(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $stmt = $this->pdo->prepare('UPDATE hotels SET active=1 WHERE id=:id');
            $stmt->execute(['id' => $id]);
            $_SESSION['flash'] = 'Hotel enabled.';
        }
        $this->redirect('/admin/hotels');
    }

    public function disable(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $stmt = $this->pdo->prepare('UPDATE hotels SET active=0 WHERE id=:id');
            $stmt->execute(['id' => $id]);
            $_SESSION['flash'] = 'Hotel disabled.';
        }
        $this->redirect('/admin/hotels');
    }

    public function toggleFlag(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        $field = trim($_POST['field'] ?? '');
        $value = isset($_POST['value']) ? (int)$_POST['value'] : null;
        $allowed = ['visible_customer','visible_agent','featured_customer','featured_agent'];
        if ($id <= 0 || $value === null || !in_array($field, $allowed, true)) {
            $_SESSION['errors'][] = 'Invalid toggle request.';
            $this->redirect('/admin/hotels');
            return;
        }
        // Build safe SQL with whitelisted column
        $sql = sprintf('UPDATE hotels SET %s = :val WHERE id = :id', $field);
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute(['val' => ($value ? 1 : 0), 'id' => $id]);
            $_SESSION['flash'] = ucfirst(str_replace('_',' ', $field)).' set to '.(($value?1:0)).'.';
        } catch (\Throwable $e) {
            $_SESSION['errors'][] = 'Update failed: '.$e->getMessage();
        }
        $this->redirect('/admin/hotels');
    }

    public function unlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $master = trim($_POST['master_password'] ?? '');
        if ($this->verifyMaster($master)) {
            $_SESSION['admin_hotels_unlocked'] = 1;
            $_SESSION['flash'] = 'Hotels unlocked for update/delete.';
        } else {
            $_SESSION['errors'][] = 'Invalid update password.';
        }
        $this->redirect('/admin/hotels');
    }

    private function verifyMaster(string $pwd): bool
    {
        // Reuse admin password from environment or stored hash; placeholder check
        $expected = $_ENV['MASTER_PASSWORD'] ?? ($_ENV['ADMIN_PASSWORD'] ?? '');
        return $expected !== '' && hash('sha256', $pwd) === hash('sha256', $expected);
    }

    public function calendarSave(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        $roomId = (int)($_POST['room_id'] ?? 0);
        $ratePlanId = (int)($_POST['rate_plan_id'] ?? 0);
        $ym = $_POST['ym'] ?? '';
        if ($hotelId <= 0 || $roomId <= 0 || !$ym || !preg_match('/^\d{4}-\d{2}$/', $ym)) {
            $this->redirect('/admin/hotels');
            return;
        }
        $dates = $_POST['date'] ?? [];
        $allotments = $_POST['allotment'] ?? [];
        $minStays = $_POST['min_stay'] ?? [];
        $maxStays = $_POST['max_stay'] ?? [];
        $closed = $_POST['closed'] ?? [];
        $cta = $_POST['cta'] ?? [];
        $ctd = $_POST['ctd'] ?? [];
        $prices = $_POST['price'] ?? [];
        $vendorCosts = $_POST['vendor_cost'] ?? [];
        $agentPrices = $_POST['agent_price'] ?? [];
        $customerPrices = $_POST['customer_price'] ?? [];
        $taxIncluded = $_POST['tax_included'] ?? [];

        $allocIns = $this->pdo->prepare(
            'INSERT INTO room_allocations (room_id, date, allotment, min_stay, max_stay, closed, cta, ctd) VALUES (:room_id, :date, :allotment, :min_stay, :max_stay, :closed, :cta, :ctd)
             ON DUPLICATE KEY UPDATE allotment=VALUES(allotment), min_stay=VALUES(min_stay), max_stay=VALUES(max_stay), closed=VALUES(closed), cta=VALUES(cta), ctd=VALUES(ctd)'
        );
        $priceIns = null;
        if ($ratePlanId > 0) {
            $priceIns = $this->pdo->prepare(
                'INSERT INTO room_prices (room_id, rate_plan_id, date, price, vendor_cost, agent_price, customer_price, tax_included)
                 VALUES (:room_id, :rate_plan_id, :date, :price, :vendor_cost, :agent_price, :customer_price, :tax_included)
                 ON DUPLICATE KEY UPDATE price=VALUES(price), vendor_cost=VALUES(vendor_cost), agent_price=VALUES(agent_price), customer_price=VALUES(customer_price), tax_included=VALUES(tax_included)'
            );
        }

        foreach ($dates as $i => $d) {
            if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $d)) continue;
            $allocIns->execute([
                'room_id' => $roomId,
                'date' => $d,
                'allotment' => ($allotments[$i] === '' ? 0 : (int)$allotments[$i]),
                'min_stay' => ($minStays[$i] === '' ? null : (int)$minStays[$i]),
                'max_stay' => ($maxStays[$i] === '' ? null : (int)$maxStays[$i]),
                'closed' => isset($closed[$i]) ? 1 : 0,
                'cta' => isset($cta[$i]) ? 1 : 0,
                'ctd' => isset($ctd[$i]) ? 1 : 0,
            ]);
            if ($priceIns) {
                $priceIns->execute([
                    'room_id' => $roomId,
                    'rate_plan_id' => $ratePlanId,
                    'date' => $d,
                    'price' => ($prices[$i] === '' ? 0 : (float)$prices[$i]),
                    'vendor_cost' => ($vendorCosts[$i] === '' ? null : (float)$vendorCosts[$i]),
                    'agent_price' => ($agentPrices[$i] === '' ? null : (float)$agentPrices[$i]),
                    'customer_price' => ($customerPrices[$i] === '' ? null : (float)$customerPrices[$i]),
                    'tax_included' => isset($taxIncluded[$i]) ? 1 : 0,
                ]);
            }
        }

        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=calendar&room_id='.$roomId.'&rate_plan_id='.$ratePlanId.'&ym='.urlencode($ym));
    }

    public function policySave(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        if ($hotelId <= 0) { $this->redirect('/admin/hotels'); return; }
        $data = [
            'hotel_id' => $hotelId,
            'checkin_time' => trim($_POST['checkin_time'] ?? ''),
            'checkout_time' => trim($_POST['checkout_time'] ?? ''),
            'child_policy' => trim($_POST['child_policy'] ?? ''),
            'pet_policy' => trim($_POST['pet_policy'] ?? ''),
            'terms_html' => $_POST['terms_html'] ?? '',
        ];
        // Upsert by hotel_id
        $existsStmt = $this->pdo->prepare('SELECT id FROM hotel_policies WHERE hotel_id = :hid');
        $existsStmt->execute(['hid' => $hotelId]);
        $exists = $existsStmt->fetchColumn();
        if ($exists) {
            $upd = $this->pdo->prepare('UPDATE hotel_policies SET checkin_time=:checkin_time, checkout_time=:checkout_time, child_policy=:child_policy, pet_policy=:pet_policy, terms_html=:terms_html WHERE hotel_id=:hotel_id');
            $upd->execute($data);
        } else {
            $ins = $this->pdo->prepare('INSERT INTO hotel_policies (hotel_id, checkin_time, checkout_time, child_policy, pet_policy, terms_html) VALUES (:hotel_id, :checkin_time, :checkout_time, :child_policy, :pet_policy, :terms_html)');
            $ins->execute($data);
        }
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=policies');
    }

    public function imageUpload(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        if ($hotelId <= 0) { $this->redirect('/admin/hotels'); return; }
        $roomId = (int)($_POST['room_id'] ?? 0) ?: null;
        $caption = trim($_POST['caption'] ?? '');
        $sortOrder = trim($_POST['sort_order'] ?? '');
        $sortOrder = ($sortOrder === '' ? null : (int)$sortOrder);
        $dest = dirname(__DIR__,2).DIRECTORY_SEPARATOR.'public'.DIRECTORY_SEPARATOR.'assets'.DIRECTORY_SEPARATOR.'uploads'.DIRECTORY_SEPARATOR.'hotels';
        $res = \App\Core\Upload::image($_FILES['image'] ?? [], $dest);
        if (!($res['ok'] ?? false)) {
            $_SESSION['errors'][] = 'Image upload failed: '.($res['error'] ?? 'unknown');
            $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=images');
            return;
        }
        // Store relative path from public/
        $abs = $res['path'];
        $publicDir = realpath(dirname(__DIR__,2).DIRECTORY_SEPARATOR.'public');
        $rel = str_replace($publicDir, '', realpath($abs));
        if ($rel === $abs) { // fallback: make web path under /assets/uploads/hotels/
            $rel = '/assets/uploads/hotels/'.$res['name'];
        }
        $stmt = $this->pdo->prepare('INSERT INTO hotel_images (hotel_id, room_id, file_path, caption, sort_order) VALUES (:hotel_id, :room_id, :file_path, :caption, :sort_order)');
        $stmt->execute([
            'hotel_id' => $hotelId,
            'room_id' => $roomId,
            'file_path' => $rel,
            'caption' => $caption,
            'sort_order' => $sortOrder,
        ]);
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=images');
    }

    public function imageDelete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        $imgId = (int)($_POST['id'] ?? 0);
        if ($hotelId <= 0 || $imgId <= 0) { $this->redirect('/admin/hotels'); return; }
        // Get image row first
        $row = $this->pdo->prepare('SELECT file_path FROM hotel_images WHERE id=:id AND hotel_id=:hid');
        $row->execute(['id' => $imgId, 'hid' => $hotelId]);
        $img = $row->fetch(PDO::FETCH_ASSOC);
        if ($img) {
            $del = $this->pdo->prepare('DELETE FROM hotel_images WHERE id=:id AND hotel_id=:hid');
            $del->execute(['id' => $imgId, 'hid' => $hotelId]);
            // Try delete physical file under public/
            $publicDir = realpath(dirname(__DIR__,2).DIRECTORY_SEPARATOR.'public');
            $path = $publicDir . str_replace(['..'], '', $img['file_path']);
            if (is_file($path)) @unlink($path);
        }
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=images');
    }

    // ===== Lightweight Admin JSON: flags status
    public function flags(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: application/json');
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { http_response_code(400); echo json_encode(['ok'=>false,'error'=>'Missing id']); return; }
        try {
            $stmt = $this->pdo->prepare('SELECT id, visible_customer, visible_agent, featured_customer, featured_agent FROM hotels WHERE id=:id');
            $stmt->execute(['id' => $id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$row) { http_response_code(404); echo json_encode(['ok'=>false,'error'=>'Hotel not found']); return; }
            echo json_encode(['ok'=>true,'data'=>[
                'id' => (int)$row['id'],
                'visible_customer' => isset($row['visible_customer']) ? (int)$row['visible_customer'] : null,
                'visible_agent' => isset($row['visible_agent']) ? (int)$row['visible_agent'] : null,
                'featured_customer' => isset($row['featured_customer']) ? (int)$row['featured_customer'] : null,
                'featured_agent' => isset($row['featured_agent']) ? (int)$row['featured_agent'] : null,
            ]]);
        } catch (\Throwable $e) {
            http_response_code(500);
            echo json_encode(['ok'=>false,'error'=>'Query failed']);
        }
    }

    // ===== Rooms CRUD =====
    public function roomStore(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        if ($hotelId <= 0) { $this->redirect('/admin/hotels'); return; }
        $stmt = $this->pdo->prepare('INSERT INTO hotel_rooms (hotel_id, name, occupancy_adults, occupancy_children, size_sqm, bed_info, active) VALUES (:hotel_id, :name, :adults, :children, :size, :bed, :active)');
        $stmt->execute([
            'hotel_id' => $hotelId,
            'name' => trim($_POST['name'] ?? ''),
            'adults' => (int)($_POST['occupancy_adults'] ?? 2),
            'children' => (int)($_POST['occupancy_children'] ?? 0),
            'size' => ($_POST['size_sqm'] === '' ? null : (int)$_POST['size_sqm']),
            'bed' => trim($_POST['bed_info'] ?? ''),
            'active' => (int)($_POST['active'] ?? 1),
        ]);
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=rooms');
    }

    public function roomUpdate(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0 || $hotelId <= 0) { $this->redirect('/admin/hotels'); return; }
        $stmt = $this->pdo->prepare('UPDATE hotel_rooms SET name=:name, occupancy_adults=:adults, occupancy_children=:children, size_sqm=:size, bed_info=:bed, active=:active WHERE id=:id AND hotel_id=:hotel_id');
        $stmt->execute([
            'id' => $id,
            'hotel_id' => $hotelId,
            'name' => trim($_POST['name'] ?? ''),
            'adults' => (int)($_POST['occupancy_adults'] ?? 2),
            'children' => (int)($_POST['occupancy_children'] ?? 0),
            'size' => ($_POST['size_sqm'] === '' ? null : (int)$_POST['size_sqm']),
            'bed' => trim($_POST['bed_info'] ?? ''),
            'active' => (int)($_POST['active'] ?? 1),
        ]);
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=rooms');
    }

    public function roomDelete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0 && $hotelId > 0) {
            $stmt = $this->pdo->prepare('DELETE FROM hotel_rooms WHERE id=:id AND hotel_id=:hotel_id');
            $stmt->execute(['id' => $id, 'hotel_id' => $hotelId]);
        }
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=rooms');
    }

    // ===== Rate Plans CRUD =====
    public function ratePlanStore(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        if ($hotelId <= 0) { $this->redirect('/admin/hotels'); return; }
        $stmt = $this->pdo->prepare('INSERT INTO rate_plans (hotel_id, name, meal_plan, refundable, refund_window_days, advance_purchase_days, active) VALUES (:hotel_id, :name, :meal, :ref, :window, :ap, :active)');
        $stmt->execute([
            'hotel_id' => $hotelId,
            'name' => trim($_POST['name'] ?? ''),
            'meal' => $_POST['meal_plan'] ?? 'RO',
            'ref' => (int)($_POST['refundable'] ?? 1),
            'window' => ($_POST['refund_window_days'] === '' ? null : (int)$_POST['refund_window_days']),
            'ap' => ($_POST['advance_purchase_days'] === '' ? null : (int)$_POST['advance_purchase_days']),
            'active' => (int)($_POST['active'] ?? 1),
        ]);
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=rateplans');
    }

    public function ratePlanUpdate(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0 || $hotelId <= 0) { $this->redirect('/admin/hotels'); return; }
        $stmt = $this->pdo->prepare('UPDATE rate_plans SET name=:name, meal_plan=:meal, refundable=:ref, refund_window_days=:window, advance_purchase_days=:ap, active=:active WHERE id=:id AND hotel_id=:hotel_id');
        $stmt->execute([
            'id' => $id,
            'hotel_id' => $hotelId,
            'name' => trim($_POST['name'] ?? ''),
            'meal' => $_POST['meal_plan'] ?? 'RO',
            'ref' => (int)($_POST['refundable'] ?? 1),
            'window' => ($_POST['refund_window_days'] === '' ? null : (int)$_POST['refund_window_days']),
            'ap' => ($_POST['advance_purchase_days'] === '' ? null : (int)$_POST['advance_purchase_days']),
            'active' => (int)($_POST['active'] ?? 1),
        ]);
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=rateplans');
    }

    public function ratePlanDelete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $hotelId = (int)($_POST['hotel_id'] ?? 0);
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0 && $hotelId > 0) {
            $stmt = $this->pdo->prepare('DELETE FROM rate_plans WHERE id=:id AND hotel_id=:hotel_id');
            $stmt->execute(['id' => $id, 'hotel_id' => $hotelId]);
        }
        $this->redirect('/admin/hotels/edit?id='.$hotelId.'&tab=rateplans');
    }
}
