<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;

class CompanyBankAccountsAdminController extends Controller
{
    private function ensureAdmin(): void
    {
        if (empty($_SESSION['user']) || (($_SESSION['user']['role'] ?? '') !== 'admin' && ($_SESSION['user']['role'] ?? '') !== 'Admin')) {
            header('Location: ' . rtrim(\APP_URL, '/') . '/login/admin');
            exit;
        }
    }

    public function index(): void
    {
        $this->ensureAdmin();
        $q = trim((string)($_GET['q'] ?? ''));
        $onlyActive = isset($_GET['active']) ? (int)$_GET['active'] : null; // 1 or 0 or null
        $where = [];$params=[];
        if ($q !== '') { $where[] = '(bank_name LIKE :q OR account_name LIKE :q OR account_number LIKE :q)'; $params[':q'] = "%$q%"; }
        if ($onlyActive !== null) { $where[] = 'active = :a'; $params[':a'] = $onlyActive; }
        $sql = 'SELECT * FROM company_bank_accounts';
        if ($where) { $sql .= ' WHERE ' . implode(' AND ', $where); }
        $sql .= ' ORDER BY active DESC, bank_name, account_number';
        $st = $this->pdo->prepare($sql);
        $st->execute($params);
        $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        $this->view('admin/company_bank_accounts_index', [ 'title' => 'Company Bank Accounts', 'rows' => $rows, 'q' => $q, 'active' => $onlyActive ]);
    }

    public function create(): void
    {
        $this->ensureAdmin();
        $this->view('admin/company_bank_accounts_form', [ 'title' => 'Add Bank Account', 'mode' => 'create' ]);
    }

    public function store(): void
    {
        $this->ensureAdmin();
        Security::requireCsrf();
        $data = [
            'bank_name' => trim((string)($_POST['bank_name'] ?? '')),
            'account_name' => trim((string)($_POST['account_name'] ?? '')),
            'account_number' => trim((string)($_POST['account_number'] ?? '')),
            'branch' => trim((string)($_POST['branch'] ?? '')),
            'swift_code' => trim((string)($_POST['swift_code'] ?? '')),
            'ifsc' => trim((string)($_POST['ifsc'] ?? '')),
            'currency' => strtoupper(trim((string)($_POST['currency'] ?? 'THB')) ?: 'THB'),
            'active' => (int)($_POST['active'] ?? 1),
            'notes' => trim((string)($_POST['notes'] ?? '')),
        ];
        if ($data['bank_name'] === '' || $data['account_name'] === '' || $data['account_number'] === '') {
            $_SESSION['flash_error'] = 'Bank name, Account name and Account number are required.';
            $this->redirect('/admin/bank-accounts/create');
            return;
        }
        $sql = 'INSERT INTO company_bank_accounts (bank_name, account_name, account_number, branch, swift_code, ifsc, currency, active, notes, created_at) VALUES (:bank_name,:account_name,:account_number,:branch,:swift_code,:ifsc,:currency,:active,:notes,NOW())';
        $this->pdo->prepare($sql)->execute([
            ':bank_name'=>$data['bank_name'], ':account_name'=>$data['account_name'], ':account_number'=>$data['account_number'], ':branch'=>($data['branch']?:null), ':swift_code'=>($data['swift_code']?:null), ':ifsc'=>($data['ifsc']?:null), ':currency'=>$data['currency'], ':active'=>$data['active'], ':notes'=>($data['notes']?:null)
        ]);
        $_SESSION['flash_success'] = 'Bank account created.';
        $this->redirect('/admin/bank-accounts');
    }

    public function edit(): void
    {
        $this->ensureAdmin();
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/bank-accounts'); return; }
        $st = $this->pdo->prepare('SELECT * FROM company_bank_accounts WHERE id = :id');
        $st->execute([':id'=>$id]);
        $row = $st->fetch(\PDO::FETCH_ASSOC);
        if (!$row) { $this->redirect('/admin/bank-accounts'); return; }
        $this->view('admin/company_bank_accounts_form', [ 'title' => 'Edit Bank Account', 'mode' => 'edit', 'row' => $row ]);
    }

    public function update(): void
    {
        $this->ensureAdmin();
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/bank-accounts'); return; }
        $data = [
            'bank_name' => trim((string)($_POST['bank_name'] ?? '')),
            'account_name' => trim((string)($_POST['account_name'] ?? '')),
            'account_number' => trim((string)($_POST['account_number'] ?? '')),
            'branch' => trim((string)($_POST['branch'] ?? '')),
            'swift_code' => trim((string)($_POST['swift_code'] ?? '')),
            'ifsc' => trim((string)($_POST['ifsc'] ?? '')),
            'currency' => strtoupper(trim((string)($_POST['currency'] ?? 'THB')) ?: 'THB'),
            'active' => (int)($_POST['active'] ?? 1),
            'notes' => trim((string)($_POST['notes'] ?? '')),
        ];
        $sql = 'UPDATE company_bank_accounts SET bank_name=:bank_name, account_name=:account_name, account_number=:account_number, branch=:branch, swift_code=:swift_code, ifsc=:ifsc, currency=:currency, active=:active, notes=:notes, updated_at=NOW() WHERE id=:id';
        $this->pdo->prepare($sql)->execute([
            ':bank_name'=>$data['bank_name'], ':account_name'=>$data['account_name'], ':account_number'=>$data['account_number'], ':branch'=>($data['branch']?:null), ':swift_code'=>($data['swift_code']?:null), ':ifsc'=>($data['ifsc']?:null), ':currency'=>$data['currency'], ':active'=>$data['active'], ':notes'=>($data['notes']?:null), ':id'=>$id
        ]);
        $_SESSION['flash_success'] = 'Bank account updated.';
        $this->redirect('/admin/bank-accounts/edit?id='.$id);
    }

    public function toggle(): void
    {
        $this->ensureAdmin();
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0); if ($id <= 0) { $this->redirect('/admin/bank-accounts'); return; }
        $this->pdo->prepare('UPDATE company_bank_accounts SET active = 1 - active, updated_at = NOW() WHERE id = :id')->execute([':id'=>$id]);
        $_SESSION['flash_success'] = 'Status updated.';
        $this->redirect('/admin/bank-accounts');
    }

    public function delete(): void
    {
        $this->ensureAdmin();
        Security::requireCsrf();
        $id = (int)($_POST['id'] ?? 0); if ($id <= 0) { $this->redirect('/admin/bank-accounts'); return; }
        $this->pdo->prepare('DELETE FROM company_bank_accounts WHERE id = :id')->execute([':id'=>$id]);
        $_SESSION['flash_success'] = 'Bank account deleted.';
        $this->redirect('/admin/bank-accounts');
    }
}
