<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Auth;
use App\Core\Security;

class CategoriesController extends Controller
{
    /**
     * Modules management page (repurposed from categories).
     * Shows an Update Password unlock form first, then list + CRUD once unlocked.
     */
    public function create(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $unlocked = isset($_SESSION['modules_unlocked']) && $_SESSION['modules_unlocked'] === true;

        if (!$unlocked) {
            $this->view('admin/categories_create', compact('csrf', 'unlocked'));
            return;
        }

        // Ensure modules table exists
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS modules (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(100) NOT NULL UNIQUE,
            slug VARCHAR(100) NOT NULL UNIQUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB");

        // Load all modules
        $modules = $this->pdo->query('SELECT id, name, slug FROM modules ORDER BY name')->fetchAll();
        $this->view('admin/categories_create', compact('csrf', 'unlocked', 'modules'));
    }

    public function unlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $_SESSION['modules_unlocked'] = true;
        $this->redirect('/admin/categories/create');
    }

    public function store(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $name = trim((string)($_POST['name'] ?? ''));
        $errors = [];
        if ($name === '') { $errors[] = 'Module name is required.'; }
        if ($errors) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old'] = ['name' => $name];
            $this->redirect('/admin/categories/create');
        }

        $slug = strtolower(preg_replace('/[^a-z0-9]+/i', '-', $name));
        $slug = trim($slug, '-');

        $this->pdo->exec("CREATE TABLE IF NOT EXISTS modules (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(100) NOT NULL UNIQUE,
            slug VARCHAR(100) NOT NULL UNIQUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB");

        // Insert if not exists
        $stmt = $this->pdo->prepare('INSERT INTO modules (name, slug) VALUES (?, ?)');
        $ok = false;
        try { $ok = $stmt->execute([$name, $slug]); } catch (\Throwable $e) { $ok = false; }
        $_SESSION['flash'] = $ok ? 'Module added.' : 'Module already exists or failed to add.';
        unset($_SESSION['old']);
        $this->redirect('/admin/categories/create');
    }

    public function update(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();

        $id = (int)($_POST['id'] ?? 0);
        $name = trim((string)($_POST['name'] ?? ''));
        if ($id <= 0 || $name === '') {
            $_SESSION['errors'] = ['Valid ID and name are required.'];
            $this->redirect('/admin/categories/create');
        }
        $slug = strtolower(preg_replace('/[^a-z0-9]+/i', '-', $name));
        $slug = trim($slug, '-');
        $stmt = $this->pdo->prepare('UPDATE modules SET name = ?, slug = ? WHERE id = ?');
        $ok = $stmt->execute([$name, $slug, $id]);
        $_SESSION['flash'] = ($ok && $stmt->rowCount() > 0) ? 'Module updated.' : 'No changes made.';
        $this->redirect('/admin/categories/create');
    }

    public function delete(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            $_SESSION['errors'] = ['Valid ID is required.'];
            $this->redirect('/admin/categories/create');
        }
        $stmt = $this->pdo->prepare('DELETE FROM modules WHERE id = ?');
        $ok = $stmt->execute([$id]);
        $_SESSION['flash'] = ($ok && $stmt->rowCount() > 0) ? 'Module deleted.' : 'No module removed.';
        $this->redirect('/admin/categories/create');
    }
}
