<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;

class AuthController extends Controller
{
    public function login(): void
    {
        // Show role selector for clarity
        $this->view('auth/select', ['title' => 'Login']);
    }

    public function doLogin(): void
    {
        $this->handleLogin(null, '/login');
    }

    public function logout(): void
    {
        session_destroy();
        header('Location: ' . rtrim(\APP_URL, '/') . '/login');
    }

    // Role selection and role-locked login screens
    public function select(): void
    {
        $this->view('auth/select', ['title' => 'Login']);
    }

    public function loginAdmin(): void { $this->view('auth/login_admin', ['title' => 'Admin Login']); }
    public function loginEmployee(): void { $this->view('auth/login_employee', ['title' => 'Employee Login']); }
    public function loginPartner(): void { $this->view('auth/login_partner', ['title' => 'Channel Partner Login']); }
    public function loginAgent(): void { $this->view('auth/login_agent', ['title' => 'B2B Login']); }
    public function loginCustomer(): void { $this->view('auth/login_customer', ['title' => 'B2C Login']); }

    public function doLoginAdmin(): void { $this->handleLogin('Admin', '/login/admin'); }
    public function doLoginEmployee(): void { $this->handleLogin('Employee', '/login/employee'); }
    public function doLoginPartner(): void { $this->handleLogin('Channel Partner', '/login/partner'); }
    public function doLoginAgent(): void { $this->handleLogin('B2B Agent', '/login/agent'); }
    public function doLoginCustomer(): void { $this->handleLogin('Customer', '/login/customer'); }

    private function handleLogin(?string $expectedRole, string $fallbackPath): void
    {
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $csrf = $_POST['csrf'] ?? '';
        if (!Security::verifyCsrf($csrf)) {
            http_response_code(400);
            echo 'Invalid CSRF';
            return;
        }
        $stmt = $this->pdo->prepare('SELECT * FROM users WHERE email = :email LIMIT 1');
        $stmt->execute(['email' => $email]);
        $user = $stmt->fetch();
        $ok = $user && password_verify($password, $user['password']);
        if ($ok && $expectedRole !== null && strcasecmp((string)$user['role'], $expectedRole) !== 0) {
            $ok = false; // role mismatch
        }
        if ($ok) {
            // Enrich session with agent profile data for B2B Agents
            if (isset($user['role']) && strcasecmp((string)$user['role'], 'B2B Agent') === 0) {
                try {
                    $stmtP = $this->pdo->prepare('SELECT company, agency_name, phone FROM agent_profiles WHERE user_id = :uid LIMIT 1');
                    $stmtP->execute(['uid' => $user['id']]);
                    $profile = $stmtP->fetch();
                    if ($profile) {
                        // Map profile fields into session-friendly keys
                        if (!empty($profile['company'])) { $user['company'] = $profile['company']; }
                        if (!empty($profile['agency_name'])) { $user['agency_name'] = $profile['agency_name']; }
                        if (!empty($profile['phone'])) { $user['phone'] = $profile['phone']; }
                    }
                } catch (\Throwable $e) {
                    // Silent: do not block login if profile missing
                }
            }
            unset($user['password']);
            $_SESSION['user'] = $user;
            // Redirect by role
            $role = (string)($user['role'] ?? '');
            $dest = '/';
            if (strcasecmp($role, 'Admin') === 0) { $dest = '/admin'; }
            elseif (strcasecmp($role, 'Employee') === 0) { $dest = '/employee'; }
            elseif (strcasecmp($role, 'Channel Partner') === 0) { $dest = '/partner'; }
            elseif (strcasecmp($role, 'B2B Agent') === 0) { $dest = '/agent'; }
            elseif (strcasecmp($role, 'Customer') === 0) { $dest = '/b2c'; }
            header('Location: ' . rtrim(\APP_URL, '/') . $dest);
            return;
        }
        $_SESSION['flash'] = 'Invalid credentials' . ($expectedRole ? ' for ' . htmlspecialchars($expectedRole) : '');
        header('Location: ' . rtrim(\APP_URL, '/') . $fallbackPath);
    }
}
