<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;
use App\Core\Upload;
use App\Core\FileCrypto;
use App\Core\Auth;

class AgentKycController extends Controller
{
    private function requireAgent(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
        if (empty($_SESSION['agent']) || empty($_SESSION['agent']['id'])) {
            // Preserve UX for agent area by redirecting to agent login
            header('Location: /login/agent');
            exit;
        }
    }
    /**
     * Decrypt and stream a stored encrypted file (.enc) belonging to the logged-in agent.
     * $storageRel must start with 'storage/'.
     */
    private function streamDecrypted(string $storageRel, ?string $downloadName = null, ?string $fallbackMime = 'application/octet-stream'): void
    {
        // Normalize and validate path: allow optional leading slash
        $storageRel = ltrim($storageRel, "/\\");
        if (strpos($storageRel, 'storage/') !== 0) { http_response_code(400); echo 'Bad path'; return; }
        $root = dirname(__DIR__, 2);
        $abs = $root . DIRECTORY_SEPARATOR . str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $storageRel);
        // Ensure resolved path remains under storage directory
        $storageRoot = $root . DIRECTORY_SEPARATOR . 'storage' . DIRECTORY_SEPARATOR;
        $absReal = realpath(dirname($abs));
        if ($absReal === false || strpos($absReal . DIRECTORY_SEPARATOR, $storageRoot) !== 0) { http_response_code(400); echo 'Bad path'; return; }
        if (!is_file($abs)) { http_response_code(404); echo 'Not found'; return; }
        $metaAbs = preg_replace('/\.enc$/', '.meta.json', $abs);
        if (!is_file($metaAbs)) {
            $altMeta = $abs . '.meta.json';
            if (is_file($altMeta)) { $metaAbs = $altMeta; } else { http_response_code(404); echo 'Meta missing'; return; }
        }
        $meta = json_decode((string)file_get_contents($metaAbs), true) ?: [];
        $iv = base64_decode($meta['iv'] ?? '', true);
        $tag = base64_decode($meta['tag'] ?? '', true);
        if (!$iv || !$tag) { http_response_code(500); echo 'Meta invalid'; return; }
        $cipher = file_get_contents($abs);
        if ($cipher === false) { http_response_code(500); echo 'Read error'; return; }
        // Decrypt using same key as FileCrypto
        try {
            $keyB64 = env('KYC_ENC_KEY', '');
            if ($keyB64 === '') { throw new \RuntimeException('KYC_ENC_KEY missing'); }
            $key = base64_decode($keyB64, true);
            if ($key === false || strlen($key) !== 32) { throw new \RuntimeException('Invalid KYC_ENC_KEY'); }
        } catch (\Throwable $e) { http_response_code(500); echo 'Key error'; return; }
        $plain = openssl_decrypt($cipher, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
        if ($plain === false) { http_response_code(500); echo 'Decrypt error'; return; }
        $mime = $meta['mime'] ?? $fallbackMime ?? 'application/octet-stream';
        $name = $downloadName ?: ($meta['original_name'] ?? 'document');
        $download = isset($_GET['download']) && (string)$_GET['download'] === '1';
        header('X-Content-Type-Options: nosniff');
        header('Content-Type: ' . $mime);
        header('Content-Disposition: ' . ($download ? 'attachment' : 'inline') . '; filename="' . addslashes($name) . '"');
        header('Content-Length: ' . strlen($plain));
        echo $plain;
        exit;
    }
    private function userId(): ?int
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
        $a = $_SESSION['agent'] ?? null;
        if (!$a) return null;
        return isset($a['id']) ? (int)$a['id'] : null;
    }

    // GET /agent/kyc
    public function index(): void
    {
        $this->requireAgent();
        $uid = $this->userId();
        // Load existing KYC row for prefill
        $st = $this->pdo->prepare('SELECT * FROM agent_kyc WHERE user_id=:u ORDER BY id DESC LIMIT 1');
        $st->execute([':u'=>$uid]);
        $kyc = $st->fetch();

        $docs = [];
        try {
            $q = $this->pdo->prepare('SELECT * FROM agent_business_docs WHERE user_id=:u');
            $q->execute([':u'=>$uid]);
            $docs = $q->fetchAll();
        } catch (\Throwable $e) { /* table exists in schema; ignore errors in fallback */ }

        // Determine business type and per-type KYC flags (from env)
        // Build a safe SELECT based on available columns
        $cols = ['business_type','company','country','state','city','pincode','address'];
        try {
            $crows = $this->pdo->query('SHOW COLUMNS FROM agent_profiles')->fetchAll(\PDO::FETCH_ASSOC);
            $present = array_map(function($r){ return (string)$r['Field']; }, $crows ?: []);
        } catch (\Throwable $_) { $present = ['business_type','company']; }
        $selectCols = array_values(array_intersect($cols, $present));
        if (!$selectCols) { $selectCols = ['business_type','company']; }
        $bpStmt = $this->pdo->prepare('SELECT '.implode(',', $selectCols).' FROM agent_profiles WHERE user_id=:u LIMIT 1');
        $bpStmt->execute([':u'=>$uid]);
        $bp = $bpStmt->fetch(\PDO::FETCH_ASSOC) ?: [];
        $bt = strtolower((string)($bp['business_type'] ?? 'freelancer'));
        $map = [ 'freelancer' => 'FREELANCER', 'sole_proprietor' => 'SP', 'partnership' => 'LLP', 'company' => 'CO' ];
        $key = $map[$bt] ?? 'FREELANCER';
        $NEED_COMPANY = ($bt !== 'freelancer');

        $flags = [
            'selfie_enable' => (bool)env("BT_{$key}_SELFIE_ENABLE", false),
            'selfie_require' => (bool)env("BT_{$key}_SELFIE_REQUIRE", false),
            'address_enable' => (bool)env("BT_{$key}_ADDRESS_PROOF_ENABLE", false),
            'address_require' => (bool)env("BT_{$key}_ADDRESS_PROOF_REQUIRE", false),
            // Government ID granular flags
            'id_type_require' => (bool)env("BT_{$key}_IDTYPE_REQUIRE", true),
            'id_number_require' => (bool)env("BT_{$key}_IDNUM_REQUIRE", true),
            'id_front_require' => (bool)env("BT_{$key}_FRONT_REQUIRE", true),
            'id_back_enable' => (bool)env("BT_{$key}_BACK_ENABLE", true),
            'id_back_require' => (bool)env("BT_{$key}_BACK_REQUIRE", false),
            'iata_enable' => (bool)env("BT_{$key}_IATA_ENABLE", false),
            'iata_require_code' => (bool)env("BT_{$key}_IATA_REQUIRE_CODE", false),
            'gst_number_enable' => (bool)env("BT_{$key}_GST_NUMBER_ENABLE", false),
            'gst_number_require' => (bool)env("BT_{$key}_GST_NUMBER_REQUIRE", false),
            'gst_company_enable' => (bool)env("BT_{$key}_GST_COMPANY_ENABLE", false),
            'gst_company_require' => (bool)env("BT_{$key}_GST_COMPANY_REQUIRE", false),
            // Registration docs toggles via override flags
            'sp_reg_enable' => (bool)env('BT_SP_OVERRIDE_REG', false),
            'llp_reg_enable' => (bool)env('BT_LLP_OVERRIDE_REG', false),
            'co_reg_enable' => (bool)env('BT_CO_OVERRIDE_REG', false),
        ];

        $this->view('agent/kyc/index', [
            'title' => 'Agent KYC',
            'csrf' => Security::csrfToken(),
            'kyc' => $kyc ?: [],
            'docs' => $docs,
            'kyc_flags' => $flags,
            'business_type' => $bt,
            'profile' => $bp,
        ]);
    }

    // POST /agent/kyc/submit
    public function submit(): void
    {
        $this->requireAgent();
        if (!Security::verifyCsrf((string)($_POST['csrf'] ?? ''))) { http_response_code(400); echo 'Invalid CSRF'; return; }
        $uid = $this->userId();

        $idType = trim((string)($_POST['id_type'] ?? 'passport'));
        $idNumber = trim((string)($_POST['id_number'] ?? ''));
        $errors = [];

        $dirKyc = dirname(__DIR__,2) . '/storage/kyc';
        $dirBiz = dirname(__DIR__,2) . '/storage/business_docs';
        $paths = ['front'=>null,'back'=>null,'selfie'=>null,'address'=>null];

        // Load flags again for validation
        $btStmt = $this->pdo->prepare('SELECT business_type FROM agent_profiles WHERE user_id=:u LIMIT 1');
        $btStmt->execute([':u'=>$uid]);
        $bt = strtolower((string)($btStmt->fetchColumn() ?: 'freelancer'));
        $map = [ 'freelancer' => 'FREELANCER', 'sole_proprietor' => 'SP', 'partnership' => 'LLP', 'company' => 'CO' ];
        $key = $map[$bt] ?? 'FREELANCER';
        $SELFIE_REQ = (bool)env("BT_{$key}_SELFIE_REQUIRE", false);
        $ADDR_REQ = (bool)env("BT_{$key}_ADDRESS_PROOF_REQUIRE", false);
        $IDTYPE_REQ = (bool)env("BT_{$key}_IDTYPE_REQUIRE", true);
        $IDNUM_REQ = (bool)env("BT_{$key}_IDNUM_REQUIRE", true);
        $FRONT_REQ = (bool)env("BT_{$key}_FRONT_REQUIRE", true);
        $BACK_ENABLE = (bool)env("BT_{$key}_BACK_ENABLE", true);
        $BACK_REQ = (bool)env("BT_{$key}_BACK_REQUIRE", false);
        $IATA_ENABLE = (bool)env("BT_{$key}_IATA_ENABLE", false);
        $IATA_CODE_REQ_YES = (bool)env("BT_{$key}_IATA_REQUIRE_CODE", false);
        $GST_NUM_ENABLE = (bool)env("BT_{$key}_GST_NUMBER_ENABLE", false);
        $GST_NUM_REQ = (bool)env("BT_{$key}_GST_NUMBER_REQUIRE", false);
        $GST_COM_ENABLE = (bool)env("BT_{$key}_GST_COMPANY_ENABLE", false);
        $GST_COM_REQ = (bool)env("BT_{$key}_GST_COMPANY_REQUIRE", false);

        // Fetch existing latest KYC to know if files exist
        $prev = null;
        try {
            $stPrev = $this->pdo->prepare('SELECT id, doc_front_path, doc_back_path FROM agent_kyc WHERE user_id=:u ORDER BY id DESC LIMIT 1');
            $stPrev->execute([':u'=>$uid]);
            $prev = $stPrev->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $_) { $prev = null; }

        // Handle uploads (optional except per flags) with AES-256-GCM at-rest encryption
        $enc = function(array $uploadResult, string $relDir) use (&$errors) {
            try {
                $absTmp = $uploadResult['path'];
                $mime = $uploadResult['mime'];
                $nameEnc = bin2hex(random_bytes(12)) . '.enc';
                $absEnc = dirname($absTmp) . DIRECTORY_SEPARATOR . $nameEnc;
                FileCrypto::encryptFile($absTmp, $absEnc, $mime, $meta);
                @unlink($absTmp);
                return rtrim($relDir,'/').'/'.$nameEnc; // store relative web path
            } catch (\Throwable $e) {
                $errors[] = 'Encrypt failed: ' . $e->getMessage();
                return null;
            }
        };

        if (!empty($_FILES['doc_front']) && ($_FILES['doc_front']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
            $up = Upload::image($_FILES['doc_front'], $dirKyc);
            if ($up['ok']) { $p = $enc($up, 'storage/kyc'); if ($p) $paths['front'] = $p; } else { $errors[] = 'Front ID: '.$up['error']; }
        }
        if (!empty($_FILES['doc_back']) && ($_FILES['doc_back']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
            $up = Upload::image($_FILES['doc_back'], $dirKyc);
            if ($up['ok']) { $p = $enc($up, 'storage/kyc'); if ($p) $paths['back'] = $p; } else { $errors[] = 'Back ID: '.$up['error']; }
        }
        if (!empty($_FILES['doc_selfie']) && ($_FILES['doc_selfie']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
            $up = Upload::image($_FILES['doc_selfie'], $dirBiz);
            if ($up['ok']) { $p = $enc($up, 'storage/business_docs'); if ($p) $paths['selfie'] = $p; } else { $errors[] = 'Selfie: '.$up['error']; }
        }
        if (!empty($_FILES['doc_address']) && ($_FILES['doc_address']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
            $up = Upload::image($_FILES['doc_address'], $dirBiz);
            if ($up['ok']) { $p = $enc($up, 'storage/business_docs'); if ($p) $paths['address'] = $p; } else { $errors[] = 'Address proof: '.$up['error']; }
        }

        // Registration docs by business type
        $regDocs = [];
        if ($bt === 'sole_proprietor' && (bool)env('BT_SP_OVERRIDE_REG', false) && !empty($_FILES['doc_registration_certificate']) && ($_FILES['doc_registration_certificate']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
            $up = Upload::image($_FILES['doc_registration_certificate'], $dirBiz);
            if ($up['ok']) { $p = $enc($up, 'storage/business_docs'); if ($p) $regDocs[] = ['type' => 'registration_certificate', 'path' => $p]; } else { $errors[] = 'Registration certificate: '.$up['error']; }
        }
        if ($bt === 'partnership' && (bool)env('BT_LLP_OVERRIDE_REG', false) && !empty($_FILES['doc_partnership_deed']) && ($_FILES['doc_partnership_deed']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
            $up = Upload::image($_FILES['doc_partnership_deed'], $dirBiz);
            if ($up['ok']) { $p = $enc($up, 'storage/business_docs'); if ($p) $regDocs[] = ['type' => 'partnership_deed', 'path' => $p]; } else { $errors[] = 'Partnership deed: '.$up['error']; }
        }
        if ($bt === 'company' && (bool)env('BT_CO_OVERRIDE_REG', true) && !empty($_FILES['doc_incorporation_certificate']) && ($_FILES['doc_incorporation_certificate']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
            $up = Upload::image($_FILES['doc_incorporation_certificate'], $dirBiz);
            if ($up['ok']) { $p = $enc($up, 'storage/business_docs'); if ($p) $regDocs[] = ['type' => 'incorporation_certificate', 'path' => $p]; } else { $errors[] = 'Incorporation certificate: '.$up['error']; }
        }

        // Required validations per flags (if missing)
        $company = trim((string)($_POST['company'] ?? ''));
        if ($NEED_COMPANY && $company === '') { $errors[] = 'Company / Agency name is required for your business type'; }
        if ($IDNUM_REQ && $idNumber === '') { $errors[] = 'ID number is required'; }
        if ($FRONT_REQ && (($paths['front'] === null) && (empty($prev['doc_front_path'])))) {
            $errors[] = 'Front ID is required for your business type';
        }
        if ($BACK_ENABLE && $BACK_REQ && (($paths['back'] === null) && (empty($prev['doc_back_path'])))) {
            $errors[] = 'Back ID is required for your business type';
        }
        if ($SELFIE_REQ && (($_FILES['doc_selfie']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE)) {
            $errors[] = 'Selfie is required for your business type';
        }
        if ($ADDR_REQ && (($_FILES['doc_address']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE)) {
            $errors[] = 'Address proof is required for your business type';
        }
        // Address text fields required when address proof required and columns exist
        if ($ADDR_REQ) {
            try {
                $crows3 = $this->pdo->query('SHOW COLUMNS FROM agent_profiles')->fetchAll(\PDO::FETCH_ASSOC);
                $present3 = array_map(function($r){ return (string)$r['Field']; }, $crows3 ?: []);
            } catch (\Throwable $_) { $present3 = []; }
            $country = trim((string)($_POST['country'] ?? ''));
            $state = trim((string)($_POST['state'] ?? ''));
            $city = trim((string)($_POST['city'] ?? ''));
            $pincode = trim((string)($_POST['pincode'] ?? ''));
            $address = trim((string)($_POST['address'] ?? ''));
            if (in_array('country',$present3,true) && $country === '') { $errors[] = 'Country is required'; }
            if (in_array('state',$present3,true) && $state === '') { $errors[] = 'State/Province is required'; }
            if (in_array('city',$present3,true) && $city === '') { $errors[] = 'City is required'; }
            if (in_array('pincode',$present3,true) && $pincode === '') { $errors[] = 'Pincode is required'; }
            if (in_array('address',$present3,true) && $address === '') { $errors[] = 'Address is required'; }
        }

        if ($errors) { $_SESSION['errors'] = $errors; header('Location: /agent/kyc'); return; }

        // Update agent_profiles with IATA/GST if enabled
        try {
            $fields = [];
            $params = [':u'=>$uid];
            if ($NEED_COMPANY) { $fields[] = 'company = :co'; $params[':co'] = $company; }
            if ($IATA_ENABLE) {
                $iataRegistered = (string)($_POST['iata_registered'] ?? '0') === '1' ? 1 : 0;
                $iataCode = trim((string)($_POST['iata_code'] ?? ''));
                if ($iataRegistered && $IATA_CODE_REQ_YES && $iataCode === '') { $_SESSION['errors'] = ['IATA code is required']; header('Location: /agent/kyc'); return; }
                $fields[] = 'iata_registered = :ir'; $params[':ir'] = $iataRegistered;
                $fields[] = 'iata_code = :ic'; $params[':ic'] = $iataCode !== '' ? $iataCode : null;
            }
            if ($GST_NUM_ENABLE) { $fields[] = 'gst_number = :gn'; $params[':gn'] = trim((string)($_POST['gst_number'] ?? '')); if ($GST_NUM_REQ && $params[':gn'] === '') { $_SESSION['errors'] = ['GST Number is required']; header('Location: /agent/kyc'); return; } }
            if ($GST_COM_ENABLE) { $fields[] = 'gst_company = :gc'; $params[':gc'] = trim((string)($_POST['gst_company'] ?? '')); if ($GST_COM_REQ && $params[':gc'] === '') { $_SESSION['errors'] = ['GST Company is required']; header('Location: /agent/kyc'); return; } }
            // Persist address fields if provided (optional), but only for columns that exist
            try {
                $crows2 = $this->pdo->query('SHOW COLUMNS FROM agent_profiles')->fetchAll(\PDO::FETCH_ASSOC);
                $present2 = array_map(function($r){ return (string)$r['Field']; }, $crows2 ?: []);
            } catch (\Throwable $_) { $present2 = []; }
            $country = trim((string)($_POST['country'] ?? ''));
            $state = trim((string)($_POST['state'] ?? ''));
            $city = trim((string)($_POST['city'] ?? ''));
            $pincode = trim((string)($_POST['pincode'] ?? ''));
            $address = trim((string)($_POST['address'] ?? ''));
            if ($country !== '' && in_array('country',$present2,true)) { $fields[] = 'country = :ct'; $params[':ct'] = $country; }
            if ($state !== ''   && in_array('state',$present2,true))   { $fields[] = 'state = :st'; $params[':st'] = $state; }
            if ($city !== ''    && in_array('city',$present2,true))    { $fields[] = 'city = :ci'; $params[':ci'] = $city; }
            if ($pincode !== '' && in_array('pincode',$present2,true)) { $fields[] = 'pincode = :pc'; $params[':pc'] = $pincode; }
            if ($address !== '' && in_array('address',$present2,true)) { $fields[] = 'address = :ad'; $params[':ad'] = $address; }

            if ($fields) {
                $sql = 'UPDATE agent_profiles SET ' . implode(',', $fields) . ' WHERE user_id = :u';
                $this->pdo->prepare($sql)->execute($params);
            }
        } catch (\Throwable $_) { /* ignore */ }

        // Upsert agent_kyc
        $st = $this->pdo->prepare('SELECT id FROM agent_kyc WHERE user_id=:u ORDER BY id DESC LIMIT 1');
        $st->execute([':u'=>$uid]);
        $row = $st->fetch();
        if ($row) {
            $sql = 'UPDATE agent_kyc SET id_type=:t, id_number=:n, doc_front_path=COALESCE(:f,doc_front_path), doc_back_path=COALESCE(:b,doc_back_path), status=\'pending\' WHERE id=:id';
            $this->pdo->prepare($sql)->execute([':t'=>$idType, ':n'=>$idNumber, ':f'=>$paths['front'], ':b'=>$paths['back'], ':id'=>$row['id']]);
        } else {
            $sql = 'INSERT INTO agent_kyc (user_id,id_type,id_number,doc_front_path,doc_back_path,status,created_at) VALUES (:u,:t,:n,:f,:b,\'pending\',NOW())';
            $this->pdo->prepare($sql)->execute([':u'=>$uid, ':t'=>$idType, ':n'=>$idNumber, ':f'=>$paths['front'], ':b'=>$paths['back']]);
        }
        // Insert business docs if present
        if ($paths['selfie']) {
            $this->pdo->prepare('INSERT INTO agent_business_docs (user_id,doc_type,file_path,created_at) VALUES (:u,\'selfie\',:p,NOW())')->execute([':u'=>$uid, ':p'=>$paths['selfie']]);
        }
        if ($paths['address']) {
            $this->pdo->prepare('INSERT INTO agent_business_docs (user_id,doc_type,file_path,created_at) VALUES (:u,\'address_proof\',:p,NOW())')->execute([':u'=>$uid, ':p'=>$paths['address']]);
        }
        foreach ($regDocs as $rd) {
            $this->pdo->prepare('INSERT INTO agent_business_docs (user_id,doc_type,file_path,created_at) VALUES (:u,:t,:p,NOW())')->execute([':u'=>$uid, ':t'=>$rd['type'], ':p'=>$rd['path']]);
        }

        $_SESSION['flash'] = 'KYC submitted. Status set to pending.';
        header('Location: /agent/kyc/status');
    }

    // GET /agent/kyc/status
    public function status(): void
    {
        $uid = $this->userId();
        if (!$uid) { $_SESSION['flash'] = 'Please login as Agent'; header('Location: /login/agent'); return; }
        $st = $this->pdo->prepare('SELECT * FROM agent_kyc WHERE user_id=:u ORDER BY id DESC LIMIT 1');
        $st->execute([':u'=>$uid]);
        $kyc = $st->fetch();
        $q = $this->pdo->prepare('SELECT * FROM agent_business_docs WHERE user_id=:u');
        $q->execute([':u'=>$uid]);
        $docs = $q->fetchAll();
        $this->view('agent/kyc/status', [ 'title' => 'KYC Status', 'kyc' => $kyc ?: [], 'docs' => $docs ]);
    }

    // GET /b2b/agent/kyc/file?id=KYC_ID&side=front|back
    public function file(): void
    {
        $this->requireAgent();
        $uid = $this->userId();
        if (!$uid) { http_response_code(403); echo 'Forbidden'; return; }
        $id = (int)($_GET['id'] ?? 0);
        $side = ($_GET['side'] ?? 'front') === 'back' ? 'back' : 'front';
        if ($id <= 0) { http_response_code(404); echo 'Not found'; return; }
        $col = $side === 'front' ? 'doc_front_path' : 'doc_back_path';
        $stmt = $this->pdo->prepare("SELECT {$col} AS path FROM agent_kyc WHERE id = :id AND user_id = :u LIMIT 1");
        $stmt->execute([':id'=>$id, ':u'=>$uid]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$row || empty($row['path'])) { http_response_code(404); echo 'File not found'; return; }
        $this->streamDecrypted($row['path'], $side . '.document');
    }

    // GET /b2b/agent/kyc/doc?id=DOC_ID
    public function doc(): void
    {
        $this->requireAgent();
        $uid = $this->userId();
        if (!$uid) { http_response_code(403); echo 'Forbidden'; return; }
        $docId = (int)($_GET['id'] ?? 0);
        if ($docId <= 0) { http_response_code(404); echo 'Not found'; return; }
        $stmt = $this->pdo->prepare('SELECT doc_type, file_path FROM agent_business_docs WHERE id = :id AND user_id = :u LIMIT 1');
        $stmt->execute([':id'=>$docId, ':u'=>$uid]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$row || empty($row['file_path'])) { http_response_code(404); echo 'File not found'; return; }
        $name = ($row['doc_type'] ?? 'document') . '.file';
        $this->streamDecrypted($row['file_path'], $name);
    }

    // GET /b2b/agent/kyc/check (and /agent/kyc/check)
    public function check(): void
    {
        $this->requireAgent();
        header('Content-Type: application/json');
        try {
            $uid = $this->userId();
            if (!$uid) { echo json_encode(['ok'=>false,'status'=>'unauth']); return; }
            $st = $this->pdo->prepare('SELECT status FROM agent_kyc WHERE user_id=:u ORDER BY id DESC LIMIT 1');
            $st->execute([':u'=>$uid]);
            $status = (string)($st->fetchColumn() ?: '');
            if ($status === '') { echo json_encode(['ok'=>true,'status'=>'none']); return; }
            echo json_encode(['ok'=>true,'status'=>$status]);
        } catch (\Throwable $e) {
            echo json_encode(['ok'=>false,'status'=>'error']);
        }
    }
}
