<?php
namespace App\Controllers\Agent\Auth;

use App\Controllers\Agent\AgentController;
use App\Core\Security;
use App\Core\Mail\MailService;
use App\Core\Mail\Envelope;
use DateTimeImmutable;
use DateInterval;

class PasswordResetController extends AgentController
{
    protected $mailService;
    
    public function __construct()
    {
        parent::__construct();
        $this->mailService = new MailService($this->pdo);
    }

    // Show the forgot password form
    public function showForgotPasswordForm()
    {
        $this->view('agent/auth/forgot_password', [
            'title' => 'Forgot Password',
            'csrf' => Security::csrfToken(),
        ]);
    }

    // Process the forgot password form
    public function sendResetLink()
    {
        if (!Security::verifyCsrf((string)($_POST['csrf_token'] ?? ''))) {
            $_SESSION['forgot_password_error'] = 'Invalid CSRF token. Please try again.';
            $this->redirect('/agent/forgot-password');
            return;
        }

        $email = filter_var(trim((string)($_POST['email'] ?? '')), FILTER_VALIDATE_EMAIL);
        
        if (!$email) {
            $_SESSION['forgot_password_error'] = 'Please enter a valid email address.';
            $this->redirect('/agent/forgot-password');
            return;
        }

        // Find user by email
        $user = $this->pdo->prepare('SELECT id, name, email FROM users WHERE email = :email AND role = "B2B Agent" LIMIT 1');
        $user->execute([':email' => $email]);
        $user = $user->fetch(\PDO::FETCH_ASSOC);

        if ($user) {
            // Generate reset token (valid for 1 hour)
            $token = bin2hex(random_bytes(32));
            $expires = (new DateTimeImmutable())->add(new DateInterval('PT1H'))->format('Y-m-d H:i:s');
            
            // Store token in database
            $this->pdo->prepare('INSERT INTO password_resets (email, token, created_at, expires_at) VALUES (:email, :token, NOW(), :expires)')
                ->execute([
                    ':email' => $email,
                    ':token' => password_hash($token, PASSWORD_DEFAULT),
                    ':expires' => $expires
                ]);

            // Send reset email
            $resetUrl = url("/agent/reset-password?email=" . urlencode($email) . "&token=" . urlencode($token));
            
            $mail = new Envelope(
                $email,
                'Password Reset Request - B2B Thai Hub',
                $this->viewContent('emails/agent/password_reset', [
                    'name' => $user['name'],
                    'resetUrl' => $resetUrl,
                    'expiresIn' => '1 hour'
                ]),
                '', // Plain text version (optional)
                'noreply@b2bthaihub.com',
                'B2B Thai Hub'
            );
            
            $this->mailService->send($mail);
        }

        // Always show success to prevent email enumeration
        $_SESSION['forgot_password_success'] = 'If an account exists with this email, you will receive a password reset link.';
        $this->redirect('/agent/forgot-password');
    }

    // Show the password reset form
    public function showResetForm()
    {
        $email = $_GET['email'] ?? '';
        $token = $_GET['token'] ?? '';
        
        if (!$email || !$token) {
            $_SESSION['login_error'] = 'Invalid password reset link.';
            $this->redirect('/b2b/agent/login');
            return;
        }
        
        // Verify token is valid and not expired
        $reset = $this->pdo->prepare('
            SELECT * FROM password_resets 
            WHERE email = :email 
            AND expires_at > NOW() 
            AND used_at IS NULL
            ORDER BY created_at DESC 
            LIMIT 1
        ');
        $reset->execute([':email' => $email]);
        $reset = $reset->fetch(\PDO::FETCH_ASSOC);
        
        if (!$reset || !password_verify($token, $reset['token'])) {
            $_SESSION['login_error'] = 'Invalid or expired password reset link. Please try again.';
            $this->redirect('/b2b/agent/login');
            return;
        }
        
        $this->view('agent/auth/reset_password', [
            'title' => 'Reset Password',
            'email' => $email,
            'token' => $token,
            'csrf' => Security::csrfToken(),
        ]);
    }

    // Process the password reset form
    public function reset()
    {
        if (!Security::verifyCsrf((string)($_POST['csrf_token'] ?? ''))) {
            $_SESSION['reset_password_error'] = 'Invalid CSRF token. Please try again.';
            $this->redirect()->back();
            return;
        }

        $email = filter_var(trim((string)($_POST['email'] ?? '')), FILTER_VALIDATE_EMAIL);
        $token = trim((string)($_POST['token'] ?? ''));
        $password = $_POST['password'] ?? '';
        $passwordConfirm = $_POST['password_confirmation'] ?? '';

        // Validate input
        if (!$email || !$token) {
            $_SESSION['reset_password_error'] = 'Invalid request.';
            $this->redirect('/b2b/agent/login');
            return;
        }

        if (strlen($password) < 8) {
            $_SESSION['reset_password_error'] = 'Password must be at least 8 characters long.';
            $this->redirect()->back();
            return;
        }

        if ($password !== $passwordConfirm) {
            $_SESSION['reset_password_error'] = 'Passwords do not match.';
            $this->redirect()->back();
            return;
        }

        // Verify token is valid and not expired
        $reset = $this->pdo->prepare('
            SELECT * FROM password_resets 
            WHERE email = :email 
            AND expires_at > NOW() 
            AND used_at IS NULL
            ORDER BY created_at DESC 
            LIMIT 1
        ');
        $reset->execute([':email' => $email]);
        $reset = $reset->fetch(\PDO::FETCH_ASSOC);
        
        if (!$reset || !password_verify($token, $reset['token'])) {
            $_SESSION['login_error'] = 'Invalid or expired password reset link. Please try again.';
            $this->redirect('/b2b/agent/login');
            return;
        }

        // Update user's password
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $this->pdo->prepare('UPDATE users SET password = :password WHERE email = :email')
            ->execute([':password' => $hashedPassword, ':email' => $email]);

        // Mark token as used
        $this->pdo->prepare('UPDATE password_resets SET used_at = NOW() WHERE id = :id')
            ->execute([':id' => $reset['id']]);

        // Send confirmation email
        $user = $this->pdo->prepare('SELECT name, email FROM users WHERE email = :email LIMIT 1');
        $user->execute([':email' => $email]);
        $user = $user->fetch(\PDO::FETCH_ASSOC);

        if ($user) {
            $mail = new Envelope(
                $email,
                'Password Updated - B2B Thai Hub',
                $this->viewContent('emails/agent/password_updated', [
                    'name' => $user['name']
                ]),
                '', // Plain text version (optional)
                'noreply@b2bthaihub.com',
                'B2B Thai Hub'
            );
            
            $this->mailService->send($mail);
        }

        $_SESSION['login_success'] = 'Your password has been reset successfully. You can now log in with your new password.';
        $this->redirect('/b2b/agent/login');
    }
}
