<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Security;
use App\Core\Session;
use App\Core\Auth;
use App\Services\WalletService;
use App\Services\ChannelPartnerWalletService;

class AdminController extends Controller
{
    public function index(): void
    {
        Auth::requireRole(['Admin']);
        // Basic stats
        $users = (int)$this->pdo->query('SELECT COUNT(*) FROM users')->fetchColumn();
        $bookings = (int)$this->pdo->query('SELECT COUNT(*) FROM bookings')->fetchColumn();
        $pendingWallets = (int)$this->pdo->query("SELECT COUNT(*) FROM wallet_ledger WHERE status='pending'")->fetchColumn();
        // Pending wallet deposit requests (new flow)
        try {
            $pendingDeposits = (int)$this->pdo->query("SELECT COUNT(*) FROM wallet_deposit_requests WHERE status IN ('pending','bank_assigned','under_verification')")->fetchColumn();
        } catch (\Throwable $_) {
            $pendingDeposits = 0;
        }
        $this->view('admin/index', compact('users','bookings','pendingWallets','pendingDeposits'));
    }

    // ===== Admin: Support =====
    // GET /admin/support
    public function supportIndex(): void
    {
        Auth::requireRole(['Admin']);
        $status = trim((string)($_GET['status'] ?? ''));
        $where = [];$params=[];
        if ($status !== '') { $where[] = 't.status = :st'; $params[':st'] = $status; }
        $sql = 'SELECT t.*, u.email AS user_email FROM support_tickets t LEFT JOIN users u ON u.id = t.user_id';
        if ($where) { $sql .= ' WHERE ' . implode(' AND ', $where); }
        $sql .= " ORDER BY FIELD(t.status,'open','in_progress','resolved','closed'), COALESCE(t.updated_at, t.created_at) DESC LIMIT 200";
        $tickets = [];
        try { $st = $this->pdo->prepare($sql); $st->execute($params); $tickets = $st->fetchAll(\PDO::FETCH_ASSOC) ?: []; } catch (\Throwable $_) { $tickets = []; }
        $csrf = Security::csrfToken();
        $this->view('admin/support_index', ['title'=>'Support Tickets','tickets'=>$tickets,'csrf'=>$csrf]);
    }

    // GET /admin/support/view?id=
    public function supportView(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { $this->redirect('/admin/support'); return; }
        $ticket = null; $replies = [];
        try {
            $st = $this->pdo->prepare('SELECT t.*, u.email AS user_email FROM support_tickets t LEFT JOIN users u ON u.id=t.user_id WHERE t.id=:id');
            $st->execute([':id'=>$id]);
            $ticket = $st->fetch(\PDO::FETCH_ASSOC);
        } catch (\Throwable $_) { $ticket = null; }
        try {
            $sr = $this->pdo->prepare('SELECT * FROM support_replies WHERE ticket_id = :id ORDER BY id ASC');
            $sr->execute([':id'=>$id]);
            $replies = $sr->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $replies = []; }
        if (!$ticket) { $this->redirect('/admin/support'); return; }
        $csrf = Security::csrfToken();
        $this->view('admin/support_view', ['title'=>'Ticket #'.$id,'ticket'=>$ticket,'replies'=>$replies,'csrf'=>$csrf]);
    }

    // POST /admin/support/reply
    public function supportReply(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['ticket_id'] ?? 0);
        $message = trim((string)($_POST['message'] ?? ''));
        if ($id <= 0 || $message === '') { $this->redirect('/admin/support'); return; }
        try {
            $ins = $this->pdo->prepare('INSERT INTO support_replies (ticket_id, user_id, agent_id, role, message, created_at) VALUES (:tid, NULL, NULL, "admin", :msg, NOW())');
            $ins->execute([':tid'=>$id, ':msg'=>$message]);
            $this->pdo->prepare("UPDATE support_tickets SET updated_at = NOW(), last_replied_at = NOW(), status = IF(status='open','in_progress',status) WHERE id = :id")
                ->execute([':id'=>$id]);
        } catch (\Throwable $_) { /* ignore */ }
        $this->redirect('/admin/support/view?id='.$id);
    }

    // POST /admin/support/status
    public function supportStatus(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $id = (int)($_POST['ticket_id'] ?? 0);
        $status = (string)($_POST['status'] ?? 'open');
        if ($id <= 0) { $this->redirect('/admin/support'); return; }
        try { $this->pdo->prepare('UPDATE support_tickets SET status=:s, updated_at=NOW() WHERE id=:id')->execute([':s'=>$status, ':id'=>$id]); } catch (\Throwable $_) {}
        $this->redirect('/admin/support/view?id='.$id);
    }

    // ===== Admin: Vendor Payouts =====
    // GET /admin/vendor-payouts
    public function vendorPayouts(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $vendorId = isset($_GET['vendor_id']) ? (int)$_GET['vendor_id'] : 0;
        $status = isset($_GET['status']) ? (string)$_GET['status'] : 'pending'; // pending|cleared|all
        $from   = trim((string)($_GET['from'] ?? ''));
        $to     = trim((string)($_GET['to'] ?? ''));

        // Vendors list for filter
        try { $vendors = $this->pdo->query('SELECT id, name FROM vendors ORDER BY name')->fetchAll(\PDO::FETCH_ASSOC) ?: []; }
        catch (\Throwable $_) { $vendors = []; }

        // Build where for summary (group by vendor)
        $where = ["b.payment_status='paid'"];
        $args = [];
        if ($status === 'pending') { $where[] = "COALESCE(b.vendor_pay_status,'pending')='pending'"; }
        elseif ($status === 'cleared') { $where[] = "COALESCE(b.vendor_pay_status,'pending')='cleared'"; }
        if ($from !== '') { $where[] = 'b.trip_date >= :from'; $args[':from'] = $from; }
        if ($to !== '')   { $where[] = 'b.trip_date <= :to';   $args[':to']   = $to; }
        $whereSql = 'WHERE ' . implode(' AND ', $where);

        $summary = [];
        try {
            $sql = "SELECT COALESCE(b.vendor_id, t.vendor_id) AS vendor_id, v.name AS vendor_name,
                           SUM(COALESCE(b.vendor_cost, b.amount_total, 0)) AS total_amount,
                           SUM(CASE WHEN COALESCE(b.vendor_pay_status,'pending')='pending' THEN COALESCE(b.vendor_cost, b.amount_total, 0) ELSE 0 END) AS pending_amount,
                           SUM(CASE WHEN COALESCE(b.vendor_pay_status,'pending')='cleared' THEN COALESCE(b.vendor_cost, b.amount_total, 0) ELSE 0 END) AS cleared_amount
                    FROM taxi_bookings b
                    LEFT JOIN taxis t ON t.id=b.taxi_id
                    LEFT JOIN vendors v ON v.id = COALESCE(b.vendor_id, t.vendor_id)
                    $whereSql
                    GROUP BY COALESCE(b.vendor_id, t.vendor_id), v.name
                    ORDER BY v.name";
            $st = $this->pdo->prepare($sql); $st->execute($args); $summary = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $summary = []; }

        // Optional detail list for specific vendor
        $rows = [];
        if ($vendorId > 0) {
            $whereV = [$whereSql, 'COALESCE(b.vendor_id, t.vendor_id) = :vid'];
            $argsV = $args; $argsV[':vid'] = $vendorId;
            try {
                $sql2 = "SELECT b.id, b.booking_code, b.trip_date, b.pickup_time, b.pax, b.status, b.payment_status,
                                COALESCE(b.vendor_cost, b.amount_total, 0) AS vendor_cost, b.vendor_currency,
                                b.vendor_pay_status, b.vendor_paid_at
                         FROM taxi_bookings b
                         LEFT JOIN taxis t ON t.id=b.taxi_id
                         " . implode(' AND ', $whereV) . "
                         ORDER BY b.trip_date DESC, b.id DESC
                         LIMIT 500";
                $st2 = $this->pdo->prepare($sql2); $st2->execute($argsV); $rows = $st2->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            } catch (\Throwable $_) { $rows = []; }
        }

        $this->view('admin/vendor_payouts', [
            'title' => 'Vendor Payouts',
            'csrf' => $csrf,
            'vendors' => $vendors,
            'filters' => ['vendor_id'=>$vendorId,'status'=>$status,'from'=>$from,'to'=>$to],
            'summary' => $summary,
            'rows' => $rows,
        ]);
    }

    // POST /admin/vendor-payouts/mark-paid
    public function vendorPayoutsMarkPaid(): void
    {
        Auth::requireRole(['Admin']); Security::requireCsrf();
        $ids = isset($_POST['booking_ids']) ? (array)$_POST['booking_ids'] : [];
        $invoice = trim((string)($_POST['invoice_no'] ?? ''));
        $note = trim((string)($_POST['note'] ?? ''));
        // Optional bank details passthrough
        $bank = [
            'bank_name' => trim((string)($_POST['bank_name'] ?? '')),
            'bank_account_name' => trim((string)($_POST['bank_account_name'] ?? '')),
            'bank_account_no' => trim((string)($_POST['bank_account_no'] ?? '')),
            'bank_ifsc_swift' => trim((string)($_POST['bank_ifsc_swift'] ?? '')),
        ];
        $bankLine = [];
        foreach ($bank as $k=>$v) { if ($v !== '') { $bankLine[] = $k.': '.$v; } }
        if ($bankLine) { $note = ($note!==''?$note."\n":'') . '[BANK] ' . implode('; ', $bankLine); }
        $paidAt = trim((string)($_POST['paid_at'] ?? ''));
        $method = strtolower(trim((string)($_POST['vendor_pay_method'] ?? '')));
        $txnId  = trim((string)($_POST['vendor_pay_txn_id'] ?? ''));
        $allowedMethods = ['bank_transfer','cash','other'];
        if (!in_array($method, $allowedMethods, true)) { $method = null; }
        if (!$ids) { $this->redirect('/admin/vendor-payouts'); return; }
        $ids = array_values(array_unique(array_map('intval', $ids)));
        if (!$ids) { $this->redirect('/admin/vendor-payouts'); return; }
        $in = implode(',', array_fill(0, count($ids), '?'));
        try {
            // Attempt with extra columns. Block payouts for cancelled/refunded or unpaid bookings.
            $sql = $paidAt !== ''
                ? "UPDATE taxi_bookings SET vendor_pay_status='cleared', vendor_paid_at=?, vendor_invoice_no=?, vendor_pay_note=?, vendor_pay_method=?, vendor_pay_txn_id=? WHERE id IN ($in) AND payment_status='paid' AND COALESCE(status,'') NOT IN ('cancelled','refunded')"
                : "UPDATE taxi_bookings SET vendor_pay_status='cleared', vendor_paid_at=NOW(), vendor_invoice_no=?, vendor_pay_note=?, vendor_pay_method=?, vendor_pay_txn_id=? WHERE id IN ($in) AND payment_status='paid' AND COALESCE(status,'') NOT IN ('cancelled','refunded')";
            $paramsHead = $paidAt !== ''
                ? [$paidAt, ($invoice!==''?$invoice:null), ($note!==''?$note:null), $method, ($txnId!==''?$txnId:null)]
                : [($invoice!==''?$invoice:null), ($note!==''?$note:null), $method, ($txnId!==''?$txnId:null)];
            $params = array_merge($paramsHead, $ids);
            $st = $this->pdo->prepare($sql); $st->execute($params);
            if ($st->rowCount() > 0) {
                $_SESSION["flash"] = 'Vendor payout marked as cleared.';
            } else {
                $_SESSION["flash"] = 'No bookings eligible for payout (must be paid and not cancelled/refunded).';
            }
        } catch (\Throwable $e1) {
            // Fallback if extra columns do not exist
            try {
                $sql2 = $paidAt !== ''
                    ? "UPDATE taxi_bookings SET vendor_pay_status='cleared', vendor_paid_at=?, vendor_invoice_no=?, vendor_pay_note=? WHERE id IN ($in) AND payment_status='paid' AND COALESCE(status,'') NOT IN ('cancelled','refunded')"
                    : "UPDATE taxi_bookings SET vendor_pay_status='cleared', vendor_paid_at=NOW(), vendor_invoice_no=?, vendor_pay_note=? WHERE id IN ($in) AND payment_status='paid' AND COALESCE(status,'') NOT IN ('cancelled','refunded')";
                $paramsHead2 = $paidAt !== ''
                    ? [$paidAt, ($invoice!==''?$invoice:null), ($note!==''?$note:null)]
                    : [($invoice!==''?$invoice:null), ($note!==''?$note:null)];
                $params2 = array_merge($paramsHead2, $ids);
                $st2 = $this->pdo->prepare($sql2); $st2->execute($params2);
                if ($st2->rowCount() > 0) {
                    $_SESSION['flash'] = 'Vendor payout marked as cleared.';
                } else {
                    $_SESSION['flash'] = 'No bookings eligible for payout (must be paid and not cancelled/refunded).';
                }
            } catch (\Throwable $e2) {
                $_SESSION['flash'] = 'Failed to mark vendor payout. Please check DB patches.';
            }
        }
        $redir = isset($_POST['redirect_to']) ? (string)$_POST['redirect_to'] : '';
        $this->redirect($redir !== '' ? $redir : '/admin/vendor-payouts');
    }

    // POST /admin/vendor-payouts/mark-paid-vendor
    public function vendorPayoutsMarkPaidVendor(): void
    {
        Auth::requireRole(['Admin']); Security::requireCsrf();
        $vendorId = (int)($_POST['vendor_id'] ?? 0);
        $from   = trim((string)($_POST['from'] ?? ''));
        $to     = trim((string)($_POST['to'] ?? ''));
        $invoice = trim((string)($_POST['invoice_no'] ?? ''));
        $note = trim((string)($_POST['note'] ?? ''));
        $paidAt = trim((string)($_POST['paid_at'] ?? ''));
        $method = strtolower(trim((string)($_POST['vendor_pay_method'] ?? '')));
        $txnId  = trim((string)($_POST['vendor_pay_txn_id'] ?? ''));
        $allowedMethods = ['bank_transfer','cash','other'];
        if (!in_array($method, $allowedMethods, true)) { $method = null; }
        if ($vendorId <= 0) { $this->redirect('/admin/vendor-payouts'); return; }
        $where = ["payment_status='paid'", "COALESCE(vendor_pay_status,'pending')='pending'", '(COALESCE(vendor_id, 0) = :vid OR 1=0)'];
        $args = [':vid'=>$vendorId];
        // Using taxi join for safety (if booking.vendor_id null)
        try {
            $sql = "UPDATE taxi_bookings b
                    LEFT JOIN taxis t ON t.id=b.taxi_id
                    SET b.vendor_pay_status='cleared', b.vendor_paid_at=" . ($paidAt!==''? ':paidAt' : 'NOW()') . ", b.vendor_invoice_no=:inv, b.vendor_pay_note=:note, b.vendor_pay_method=:method, b.vendor_pay_txn_id=:tx
                    WHERE b.payment_status='paid' AND COALESCE(b.vendor_pay_status,'pending')='pending' AND COALESCE(b.status,'') NOT IN ('cancelled','refunded')
                      AND COALESCE(b.vendor_id, t.vendor_id) = :vid" .
                      ($from!=='' ? ' AND b.trip_date >= :from' : '') .
                      ($to!==''   ? ' AND b.trip_date <= :to'   : '');
            $st = $this->pdo->prepare($sql);
            $params = [':inv'=>($invoice!==''?$invoice:null), ':note'=>($note!==''?$note:null), ':method'=>$method, ':tx'=>($txnId!==''?$txnId:null), ':vid'=>$vendorId];
            if ($paidAt!=='') { $params[':paidAt'] = $paidAt; }
            if ($from!=='') $params[':from']=$from; if ($to!=='') $params[':to']=$to;
            $st->execute($params);
            $_SESSION['flash'] = 'Vendor payout marked as cleared.';
        } catch (\Throwable $e1) {
            // Fallback without extra columns
            try {
                $sql2 = "UPDATE taxi_bookings b
                         LEFT JOIN taxis t ON t.id=b.taxi_id
                         SET b.vendor_pay_status='cleared', b.vendor_paid_at=" . ($paidAt!==''? ':paidAt' : 'NOW()') . ", b.vendor_invoice_no=:inv, b.vendor_pay_note=:note
                         WHERE b.payment_status='paid' AND COALESCE(b.vendor_pay_status,'pending')='pending' AND COALESCE(b.status,'') NOT IN ('cancelled','refunded')
                           AND COALESCE(b.vendor_id, t.vendor_id) = :vid" .
                           ($from!=='' ? ' AND b.trip_date >= :from' : '') .
                           ($to!==''   ? ' AND b.trip_date <= :to'   : '');
                $st2 = $this->pdo->prepare($sql2);
                $params2 = [':inv'=>($invoice!==''?$invoice:null), ':note'=>($note!==''?$note:null), ':vid'=>$vendorId];
                if ($paidAt!=='') { $params2[':paidAt'] = $paidAt; }
                if ($from!=='') $params2[':from']=$from; if ($to!=='') $params2[':to']=$to;
                $st2->execute($params2);
                $_SESSION['flash'] = 'Vendor payout marked as cleared.';
            } catch (\Throwable $e2) {
                $_SESSION['flash'] = 'Failed to mark vendor payout. Please check DB patches.';
            }
        }
        $this->redirect('/admin/vendor-payouts?vendor_id='.$vendorId);
    }

    // ===== Admin: Taxi Bookings =====
    // GET /admin/booking/taxi
    public function bookingTaxi(): void
    {
        Auth::requireRole(['Admin']);
        $status = isset($_GET['status']) ? (string)$_GET['status'] : '';
        $payStatus = isset($_GET['payment_status']) ? (string)$_GET['payment_status'] : '';
        $date = isset($_GET['date']) ? (string)$_GET['date'] : '';
        $vendorId = isset($_GET['vendor_id']) ? (int)$_GET['vendor_id'] : 0;
        $where = [];$args=[];
        if ($status !== '') { $where[] = "b.status = :st"; $args[':st']=$status; }
        if ($payStatus !== '') { $where[] = "b.payment_status = :ps"; $args[':ps']=$payStatus; }
        if ($date !== '') { $where[] = "b.trip_date = :dt"; $args[':dt']=$date; }
        if ($vendorId > 0) { $where[] = "COALESCE(b.vendor_id, t.vendor_id) = :vid"; $args[':vid']=$vendorId; }
        $whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';
        $rows = [];
        try {
            $sql = "SELECT b.id, b.booking_code, b.agent_id, u.name AS agent_name, u.email AS agent_email,
                           b.taxi_id, t.name AS ride_name, t.vehicle_type AS vehicle_type,
                           COALESCE(vb.id, vt.id)   AS vendor_id,
                           COALESCE(vb.name, vt.name) AS vendor_name,
                           b.trip_date, b.pickup_time, b.pax, b.amount_total, b.currency,
                           b.vendor_cost, b.vendor_currency,
                           b.vendor_pay_status, b.vendor_paid_at,
                           b.status, b.payment_status, b.payment_method, b.gateway_name, b.payment_txn_id, b.created_at,
                           b.from_text, b.to_text, b.completed_at,
                           (SELECT e.event_type FROM taxi_booking_events e 
                            WHERE e.booking_id = b.id 
                            ORDER BY e.id DESC LIMIT 1) AS audit_status
                    FROM taxi_bookings b
                    LEFT JOIN taxis t ON t.id=b.taxi_id
                    LEFT JOIN users u ON u.id = b.agent_id
                    LEFT JOIN vendors vb ON vb.id = b.vendor_id
                    LEFT JOIN vendors vt ON vt.id = t.vendor_id
                    $whereSql
                    ORDER BY b.id DESC LIMIT 200";
            $st = $this->pdo->prepare($sql); 
            $st->execute($args); 
            $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
            
            // Debug: Log the query and results
            error_log("Taxi Bookings Query: " . $sql);
            error_log("Taxi Bookings Params: " . print_r($args, true));
            error_log("Taxi Bookings Found: " . count($rows));
            
        } catch (\Throwable $e) { 
            error_log("Error fetching taxi bookings: " . $e->getMessage());
            $rows = []; 
        }
        // Load vendors for filter dropdown
        $vendors = [];
        try { $vendors = $this->pdo->query('SELECT id, name FROM vendors ORDER BY name')->fetchAll(\PDO::FETCH_ASSOC) ?: []; } catch (\Throwable $_) { $vendors = []; }
        // Compute vendor stats (counts per vendor) with current filters except vendor filter
        $vendorStats = [];
        try {
            $whereStats = [];
            $argsStats = [];
            if ($status !== '') { $whereStats[] = "b.status = :st"; $argsStats[':st'] = $status; }
            if ($payStatus !== '') { $whereStats[] = "b.payment_status = :ps"; $argsStats[':ps'] = $payStatus; }
            if ($date !== '') { $whereStats[] = "b.trip_date = :dt"; $argsStats[':dt'] = $date; }
            $whereStatsSql = $whereStats ? ('WHERE '.implode(' AND ',$whereStats)) : '';
            $sqlS = "SELECT COALESCE(vb.id, vt.id) AS vendor_id, COALESCE(vb.name, vt.name) AS vendor_name, COUNT(*) AS cnt
                     FROM taxi_bookings b
                     LEFT JOIN taxis t ON t.id=b.taxi_id
                     LEFT JOIN vendors vb ON vb.id = b.vendor_id
                     LEFT JOIN vendors vt ON vt.id = t.vendor_id
                     $whereStatsSql
                     GROUP BY COALESCE(vb.id, vt.id), COALESCE(vb.name, vt.name)
                     ORDER BY cnt DESC";
            $stS = $this->pdo->prepare($sqlS); $stS->execute($argsStats);
            $vendorStats = $stS->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $vendorStats = []; }
        $csrf = Security::csrfToken();
        $this->view('admin/taxi_bookings_index', [ 'title'=>'Taxi Bookings','rows'=>$rows,'vendors'=>$vendors,'vendor_stats'=>$vendorStats,'filters'=>['status'=>$status,'payment_status'=>$payStatus,'date'=>$date,'vendor_id'=>$vendorId],'csrf'=>$csrf ]);
    }

    // GET /admin/booking/taxi/edit?id=
    public function bookingTaxiEdit(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0); 
        if ($id <= 0) { 
            $this->redirect('/admin/booking/taxi'); 
            return; 
        }
        
        try {
            $sql = 'SELECT b.*, 
                           t.name AS ride_name,
                           t.vehicle_type AS vehicle_type,
                           u.name AS agent_name, 
                           u.email AS agent_email,
                           COALESCE(vb.id, vt.id) AS vendor_id,
                           COALESCE(vb.name, vt.name) AS vendor_name
                    FROM taxi_bookings b
                    LEFT JOIN taxis t ON t.id = b.taxi_id
                    LEFT JOIN users u ON u.id = b.agent_id
                    LEFT JOIN vendors vb ON vb.id = b.vendor_id
                    LEFT JOIN vendors vt ON vt.id = t.vendor_id
                    WHERE b.id = :id';
            $st = $this->pdo->prepare($sql);
            $st->execute([':id' => $id]);
            $booking = $st->fetch(\PDO::FETCH_ASSOC);
        } catch (\Throwable $e) {
            error_log('Error fetching booking: ' . $e->getMessage());
            $booking = null;
        }
        
        if (!$booking) {
            $this->redirect('/admin/booking/taxi');
            return;
        }
        
        $csrf = Security::csrfToken();
        $this->view('admin/taxi_booking_edit', [
            'title' => 'Edit Taxi Booking',
            'booking' => $booking,
            'csrf' => $csrf
        ]);
    }
    
    // POST /admin/booking/taxi/update
    public function bookingTaxiUpdate(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            $this->redirect('/admin/booking/taxi');
            return;
        }
        
        // Get current booking data for comparison
        $currentBooking = [];
        try {
            $st = $this->pdo->prepare('SELECT * FROM taxi_bookings WHERE id = ?');
            $st->execute([$id]);
            $currentBooking = $st->fetch(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Exception $e) {
            error_log('Error fetching current booking data: ' . $e->getMessage());
            $currentBooking = [];
        }
        
        // Get and validate input
        $tripDate = $_POST['trip_date'] ?? '';
        $pickupTime = $_POST['pickup_time'] ?? '';
        $pax = (int)($_POST['pax'] ?? 1);
        $fromText = trim($_POST['from_text'] ?? '');
        $toText = trim($_POST['to_text'] ?? '');
        $specialInstructions = trim($_POST['special_instructions'] ?? '');
        $contactName = trim($_POST['contact_name'] ?? '');
        $contactPhone = trim($_POST['contact_phone'] ?? '');
        $contactEmail = trim($_POST['contact_email'] ?? '');
        $customerNotes = trim($_POST['customer_notes'] ?? '');
        
        // Basic validation
        if (empty($tripDate) || empty($pickupTime) || empty($fromText) || empty($toText) || 
            empty($contactName) || empty($contactPhone)) {
            Session::flash('error', 'Please fill in all required fields');
            $this->redirect("/admin/booking/taxi/edit?id=$id");
            return;
        }
        
        try {
            // Prepare update fields and parameters
            $updateFields = [];
            $params = [':id' => $id];
            $oldValues = [];
            $newValues = [];
            
            // Function to check and add field to update
            $checkField = function($field, $value, $label = null) use (&$updateFields, &$params, &$currentBooking, &$oldValues, &$newValues) {
                $label = $label ?: ucfirst(str_replace('_', ' ', $field));
                if (!array_key_exists($field, $currentBooking) || $currentBooking[$field] != $value) {
                    $updateFields[] = "$field = :$field";
                    $params[":$field"] = $value;
                    $oldValues[$label] = $currentBooking[$field] ?? '';
                    $newValues[$label] = $value;
                    return true;
                }
                return false;
            };
            
            // Check each field for changes
            $checkField('trip_date', $tripDate, 'Trip Date');
            $checkField('pickup_time', $pickupTime, 'Pickup Time');
            $checkField('pax', $pax, 'Passengers');
            $checkField('from_text', $fromText, 'From Location');
            $checkField('to_text', $toText, 'To Location');
            $checkField('special_instructions', $specialInstructions, 'Special Instructions');
            $checkField('contact_name', $contactName, 'Contact Name');
            $checkField('contact_phone', $contactPhone, 'Contact Phone');
            $checkField('contact_email', $contactEmail, 'Contact Email');
            $checkField('customer_notes', $customerNotes, 'Customer Notes');

            // Keep legacy customer_* fields in sync for backward compatibility
            // These may not exist in some environments; we'll attempt and let DB handle if columns exist
            $checkField('customer_name', $contactName, 'Customer Name');
            $checkField('customer_mobile', $contactPhone, 'Customer Mobile');
            $checkField('customer_email', $contactEmail, 'Customer Email');
            // Some schemas store notes in `notes`; sync it from customer_notes
            if ($customerNotes !== '') {
                $checkField('notes', $customerNotes, 'Notes');
            }
            
            // Only update if there are changes
            if (!empty($updateFields)) {
                $updateFields[] = 'updated_at = NOW()';
                $sql = 'UPDATE taxi_bookings SET ' . implode(', ', $updateFields) . ' WHERE id = :id';
                
                $st = $this->pdo->prepare($sql);
                $st->execute($params);
                
                // Log the update in audit trail
                $userId = $_SESSION['user_id'] ?? 0;
                if (!empty($oldValues) && !empty($newValues)) {
                    $eventType = 'taxi_booking_updated';
                    $eventTitle = 'Booking Details Updated';
                    $eventDescription = 'Updated booking details';
                    
                    // Format changes for display
                    $changes = [];
                    foreach ($newValues as $field => $newValue) {
                        $oldValue = $oldValues[$field] ?? '';
                        $changes[] = "$field: \"$oldValue\" → \"$newValue\"";
                    }
                    
                    $eventData = [
                        'changes' => $changes,
                        'old_values' => $oldValues,
                        'new_values' => $newValues
                    ];
                    
                    // Insert audit log
                    $auditSql = 'INSERT INTO audit_logs (
                        user_id, event_type, event_title, event_description, 
                        event_data, ip_address, user_agent, created_at
                    ) VALUES (
                        :user_id, :event_type, :event_title, :event_description,
                        :event_data, :ip_address, :user_agent, NOW()
                    )';
                    
                    $auditSt = $this->pdo->prepare($auditSql);
                    $auditSt->execute([
                        ':user_id' => $userId,
                        ':event_type' => $eventType,
                        ':event_title' => $eventTitle,
                        ':event_description' => $eventDescription,
                        ':event_data' => json_encode($eventData),
                        ':ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
                        ':user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
                    ]);
                    
                    // Also log in booking events
                    $eventSql = 'INSERT INTO taxi_booking_events (
                        booking_id, user_id, event_type, description, 
                        old_values, new_values, created_at
                    ) VALUES (
                        :booking_id, :user_id, :event_type, :description,
                        :old_values, :new_values, NOW()
                    )';
                    
                    $eventSt = $this->pdo->prepare($eventSql);
                    $eventSt->execute([
                        ':booking_id' => $id,
                        ':user_id' => $userId,
                        ':event_type' => 'admin_updated',
                        ':description' => 'Booking details updated by admin',
                        ':old_values' => json_encode($oldValues),
                        ':new_values' => json_encode($newValues)
                    ]);
                    
                    $this->logAuditEvent('taxi_booking', $id, 'updated', 'Booking details updated', $userId);
                }
                
                Session::flash('success', 'Booking updated successfully');
            } else {
                Session::flash('info', 'No changes were made to the booking');
            }
            
            $this->redirect("/admin/booking/taxi/view?id=$id");
            
        } catch (\Throwable $e) {
            error_log('Error updating booking: ' . $e->getMessage());
            Session::flash('error', 'Error updating booking. Please try again.');
            $this->redirect("/admin/booking/taxi/edit?id=$id");
        }
    }
    
    // GET /admin/booking/taxi/view?id=
    public function bookingTaxiView(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0); if ($id<=0){ $this->redirect('/admin/booking/taxi'); return; }
        $row=null;$events=[];$payments=[];
        // Load booking with agent and vendor names for better display in the view
        try {
            $sql = 'SELECT b.*, 
                           t.name AS ride_name,
                           t.vehicle_type AS vehicle_type,
                           u.name AS agent_name, u.email AS agent_email,
                           COALESCE(vb.id, vt.id) AS vendor_id,
                           COALESCE(vb.name, vt.name) AS vendor_name
                    FROM taxi_bookings b
                    LEFT JOIN taxis t ON t.id = b.taxi_id
                    LEFT JOIN users u ON u.id = b.agent_id
                    LEFT JOIN vendors vb ON vb.id = b.vendor_id
                    LEFT JOIN vendors vt ON vt.id = t.vendor_id
                    WHERE b.id = :id';
            $st = $this->pdo->prepare($sql);
            $st->execute([':id'=>$id]);
            $row = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
        } catch (\Throwable $_) { $row=null; }
        if (!$row){ $this->redirect('/admin/booking/taxi'); return; }
        // Load events with user name for nicer audit trail rendering
        try {
            $ev=$this->pdo->prepare('SELECT e.*, 
                                            COALESCE(u.name, v.name) AS user_name
                                     FROM taxi_booking_events e
                                     LEFT JOIN users u ON u.id = e.user_id
                                     LEFT JOIN vendors v ON v.id = e.user_id
                                     WHERE e.booking_id = :id
                                     ORDER BY e.id DESC');
            $ev->execute([':id'=>$id]);
            $events=$ev->fetchAll(\PDO::FETCH_ASSOC)?:[];
        } catch (\Throwable $_) { $events=[]; }
        try { $pv=$this->pdo->prepare('SELECT * FROM taxi_payments WHERE booking_id=:id ORDER BY id DESC'); $pv->execute([':id'=>$id]); $payments=$pv->fetchAll(\PDO::FETCH_ASSOC)?:[]; } catch (\Throwable $_) { $payments=[]; }
        $csrf = Security::csrfToken();
        $pgCfg = @require __DIR__ . '/../../config/payment_gateways.php';
        $stripeEnabled = (bool)($pgCfg['stripe']['enabled'] ?? false);
        $this->view('admin/taxi_booking_view', [ 'title'=>'Taxi Booking', 'booking'=>$row, 'events'=>$events, 'payments'=>$payments, 'csrf'=>$csrf, 'stripe_enabled'=>$stripeEnabled ]);
    }

    // GET /admin/security/agent/unlock
    public function agentUnlockForm(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        // Load currently locked users and those with pending failed attempts
        $lockedUsers = [];
        $attemptUsers = [];
        $ipBlocks = [];
        $ipBlockHistory = [];
        try {
            $st = $this->pdo->query("SELECT id, name, email, role, failed_login_attempts, locked_until FROM users WHERE (locked_until IS NOT NULL AND locked_until > NOW()) ORDER BY locked_until DESC LIMIT 200");
            $lockedUsers = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $lockedUsers = []; }
        try {
            $st2 = $this->pdo->query("SELECT id, name, email, role, failed_login_attempts, locked_until FROM users WHERE failed_login_attempts > 0 ORDER BY failed_login_attempts DESC, id DESC LIMIT 200");
            $attemptUsers = $st2->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $attemptUsers = []; }
        try {
            $st3 = $this->pdo->query("SELECT email, ip, attempts, blocked_until, last_attempt_at FROM login_ip_throttle WHERE blocked_until IS NOT NULL AND blocked_until > NOW() ORDER BY blocked_until DESC LIMIT 200");
            $ipBlocks = $st3->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $ipBlocks = []; }
        try {
            $st4 = $this->pdo->query("SELECT email, ip, attempts, blocked_until, last_attempt_at FROM login_ip_throttle ORDER BY COALESCE(blocked_until, last_attempt_at) DESC LIMIT 1000");
            $ipBlockHistory = $st4->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (\Throwable $_) { $ipBlockHistory = []; }

        $this->view('admin/security_agent_unlock', [
            'title' => 'Unlock Agent Login',
            'csrf' => $csrf,
            'locked_users' => $lockedUsers,
            'attempt_users' => $attemptUsers,
            'ip_blocks' => $ipBlocks,
            'ip_block_history' => $ipBlockHistory,
        ]);
    }

    // POST /admin/security/agent/unlock
    public function agentUnlock(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $email = strtolower(trim((string)($_POST['email'] ?? '')));
        $userId = (int)($_POST['user_id'] ?? 0);
        $redirect = (string)($_POST['redirect_to'] ?? ($_SERVER['HTTP_REFERER'] ?? '/admin'));

        if ($email === '' && $userId <= 0) {
            $_SESSION['flash'] = 'Provide either email or user_id to unlock.';
            $this->redirect($redirect);
            return;
        }

        try {
            if ($userId <= 0) {
                $st = $this->pdo->prepare("SELECT id, email FROM users WHERE email=:e LIMIT 1");
                $st->execute([':e' => $email]);
                $row = $st->fetch(\PDO::FETCH_ASSOC) ?: null;
                if (!$row) { $_SESSION['flash'] = 'User not found.'; $this->redirect($redirect); return; }
                $userId = (int)$row['id'];
                $email = (string)$row['email'];
            } else {
                // ensure email is loaded for throttle cleanup
                $st2 = $this->pdo->prepare('SELECT email FROM users WHERE id=:id LIMIT 1');
                $st2->execute([':id'=>$userId]);
                $emailDb = (string)($st2->fetchColumn() ?: '');
                if ($emailDb !== '') { $email = $emailDb; }
            }

            // Reset DB lock counters
            $this->pdo->prepare('UPDATE users SET failed_login_attempts = 0, locked_until = NULL WHERE id = :id')
                ->execute([':id' => $userId]);

            // Clear IP throttle rows for this email
            try {
                $this->pdo->prepare('UPDATE login_ip_throttle SET attempts = 0, blocked_until = NULL WHERE email = :e')
                    ->execute([':e' => $email]);
            } catch (\Throwable $_) { /* ignore if table missing */ }

            $_SESSION['flash'] = 'Agent account unlocked successfully.';
        } catch (\Throwable $e) {
            $_SESSION['flash'] = 'Failed to unlock: ' . $e->getMessage();
        }

        $this->redirect($redirect);
    }

    // POST /admin/security/ip-throttle/clear
    public function ipThrottleClear(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $email = strtolower(trim((string)($_POST['email'] ?? '')));
        $ip = trim((string)($_POST['ip'] ?? ''));
        $redirect = (string)($_POST['redirect_to'] ?? ($_SERVER['HTTP_REFERER'] ?? '/admin/security/agent/unlock'));
        if ($email === '' || $ip === '') {
            $_SESSION['flash'] = 'Email and IP are required to clear a block.';
            $this->redirect($redirect);
            return;
        }
        try {
            $st = $this->pdo->prepare('UPDATE login_ip_throttle SET attempts = 0, blocked_until = NULL WHERE email = :e AND ip = :ip');
            $st->execute([':e' => $email, ':ip' => $ip]);
            if ($st->rowCount() > 0) {
                $_SESSION['flash'] = 'IP block cleared for ' . $email . ' @ ' . $ip . '.';
            } else {
                $_SESSION['flash'] = 'No matching IP block row found.';
            }
        } catch (\Throwable $e) {
            $_SESSION['flash'] = 'Failed to clear IP block: ' . $e->getMessage();
        }
        $this->redirect($redirect);
    }

    // POST /admin/booking/taxi/cancel
    public function bookingTaxiCancel(): void
    {
        Auth::requireRole(['Admin']); Security::requireCsrf(); Security::requireMasterPassword();
        $id=(int)($_POST['id']??0); if ($id<=0){ $this->redirect('/admin/booking/taxi'); return; }
        // Block cancellation if trip is already completed
        try {
            $r = $this->pdo->prepare('SELECT completed_at FROM taxi_bookings WHERE id=:id');
            $r->execute([':id'=>$id]);
            $row = $r->fetch(\PDO::FETCH_ASSOC);
            if ($row && !empty($row['completed_at'])) {
                $_SESSION['flash'] = 'Trip already completed. Cancellation is disabled.';
                $this->redirect('/admin/booking/taxi/view?id='.$id);
                return;
            }
        } catch (\Throwable $_) { /* ignore */ }
        try {
            $this->pdo->prepare("UPDATE taxi_bookings SET status='cancelled', updated_at=NOW() WHERE id=:id")->execute([':id'=>$id]);
            $ev=$this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,:type,:note,:data,NOW())');
            $ev->execute([':bid'=>$id, ':uid'=>(int)($_SESSION['user']['id'] ?? 0), ':type'=>'cancelled', ':note'=>'Booking cancelled by admin', ':data'=>json_encode(['ip'=>($_SERVER['REMOTE_ADDR'] ?? '')])]);
        } catch (\Throwable $_) { }
        $this->redirect('/admin/booking/taxi/view?id='.$id);
    }

    // POST /admin/booking/taxi/refund (wallet)
    public function bookingTaxiRefund(): void
    {
        Auth::requireRole(['Admin']); Security::requireCsrf(); Security::requireMasterPassword();
        $id=(int)($_POST['id']??0); if ($id<=0){ $this->redirect('/admin/booking/taxi'); return; }
        $row=null; try { $st=$this->pdo->prepare('SELECT * FROM taxi_bookings WHERE id=:id'); $st->execute([':id'=>$id]); $row=$st->fetch(\PDO::FETCH_ASSOC)?:null; } catch (\Throwable $_) { }
        if (!$row){ $this->redirect('/admin/booking/taxi'); return; }
        // Block refund if trip is completed
        if (!empty($row['completed_at'])) { $_SESSION['flash']='Trip already completed. Refunds are disabled.'; $this->redirect('/admin/booking/taxi/view?id='.$id); return; }
        $amount=(float)($row['amount_total']??0); if ($amount<=0 || (string)($row['payment_method']??'')!=='wallet'){ $this->redirect('/admin/booking/taxi/view?id='.$id+'&err=refund_method'); return; }
        try {
            $agentId=(int)($row['agent_id']??0); $ws=new WalletService($this->pdo); $walletId=$ws->getOrCreateWallet($agentId);
            $ledgerId=$ws->ledgerEntry($walletId,'credit',$amount,'taxi_refund','pending',['booking_id'=>$id]); $ok=$ws->approveLedger($ledgerId);
            if ($ok){
                $this->pdo->prepare("UPDATE taxi_bookings SET status='refunded', payment_status='refunded', updated_at=NOW() WHERE id=:id")->execute([':id'=>$id]);
                $p=$this->pdo->prepare('INSERT INTO taxi_payments (booking_id, method, gateway_name, amount, currency, status, txn_id, created_at, updated_at) VALUES (:bid,\'wallet_refund\',\'wallet\',:amt,:cur,\'captured\',:tx,NOW(),NOW())');
                $p->execute([':bid'=>$id, ':amt'=>$amount, ':cur'=>(string)($row['currency']??'THB'), ':tx'=>(string)$ledgerId]);
                $ev=$this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,:type,:note,:data,NOW())');
                $ev->execute([':bid'=>$id, ':uid'=>(int)($_SESSION['user']['id'] ?? 0), ':type'=>'refunded', ':note'=>'Wallet refund executed', ':data'=>json_encode(['ledger_id'=>$ledgerId,'ip'=>($_SERVER['REMOTE_ADDR'] ?? '')])]);
            }
        } catch (\Throwable $_) { }
        $this->redirect('/admin/booking/taxi/view?id='.$id);
    }

    // POST /admin/booking/taxi/refund-gateway (Stripe)
    public function bookingTaxiRefundGateway(): void
    {
        Auth::requireRole(['Admin']); Security::requireCsrf(); Security::requireMasterPassword();
        $id=(int)($_POST['id']??0); if ($id<=0){ $this->redirect('/admin/booking/taxi'); return; }
        $row=null; try{ $st=$this->pdo->prepare('SELECT * FROM taxi_bookings WHERE id=:id'); $st->execute([':id'=>$id]); $row=$st->fetch(\PDO::FETCH_ASSOC)?:null; } catch(\Throwable $_){ }
        if (!$row){ $this->redirect('/admin/booking/taxi'); return; }
        // Block refund if trip is completed
        if (!empty($row['completed_at'])) { $_SESSION['flash']='Trip already completed. Refunds are disabled.'; $this->redirect('/admin/booking/taxi/view?id='.$id); return; }
        if ((string)($row['gateway_name']??'')!=='stripe' || (string)($row['payment_status']??'')!=='paid'){ $this->redirect('/admin/booking/taxi/view?id='.$id.'&err=refund_gateway'); return; }
        $cfg=@require __DIR__ . '/../../config/payment_gateways.php'; $secret=(string)($cfg['stripe']['secret_key'] ?? ''); if ($secret===''){ $this->redirect('/admin/booking/taxi/view?id='.$id.'&err=stripe_cfg'); return; }
        $txn=(string)($row['payment_txn_id']??''); if ($txn===''){ $this->redirect('/admin/booking/taxi/view?id='.$id.'&err=txn_missing'); return; }
        try{ $stripe=new \Stripe\StripeClient($secret); $refund=$stripe->refunds->create(['payment_intent'=>$txn]);
            $this->pdo->prepare("UPDATE taxi_bookings SET status='refunded', payment_status='refunded', updated_at=NOW() WHERE id=:id")->execute([':id'=>$id]);
            $p=$this->pdo->prepare('INSERT INTO taxi_payments (booking_id, method, gateway_name, amount, currency, status, provider_ref, raw_event, created_at, updated_at) VALUES (:bid, "gateway_refund", "stripe", 0, :cur, "captured", :ref, :raw, NOW(), NOW())');
            $p->execute([':bid'=>$id, ':cur'=>(string)($row['currency']??'THB'), ':ref'=>(string)($refund->id ?? ''), ':raw'=>json_encode($refund)]);
            $ev=$this->pdo->prepare('INSERT INTO taxi_booking_events (booking_id, user_id, event_type, note, data_json, created_at) VALUES (:bid,:uid,:type,:note,:data,NOW())');
            $ev->execute([':bid'=>$id, ':uid'=>(int)($_SESSION['user']['id'] ?? 0), ':type'=>'refunded', ':note'=>'Stripe refund executed', ':data'=>json_encode(['refund_id'=>$refund->id ?? null,'ip'=>($_SERVER['REMOTE_ADDR'] ?? '')])]);
        } catch(\Throwable $_){ }
        $this->redirect('/admin/booking/taxi/view?id='.$id);
    }

    public function walletManualForm(): void
    {
        Auth::requireRole(['Admin']);
        $users = $this->pdo->query("SELECT id, name, email FROM users ORDER BY name ASC")->fetchAll();
        $csrf = Security::csrfToken();
        $this->view('admin/wallet_manual', ['users' => $users, 'csrf' => $csrf]);
    }

    public function walletManualStore(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        $userId = (int)($_POST['user_id'] ?? 0);
        $type = $_POST['type'] ?? 'credit';
        $amount = (float)($_POST['amount'] ?? 0);
        $note = trim((string)($_POST['note'] ?? ''));
        if ($userId <= 0 || $amount <= 0 || !in_array($type, ['credit','debit'], true)) {
            $_SESSION['flash'] = 'Please fill all fields correctly.';
            $this->redirect('/admin/wallet/manual');
        }
        // Prevent manual requests targeting Admin wallets; use secure screen instead
        $r = $this->pdo->prepare("SELECT role, email FROM users WHERE id = :id LIMIT 1");
        $r->execute(['id'=>$userId]);
        $u = $r->fetch(\PDO::FETCH_ASSOC);
        if ($u && ($u['role'] ?? '') === 'Admin') {
            $_SESSION['flash'] = 'Manual admin wallet changes are blocked. Use Secure Wallet screen.';
            $this->redirect('/admin/wallet/secure');
        }
        // ensure wallet
        $svc = new WalletService($this->pdo);
        $walletId = $svc->getOrCreateWallet($userId);
        // create pending ledger
        $svc->ledgerEntry($walletId, $type, $amount, 'manual', 'pending', ['note' => $note, 'by' => 'admin']);
        $_SESSION['flash'] = 'Request created. Review or process via Secure Wallet.';
        $this->redirect('/admin/wallet/secure');
    }

    public function walletLedger(): void
    {
        Auth::requireRole(['Admin']);
        $stmt = $this->pdo->query("SELECT wl.id, wl.wallet_id, wl.type, wl.amount, wl.method, wl.status, wl.created_at,
                                          u.name as user_name, u.email
                                   FROM wallet_ledger wl
                                   JOIN wallets w ON w.id = wl.wallet_id
                                   JOIN users u ON u.id = w.user_id
                                   ORDER BY wl.id DESC LIMIT 200");
        $rows = $stmt->fetchAll();
        $this->view('admin/wallet_ledger', ['rows' => $rows]);
    }

    public function auditLogs(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        $userId = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
        $action = isset($_GET['action']) ? trim((string)$_GET['action']) : '';
        $from   = isset($_GET['from']) ? trim((string)$_GET['from']) : '';
        $to     = isset($_GET['to']) ? trim((string)$_GET['to']) : '';
        $params = [];
        $wheres = [];
        if ($userId) { $wheres[] = 'l.user_id = :uid'; $params['uid'] = $userId; }
        if ($action !== '') { $wheres[] = 'l.action = :act'; $params['act'] = $action; }
        if ($from !== '') { $wheres[] = 'l.created_at >= :from'; $params['from'] = $from.' 00:00:00'; }
        if ($to !== '') { $wheres[] = 'l.created_at <= :to'; $params['to'] = $to.' 23:59:59'; }
        $whereSql = $wheres ? ('WHERE '.implode(' AND ', $wheres)) : '';
        $sql = "SELECT l.id, l.user_id, l.ip, l.action, l.meta, l.created_at, u.name, u.email
                FROM ip_action_logs l
                LEFT JOIN users u ON u.id = l.user_id
                $whereSql
                ORDER BY l.id DESC
                LIMIT 500";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $logs = $stmt->fetchAll();
        // For filters: users list and distinct actions
        $users = $this->pdo->query("SELECT id, name, email FROM users ORDER BY name")->fetchAll();
        $actions = $this->pdo->query("SELECT DISTINCT action FROM ip_action_logs ORDER BY action")->fetchAll(\PDO::FETCH_COLUMN);
        $this->view('admin/audit_logs', compact('csrf','logs','users','actions','userId','action','from','to'));
    }

    // Security audit events (wallet payments, PIN attempts, suspicious activity)
    public function auditSecurity(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        // Filters
        $userId = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
        $orderId = isset($_GET['order_id']) ? (int)$_GET['order_id'] : null;
        $eventType = isset($_GET['event_type']) ? trim((string)$_GET['event_type']) : '';
        $action = isset($_GET['action']) ? trim((string)$_GET['action']) : '';
        $statusCode = isset($_GET['status_code']) ? (int)$_GET['status_code'] : null;
        $ipLike = isset($_GET['ip']) ? trim((string)$_GET['ip']) : '';
        $from   = isset($_GET['from']) ? trim((string)$_GET['from']) : '';
        $to     = isset($_GET['to']) ? trim((string)$_GET['to']) : '';

        $params = [];
        $wheres = [];
        if ($userId) { $wheres[] = 'e.user_id = :uid'; $params['uid'] = $userId; }
        if ($orderId) { $wheres[] = 'e.order_id = :oid'; $params['oid'] = $orderId; }
        if ($eventType !== '') { $wheres[] = 'e.event_type = :etype'; $params['etype'] = $eventType; }
        if ($action !== '') { $wheres[] = 'e.action = :act'; $params['act'] = $action; }
        if ($statusCode) { $wheres[] = 'e.status_code = :sc'; $params['sc'] = $statusCode; }
        if ($ipLike !== '') { $wheres[] = 'e.ip LIKE :ip'; $params['ip'] = '%'.$ipLike.'%'; }
        if ($from !== '') { $wheres[] = 'e.created_at >= :from'; $params['from'] = $from.' 00:00:00'; }
        if ($to !== '') { $wheres[] = 'e.created_at <= :to'; $params['to'] = $to.' 23:59:59'; }
        $whereSql = $wheres ? ('WHERE '.implode(' AND ', $wheres)) : '';

        $sql = "SELECT e.id, e.user_id, e.agent_id, e.order_id, e.event_type, e.action, e.message, e.details, e.route, e.method, e.ip, e.user_agent, e.status_code, e.created_at,
                       u.name AS user_name, u.email AS user_email
                FROM security_audit_events e
                LEFT JOIN users u ON u.id = e.user_id
                $whereSql
                ORDER BY e.id DESC
                LIMIT 500";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll();
        // For filters menus
        $users = $this->pdo->query("SELECT id, name, email FROM users ORDER BY name")->fetchAll();
        $types = $this->pdo->query("SELECT DISTINCT event_type FROM security_audit_events ORDER BY event_type")->fetchAll(\PDO::FETCH_COLUMN);
        $actions = $this->pdo->query("SELECT DISTINCT action FROM security_audit_events ORDER BY action")->fetchAll(\PDO::FETCH_COLUMN);
        $this->view('admin/audit_security', compact('csrf','rows','users','types','actions','userId','orderId','eventType','action','statusCode','from','to','ipLike'));
    }

    // ========= SECURE WALLET OPS =========
    public function walletSecure(): void
    {
        Auth::requireRole(['Admin']);
        $csrf = Security::csrfToken();
        // load lists for selects
        $agents = $this->pdo->query("SELECT id, name, email FROM users WHERE role='B2B Agent' AND status='Active' ORDER BY name")->fetchAll();
        $partners = $this->pdo->query("SELECT cp.id, cp.name, IFNULL(w.balance, 0) AS balance
                                       FROM channel_partners cp
                                       LEFT JOIN channel_partner_wallet w ON w.partner_id = cp.id
                                       WHERE cp.status='Active' ORDER BY cp.name")->fetchAll();
        // current admin wallet balance
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $stmtBal = $this->pdo->prepare('SELECT balance FROM wallets WHERE user_id = :u LIMIT 1');
        $stmtBal->execute(['u' => $adminId]);
        $rowBal = $stmtBal->fetch(\PDO::FETCH_ASSOC);
        $adminBalance = $rowBal ? (float)$rowBal['balance'] : 0.0;
        // recent transfers (last 20 meaningful events)
        // 1) wallet_ledger credits for flows admin_to_agent, partner_to_agent, treasury_seed, and reversal credits back to Admin
        $wl = $this->pdo->query("SELECT wl.id, wl.amount, wl.method, wl.status, wl.meta, wl.created_at, u.id as user_id, u.name as user_name, u.email
                              FROM wallet_ledger wl
                              JOIN wallets w ON w.id = wl.wallet_id
                              JOIN users u ON u.id = w.user_id
                              WHERE wl.status='approved' AND wl.type='credit'
                              ORDER BY wl.id DESC LIMIT 50")->fetchAll();
        $recent = [];
        foreach ($wl as $row) {
            $meta = json_decode($row['meta'] ?? 'null', true) ?: [];
            $flow = $meta['flow'] ?? ($meta['reason'] ?? '');
            if (in_array($flow, ['admin_to_agent','partner_to_agent','treasury_seed'], true)) {
                $recent[] = [
                    'source' => ($flow === 'admin_to_agent' ? 'Admin' : ($flow === 'partner_to_agent' ? 'Partner' : 'System')),
                    'target' => $row['user_name'].' ('.$row['email'].')',
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                    'meta'   => $meta,
                ];
            } elseif ($flow === 'reverse_admin_to_agent') {
                // Credit back to Admin; source is the Agent (from meta)
                $agentLabel = 'Agent';
                $agentId = (int)($meta['agent_user_id'] ?? 0);
                if ($agentId > 0) {
                    $s = $this->pdo->prepare("SELECT name, email FROM users WHERE id=:id LIMIT 1");
                    $s->execute(['id' => $agentId]);
                    if ($u = $s->fetch(\PDO::FETCH_ASSOC)) {
                        $agentLabel = ($u['name'] ?? 'Agent').' ('.($u['email'] ?? '').')';
                    }
                }
                $recent[] = [
                    'source' => $agentLabel,
                    'target' => 'Admin',
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                    'meta'   => $meta,
                ];
            }
        }
        // 2) channel_partner_wallet_ledger entries for admin_to_partner (credit), reverse_admin_to_partner (debit), reverse_partner_to_agent (credit)
        $cp = $this->pdo->query("SELECT l.id, l.amount, l.type, l.method, l.status, l.meta, l.created_at, p.id as partner_id, p.name as partner_name
                              FROM channel_partner_wallet_ledger l
                              JOIN channel_partner_wallet w ON w.id = l.wallet_id
                              JOIN channel_partners p ON p.id = w.partner_id
                              WHERE l.status='approved' AND l.type IN ('credit','debit')
                              ORDER BY l.id DESC LIMIT 50")->fetchAll();
        foreach ($cp as $row) {
            $meta = json_decode($row['meta'] ?? 'null', true) ?: [];
            $flow = $meta['flow'] ?? '';
            if ($flow === 'admin_to_partner' && ($row['type'] ?? '') === 'credit') {
                $recent[] = [
                    'source' => 'Admin',
                    'target' => $row['partner_name'],
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                    'meta'   => $meta,
                ];
            } elseif ($flow === 'reverse_admin_to_partner' && ($row['type'] ?? '') === 'debit') {
                // Debit from Partner back to Admin
                $recent[] = [
                    'source' => $row['partner_name'],
                    'target' => 'Admin',
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                    'meta'   => $meta,
                ];
            } elseif ($flow === 'reverse_partner_to_agent' && ($row['type'] ?? '') === 'credit') {
                // Credit to Partner from Agent
                $agentLabel = 'Agent';
                $agentId = (int)($meta['agent_user_id'] ?? 0);
                if ($agentId > 0) {
                    $s = $this->pdo->prepare("SELECT name, email FROM users WHERE id=:id LIMIT 1");
                    $s->execute(['id' => $agentId]);
                    if ($u = $s->fetch(\PDO::FETCH_ASSOC)) {
                        $agentLabel = ($u['name'] ?? 'Agent').' ('.($u['email'] ?? '').')';
                    }
                }
                $recent[] = [
                    'source' => $agentLabel,
                    'target' => $row['partner_name'],
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                    'meta'   => $meta,
                ];
            }
        }
        // sort by created_at desc and cap 20
        usort($recent, function($a,$b){ return strcmp($b['created_at'], $a['created_at']); });
        $recent = array_slice($recent, 0, 20);

        $this->view('admin/wallet_secure', compact('csrf','agents','partners','recent','adminBalance'));
    }

    // One-time or repeated seed of treasury admin wallet (admin3@example.com). Block general admin self-loads.
    public function walletSeed(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $email = trim((string)($_POST['email'] ?? 'admin3@example.com'));
        $amount = (float)($_POST['amount'] ?? 10000);
        if ($amount <= 0) { $_SESSION['flash'] = 'Invalid amount'; $this->redirect('/admin/wallet/secure'); }
        $stmt = $this->pdo->prepare("SELECT id, role FROM users WHERE email = :e LIMIT 1");
        $stmt->execute(['e'=>$email]);
        $user = $stmt->fetch(\PDO::FETCH_ASSOC);
        if (!$user) { $_SESSION['flash'] = 'Treasury user not found'; $this->redirect('/admin/wallet/secure'); }
        // Prevent general admin self-load by policy: deny list of emails
        $denyEmails = ['admin3@example.com','admin@example.com'];
        if (in_array(strtolower($email), $denyEmails, true)) {
            // Allow only via this seed endpoint (we are here), so continue
        }
        $svc = new WalletService($this->pdo);
        $ok = $svc->creditApproved((int)$user['id'], $amount, 'manual', [
            'reason' => 'treasury_seed',
            'by' => 'admin_controller',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'at' => date('c'),
        ]);
        // IP log
        $this->logIp('wallet_seed', ['email'=>$email,'amount'=>$amount,'ok'=>$ok]);
        $_SESSION['flash'] = $ok ? 'Treasury seeded.' : 'Seed failed.';
        $this->redirect('/admin/wallet/secure');
    }

    // Admin -> Agent transfer (direct B2B)
    public function transferAdminToAgent(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $fromEmail = (string)($_SESSION['user']['email'] ?? '');
        $toUserId = (int)($_POST['agent_user_id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        if ($adminId <= 0 || $toUserId <= 0 || $amount <= 0) { $_SESSION['flash'] = 'Invalid input'; $this->redirect('/admin/wallet/secure'); }
        // Perform transfer
        $svc = new WalletService($this->pdo);
        $ok = $svc->transferApproved($adminId, $toUserId, $amount, 'manual_transfer', [
            'flow' => 'admin_to_agent',
            'from_admin_id' => $adminId,
            'from_admin_email' => $fromEmail,
            'to_user_id' => $toUserId,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'at' => date('c'),
        ]);
        $this->logIp('transfer_admin_to_agent', ['from'=>$fromEmail,'to_user_id'=>$toUserId,'amount'=>$amount,'ok'=>$ok]);
        // Also log to wallet_transactions so it shows in agent history
        if ($ok) {
            try {
                $metaJson = json_encode([
                    'flow' => 'admin_to_agent',
                    'from_admin_id' => $adminId,
                    'from_admin_email' => $fromEmail,
                    'to_user_id' => $toUserId,
                ]);
                $st = $this->pdo->prepare("SELECT insert_wallet_transaction(?, 'credit', 'admin_credit', ?, 'Admin \xE2\x86\x92 Agent Transfer', ?, 'admin_action', NULL, ?)");
                $st->execute([
                    $toUserId,
                    $amount,
                    'ADMIN_TO_AGENT',
                    $metaJson,
                ]);
            } catch (\Throwable $e) { error_log('admin_to_agent wallet_tx insert failed: '.$e->getMessage()); }
        }
        $_SESSION['flash'] = $ok ? 'Transfer completed' : 'Transfer failed';
        $this->redirect('/admin/wallet/secure');
    }

    // ======== REVERSALS (Compensating Entries) ========
    // Reverse Admin → Agent (Agent → Admin)
    public function reverseAdminAgent(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $fromEmail = (string)($_SESSION['user']['email'] ?? '');
        $agentUserId = (int)($_POST['agent_user_id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $reason = trim((string)($_POST['reason'] ?? '')); 
        if ($adminId <= 0 || $agentUserId <= 0 || $amount <= 0 || $reason === '') { $_SESSION['flash'] = 'Invalid input'; $this->redirect('/admin/wallet/secure'); }
        $svc = new WalletService($this->pdo);
        $ok = $svc->reverseTransferApproved($adminId, $agentUserId, $amount, $reason, [
            'flow' => 'reverse_admin_to_agent',
            'reversal_of_flow' => 'admin_to_agent',
            'from_admin_id' => $adminId,
            'from_admin_email' => $fromEmail,
            'agent_user_id' => $agentUserId,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'at' => date('c'),
        ]);
        $this->logIp('reverse_admin_agent', ['from'=>$fromEmail,'agent_user_id'=>$agentUserId,'amount'=>$amount,'ok'=>$ok,'reason'=>$reason]);
        // Log debit on agent wallet_transactions for visibility
        if ($ok) {
            try {
                $metaJson = json_encode([
                    'flow' => 'reverse_admin_to_agent',
                    'reason' => $reason,
                    'from_admin_id' => $adminId,
                    'from_admin_email' => $fromEmail,
                ]);
                $st = $this->pdo->prepare("SELECT insert_wallet_transaction(?, 'debit', 'adjustment', ?, 'Reversal: Admin \x2190 Agent', ?, 'admin_action', NULL, ?)");
                $st->execute([
                    $agentUserId,
                    $amount,
                    'REVERSE_ADMIN_AGENT',
                    $metaJson,
                ]);
            } catch (\Throwable $e) { error_log('reverse_admin_agent wallet_tx insert failed: '.$e->getMessage()); }
        }
        $_SESSION['flash'] = $ok ? 'Reversal completed' : 'Reversal failed (insufficient funds on recipient or other error)';
        $this->redirect('/admin/wallet/secure');
    }

    // Reverse Partner → Agent (Agent → Partner)
    public function reversePartnerAgent(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $partnerId = (int)($_POST['partner_id'] ?? 0);
        $agentUserId = (int)($_POST['agent_user_id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $reason = trim((string)($_POST['reason'] ?? ''));
        if ($partnerId <= 0 || $agentUserId <= 0 || $amount <= 0 || $reason === '') { $_SESSION['flash'] = 'Invalid input'; $this->redirect('/admin/wallet/secure'); }
        $cp = new ChannelPartnerWalletService($this->pdo);
        $ok = $cp->reversePartnerToUserApproved($partnerId, $agentUserId, $amount, $reason, [
            'flow' => 'reverse_partner_to_agent',
            'reversal_of_flow' => 'partner_to_agent',
            'partner_id' => $partnerId,
            'agent_user_id' => $agentUserId,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'at' => date('c'),
        ]);
        $this->logIp('reverse_partner_agent', ['partner_id'=>$partnerId,'agent_user_id'=>$agentUserId,'amount'=>$amount,'ok'=>$ok,'reason'=>$reason]);
        $_SESSION['flash'] = $ok ? 'Reversal completed' : 'Reversal failed (insufficient funds on recipient or other error)';
        $this->redirect('/admin/wallet/secure');
    }

    // Reverse Admin → Partner (Partner → Admin)
    public function reverseAdminPartner(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $fromEmail = (string)($_SESSION['user']['email'] ?? '');
        $partnerId = (int)($_POST['partner_id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $reason = trim((string)($_POST['reason'] ?? ''));
        if ($adminId <= 0 || $partnerId <= 0 || $amount <= 0 || $reason === '') { $_SESSION['flash'] = 'Invalid input'; $this->redirect('/admin/wallet/secure'); }
        $cp = new ChannelPartnerWalletService($this->pdo);
        $ok = $cp->reverseUserToPartnerApproved($adminId, $partnerId, $amount, $reason, [
            'flow' => 'reverse_admin_to_partner',
            'reversal_of_flow' => 'admin_to_partner',
            'from_admin_id' => $adminId,
            'from_admin_email' => $fromEmail,
            'partner_id' => $partnerId,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'at' => date('c'),
        ]);
        $this->logIp('reverse_admin_partner', ['from'=>$fromEmail,'partner_id'=>$partnerId,'amount'=>$amount,'ok'=>$ok,'reason'=>$reason]);
        $_SESSION['flash'] = $ok ? 'Reversal completed' : 'Reversal failed (insufficient funds on recipient or other error)';
        $this->redirect('/admin/wallet/secure');
    }

    // Admin -> Channel Partner transfer
    public function transferAdminToPartner(): void
    {
        Auth::requireRole(['Admin']);
        Security::requireCsrf();
        Security::requireMasterPassword();
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $fromEmail = (string)($_SESSION['user']['email'] ?? '');
        $partnerId = (int)($_POST['partner_id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        if ($adminId <= 0 || $partnerId <= 0 || $amount <= 0) { $_SESSION['flash'] = 'Invalid input'; $this->redirect('/admin/wallet/secure'); }
        // Move funds from admin(user wallet) to partner(wallet)
        $cp = new ChannelPartnerWalletService($this->pdo);
        $ok = $cp->transferFromUserApproved($adminId, $partnerId, $amount, 'manual_transfer', [
            'flow' => 'admin_to_partner',
            'from_admin_id' => $adminId,
            'from_admin_email' => $fromEmail,
            'partner_id' => $partnerId,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'at' => date('c'),
        ]);
        $this->logIp('transfer_admin_to_partner', ['from'=>$fromEmail,'partner_id'=>$partnerId,'amount'=>$amount,'ok'=>$ok]);
        $_SESSION['flash'] = $ok ? 'Transfer completed' : 'Transfer failed';
        $this->redirect('/admin/wallet/secure');
    }

    // Channel Partner -> Agent (B2B via partner)
    public function transferPartnerToAgent(): void
    {
        Auth::requireRole(['Admin']); // Only admin can initiate partner payouts per spec
        Security::requireCsrf();
        Security::requireMasterPassword();
        $partnerId = (int)($_POST['partner_id'] ?? 0);
        $toUserId = (int)($_POST['agent_user_id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $txPassword = (string)($_POST['transaction_password'] ?? '');
        if ($partnerId <= 0 || $toUserId <= 0 || $amount <= 0) { $_SESSION['flash'] = 'Invalid input'; $this->redirect('/admin/wallet/secure'); }
        if ($txPassword === '') { $_SESSION['flash'] = 'Transaction password is required for Partner → Agent transfers.'; $this->redirect('/admin/wallet/secure'); }
        // Ensure partners have a transaction password column and validate it
        $this->ensurePartnerTxPasswordColumn();
        try {
            $sp = $this->pdo->prepare('SELECT transaction_password_hash FROM channel_partners WHERE id = :id LIMIT 1');
            $sp->execute(['id' => $partnerId]);
            $prow = $sp->fetch(\PDO::FETCH_ASSOC);
            $stored = (string)($prow['transaction_password_hash'] ?? '');
            if ($stored === '' || !password_verify($txPassword, $stored)) {
                $_SESSION['flash'] = 'Invalid transaction password for the selected Channel Partner.';
                $this->redirect('/admin/wallet/secure');
            }
        } catch (\Throwable $e) {
            $_SESSION['flash'] = 'Unable to validate partner transaction password.';
            $this->redirect('/admin/wallet/secure');
        }
        // Enforce that the agent belongs to the selected partner (users.partner_id)
        $this->ensureUsersPartnerIdColumn();
        $chk = $this->pdo->prepare("SELECT partner_id FROM users WHERE id=:id AND role='B2B Agent' LIMIT 1");
        $chk->execute(['id'=>$toUserId]);
        $row = $chk->fetch(\PDO::FETCH_ASSOC);
        if (!$row || (int)($row['partner_id'] ?? 0) !== $partnerId) {
            $_SESSION['flash'] = 'Selected agent is not mapped under this partner.';
            $this->redirect('/admin/wallet/secure');
        }
        $cp = new ChannelPartnerWalletService($this->pdo);
        $ok = $cp->transferToUserApproved($partnerId, $toUserId, $amount, 'manual_transfer', [
            'flow' => 'partner_to_agent',
            'partner_id' => $partnerId,
            'to_user_id' => $toUserId,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
            'at' => date('c'),
        ]);
        $this->logIp('transfer_partner_to_agent', ['partner_id'=>$partnerId,'to_user_id'=>$toUserId,'amount'=>$amount,'ok'=>$ok]);
        // Log credit on agent wallet_transactions for visibility
        if ($ok) {
            try {
                $metaJson = json_encode([
                    'flow' => 'partner_to_agent',
                    'partner_id' => $partnerId,
                    'to_user_id' => $toUserId,
                ]);
                $st = $this->pdo->prepare("SELECT insert_wallet_transaction(?, 'credit', 'transfer_in', ?, 'Partner \x2192 Agent Transfer', ?, 'transfer', NULL, ?)");
                $st->execute([
                    $toUserId,
                    $amount,
                    'PARTNER_TO_AGENT',
                    $metaJson,
                ]);
            } catch (\Throwable $e) { error_log('partner_to_agent wallet_tx insert failed: '.$e->getMessage()); }
        }
        $_SESSION['flash'] = $ok ? 'Transfer completed' : 'Transfer failed';
        $this->redirect('/admin/wallet/secure');
    }

    // API: Agents by Partner (for modal dynamic select)
    public function apiAgentsByPartner(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: application/json');
        $partnerId = (int)($_GET['partner_id'] ?? 0);
        if ($partnerId <= 0) { echo json_encode(['ok'=>false,'error'=>'invalid_partner']); return; }
        // Ensure schema has users.partner_id
        $this->ensureUsersPartnerIdColumn();
        $stmt = $this->pdo->prepare("SELECT id, name, email FROM users WHERE role='B2B Agent' AND status='Active' AND partner_id = :pid ORDER BY name");
        $stmt->execute(['pid'=>$partnerId]);
        $agents = $stmt->fetchAll();
        echo json_encode(['ok'=>true,'data'=>$agents]);
    }

    // API: User wallet balance by user_id
    public function apiUserBalance(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: application/json');
        $userId = (int)($_GET['user_id'] ?? 0);
        if ($userId <= 0) { echo json_encode(['ok'=>false,'error'=>'invalid_user']); return; }
        $stmt = $this->pdo->prepare('SELECT balance FROM wallets WHERE user_id = :u LIMIT 1');
        $stmt->execute(['u'=>$userId]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        $balance = $row ? (float)$row['balance'] : 0.0;
        echo json_encode(['ok'=>true,'user_id'=>$userId,'balance'=>$balance]);
    }

    // API: Channel partner wallet balance by partner_id
    public function apiPartnerBalance(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: application/json');
        $partnerId = (int)($_GET['partner_id'] ?? 0);
        if ($partnerId <= 0) { echo json_encode(['ok'=>false,'error'=>'invalid_partner']); return; }
        $stmt = $this->pdo->prepare('SELECT balance FROM channel_partner_wallet WHERE partner_id = :p LIMIT 1');
        $stmt->execute(['p'=>$partnerId]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        $balance = $row ? (float)$row['balance'] : 0.0;
        echo json_encode(['ok'=>true,'partner_id'=>$partnerId,'balance'=>$balance]);
    }

    // API: Stats for Secure Wallet page (balances + recent transfers)
    public function apiWalletSecureStats(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: application/json');
        // Admin balance
        $adminId = (int)($_SESSION['user']['id'] ?? 0);
        $stmtBal = $this->pdo->prepare('SELECT balance FROM wallets WHERE user_id = :u LIMIT 1');
        $stmtBal->execute(['u' => $adminId]);
        $rowBal = $stmtBal->fetch(\PDO::FETCH_ASSOC);
        $adminBalance = $rowBal ? (float)$rowBal['balance'] : 0.0;
        // Partner balances
        $partners = $this->pdo->query("SELECT cp.id, cp.name, IFNULL(w.balance, 0) AS balance
                                       FROM channel_partners cp
                                       LEFT JOIN channel_partner_wallet w ON w.partner_id = cp.id
                                       WHERE cp.status='Active' ORDER BY cp.name")->fetchAll();
        // Recent transfers (reuse logic from walletSecure)
        $recent = [];
        $wl = $this->pdo->query("SELECT wl.id, wl.amount, wl.method, wl.status, wl.meta, wl.created_at, u.id as user_id, u.name as user_name, u.email
                              FROM wallet_ledger wl
                              JOIN wallets w ON w.id = wl.wallet_id
                              JOIN users u ON u.id = w.user_id
                              WHERE wl.status='approved' AND wl.type='credit'
                              ORDER BY wl.id DESC LIMIT 50")->fetchAll();
        foreach ($wl as $row) {
            $meta = json_decode($row['meta'] ?? 'null', true) ?: [];
            $flow = $meta['flow'] ?? ($meta['reason'] ?? '');
            if (in_array($flow, ['admin_to_agent','partner_to_agent','treasury_seed'], true)) {
                $recent[] = [
                    'source' => ($flow === 'admin_to_agent' ? 'Admin' : ($flow === 'partner_to_agent' ? 'Partner' : 'System')),
                    'target' => $row['user_name'].' ('.$row['email'].')',
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                ];
            } elseif ($flow === 'reverse_admin_to_agent') {
                // Credit back to Admin; show as Agent -> Admin
                $agentLabel = 'Agent';
                $agentId = (int)($meta['agent_user_id'] ?? 0);
                if ($agentId > 0) {
                    $s = $this->pdo->prepare("SELECT name, email FROM users WHERE id=:id LIMIT 1");
                    $s->execute(['id' => $agentId]);
                    if ($u = $s->fetch(\PDO::FETCH_ASSOC)) {
                        $agentLabel = ($u['name'] ?? 'Agent').' ('.($u['email'] ?? '').')';
                    }
                }
                $recent[] = [
                    'source' => $agentLabel,
                    'target' => 'Admin',
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                ];
            }
        }
        $cp = $this->pdo->query("SELECT l.id, l.amount, l.type, l.method, l.status, l.meta, l.created_at, p.id as partner_id, p.name as partner_name
                              FROM channel_partner_wallet_ledger l
                              JOIN channel_partner_wallet w ON w.id = l.wallet_id
                              JOIN channel_partners p ON p.id = w.partner_id
                              WHERE l.status='approved' AND l.type IN ('credit','debit')
                              ORDER BY l.id DESC LIMIT 50")->fetchAll();
        foreach ($cp as $row) {
            $meta = json_decode($row['meta'] ?? 'null', true) ?: [];
            $flow = $meta['flow'] ?? '';
            if ($flow === 'admin_to_partner' && ($row['type'] ?? '') === 'credit') {
                $recent[] = [
                    'source' => 'Admin',
                    'target' => $row['partner_name'],
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                ];
            } elseif ($flow === 'reverse_admin_to_partner' && ($row['type'] ?? '') === 'debit') {
                $recent[] = [
                    'source' => $row['partner_name'],
                    'target' => 'Admin',
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                ];
            } elseif ($flow === 'reverse_partner_to_agent' && ($row['type'] ?? '') === 'credit') {
                $agentLabel = 'Agent';
                $agentId = (int)($meta['agent_user_id'] ?? 0);
                if ($agentId > 0) {
                    $s = $this->pdo->prepare("SELECT name, email FROM users WHERE id=:id LIMIT 1");
                    $s->execute(['id' => $agentId]);
                    if ($u = $s->fetch(\PDO::FETCH_ASSOC)) {
                        $agentLabel = ($u['name'] ?? 'Agent').' ('.($u['email'] ?? '').')';
                    }
                }
                $recent[] = [
                    'source' => $agentLabel,
                    'target' => $row['partner_name'],
                    'amount' => (float)$row['amount'],
                    'flow'   => $flow,
                    'created_at' => $row['created_at'],
                    'status' => $row['status'],
                ];
            }
        }
        usort($recent, function($a,$b){ return strcmp($b['created_at'], $a['created_at']); });
        $recent = array_slice($recent, 0, 20);
        echo json_encode(['ok'=>true,'adminBalance'=>$adminBalance,'partners'=>$partners,'recent'=>$recent]);
    }

    // API: New agent hotel bookings ping
    public function apiNewAgentHotelBookings(): void
    {
        Auth::requireRole(['Admin']);
        header('Content-Type: application/json');
        $sinceId = isset($_GET['since_id']) ? (int)$_GET['since_id'] : 0;
        // Determine last booking id for hotel module
        $lastId = (int)$this->pdo->query("SELECT IFNULL(MAX(id),0) FROM bookings WHERE module='hotel'")->fetchColumn();
        if ($sinceId <= 0) {
            echo json_encode(['ok'=>true,'last_id'=>$lastId,'new_count'=>0,'items'=>[]]);
            return;
        }
        // Count new bookings created by B2B Agents after since_id
        $sqlCount = "SELECT COUNT(*)
                     FROM bookings b
                     JOIN users u ON u.id = b.user_id
                     WHERE b.module='hotel' AND b.id > :sid AND u.role='B2B Agent'";
        $st = $this->pdo->prepare($sqlCount);
        $st->execute(['sid'=>$sinceId]);
        $newCount = (int)$st->fetchColumn();
        $items = [];
        if ($newCount > 0) {
            $sqlItems = "SELECT b.id, b.created_at, b.pax, b.price, u.name AS agent_name
                         FROM bookings b
                         JOIN users u ON u.id = b.user_id
                         WHERE b.module='hotel' AND b.id > :sid AND u.role='B2B Agent'
                         ORDER BY b.id DESC
                         LIMIT 5";
            $si = $this->pdo->prepare($sqlItems);
            $si->execute(['sid'=>$sinceId]);
            $items = $si->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        }
        echo json_encode(['ok'=>true,'last_id'=>$lastId,'new_count'=>$newCount,'items'=>$items]);
    }

    // ===== Admin: Payments listing =====
    public function payments(): void
    {
        Auth::requireRole(['Admin']);
        // Filters
        $status   = isset($_GET['status']) ? trim((string)$_GET['status']) : '';
        $gateway  = isset($_GET['gateway']) ? trim((string)$_GET['gateway']) : '';
        $orderId  = isset($_GET['order_id']) ? (int)$_GET['order_id'] : null;
        $userId   = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
        $from     = isset($_GET['from']) ? trim((string)$_GET['from']) : '';
        $to       = isset($_GET['to']) ? trim((string)$_GET['to']) : '';

        $params = [];
        $wheres = [];
        if ($status !== '') { $wheres[] = 'p.status = :st'; $params['st'] = $status; }
        if ($gateway !== '') { $wheres[] = 'p.gateway = :gw'; $params['gw'] = $gateway; }
        if ($orderId) { $wheres[] = 'p.order_id = :oid'; $params['oid'] = $orderId; }
        if ($userId) { $wheres[] = 'p.user_id = :uid'; $params['uid'] = $userId; }
        if ($from !== '') { $wheres[] = 'p.created_at >= :from'; $params['from'] = $from.' 00:00:00'; }
        if ($to !== '') { $wheres[] = 'p.created_at <= :to'; $params['to'] = $to.' 23:59:59'; }
        $whereSql = $wheres ? ('WHERE '.implode(' AND ', $wheres)) : '';

        $sql = "SELECT p.id, p.order_id, p.user_id, p.gateway, p.amount, p.currency, p.status, p.provider_ref, p.event_id, p.idempotency_key, p.created_at,
                       o.payment_status AS order_payment_status,
                       u.name AS user_name, u.email AS user_email
                FROM payments p
                LEFT JOIN orders o ON o.id = p.order_id
                LEFT JOIN users u ON u.id = p.user_id
                $whereSql
                ORDER BY p.id DESC
                LIMIT 500";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll();

        // Distinct gateways and statuses for filters
        try {
            $gateways = $this->pdo->query("SELECT DISTINCT gateway FROM payments ORDER BY gateway")->fetchAll(\PDO::FETCH_COLUMN) ?: [];
            $statuses = $this->pdo->query("SELECT DISTINCT status FROM payments ORDER BY status")->fetchAll(\PDO::FETCH_COLUMN) ?: [];
        } catch (\Throwable $e) {
            $gateways = ['stripe'];
            $statuses = ['pending','succeeded','failed','canceled'];
        }

        $this->view('admin/payments', compact('rows','status','gateway','orderId','userId','from','to','gateways','statuses'));
    }

    // ===== Admin: Order view (read-only) =====
    public function orderView(): void
    {
        Auth::requireRole(['Admin']);
        $orderId = (int)($_GET['id'] ?? 0);
        if ($orderId <= 0) { http_response_code(400); echo 'Missing id'; return; }

        // Fetch order core fields
        $sql = 'SELECT o.*, u.name AS user_name, u.email AS user_email FROM orders o LEFT JOIN users u ON u.id = o.user_id WHERE o.id = :id LIMIT 1';
        $st = $this->pdo->prepare($sql);
        $st->execute(['id' => $orderId]);
        $order = $st->fetch();
        if (!$order) { http_response_code(404); echo 'Order not found'; return; }

        // Fetch payments for this order
        $ps = $this->pdo->prepare('SELECT id, gateway, amount, currency, status, provider_ref, created_at FROM payments WHERE order_id = :id ORDER BY id DESC');
        $ps->execute(['id' => $orderId]);
        $payments = $ps->fetchAll();

        // Attempt to fetch order items if table exists and enrich details for known modules
        $orderItems = [];
        $enrichedItems = [];
        try {
            $chk = $this->pdo->query("SHOW TABLES LIKE 'order_items'")->fetchColumn();
            if ($chk) {
                $is = $this->pdo->prepare('SELECT id, order_id, module, item_id, variant_id, price_id, qty, unit_price, line_total, currency, booking_id FROM order_items WHERE order_id = :id ORDER BY id');
                $is->execute(['id' => $orderId]);
                $orderItems = $is->fetchAll(\PDO::FETCH_ASSOC);

                foreach ($orderItems as $row) {
                    $row['details'] = null;
                    $module = strtolower((string)($row['module'] ?? ''));
                    if ($module === 'hotel' && !empty($row['booking_id'])) {
                        // Try to fetch hotel booking with hotel and room(s) across legacy/new schemas
                        try {
                            // Detect schema
                            $hasRoomId = $this->hasColumn('hotel_bookings', 'room_id');
                            $hasGuests = $this->hasColumn('hotel_bookings', 'guests_json');
                            $hasRoomsQty = $this->hasColumn('hotel_bookings', 'rooms_qty') || $this->hasColumn('hotel_bookings', 'rooms');
                            $roomJoin = '';
                            $roomSelect = '';
                            if ($hasRoomId) {
                                $roomSelect = ', r.name AS room_name';
                                $roomJoin = ' LEFT JOIN hotel_rooms r ON r.id = hb.room_id ';
                            } else {
                                // Aggregate room names from hotel_booking_rooms if available
                                $roomSelect = ", (
                                    SELECT GROUP_CONCAT(DISTINCT hr.name SEPARATOR ', ') FROM hotel_booking_rooms hbr
                                    LEFT JOIN hotel_rooms hr ON hr.id = hbr.room_id
                                    WHERE hbr.booking_id = hb.id
                                ) AS room_name";
                            }
                            $bsql = "SELECT hb.*, h.name AS hotel_name, h.city AS hotel_city, h.country AS hotel_country $roomSelect
                                     FROM hotel_bookings hb
                                     LEFT JOIN hotels h ON h.id = hb.hotel_id
                                     $roomJoin
                                     WHERE hb.id = :bid LIMIT 1";
                            $bs = $this->pdo->prepare($bsql);
                            $bs->execute(['bid' => (int)$row['booking_id']]);
                            if ($b = $bs->fetch(\PDO::FETCH_ASSOC)) {
                                // Normalize totals/rooms fields
                                if (!isset($b['total_price']) && isset($b['total_amount'])) { $b['total_price'] = $b['total_amount']; }
                                if (!isset($b['rooms_qty']) && isset($b['rooms'])) { $b['rooms_qty'] = $b['rooms']; }
                                // Parse guests JSON (legacy only)
                                $guests = null;
                                if ($hasGuests && !empty($b['guests_json'])) {
                                    $decoded = json_decode($b['guests_json'], true);
                                    if (json_last_error() === JSON_ERROR_NONE) { $guests = $decoded; }
                                }
                                $row['details'] = [
                                    'type' => 'hotel',
                                    'booking' => $b,
                                    'guests' => $guests,
                                ];
                            } else {
                                // Fallback: generic bookings table (module=hotel)
                                try {
                                    $qb = $this->pdo->prepare("SELECT b.*, h.name AS hotel_name, h.city AS hotel_city, h.country AS hotel_country FROM bookings b LEFT JOIN hotels h ON h.id = b.item_id WHERE b.id = :id AND b.module = 'hotel' LIMIT 1");
                                    $qb->execute(['id' => (int)$row['booking_id']]);
                                    $bk = $qb->fetch(\PDO::FETCH_ASSOC) ?: null;
                                    if ($bk) {
                                        // Derive details
                                        $checkin = $bk['check_in_date'] ?? null; $checkout = $bk['check_out_date'] ?? null; $nights = $bk['nights'] ?? null; $roomsQty = $bk['rooms'] ?? null; $roomName = null; $currency = $bk['currency'] ?? null;
                                        $details = [];
                                        try { $details = json_decode((string)($bk['details_json'] ?? 'null'), true); } catch (\Throwable $e) { $details = []; }
                                        if (is_array($details)) {
                                            $checkin = $checkin ?: ($details['check_in_date'] ?? null);
                                            $checkout = $checkout ?: ($details['check_out_date'] ?? null);
                                            $nights = $nights ?: ($details['nights'] ?? null);
                                            $roomsQty = $roomsQty ?: ($details['rooms'] ?? null);
                                            $rid = $details['room_id'] ?? null;
                                            if ($rid) {
                                                try { $rn = $this->pdo->prepare('SELECT name FROM hotel_rooms WHERE id = :rid LIMIT 1'); $rn->execute(['rid' => (int)$rid]); $roomName = $rn->fetchColumn() ?: null; } catch (\Throwable $e) { /* ignore */ }
                                            }
                                        }
                                        // Payment status via linked order (if exists)
                                        $paymentStatus = null; $orderCurrency = null;
                                        try {
                                            $qo = $this->pdo->prepare('SELECT o.status, o.currency FROM orders o JOIN order_items oi ON oi.order_id = o.id WHERE oi.booking_id = :bid LIMIT 1');
                                            $qo->execute(['bid' => (int)$row['booking_id']]);
                                            if ($ord = $qo->fetch(\PDO::FETCH_ASSOC)) {
                                                $orderCurrency = $ord['currency'] ?? null;
                                                $st = strtolower((string)($ord['status'] ?? ''));
                                                $paymentStatus = in_array($st, ['paid','confirmed','completed'], true) ? 'paid' : 'unpaid';
                                            }
                                        } catch (\Throwable $e) { /* ignore */ }
                                        // Contact info from details_json.contact
                                        $customerName = null; $customerEmail = null; $customerMobile = null; $customerWhatsapp = null; $specialRequests = null; $guestsAdults = null; $guestsChildren = null;
                                        if (is_array($details)) {
                                            if (isset($details['contact']) && is_array($details['contact'])) {
                                                $customerName = trim((string)($details['contact']['name'] ?? '')) ?: null;
                                                $customerEmail = trim((string)($details['contact']['email'] ?? '')) ?: null;
                                                $customerMobile = trim((string)($details['contact']['mobile'] ?? '')) ?: null;
                                                $customerWhatsapp = trim((string)($details['contact']['whatsapp'] ?? '')) ?: null;
                                            }
                                            if (isset($details['special_requests'])) { $specialRequests = (string)$details['special_requests']; }
                                            if (isset($details['guests']) && is_array($details['guests'])) {
                                                $guestsAdults = isset($details['guests']['adults']) ? (int)$details['guests']['adults'] : null;
                                                $guestsChildren = isset($details['guests']['children']) ? (int)$details['guests']['children'] : null;
                                            }
                                        }
                                        // Try to overlay contact from linked order if available
                                        try {
                                            $qoc = $this->pdo->prepare('SELECT o.customer_name, o.customer_email, o.customer_mobile, o.customer_whatsapp FROM orders o JOIN order_items oi ON oi.order_id = o.id WHERE oi.booking_id = :bid LIMIT 1');
                                            $qoc->execute(['bid' => (int)$row['booking_id']]);
                                            if ($oc = $qoc->fetch(\PDO::FETCH_ASSOC)) {
                                                $customerName = $customerName ?: ($oc['customer_name'] ?? null);
                                                $customerEmail = $customerEmail ?: ($oc['customer_email'] ?? null);
                                                $customerMobile = $customerMobile ?: ($oc['customer_mobile'] ?? null);
                                                $customerWhatsapp = $customerWhatsapp ?: ($oc['customer_whatsapp'] ?? null);
                                            }
                                        } catch (\Throwable $e) { /* ignore */ }

                                        $assembled = [
                                            'id' => (int)$bk['id'],
                                            'hotel_name' => $bk['hotel_name'] ?? null,
                                            'hotel_city' => $bk['hotel_city'] ?? null,
                                            'hotel_country' => $bk['hotel_country'] ?? null,
                                            'room_name' => $roomName,
                                            'booking_code' => $bk['booking_code'] ?? ('BK'.$bk['id']),
                                            'checkin' => $checkin,
                                            'checkout' => $checkout,
                                            'nights' => $nights,
                                            'rooms_qty' => $roomsQty,
                                            'status' => $bk['status'] ?? 'pending',
                                            'payment_status' => $paymentStatus,
                                            'total_price' => $bk['price'] ?? 0,
                                            'currency' => $orderCurrency ?: ($currency ?: 'THB'),
                                            'customer_name' => $customerName,
                                            'customer_email' => $customerEmail,
                                            'customer_mobile' => $customerMobile,
                                            'customer_whatsapp' => $customerWhatsapp,
                                            'special_requests' => $specialRequests,
                                            'guests_adults' => $guestsAdults,
                                            'guests_children' => $guestsChildren,
                                            'created_at' => $bk['created_at'] ?? null,
                                            'updated_at' => $bk['updated_at'] ?? null,
                                        ];
                                        $row['details'] = [
                                            'type' => 'hotel',
                                            'booking' => $assembled,
                                            'guests' => null,
                                        ];
                                    }
                                } catch (\Throwable $e) { /* ignore */ }
                            }
                        } catch (\Throwable $e) {
                            // ignore enrichment failure; show base row
                        }
                    }
                    $enrichedItems[] = $row;
                }
            }
        } catch (\Throwable $e) { /* ignore if table not found */ }

        $orderItems = $enrichedItems ?: $orderItems; // prefer enriched when available

        $this->view('admin/order_view', compact('order','payments','orderItems'));
    }

    public function bookingHotelView(): void
    {
        Auth::requireRole(['Admin']);
        $id = (int)($_GET['id'] ?? 0);
        if ($id <= 0) { http_response_code(400); echo 'Missing id'; return; }

        // Detect schema and build appropriate query
        $hasRoomId = $this->hasColumn('hotel_bookings', 'room_id');
        $roomJoin = '';
        $roomSelect = '';
        if ($hasRoomId) {
            $roomSelect = ', r.name AS room_name';
            $roomJoin = ' LEFT JOIN hotel_rooms r ON r.id = hb.room_id ';
        } else {
            $roomSelect = ", (
                SELECT GROUP_CONCAT(DISTINCT hr.name SEPARATOR ', ') FROM hotel_booking_rooms hbr
                LEFT JOIN hotel_rooms hr ON hr.id = hbr.room_id
                WHERE hbr.booking_id = hb.id
            ) AS room_name";
        }

        $sql = "SELECT hb.*, h.name AS hotel_name, h.city AS hotel_city, h.country AS hotel_country $roomSelect
                FROM hotel_bookings hb
                LEFT JOIN hotels h ON h.id = hb.hotel_id
                $roomJoin
                WHERE hb.id = :id LIMIT 1";
        $st = $this->pdo->prepare($sql);
        $st->execute(['id' => $id]);
        $booking = $st->fetch(\PDO::FETCH_ASSOC);
        if (!$booking) {
            // Fallback: try generic bookings table
            try {
                $qb = $this->pdo->prepare("SELECT b.*, h.name AS hotel_name, h.city AS hotel_city, h.country AS hotel_country FROM bookings b LEFT JOIN hotels h ON h.id = b.item_id WHERE b.id = :id AND b.module = 'hotel' LIMIT 1");
                $qb->execute(['id' => $id]);
                $bk = $qb->fetch(\PDO::FETCH_ASSOC) ?: null;
                if ($bk) {
                    $details = [];
                    try { $details = json_decode((string)($bk['details_json'] ?? 'null'), true); } catch (\Throwable $e) { $details = []; }
                    $checkin = $details['check_in_date'] ?? null;
                    $checkout = $details['check_out_date'] ?? null;
                    $nights = $details['nights'] ?? null;
                    $roomsQty = $details['rooms'] ?? null;
                    $roomName = null;
                    $rid = $details['room_id'] ?? null;
                    if ($rid) {
                        try { $rn = $this->pdo->prepare('SELECT name FROM hotel_rooms WHERE id = :rid LIMIT 1'); $rn->execute(['rid' => (int)$rid]); $roomName = $rn->fetchColumn() ?: null; } catch (\Throwable $e) { /* ignore */ }
                    }
                    // Determine payment status and currency via order (if any)
                    $paymentStatus = null; $orderCurrency = null;
                    try {
                        $qo = $this->pdo->prepare('SELECT o.status, o.currency FROM orders o JOIN order_items oi ON oi.order_id = o.id WHERE oi.booking_id = :bid LIMIT 1');
                        $qo->execute(['bid' => $id]);
                        if ($ord = $qo->fetch(\PDO::FETCH_ASSOC)) {
                            $orderCurrency = $ord['currency'] ?? null;
                            $stt = strtolower((string)($ord['status'] ?? ''));
                            $paymentStatus = in_array($stt, ['paid','confirmed','completed'], true) ? 'paid' : 'unpaid';
                        }
                    } catch (\Throwable $e) { /* ignore */ }
                    // Contact info and guests summary
                    $customerName = null; $customerEmail = null; $customerMobile = null; $customerWhatsapp = null; $specialRequests = null; $guestsAdults = null; $guestsChildren = null;
                    if (is_array($details)) {
                        if (isset($details['contact']) && is_array($details['contact'])) {
                            $customerName = trim((string)($details['contact']['name'] ?? '')) ?: null;
                            $customerEmail = trim((string)($details['contact']['email'] ?? '')) ?: null;
                            $customerMobile = trim((string)($details['contact']['mobile'] ?? '')) ?: null;
                            $customerWhatsapp = trim((string)($details['contact']['whatsapp'] ?? '')) ?: null;
                        }
                        if (isset($details['special_requests'])) { $specialRequests = (string)$details['special_requests']; }
                        if (isset($details['guests']) && is_array($details['guests'])) {
                            $guestsAdults = isset($details['guests']['adults']) ? (int)$details['guests']['adults'] : null;
                            $guestsChildren = isset($details['guests']['children']) ? (int)$details['guests']['children'] : null;
                        }
                    }
                    // Try to overlay contact from order if present
                    try {
                        $qoc = $this->pdo->prepare('SELECT o.customer_name, o.customer_email, o.customer_mobile, o.customer_whatsapp FROM orders o JOIN order_items oi ON oi.order_id = o.id WHERE oi.booking_id = :bid LIMIT 1');
                        $qoc->execute(['bid' => $id]);
                        if ($oc = $qoc->fetch(\PDO::FETCH_ASSOC)) {
                            $customerName = $customerName ?: ($oc['customer_name'] ?? null);
                            $customerEmail = $customerEmail ?: ($oc['customer_email'] ?? null);
                            $customerMobile = $customerMobile ?: ($oc['customer_mobile'] ?? null);
                            $customerWhatsapp = $customerWhatsapp ?: ($oc['customer_whatsapp'] ?? null);
                        }
                    } catch (\Throwable $e) { /* ignore */ }

                    // Find linked order id (if any)
                    $orderId = null;
                    try {
                        $qo2 = $this->pdo->prepare('SELECT oi.order_id FROM order_items oi WHERE oi.booking_id = :bid LIMIT 1');
                        $qo2->execute(['bid' => $id]);
                        $orderId = (int)($qo2->fetchColumn() ?: 0);
                    } catch (\Throwable $e) { $orderId = 0; }

                    $booking = [
                        'id' => (int)$bk['id'],
                        'hotel_name' => $bk['hotel_name'] ?? null,
                        'hotel_city' => $bk['hotel_city'] ?? null,
                        'hotel_country' => $bk['hotel_country'] ?? null,
                        'room_name' => $roomName,
                        'booking_code' => $bk['booking_code'] ?? ('BK'.$bk['id']),
                        'checkin' => $checkin,
                        'checkout' => $checkout,
                        'nights' => $nights,
                        'rooms_qty' => $roomsQty,
                        'status' => $bk['status'] ?? 'pending',
                        'payment_status' => $paymentStatus,
                        'total_price' => $bk['price'] ?? 0,
                        'currency' => $orderCurrency ?: ($bk['currency'] ?? 'THB'),
                        'customer_name' => $customerName,
                        'customer_email' => $customerEmail,
                        'customer_mobile' => $customerMobile,
                        'customer_whatsapp' => $customerWhatsapp,
                        'special_requests' => $specialRequests,
                        'guests_adults' => $guestsAdults,
                        'guests_children' => $guestsChildren,
                        'created_at' => $bk['created_at'] ?? null,
                        'updated_at' => $bk['updated_at'] ?? null,
                    ];
                    // Build guests list from details_json.guests_list if present
                    $guests = [];
                    if (is_array($details) && isset($details['guests_list']) && is_array($details['guests_list'])) {
                        foreach ($details['guests_list'] as $g) {
                            if (!is_array($g)) continue;
                            $guests[] = [
                                'title' => $g['title'] ?? null,
                                'first_name' => $g['first_name'] ?? ($g['fname'] ?? null),
                                'last_name' => $g['last_name'] ?? ($g['lname'] ?? null),
                                'age' => isset($g['age']) ? (int)$g['age'] : null,
                            ];
                        }
                    }

                    // If still empty, try normalized booking_guests table
                    if (empty($guests)) {
                        try {
                            $qbg = $this->pdo->prepare('SELECT room_index, guest_index, type, full_name FROM booking_guests WHERE booking_id = :bid ORDER BY room_index, guest_index, id');
                            $qbg->execute(['bid' => $id]);
                            $rows = $qbg->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                            foreach ($rows as $r) {
                                $full = trim((string)($r['full_name'] ?? ''));
                                if ($full === '') { continue; }
                                $first = $full; $last = null;
                                if (strpos($full, ' ') !== false) {
                                    $parts = preg_split('/\s+/', $full);
                                    if ($parts && count($parts) > 1) {
                                        $first = array_shift($parts);
                                        $last = implode(' ', $parts);
                                    }
                                }
                                $guests[] = [ 'title' => null, 'first_name' => $first, 'last_name' => $last, 'age' => null ];
                            }
                        } catch (\Throwable $e) { /* ignore */ }
                    }

                    // Fetch agency (user) info via linked order user_id or booking user_id
                    $agency = null;
                    try {
                        $uid = null;
                        if ($orderId > 0) {
                            $qou = $this->pdo->prepare('SELECT user_id FROM orders WHERE id = :oid LIMIT 1');
                            $qou->execute(['oid' => $orderId]);
                            $uid = (int)($qou->fetchColumn() ?: 0);
                        }
                        if (!$uid) {
                            $uid = (int)($bk['user_id'] ?? 0);
                        }
                        if ($uid > 0) {
                            $qu = $this->pdo->prepare('SELECT id, name, email, mobile, whatsapp, company FROM users WHERE id = :id LIMIT 1');
                            $qu->execute(['id' => $uid]);
                            $u = $qu->fetch(\PDO::FETCH_ASSOC) ?: null;
                            if ($u) { $agency = $u; }
                        }
                    } catch (\Throwable $e) { $agency = null; }

                    // If no named guests but counts exist, synthesize a readable list
                    if (empty($guests)) {
                        $ad = isset($booking['guests_adults']) ? (int)$booking['guests_adults'] : 0;
                        $ch = isset($booking['guests_children']) ? (int)$booking['guests_children'] : 0;
                        $total = $ad + $ch;
                        if ($total > 0) {
                            $lead = null;
                            if (!empty($booking['customer_name'])) { $lead = trim((string)$booking['customer_name']); }
                            elseif (is_array($details) && isset($details['contact']['name'])) { $lead = trim((string)$details['contact']['name']); }
                            if ($lead && $lead !== '') {
                                $guests[] = [ 'title' => null, 'first_name' => $lead, 'last_name' => null, 'age' => null ];
                                for ($i = 2; $i <= $total; $i++) { $guests[] = [ 'title' => null, 'first_name' => 'Guest '.$i, 'last_name' => null, 'age' => null ]; }
                            } else {
                                for ($i = 1; $i <= $total; $i++) { $guests[] = [ 'title' => null, 'first_name' => 'Guest '.$i, 'last_name' => null, 'age' => null ]; }
                            }
                        }
                    }

                    $this->view('admin/booking_hotel_view', compact('booking','guests','orderId','agency'));
                    return;
                }
            } catch (\Throwable $e) { /* ignore */ }
            http_response_code(404); echo 'Booking not found'; return;
        }

        // Normalize for view
        if (!isset($booking['total_price']) && isset($booking['total_amount'])) { $booking['total_price'] = $booking['total_amount']; }
        if (!isset($booking['rooms_qty']) && isset($booking['rooms'])) { $booking['rooms_qty'] = $booking['rooms']; }

        $guests = [];
        if ($this->hasColumn('hotel_bookings', 'guests_json') && !empty($booking['guests_json'])) {
            $decoded = json_decode($booking['guests_json'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) { $guests = $decoded; }
        }

        // Try building guests from a related generic bookings row if still empty
        if (empty($guests)) {
            try {
                $qb = $this->pdo->prepare('SELECT details_json, user_id FROM bookings WHERE id = :id AND module = "hotel" LIMIT 1');
                $qb->execute(['id' => $id]);
                if ($bkr = $qb->fetch(\PDO::FETCH_ASSOC)) {
                    $d = json_decode((string)($bkr['details_json'] ?? 'null'), true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($d) && isset($d['guests_list']) && is_array($d['guests_list'])) {
                        foreach ($d['guests_list'] as $g) {
                            if (!is_array($g)) continue;
                            $guests[] = [
                                'title' => $g['title'] ?? null,
                                'first_name' => $g['first_name'] ?? ($g['fname'] ?? null),
                                'last_name' => $g['last_name'] ?? ($g['lname'] ?? null),
                                'age' => isset($g['age']) ? (int)$g['age'] : null,
                            ];
                        }
                    }
                }
            } catch (\Throwable $e) { /* ignore */ }
        }

        // If still empty, try normalized booking_guests for this booking id
        if (empty($guests)) {
            try {
                $qbg = $this->pdo->prepare('SELECT room_index, guest_index, type, full_name FROM booking_guests WHERE booking_id = :bid ORDER BY room_index, guest_index, id');
                $qbg->execute(['bid' => $id]);
                $rows = $qbg->fetchAll(\PDO::FETCH_ASSOC) ?: [];
                foreach ($rows as $r) {
                    $full = trim((string)($r['full_name'] ?? ''));
                    if ($full === '') { continue; }
                    $first = $full; $last = null;
                    if (strpos($full, ' ') !== false) {
                        $parts = preg_split('/\s+/', $full);
                        if ($parts && count($parts) > 1) {
                            $first = array_shift($parts);
                            $last = implode(' ', $parts);
                        }
                    }
                    $guests[] = [ 'title' => null, 'first_name' => $first, 'last_name' => $last, 'age' => null ];
                }
            } catch (\Throwable $e) { /* ignore */ }
        }

        // Find linked order id (if any)
        $orderId = null;
        try {
            $qo2 = $this->pdo->prepare('SELECT oi.order_id FROM order_items oi WHERE oi.booking_id = :bid LIMIT 1');
            $qo2->execute(['bid' => $id]);
            $orderId = (int)($qo2->fetchColumn() ?: 0);
        } catch (\Throwable $e) { $orderId = 0; }

        // If no named guests but counts exist, synthesize a readable list
        if (empty($guests)) {
            $ad = isset($booking['guests_adults']) ? (int)$booking['guests_adults'] : 0;
            $ch = isset($booking['guests_children']) ? (int)$booking['guests_children'] : 0;
            $total = $ad + $ch;
            if ($total > 0) {
                $lead = null;
                if (!empty($booking['customer_name'])) { $lead = trim((string)$booking['customer_name']); }
                else {
                    try {
                        $bkr = $this->pdo->prepare('SELECT details_json FROM bookings WHERE id = :id AND module = "hotel" LIMIT 1');
                        $bkr->execute(['id' => $id]);
                        $drow = $bkr->fetch(\PDO::FETCH_ASSOC) ?: null;
                        if ($drow) {
                            $dj = json_decode((string)($drow['details_json'] ?? 'null'), true);
                            if (is_array($dj) && isset($dj['contact']['name'])) { $lead = trim((string)$dj['contact']['name']); }
                        }
                    } catch (\Throwable $e) { /* ignore */ }
                }
                if ($lead && $lead !== '') {
                    $guests[] = [ 'title' => null, 'first_name' => $lead, 'last_name' => null, 'age' => null ];
                    for ($i = 2; $i <= $total; $i++) { $guests[] = [ 'title' => null, 'first_name' => 'Guest '.$i, 'last_name' => null, 'age' => null ]; }
                } else {
                    for ($i = 1; $i <= $total; $i++) { $guests[] = [ 'title' => null, 'first_name' => 'Guest '.$i, 'last_name' => null, 'age' => null ]; }
                }
            }
        }

        // Fetch agency (user) info
        $agency = null;
        try {
            $uid = null;
            if ($orderId > 0) {
                $qou = $this->pdo->prepare('SELECT user_id FROM orders WHERE id = :oid LIMIT 1');
                $qou->execute(['oid' => $orderId]);
                $uid = (int)($qou->fetchColumn() ?: 0);
            }
            if (!$uid) {
                // try hotel_bookings.user_id, else generic bookings.user_id
                $uid = isset($booking['user_id']) ? (int)$booking['user_id'] : 0;
                if (!$uid) {
                    $qb = $this->pdo->prepare('SELECT user_id FROM bookings WHERE id = :id AND module = "hotel" LIMIT 1');
                    $qb->execute(['id' => $id]);
                    $uid = (int)($qb->fetchColumn() ?: 0);
                }
            }
            if ($uid > 0) {
                $qu = $this->pdo->prepare('SELECT id, name, email, mobile, whatsapp, company FROM users WHERE id = :id LIMIT 1');
                $qu->execute(['id' => $uid]);
                $u = $qu->fetch(\PDO::FETCH_ASSOC) ?: null;
                if ($u) { $agency = $u; }
            }
        } catch (\Throwable $e) { $agency = null; }

        $this->view('admin/booking_hotel_view', compact('booking','guests','orderId','agency'));
    }

    private function hasColumn(string $table, string $column): bool
    {
        try {
            $st = $this->pdo->prepare("SHOW COLUMNS FROM `$table` LIKE :c");
            $st->execute(['c' => $column]);
            return (bool)$st->fetch();
        } catch (\Throwable $e) {
            return false;
        }
    }

    // Helper to add users.partner_id if missing (idempotent)
    private function ensureUsersPartnerIdColumn(): void
    {
        try {
            $this->pdo->exec("ALTER TABLE users ADD COLUMN IF NOT EXISTS partner_id INT NULL");
        } catch (\Throwable $e) {
            // Silently ignore if DB version doesn't support IF NOT EXISTS or column exists
            try {
                // Fallback: detect existence
                $chk = $this->pdo->query("SHOW COLUMNS FROM users LIKE 'partner_id'");
                if (!$chk || !$chk->fetch()) {
                    // Try standard add without IF NOT EXISTS
                    $this->pdo->exec("ALTER TABLE users ADD COLUMN partner_id INT NULL");
                }
            } catch (\Throwable $e2) {
                // If cannot add, continue; API will just return empty and transfers will fail mapping check
            }
        }
    }

    // Helper to add channel_partners.transaction_password_hash if missing (idempotent)
    private function ensurePartnerTxPasswordColumn(): void
    {
        try {
            $this->pdo->exec("ALTER TABLE channel_partners ADD COLUMN IF NOT EXISTS transaction_password_hash VARCHAR(255) NULL");
        } catch (\Throwable $e) {
            try {
                $chk = $this->pdo->query("SHOW COLUMNS FROM channel_partners LIKE 'transaction_password_hash'");
                if (!$chk || !$chk->fetch()) {
                    $this->pdo->exec("ALTER TABLE channel_partners ADD COLUMN transaction_password_hash VARCHAR(255) NULL");
                }
            } catch (\Throwable $e2) {
                // ignore; validation will fail gracefully if column truly missing
            }
        }
    }

    private function logIp(string $action, array $meta): void
    {
        $uid = $_SESSION['user']['id'] ?? null;
        $ip = $_SERVER['REMOTE_ADDR'] ?? null;
        $xff = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? null;
        $ua  = $_SERVER['HTTP_USER_AGENT'] ?? null;
        $uri = $_SERVER['REQUEST_URI'] ?? null;
        $meta = array_merge([
            'ip' => $ip,
            'x_forwarded_for' => $xff,
            'user_agent' => $ua,
            'request_uri' => $uri,
            'at' => date('c'),
        ], $meta);
        $stmt = $this->pdo->prepare('INSERT INTO ip_action_logs (user_id, ip, action, meta) VALUES (:u,:i,:a,:m)');
        $stmt->execute(['u'=>$uid,'i'=>$ip,'a'=>$action,'m'=>json_encode($meta)]);
    }
}
